/*
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: lvmregmgr
 * File: lvmregmgr.c
 *
 * Description: This file contains all of the required engine-plugin APIs
 *              for the LVM region manager. This is also where all LVM global
 *              data is declared.
 */


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>
#include "lvmregmgr.h"


// Global variables
static engine_mode_t		open_mode;		// Still don't know what this is for. :)
engine_functions_t		* lvm_engine;		// The Engine's internal API set.
plugin_record_t			* lvm_plugin;		// Used to identify containers and regions as owned by LVM
dlist_t				lvm_group_list;		// List of volume groups.
lvm_pv_remove_ioctl_t		* lvm_pv_remove_list;	// List of PVs that need to be removed in the corresponding group in the kernel.


/* Function: lvm_setup_evms_plugin
 *
 *	This function gets called shortly after the plugin is loaded by the
 *	Engine. It performs all tasks that are necessary before the initial
 *	discovery pass.
 */
static int lvm_setup_evms_plugin(engine_mode_t		mode,
				engine_functions_t	* functions)
{
	// Initialize global variables.
	open_mode		= mode;
	lvm_engine		= functions;
	lvm_plugin		= &LVM_Plugin;
	lvm_pv_remove_list	= NULL;

	// Allocate the global group list.
	lvm_group_list = CreateList();
	if ( ! lvm_group_list ) {
		LOG_CRITICAL("Error creating LVM container list\n");
		return ENOMEM;
	}

	// Register the LVM namespace so no-one tries to steal it.
	lvm_engine->register_name(EVMS_DEV_NODE_PATH "lvm");

	return 0;
}


/* Function: lvm_cleanup_evms_plugin
 *
 *	Clean up all volume groups from memory before closing the engine.
 */
static void lvm_cleanup_evms_plugin( void )
{
	lvm_volume_group_t	* group;
	TAG			tag;
	int			size;

	LOG_ENTRY;

	GoToStartOfList(lvm_group_list);
	while ( ! BlindExtractObject(lvm_group_list, &size, &tag, NULL, (void**)&group) ) {
		//lvm_deallocate_volume_group(group);
	}

	DestroyList(&lvm_group_list, FALSE);

	LOG_EXIT(0);
}


/****** Region Checking Functions ******/


/* All of the following lvm_can_ functions return 0 if they are able to
 * perform the specified action, or non-zero if they cannot.
 */


/* Function: lvm_can_delete
 *
 *	Can we remove the specified LVM logical volume, and consolidate the
 *	space with the freespace volume?
 */
static int lvm_can_delete( storage_object_t * region )
{
	lvm_logical_volume_t * volume = region->private_data;

	LOG_ENTRY;

	// The region must be a regular (non-freespace) volume,
	// and must not have any snapshots.
	if ( region->data_type != DATA_TYPE ) {
		LOG("Cannot delete freespace region %s.\n", region->name);
		RETURN(EINVAL);
	}

	if ( volume->lv->lv_access & LV_SNAPSHOT_ORG ) {
		LOG("Cannot delete snapshotted region %s.\n", region->name);
		RETURN(EINVAL);
	}

	RETURN(0);
}


/* Function: lvm_can_expand
 *
 *	Can this region be expanded, using freespace from its group? If so,
 *	add the region to the expansion_points output list.
 */
static int lvm_can_expand(storage_object_t	* region,
			u_int64_t		* expand_limit,
			dlist_t			expansion_points )
{
	lvm_logical_volume_t	* volume = region->private_data;
	lvm_volume_group_t	* group = volume->group;
	expand_object_info_t	* expand_point;
	void			* handle;
	int			rc = 0;

	LOG_ENTRY;

	// Don't expand the freespace region, or a volume involved in
	// snapshotting.
	if ( region->data_type != DATA_TYPE ) {
		LOG("Cannot expand freespace region %s.\n", region->name);
		RETURN(EINVAL);
	}
	if ( volume->lv->lv_access & LV_SNAPSHOT ) {
		LOG("Cannot expand snapshot region %s.\n", region->name);
		RETURN(EINVAL);
	}
	if ( volume->lv->lv_access & LV_SNAPSHOT_ORG ) {
		LOG("Cannot expand region %s while being snapshotted.\n", region->name);
		RETURN(EINVAL);
	}

	// Must be space available in the group.
	if ( ! group->freespace->lv->lv_allocated_le ) {
		LOG("No freespace left in container %s.\n", group->container->name);
		RETURN(EINVAL);
	}

	// Create a new expansion info object.
	expand_point = lvm_engine->engine_alloc(sizeof(expand_object_info_t));
	if ( ! expand_point ) {
		LOG_CRITICAL("Memory error creating expansion info object.\n");
		RETURN(ENOMEM);
	}

	expand_point->object = region;
	if ( *expand_limit < group->freespace->region->size ) {
		expand_point->max_expand_size = round_down(*expand_limit, group->vg->pe_size);
	}
	else {
		expand_point->max_expand_size = group->freespace->region->size;
	}

	// Add to the expansion points list.
	rc = InsertObject(expansion_points,
			sizeof(expand_object_info_t),
			expand_point,
			EXPAND_OBJECT_TAG, NULL,
			AppendToList, TRUE, &handle);
	if (rc) {
		LOG_SERIOUS("Error adding region %s to expansion points list.\n", region->name);
		RETURN(rc);
	}

	RETURN(0);
}


/* Function: lvm_can_expand_by
 *
 *	LVM does not allow its child segments to expand.
 */
static int lvm_can_expand_by(	storage_object_t	* child_object,
				u_int64_t		* size )
{
	LOG_ENTRY;

	RETURN(EINVAL);
}


/* Function: lvm_can_shrink
 *
 *	Can this region be shrunk, and consolidate the space with this group's
 *	freespace? If so, add this region to the shrink_points list.
 */
static int lvm_can_shrink(storage_object_t	* region,
			u_int64_t		* shrink_limit,
			dlist_t			shrink_points )
{
	lvm_logical_volume_t	* volume = region->private_data;
	lvm_volume_group_t	* group = volume->group;
	shrink_object_info_t	* shrink_point;
	void			* handle;
	int			rc;

	LOG_ENTRY;

	// Don't shrink the freespace region, or a volume involved in
	// snapshotting.
	if ( region->data_type != DATA_TYPE ) {
		LOG("Cannot shrink freespace region %s.\n", region->name);
		RETURN(EINVAL);
	}
	if ( volume->lv->lv_access & LV_SNAPSHOT ) {
		LOG("Cannot shrink snapshot region %s.\n", region->name);
		RETURN(EINVAL);
	}
	if ( volume->lv->lv_access & LV_SNAPSHOT_ORG ) {
		LOG("Cannot shrink region %s while being snapshotted.\n", region->name);
		RETURN(EINVAL);
	}

	// Create a new shrink info object.
	shrink_point = lvm_engine->engine_alloc(sizeof(shrink_object_info_t));
	if ( ! shrink_point ) {
		LOG_CRITICAL("Memory error creating shrink info object.\n");
		RETURN(ENOMEM);
	}

	shrink_point->object = region;
	shrink_point->max_shrink_size = (volume->lv->lv_allocated_le - 1) * group->vg->pe_size;
	if ( *shrink_limit < shrink_point->max_shrink_size ) {
		shrink_point->max_shrink_size = round_down(*shrink_limit, group->vg->pe_size);
	}

	// Add to the shrink points list.
	rc = InsertObject(shrink_points,
			sizeof(shrink_object_info_t),
			shrink_point,
			SHRINK_OBJECT_TAG, NULL,
			AppendToList, TRUE, &handle);
	if (rc) {
		LOG_SERIOUS("Error adding region %s to shrink points list.\n", region->name);
		RETURN(rc);
	}

	RETURN(0);
}


/* Function: lvm_can_shrink_by
 *
 *	LVM does not allow shrinking of its PVs.
 */
static int lvm_can_shrink_by(	storage_object_t	* region,
				u_int64_t		* size )
{
	LOG_ENTRY;

	RETURN(EINVAL);
}


/* Function: lvm_can_move
 */
static int lvm_can_move( storage_object_t * region )
{
	LOG_ENTRY;
	LOG("Not yet implemented\n");
	RETURN(ENOSYS);
}


/* Function: lvm_can_set_volume
 *
 *	Any non-freespace LVM region can be converted into an EVMS
 *	compatibility volume. Snapshots and snapshot originals
 *	cannot have their volumes removed.
 */
static int lvm_can_set_volume(	storage_object_t	* region,
				boolean			flag )
{
	lvm_logical_volume_t	* volume = region->private_data;

	LOG_ENTRY;

	// Can't make freespace into an EVMS volume.
	if ( region->data_type != DATA_TYPE ) {
		LOG("Cannot make a volume from freespace region %s\n", region->name);
		RETURN(EINVAL);
	}

	if ( flag == FALSE ) {
		if ( volume->lv->lv_access & LV_SNAPSHOT ) {
			LOG("Cannot remove volume from snapshot region %s\n", region->name);
			RETURN(EINVAL);
		}
		if ( volume->lv->lv_access & LV_SNAPSHOT_ORG ) {
			LOG("Cannot remove volume from snapshot original region %s\n", region->name);
			RETURN(EINVAL);
		}
	}

	RETURN(0);
}


/* Function: lvm_discover
 *
 *	segments : Input list of storage objects
 *	regions : Output list of storage objects
 *	final_call : Is this the last discovery iteration?
 *
 *	Examine all disk segments and find LVM PVs. Assemble volume groups
 *	and export all LVM logical volumes as EVMS regions.
 *
 *	All newly created regions must be added to the output list, and all
 *	segments from the input list must either be claimed or moved to the
 *	output list.
 */
static int lvm_discover(dlist_t	segments,
			dlist_t	regions,
			boolean	final_call )
{
	int rc = 0;

	LOG_ENTRY;
	LOG_DEBUG("Beginning LVM Discovery\n");

	lvm_discover_volume_groups(segments, regions);

	lvm_check_volume_groups(final_call);

	lvm_discover_logical_volumes(final_call);

	rc = lvm_export_logical_volumes(regions, final_call);

	// No cleanup in the engine. All metadata structures are
	// kept around for the life of the engine session.

	LOG_DEBUG("LVM Discovery complete.\n");
	RETURN(rc);
}



/****** Region Functions ******/



/* Function: lvm_create
 *
 *	Create a new LVM region using the specified freespace.
 */
static int lvm_create(	dlist_t		freespace_region_list,
			option_array_t	* options,
			dlist_t		new_region_list )
{
	lvm_logical_volume_t	* freespace_volume;
	lvm_volume_group_t	* group;
	lvm_lv_create_options_t	lv_opts;
	lv_disk_t		* lv;
	lvm_logical_volume_t	* new_volume;
	int			rc;

	LOG_ENTRY;

	// Get the input freespace volume and group.
	rc = lvm_get_freespace_volume(freespace_region_list, &freespace_volume);
	if (rc) {
		LOG_ERROR("Error getting freespace region from list\n");
		RETURN(EINVAL);
	}

	group = freespace_volume->group;

	// Get the input options and verify.
	rc = lvm_create_region_parse_option_array(options, group, &lv_opts);
	if (rc) {
		LOG_ERROR("Error parsing and verifying creation options\n");
		RETURN(EINVAL);
	}

	// Make sure there is enough space for the desired allocation scheme.
	rc = lvm_check_available_extents(group, &lv_opts);
	if (rc) {
		LOG_ERROR("Can not create new region in container %s using specified options\n",
			group->container->name);
		RETURN(rc);
	}

	// Initialize the appropriate LV entry.
	rc = lvm_initialize_new_lv(&lv_opts, group, &lv);
	if (rc) {
		LOG_ERROR("Error initializing the LV metadata\n");
		RETURN(rc);
	}

	// Allocate a new logical volume.
	new_volume = lvm_allocate_logical_volume(lv, group);
	if ( ! new_volume ) {
		LOG_ERROR("Error creating new region %s\n", lv_opts.lv_name);
		lvm_clear_lv(lv);
		RETURN(ENOMEM);
	}

	// Set up the LE-to-PE mappings for the new volume.
	rc = lvm_allocate_extents_to_volume(new_volume, &lv_opts);
	if (rc) {
		LOG_ERROR("Unable to allocate PEs to the new region %s\n", lv_opts.lv_name);
		lvm_clear_lv(lv);
		lvm_deallocate_logical_volume(new_volume);
		RETURN(rc);
	}

	// Update the group information
	group->volume_list[new_volume->number] = new_volume;
	group->volume_count++;
	group->vg->lv_cur++;
	group->vg->pe_allocated += new_volume->lv->lv_allocated_le;

	// Update the freespace and rebuild the LE maps.
	rc = lvm_update_freespace_volume(group);
	if (rc) {
		LOG_ERROR("Error updating freespace for container %s\n", group->container->name);
		RETURN(rc);
	}

	// Update snapshotting information
	if ( lv_opts.snapshot ) {
		rc = lvm_create_snapshot_volume(new_volume, lv_opts.snapshot_org);
		if (rc) {
			// Hmmmm...should probably try to delete this guy we just created.
			RETURN(rc);
		}
	}

	if ( lv_opts.zero ) {
		lvm_zero_first_1k(new_volume);
	}

	group->container->flags |= SCFLAG_DIRTY;
	lvm_add_object_to_list(new_volume->region, new_region_list);

	RETURN(0);
}


/* Function: lvm_delete
 *
 *	Remove the specified region and consolidate all of its space into
 *	the appropriate freespace region.
 */
static int lvm_delete(	storage_object_t	* region,
			dlist_t			children )
{
	lvm_logical_volume_t	* volume;
	lvm_volume_group_t	* group;
	int			rc = 0;

	LOG_ENTRY;

	// Check that this region can be removed.
	rc = lvm_can_delete(region);
	if (rc) {
		LOG_ERROR("Not allowed to delete region %s.\n", region->name);
		RETURN(rc);
	}

	volume = region->private_data;
	group = volume->group;

	// Special stuff to do for snapshot volumes
	if ( volume->lv->lv_access & LV_SNAPSHOT ) {
		rc = lvm_delete_snapshot_volume(volume);
		if (rc) {
			LOG_ERROR("Unable to remove snapshot region %s\n", region->name);
			RETURN(EINVAL);
		}
	}

	// Remove all the extents from the volume
	lvm_deallocate_extents_from_volume(volume);

	// Remove the parent/child associations with the PVs
	lvm_clear_child_list(region);

	// Update the group information
	group->vg->lv_cur--;
	group->vg->pe_allocated -= volume->lv->lv_allocated_le;

	// Delete the volume.
	lvm_clear_lv(volume->lv);
	lvm_deallocate_logical_volume(volume);

	// Update the freespace and rebuild the LE maps.
	rc = lvm_update_freespace_volume(group);
	if (rc) {
		LOG_ERROR("Error updating freespace for container %s\n", group->container->name);
	}

	group->container->flags |= SCFLAG_DIRTY;

	RETURN(rc);
}


/* Function: lvm_expand
 *
 *	Expand an LVM region using the freespace in that region's container.
 */
static int lvm_expand(	storage_object_t	* region,
			storage_object_t	* expand_object,
			dlist_t			input_objects,
			option_array_t		* options )
{
	lvm_volume_group_t	* group;
	lvm_logical_volume_t	* volume;
	lvm_logical_volume_t	* freespace;
	lvm_lv_expand_options_t lv_opts;
	sector_count_t		expand_delta;
	int			rc = 0;

	LOG_ENTRY;

	// If region and expand_object are not the same, someone is trying to
	// expand an object below LVM. Not allowed.
	if ( region != expand_object ) {
		LOG_ERROR("Cannot expand object %s below LVM region %s.\n", expand_object->name, region->name);
		RETURN(EINVAL);
	}

	volume = region->private_data;
	group = volume->group;

	// Get the selected freespace from the input objects list, and make
	// sure it belongs to the target object's group.
	rc = lvm_get_freespace_volume(input_objects, &freespace);
	if (rc) {
		RETURN(rc);
	}
	if ( freespace->group != group ) {
		LOG_ERROR("Target region and selected freespace are not in same group.\n");
		LOG_ERROR("Target of expand is %s.\n", volume->region->name);
		LOG_ERROR("Selected freespace is %s.\n", freespace->region->name);
		RETURN(EINVAL);
	}

	// Don't expand the freespace region, a volume involved in
	// snapshotting, or an incomplete volume.
	if ( region->data_type != DATA_TYPE ) {
		LOG_ERROR("Cannot expand freespace region %s.\n", region->name);
		RETURN(EINVAL);
	}
	if ( volume->lv->lv_access & LV_SNAPSHOT ) {
		LOG_ERROR("Cannot expand snapshot region %s.\n", region->name);
		RETURN(EINVAL);
	}
	if ( volume->lv->lv_access & LV_SNAPSHOT_ORG ) {
		LOG_ERROR("Cannot expand region %s while being snapshotted.\n", region->name);
		RETURN(EINVAL);
	}
	if ( volume->flags & LVM_LV_FLAG_INCOMPLETE ) {
		LOG_ERROR("Region %s is currently missing PVs. Cannot expand.\n", region->name);
		RETURN(EINVAL);
	}

	// Must be space available in the group.
	if ( ! freespace->lv->lv_allocated_le ) {
		LOG_ERROR("No freespace left in container %s.\n", group->container->name);
		RETURN(EINVAL);
	}

	rc = lvm_expand_region_parse_option_array(options, group, volume, &lv_opts);
	if (rc) {
		LOG_ERROR("Error parsing and verifying expand options\n");
		RETURN(rc);
	}

	// Call the engine's can_expand_by() to make sure all plugins 
	// and FSIMS above LVM approve of the expand.
	expand_delta = lv_opts.add_size;
	rc = lvm_engine->can_expand_by(region, &expand_delta);
	if (rc) {
		LOG_ERROR("A parent of LVM has disallowed the expand of region %s\n", region->name);
		RETURN(rc);
	}

	// Make sure there is enough space for the expand based on the
	// volume's allocation scheme.
	rc = lvm_check_available_expand_extents(volume, &lv_opts);
	if (rc) {
		LOG_ERROR("Can not expand region %s using specified options\n", region->name);
		RETURN(rc);
	}

	// Set up the expanded LE-to-PE mappings for the volume.
	rc = lvm_allocate_expand_extents_to_volume(volume, &lv_opts);
	if (rc) {
		LOG_ERROR("Unable to allocate enough PEs to expand region %s\n", region->name);
		RETURN(rc);
	}

	// Update the volume information.
	rc = lvm_update_expanded_volume(volume, &lv_opts);
	if (rc) {
		LOG_ERROR("Error updating expanded region %s\n", region->name);
		RETURN(rc);
	}

	// Update the group information
	group->vg->pe_allocated += lv_opts.add_extents;

	// Update the freespace and rebuild the LE maps.
	rc = lvm_update_freespace_volume(group);
	if (rc) {
		LOG_ERROR("Error updating freespace for container %s\n", group->container->name);
	}

	// Mark container dirty.
	group->container->flags |= SCFLAG_DIRTY;

	RETURN(rc);
}


/* Function: lvm_shrink
 */
static int lvm_shrink(	storage_object_t	* region,
			storage_object_t	* shrink_object,
			dlist_t			input_objects,
			option_array_t		* options )
{
	lvm_volume_group_t	* group;
	lvm_logical_volume_t	* volume;
	u_int32_t		remove_extents;
	sector_count_t		shrink_delta;
	int			rc = 0;

	LOG_ENTRY;

	// If region and shrink_object are not the same, someone is trying to
	// shrink an object below LVM. Not allowed.
	if ( region != shrink_object ) {
		LOG_ERROR("Cannot shrink object %s below LVM region %s.\n", shrink_object->name, region->name);
		RETURN(EINVAL);
	}

	volume = region->private_data;
	group = volume->group;

	// Don't shrink the freespace region, a volume involved in
	// snapshotting, or an incomplete volume.
	if ( region->data_type != DATA_TYPE ) {
		LOG_ERROR("Cannot shrink freespace region %s.\n", region->name);
		RETURN(EINVAL);
	}
	if ( volume->lv->lv_access & LV_SNAPSHOT ) {
		LOG_ERROR("Cannot shrink snapshot region %s.\n", region->name);
		RETURN(EINVAL);
	}
	if ( volume->lv->lv_access & LV_SNAPSHOT_ORG ) {
		LOG_ERROR("Cannot shrink region %s while being snapshotted.\n", region->name);
		RETURN(EINVAL);
	}
	if ( volume->flags & LVM_LV_FLAG_INCOMPLETE ) {
		LOG_ERROR("Region %s is currently missing PVs. Cannot shrink.\n", region->name);
		RETURN(EINVAL);
	}

	// Parse the options for the shrink command.
	rc = lvm_shrink_region_parse_option_array(options, group, volume, &remove_extents);
	if (rc) {
		LOG_ERROR("Error parsing and verifying shrink options\n");
		RETURN(rc);
	}

	// Call the engine's can_shrink_by() to make sure all plugins 
	// and FSIMS above LVM approve of the shrink.
	shrink_delta = remove_extents * group->vg->pe_size;
	rc = lvm_engine->can_shrink_by(region, &shrink_delta);
	if (rc) {
		LOG_ERROR("A parent of LVM has disallowed the shrink of region %s\n", region->name);
		RETURN(rc);
	}

	// Deallocate the necessary extents from the volume.
	rc = lvm_deallocate_shrink_extents_from_volume(volume, remove_extents);
	if (rc) {
		LOG_ERROR("Unable to deallocate enough PEs to shrink region %s\n", region->name);
		RETURN(rc);
	}

	// Update the volume information.
	rc = lvm_update_shrunk_volume(volume, remove_extents);
	if (rc) {
		LOG_ERROR("Error updating shrunk region %s\n", region->name);
		RETURN(rc);
	}

	// Update the group information
	group->vg->pe_allocated -= remove_extents;

	// Update the freespace and rebuild the LE maps.
	rc = lvm_update_freespace_volume(group);
	if (rc) {
		LOG_ERROR("Error updating freespace for container %s\n", group->container->name);
	}

	// Mark container dirty.
	group->container->flags |= SCFLAG_DIRTY;

	RETURN(rc);
}


/* Function: lvm_move
 */
static int lvm_move(	storage_object_t	* source,
			storage_object_t	* target,
			option_array_t		* options )
{
	LOG_ENTRY;
	LOG("Not yet implemented\n");
	RETURN(ENOSYS);
}


/* Function: lvm_set_volume
 */
static void lvm_set_volume(	storage_object_t	* region,
				boolean			flag )
{
	LOG_ENTRY;
	LOG("That's nice. :)\n");
	LOG_EXIT(0);
}


/* Function: lvm_add_sectors_to_kill_list
 *
 *	The kill_sectors list contains a list of sectors that need to be zeroed
 *	during the next commit. This function is very similar to read/write.
 */
static int lvm_add_sectors_to_kill_list( storage_object_t	* region,
					lsn_t			lsn,
					sector_count_t		count )
{
	lvm_logical_volume_t	* volume = region->private_data;
	lvm_physical_volume_t	* pv_entry = NULL;
	lsn_t			new_lsn = 0;
	sector_count_t		new_count;
	int			rc = 0;

	LOG_ENTRY;

	// Region cannot be a snapshot, a snapshot original, or
	// freespace. Don't kill past the end of the region either.
	if ( volume->lv->lv_access & (LV_SNAPSHOT|LV_SNAPSHOT_ORG) ||
	     region->data_type != DATA_TYPE ||
	     lsn + count > volume->lv->lv_size ) {
		RETURN(EIO);
	}

	// I/Os needs to deal with the possibility that a request can
	// come in that spans PEs or stripes. This is possible because
	// there is no limit on count. To fix this, we loop through
	// lvm_remap_sector and KILL_SECTORS until count reaches zero.
	// Also make sure to check for missing LE entries.
	while ( count ) {
		lvm_remap_sector(volume, lsn, count, &new_lsn, &new_count, &pv_entry);
		if ( pv_entry && new_lsn ) {
			rc = KILL_SECTORS(pv_entry->segment, new_lsn, new_count);
		}
		count	-= new_count;
		lsn	+= new_count;
	}

	RETURN(rc);
}


/* Function: lvm_commit_changes
 *
 *	All commit operations are done in commit_container_changes.
 */
static int lvm_commit_changes(	storage_object_t	* region,
				unsigned int		phase )
{
	LOG_ENTRY;
	LOG("That's nice. :)\n");
	RETURN(0);
}


/* Function: lvm_get_option_count
 *
 *	Determine the type of Task that is being performed, and return
 *	the number of options that are available for that Task.
 */
static int lvm_get_option_count( task_context_t * context )
{
	int count = 0;

	LOG_ENTRY;

	switch(context->action) {
	case EVMS_Task_Create:
		count = LVM_OPTION_PV_NAMES_INDEX + 1;
		break;
	case EVMS_Task_Create_Container:
		count = LVM_OPTION_PE_SIZE_INDEX + 1;
		break;
	case EVMS_Task_Expand_Container:
		count = 0;
		break;
	case EVMS_Task_Expand:
		count = LVM_OPTION_EXPAND_PV_NAMES_INDEX + 1;
		break;
	case EVMS_Task_Shrink:
		count = LVM_OPTION_SHRINK_SIZE_INDEX + 1;
		break;
	default:
		count = -1;
		break;
	}

	RETURN(count);
}


/* Function: lvm_init_task
 *
 *	Determine the type of Task that is being performed, and set up the
 *	context structure with the appropriate initial values.
 */
static int lvm_init_task( task_context_t * context )
{
	lvm_volume_group_t	* group;
	int			rc = 0;

	LOG_ENTRY;

	switch(context->action) {

	case EVMS_Task_Create:
		rc = lvm_create_region_allocate_option_descriptor(context->option_descriptors);
		if (!rc) {
			rc = lvm_create_region_get_acceptable(context->acceptable_objects);
			context->min_selected_objects = 1;
			context->max_selected_objects = 1;
		}
		break;

	case EVMS_Task_Create_Container:
		rc = lvm_create_container_allocate_option_descriptor(context->option_descriptors);
		if (!rc) {
			rc = lvm_create_container_get_acceptable(context->acceptable_objects);
			context->min_selected_objects = 1;
			context->max_selected_objects = MAX_PV;
		}
		break;

	case EVMS_Task_Expand_Container:
		// If the container already has the max number of PVs, then no
		// segments are acceptable.
		group = context->container->private_data;
		if ( group->vg->pv_cur >= MAX_PV ) {
			LOG_ERROR("Container %s already has maximum number of PVs (%d).\n", context->container->name, group->vg->pv_cur);
			rc = EINVAL;
			break;
		}

		rc = lvm_expand_container_allocate_option_descriptor(context->option_descriptors);
		if (!rc) {
			rc = lvm_expand_container_get_acceptable(context->container, context->acceptable_objects);
			context->min_selected_objects = 1;
			context->max_selected_objects = MAX_PV - group->vg->pv_cur;
		}
		break;

	case EVMS_Task_Expand:
		// Don't allow expanding the freespace region
		if ( context->object->data_type == FREE_SPACE_TYPE ) {
			LOG_ERROR("Cannot expand freespace region %s\n", context->object->name);
			rc = EINVAL;
			break;
		}

		rc = lvm_expand_region_allocate_option_descriptor(context->option_descriptors);
		if (!rc) {
			rc = lvm_expand_region_get_acceptable(context->object, context->acceptable_objects);
			context->min_selected_objects = 1;
			context->max_selected_objects = 1;
		}
		break;

	case EVMS_Task_Shrink:
		// Don't allow shrinking the freespace region.
		if ( context->object->data_type == FREE_SPACE_TYPE ) {
			LOG_WARNING("Cannot shrink freespace region %s\n", context->object->name);
			rc = EINVAL;
			break;
		}

		rc = lvm_shrink_region_allocate_option_descriptor(context->option_descriptors);
		if (!rc) {
			rc = lvm_shrink_region_get_acceptable(context->object, context->acceptable_objects);
			context->min_selected_objects = 1;
			context->max_selected_objects = 1;
		}
		break;

	default:
		rc = EINVAL;
		break;
	}

	RETURN(rc);
}


/* Function: lvm_set_option
 *
 *	Determine the type of Task that is being performed. Then examine the
 *	desired option (using the index), and verify that the given value is
 *	appropriate. Reset the value if necessary and possible. Adjust other
 *	options as appropriate.
 */
static int lvm_set_option(task_context_t	* context,
			u_int32_t		index,
			value_t			* value,
			task_effect_t		* effect )
{
	int rc = 0;

	LOG_ENTRY;

	*effect = 0;

	switch (context->action) {

	case EVMS_Task_Create:
		rc = lvm_create_region_set_option(context, index, value, effect);
		break;

	case EVMS_Task_Create_Container:
		rc = lvm_create_container_set_option(context, index, value, effect);
		break;

	case EVMS_Task_Expand_Container:
		rc = lvm_expand_container_set_option(context, index, value, effect);
		break;

	case EVMS_Task_Expand:
		rc = lvm_expand_region_set_option(context, index, value, effect);
		break;

	case EVMS_Task_Shrink:
		rc = lvm_shrink_region_set_option(context, index, value, effect);
		break;

	default:
		rc = EINVAL;
		break;
	}

	RETURN(rc);
}


/* Function: lvm_set_objects
 *
 *	Determine the type of task, and then validate that the objects on the
 *	"selected" list are valid for that task. If so, adjust the option
 *	descriptor as appropriate.
 */
static int lvm_set_objects(	task_context_t	* context,
				dlist_t		declined_objects,
				task_effect_t	* effect )
{
	int rc = 0;

	LOG_ENTRY;

	*effect = 0;

	switch(context->action) {

	case EVMS_Task_Create:
		rc = lvm_create_region_set_objects(context, declined_objects, effect);
		break;

	case EVMS_Task_Create_Container:
		rc = lvm_create_container_set_objects(context, declined_objects, effect);
		break;

	case EVMS_Task_Expand_Container:
		rc = lvm_expand_container_set_objects(context, declined_objects, effect);
		break;

	case EVMS_Task_Expand:
		rc = lvm_expand_region_set_objects(context, declined_objects, effect);
		break;

	case EVMS_Task_Shrink:
		rc = lvm_shrink_region_set_objects(context, declined_objects, effect);
		break;

	default:
		rc = EINVAL;
		break;
	}

	RETURN(rc);
}


/* Function: lvm_get_info
 *
 *	Return LVM-specific information about the specified region. If the
 *	name field is set, only return the "extra" information pertaining
 *	to that name.
 */
static int lvm_get_info(	storage_object_t	* region,
				char			* name,
				extended_info_array_t	** info_array )
{
	lvm_logical_volume_t	* volume = region->private_data;
	int			rc = 0;

	LOG_ENTRY;

	if ( ! name ) {
		// Default case. Return all basic info about the region.
		rc = lvm_get_volume_info(volume, info_array);
	}
	else if ( ! strcmp(name, "Extents") ) {
		// "Extra" information about LE-to-PE mapping.
		rc = lvm_get_volume_extent_info(volume, info_array);
	}
	else {
		LOG_ERROR("No support for extra region information about \"%s\"\n", name);
		rc = EINVAL;
	}
	
	RETURN(rc);
}


/* Function: lvm_set_info
 */
static int lvm_set_info(storage_object_t	* region,
			option_array_t		* options )
{
	LOG_ENTRY;
	LOG("Not yet implemented.\n");
	RETURN(ENOSYS);
}


/* Function: lvm_get_plugin_info
 *
 *	Return information about the LVM plugin. There is no "extra"
 *	information about LVM, so "name" should always be NULL.
 */
static int lvm_get_plugin_info(	char			* name,
				extended_info_array_t	** info_array )
{
	extended_info_array_t	* info = NULL;
	char			buffer[50] = {0};
	int			i = 0;
	
	LOG_ENTRY;

	if ( ! name ) {
		// Get memory for the info array
		info = lvm_engine->engine_alloc(sizeof(extended_info_array_t) + sizeof(extended_info_t)*4);
		if ( ! info ) {
			LOG_CRITICAL("Memory error creating info array\n");
			RETURN(ENOMEM);
		}

		// Short Name
		SET_STRING(info->info[i].name, "ShortName");
		SET_STRING(info->info[i].title, "Short Name");
		SET_STRING(info->info[i].desc, "A short name given to this plugin");
		info->info[i].type = EVMS_Type_String;
		SET_STRING(info->info[i].value.s, lvm_plugin->short_name);
		i++;

		// Long Name
		SET_STRING(info->info[i].name, "LongName");
		SET_STRING(info->info[i].title, "Long Name");
		SET_STRING(info->info[i].desc, "A long name given to this plugin");
		info->info[i].type = EVMS_Type_String;
		SET_STRING(info->info[i].value.s, lvm_plugin->long_name);
		i++;

		// Plugin Type
		SET_STRING(info->info[i].name, "Type");
		SET_STRING(info->info[i].title, "Plugin Type");
		SET_STRING(info->info[i].desc, "There are various types of plugins; each responsible for some kind of storage object.");
		info->info[i].type = EVMS_Type_String;
		SET_STRING(info->info[i].value.s, "Region Manager");
		i++;

		// Plugin Version
		SET_STRING(info->info[i].name, "Version");
		SET_STRING(info->info[i].title, "Plugin Version");
		SET_STRING(info->info[i].desc, "This is the version number of the plugin.");
		info->info[i].type = EVMS_Type_String;
		snprintf(buffer, 50, "%d.%d.%d", MAJOR_VERSION, MINOR_VERSION, PATCH_LEVEL);
		SET_STRING(info->info[i].value.s, buffer);
		i++;

		// Required API Version
		SET_STRING(info->info[i].name, "Required_Version");
		SET_STRING(info->info[i].title, "Required Plugin API Version");
		SET_STRING(info->info[i].desc, "This is the version of the engine that the plugin requires.It will not run on older versions of the Engine.");
		info->info[i].type = EVMS_Type_String;
		snprintf(buffer, 50, "%d.%d.%d", lvm_plugin->required_api_version.major, lvm_plugin->required_api_version.minor, lvm_plugin->required_api_version.patchlevel);
		SET_STRING(info->info[i].value.s, buffer);
		i++;

		info->count = i;
		*info_array = info;
	}
	else {
		LOG_ERROR("No support for extra plugin information about \"%s\"\n", name);
		RETURN(EINVAL);
	}

	RETURN(0);
}


/* Function: lvm_read
 *
 *	Perform a logical-to-physical remapping, and send the read down to
 *	the next plugin.
 */
static int lvm_read(	storage_object_t	* region,
			lsn_t			lsn,
			sector_count_t		count,
			void			* buffer )
{
	lvm_logical_volume_t	* volume = region->private_data;
	lvm_physical_volume_t	* pv_entry = NULL;
	lsn_t			new_lsn = 0;
	sector_count_t		new_count;
	int			rc = 0;

	LOG_ENTRY;

	// No reads allowed on snapshot volumes, incomplete volumes, or
	// freespace volumes. Don't read past the end of the volume.
	if ( volume->lv->lv_access & LV_SNAPSHOT ||
	     region->data_type != DATA_TYPE ||
	     lsn + count > volume->lv->lv_size ) {
		memset(buffer, 0, sectors_to_bytes(count));
		RETURN(0);
	}

	// I/Os needs to deal with requests that can span PEs
	// or stripes. This is possible because there is no
	// limit on count. To fix this, we loop through
	// lvm_remap_sector and READ until count reaches zero.
	// Also make sure to check for missing LE entries.
	while ( count ) {
		lvm_remap_sector(volume, lsn, count, &new_lsn, &new_count, &pv_entry);
		if ( pv_entry && new_lsn ) {
			rc = READ(pv_entry->segment, new_lsn, new_count, buffer);
		}
		else {
			memset(buffer, 0, sectors_to_bytes(new_count));
			rc = EIO;
		}
		count	-= new_count;
		lsn	+= new_count;
		buffer	= (void*)(((unsigned long)buffer) + (unsigned long)(new_count << SECTOR_SIZE_BITS));
	}

	RETURN(rc);
}


/* Function: lvm_write
 *
 *	Perform a logical-to-physical remapping, and send the write down to
 *	the next plugin.
 */
static int lvm_write(	storage_object_t	* region,
			lsn_t			lsn,
			sector_count_t		count,
			void			* buffer )
{
	lvm_logical_volume_t	* volume = region->private_data;
	lvm_physical_volume_t	* pv_entry;
	lsn_t			new_lsn;
	sector_count_t		new_count;
	int			rc = 0;

	LOG_ENTRY;

	// No writes allowed to any volumes involved in snapshotting, any
	// incomplete volumes, or any freespace volumes. Don't write past
	// the end of the volume.
	if ( volume->lv->lv_access & (LV_SNAPSHOT|LV_SNAPSHOT_ORG) ||
	     region->data_type != DATA_TYPE ||
	     lsn + count > volume->lv->lv_size ) {
		RETURN(EIO);
	}

	// I/Os needs to deal with requests that span PEs or
	// stripes. This is possible because there is no limit
	// on count. To fix this, we loop through
	// lvm_remap_sector and WRITE until count reaches zero.
	// Also check for missing LE entries.
	while ( count ) {
		lvm_remap_sector(volume, lsn, count, &new_lsn, &new_count, &pv_entry);
		if ( pv_entry && new_lsn ) {
			rc = WRITE(pv_entry->segment, new_lsn, new_count, buffer);
		}
		else {
			rc = EIO;
		}
		count	-= new_count;
		lsn	+= new_count;
		buffer	= (void*)(((unsigned long)buffer) + (unsigned long)(new_count << SECTOR_SIZE_BITS));
	}

	RETURN(rc);
}


/* Function: lvm_direct_plugin_communication
 */
static int lvm_direct_plugin_communication(	void	* thing,
						boolean	target_kernel_plugin,
						void	* arg )
{
	LOG_ENTRY;
	LOG("Not yet implemented\n");
	RETURN(ENOSYS);
}



/****** Container Functions ******/



/* Function: lvm_can_create_container
 *
 *	Can we create a new volume group from this list of segments?
 */
static int lvm_can_create_container( dlist_t objects )
{
	storage_object_t	* object;
	int			rc;

	LOG_ENTRY;

	if ( ListEmpty(objects) ) {
		RETURN(EINVAL);
	}

	// In order to create a new group, all of the specified segments must
	// not be in an LVM regular group, or else not have any parent objects.
	FOR_EACH(object, objects) {
		rc = lvm_check_segment_for_group_inclusion(object);
		if (rc) {
			LOG_WARNING("One or more objects cannot be added to an LVM container.\n");
			RETURN(EINVAL);
		}
	}

	RETURN(0);
}


/* Function: lvm_can_delete_container
 *
 *	Can we remove the specified volume group?
 */
static int lvm_can_delete_container( storage_container_t * container )
{
	lvm_volume_group_t * group = container->private_data;

	LOG_ENTRY;

	// The container must not contain any regions.
	if ( group->volume_count ) {
		LOG("Container %s is still exporting %d regions.\n", container->name, group->volume_count);
		RETURN(EINVAL);
	}

	RETURN(0);
}


/* Function: lvm_can_add_object
 *
 *	Can the specified object be added to the specified container.
 */
static int lvm_can_add_object(	storage_object_t	* object,
				storage_container_t	* container )
{
	lvm_volume_group_t	* group = container->private_data;
	u_int32_t		pe_size;
	int			rc;

	LOG_ENTRY;

	// Check that the segment can be added to a group.
	rc = lvm_check_segment_for_group_inclusion(object);
	if (rc) {
		RETURN(rc);
	}

	// Make sure the group has space for a new PV.
	if ( group->vg->pv_cur >= MAX_PV ) {
		LOG_WARNING("Container %s already has maximum number of PVs (%d).\n", container->name, MAX_PV);
		RETURN(EINVAL);
	}

	// Check that the segment is big enough to add to this group.
	pe_size = group->vg->pe_size;
	rc = lvm_check_segment_for_pe_size(object, &pe_size);
	if (rc) {
		RETURN(rc);
	}

	RETURN(0);
}


/* Function: lvm_can_remove_object
 *
 *	Can the specified object be removed from its container.
 */
static int lvm_can_remove_object( storage_object_t * object )
{
	storage_container_t	* container = object->consuming_container;
	lvm_volume_group_t	* group;
	lvm_physical_volume_t	* pv_entry;
	int			rc;

	LOG_ENTRY;

	// Check that the segment is part of an LVM container.
	if ( ! container ) {
		LOG_WARNING("Object %s is not in a container.\n", object->name);
		RETURN(EINVAL);
	}
	if ( container->plugin != lvm_plugin ) {
		LOG_WARNING("Object %s is in non-LVM container %s.\n", object->name, container->name);
		RETURN(EINVAL);
	}
	group = container->private_data;
	
	// Make sure the group has no snapshot volumes. Removing a PV from a
	// group can cause PVs to get renumbered, which could drastically
	// break the COW tables on existing snapshots.
	rc = lvm_count_snapshot_volumes(group);
	if ( rc > 0 ) {
		LOG_WARNING("Cannot remove objects from container %s\n", container->name);
		LOG_WARNING("Contains %d snapshot regions.\n", rc);
		LOG_WARNING("Removing objects may break the metadata of these snapshots.\n");
		RETURN(EINVAL);
	}

	// Make sure this PV isn't the last one in the group. To take the last
	// PV out, use the lvm_destroy_container function. We don't want empty
	// groups hanging around.
	if ( group->pv_count == 1 ) {
		LOG_WARNING("Cannot remove the last object from container %s\n", container->name);
		LOG_WARNING("Use delete_container to remove this container.\n");
		RETURN(EINVAL);
	}

	// Get the PV for this segment
	pv_entry = lvm_get_pv_for_segment(object);
	if ( ! pv_entry ) {
		LOG_ERROR("Could not find PV entry for object %s\n", object->name);
		RETURN(EINVAL);
	}

	// Make sure the PV is not in use by any volumes
	if ( pv_entry->pv->lv_cur ) {
		LOG_WARNING("Object %s is in use by %d regions\n",
			object->name, pv_entry->pv->lv_cur);
		RETURN(EINVAL);
	}

	RETURN(0);
}


/* Function: lvm_create_container
 *
 *	Create a new LVM volume group, using the specified segments.
 */
static int lvm_create_container( dlist_t			objects,
				option_array_t			* options,
				storage_container_t		** container )
{
	storage_object_t	* object;
	vg_disk_t		* vg;
	lvm_volume_group_t	* new_group;
	unsigned char		vg_name[NAME_LEN] = {0};
	u_int32_t		pe_size;
	int			rc;

	LOG_ENTRY;

	// There needs to be at least one segment. Don't create an empty group.
	if ( ListEmpty(objects) ) {
		LOG_ERROR("No objects specified.\n");
		LOG_ERROR("Cannot create an empty container\n");
		RETURN(EINVAL);
	}

	// Get the appropriate options for group creation.
	rc = lvm_create_container_parse_option_array(options, vg_name, &pe_size);
	if (rc) {
		LOG_ERROR("Error parsing and verifying options\n");
		RETURN(rc);
	}

	// Make sure every segment is valid for use in a new group, and
	// large enough for the specified pe_size.
	FOR_EACH(object, objects) {
		rc = lvm_check_segment_for_group_inclusion(object);
		if (rc) {
			LOG_ERROR("Object %s cannot be added to an LVM container.\n", object->name);
			RETURN(rc);
		}

		rc = lvm_check_segment_for_pe_size(object, &pe_size);
		if (rc) {
			LOG_ERROR("Object %s cannot be added to an LVM container.\n", object->name);
			RETURN(rc);
		}
	}

	// Allocate and initialize a vg_disk_t structure with the user options.
	vg = lvm_initialize_new_vg(pe_size);
	if ( ! vg ) {
		LOG_ERROR("Error initializing new VG metadata\n");
		RETURN(EINVAL);
	}

	// Create the volume group.
	new_group = lvm_allocate_volume_group(vg, vg_name);
	if ( ! new_group ) {
		LOG_CRITICAL("Memory error creating new container %s\n", vg_name);
		RETURN(ENOMEM);
	}

	// The group has no UUID list or LV array to read yet, so set the flags
	// to prevent spurious error messages.
	new_group->flags |= (LVM_VG_FLAG_UUID_LIST_PRESENT | LVM_VG_FLAG_LV_LIST_PRESENT);

	// Create the freespace region for the new group.
	rc = lvm_create_freespace_volume(new_group);
	if (rc) {
		lvm_deallocate_volume_group(new_group);
		RETURN(rc);
	}

	// Add each segment to the new group.
	FOR_EACH(object, objects) {
		rc = lvm_transfer_segment_to_group(object, new_group);
		if (rc) {
			lvm_deallocate_volume_group(new_group);
			RETURN(rc);
		}
	}

	// Update the freespace region
	rc = lvm_update_freespace_volume(new_group);
	if (rc) {
		LOG_SERIOUS("Error updating freespace in container %s\n", new_group->container->name);
		RETURN(rc);
	}

	// Mark the container dirty. None of this has been written to disk yet.
	new_group->container->flags |= SCFLAG_DIRTY;

	// Place the new group in the global list.
	lvm_add_group_to_list(new_group);
	*container = new_group->container;

	LOG("Successfully created new container %s\n", new_group->container->name);
	RETURN(0);
}


/* Function: lvm_add_object
 *
 *	Add the specified object to the specified (existing) container. The
 *	object must have already been removed from its previous location
 *	before this function is called. This function is not used to move a
 *	segment from one LVM container to another LVM container. The function
 *	lvm_transfer_object will be used for that purpose. Externally, this
 *	function will be used to emulate LVM's pvcreate command. No options
 *	are available for this function, since all requirements have been set
 *	by the existing container.
 */
static int lvm_add_object(	storage_object_t	* object,
				storage_container_t	* container,
				option_array_t		* options )
{
	lvm_volume_group_t	* group = container->private_data;
	lvm_physical_volume_t	* pv_entry;
	u_int32_t		pe_size;
	int			rc;

	LOG_ENTRY;

	// Check that the specified segment is not owned by LVM.
	rc = lvm_check_segment_for_group(object, NULL);
	if (rc) {
		RETURN(rc);
	}

	// Check that the segment can be added to a group.
	rc = lvm_check_segment_for_group_inclusion(object);
	if (rc) {
		RETURN(rc);
	}

	// Check that the segment is big enough to add to this group.
	pe_size = group->vg->pe_size;
	rc = lvm_check_segment_for_pe_size(object, &pe_size);
	if (rc) {
		RETURN(rc);
	}
	
	// Create a new physical volume for the segment
	pv_entry = lvm_create_pv_from_segment(object);
	if ( ! pv_entry ) {
		RETURN(ENOMEM);
	}

	// Add the new physical volume to the group.
	rc = lvm_add_new_pv_to_group(pv_entry, group);
	if (rc) {
		LOG_SERIOUS("Error adding object %s to container %s\n",
			object->name, group->container->name );
		lvm_deallocate_physical_volume(pv_entry);
		RETURN(rc);
	}

	// The freespace volume needs to be adjusted to include the space on
	// this new PV.
	rc = lvm_update_freespace_volume(group);
	if (rc) {
		LOG_SERIOUS("Error updating freespace for container %s\n", group->container->name);
		RETURN(rc);
	}

	// Mark the container dirty. None of this has been written to disk yet.
	group->container->flags |= SCFLAG_DIRTY;

	LOG("Successfully added object %s to container %s\n",
		object->name, container->name);

	RETURN(0);
}


/* Function: lvm_transfer_object
 *
 *	Move the specified segment from its current container to the specified
 *	container. Unlike add_object, this function is used to move segments
 *	around internally to LVM. This function will be used to emulate LVM's
 *	vgextend and vgreduce commands. Just like with add_object, there are
 *	no options available for this function, since all of the requirements
 *	have already been set by the target container.
 */
static int lvm_transfer_object( storage_object_t	* object,
				storage_container_t	* target_container,
				option_array_t		* options )
{
	storage_container_t	* source_container;
	lvm_volume_group_t	* source_group;
	lvm_volume_group_t	* target_group = target_container->private_data;
	lvm_physical_volume_t	* pv_entry;
	u_int32_t		pe_size;
	int			rc;

	LOG_ENTRY;

	// Check that the segment is part of an LVM container.
	rc = lvm_check_segment_for_group(object, NULL);
	if (!rc) {
		RETURN(EINVAL);
	}
	source_container = object->consuming_container;
	source_group = source_container->private_data;
	
	// Check that the source and target are not the same
	if ( source_group == target_group ) {
		LOG_ERROR("Source and target containers are the same: %s\n", source_container->name);
		RETURN(EINVAL);
	}

	// Make sure the source group has no snapshot volumes. Removing a PV
	// from a group can cause PVs to get renumbered, which could drastically
	// break the COW tables on existing snapshots.
	rc = lvm_count_snapshot_volumes(source_group);
	if ( rc > 0 ) {
		LOG_ERROR("Cannot remove objects from container %s\n", source_container->name);
		LOG_ERROR("Contains %d snapshot regions.\n", rc);
		LOG_ERROR("Removing objects may break the metadata of these snapshots.\n");
		RETURN(EINVAL);
	}

	// Make sure this PV isn't the last one in the group. To take the last
	// PV out, use the lvm_destroy_container function. We don't want empty
	// groups hanging around.
	if ( source_group->pv_count == 1 ) {
		LOG_ERROR("Cannot remove the last object from container %s\n", source_container->name);
		LOG_ERROR("Use delete_container to remove this container.\n");
		RETURN(EINVAL);
	}

	// Make sure this PV is big enough for the target group.
	pe_size = target_group->vg->pe_size;
	rc = lvm_check_segment_for_pe_size(object, &pe_size);
	if (rc) {
		RETURN(rc);
	}

	// Get the PV for this segment
	pv_entry = lvm_get_pv_for_segment(object);
	if ( ! pv_entry ) {
		LOG_ERROR("Could not find PV entry for object %s\n", object->name);
		RETURN(EINVAL);
	}

	// Make sure the PV is not in use by any volumes
	if ( pv_entry->pv->lv_cur ) {
		LOG_ERROR("Object %s is in use by %d regions\n",
			object->name, pv_entry->pv->lv_cur);
		RETURN(EINVAL);
	}

	// Queue this segment to be removed in the kernel.
	rc = lvm_add_pv_to_removal_list(pv_entry, source_group);
	if (rc) {
		LOG_SERIOUS("Error queueing request to kernel to remove object %s from container %s\n",
			object->name, source_container->name);
		RETURN(rc);
	}

	// Take the segment out of the source group.
	rc = lvm_remove_pv_from_group(pv_entry);
	if (rc) {
		LOG_SERIOUS("Error removing object %s from container %s\n",
			object->name, source_container->name);
		RETURN(rc);
	}

	// Add the segment to the target group
	rc = lvm_add_new_pv_to_group(pv_entry, target_group);
	if (rc) {
		LOG_SERIOUS("Error adding object %s to container %s\n",
			object->name, target_container->name);
		RETURN(rc);
	}

	// The freespace volumes need to be adjusted
	rc = lvm_update_freespace_volume(source_group);
	if (rc) {
		LOG_SERIOUS("Error updating freespace for container %s\n",
			source_container->name);
		RETURN(rc);
	}
	rc = lvm_update_freespace_volume(target_group);
	if (rc) {
		LOG_SERIOUS("Error updating freespace for container %s\n",
			target_container->name);
		RETURN(rc);
	}

	// Mark both containers dirty. None of this has been written to disk yet.
	source_container->flags |= SCFLAG_DIRTY;
	target_container->flags |= SCFLAG_DIRTY;

	LOG("Successfully moved object %s from container %s to container %s\n",
		object->name, source_container->name, target_container->name);

	RETURN(0);
}


/* Function: lvm_remove_object
 *
 *	Remove the specified segment from its current container. Make sure the
 * 	segment does not contain any allocated regions. In addition to removing
 *	it from the container, we want to kill any metadata that might have been
 *	on the segment.
 */
static int lvm_remove_object( storage_object_t * object )
{
	lvm_volume_group_t	* group;
	lvm_physical_volume_t	* pv_entry;
	int			rc;

	LOG_ENTRY;

	// Check that the segment is part of an LVM container.
	rc = lvm_check_segment_for_group(object, NULL);
	if (!rc) {
		RETURN(EINVAL);
	}
	group = object->consuming_container->private_data;

	// Make sure the source group has no snapshot volumes. Removing a PV
	// from a group can cause PVs to get renumbered, which could drastically
	// break the COW tables on existing snapshots.
	rc = lvm_count_snapshot_volumes(group);
	if ( rc > 0 ) {
		LOG_ERROR("Cannot remove object from container %s\n", group->container->name);
		LOG_ERROR("Contains %d snapshot regions.\n", rc);
		LOG_ERROR("Removing objects may break the metadata of these snapshots.\n");
		RETURN(EINVAL);
	}

	// Make sure this PV isn't the last one in the group. To take the last
	// PV out, use the lvm_delete_container function. We don't want empty
	// groups hanging around.
	if ( group->pv_count == 1 ) {
		LOG_ERROR("Cannot remove the last object from container %s\n", group->container->name);
		RETURN(EINVAL);
	}

	// Get the PV for this segment
	pv_entry = lvm_get_pv_for_segment(object);
	if ( ! pv_entry ) {
		LOG_ERROR("Could not find PV entry for object %s\n", object->name);
		RETURN(EINVAL);
	}

	// Make sure the PV is not in use by any volumes
	if ( pv_entry->pv->lv_cur ) {
		LOG_ERROR("Object %s is in use by %d regions\n",
			object->name, pv_entry->pv->lv_cur);
		RETURN(EINVAL);
	}

	// Queue this segment to be removed in the kernel.
	rc = lvm_add_pv_to_removal_list(pv_entry, group);
	if (rc) {
		LOG_SERIOUS("Error queueing request to kernel to remove object %s from container %s\n",
			object->name, group->container->name);
		RETURN(rc);
	}

	// Take the segment out of the source group.
	rc = lvm_remove_pv_from_group(pv_entry);
	if (rc) {
		LOG_SERIOUS("Error removing object %s from container %s\n",
			object->name, group->container->name);
		RETURN(rc);
	}

	// The freespace volume needs to be adjusted
	rc = lvm_update_freespace_volume(group);
	if (rc) {
		LOG_SERIOUS("Error updating freespace for container %s\n",
			group->container->name);
		RETURN(rc);
	}

	// Erase the PV metadata and deallocate.
	lvm_erase_pv(object);
	lvm_deallocate_physical_volume(pv_entry);

	// Mark the container dirty. None of this has been written to disk yet.
	group->container->flags |= SCFLAG_DIRTY;

	LOG("Successfully removed object %s from container %s\n",
		object->name, group->container->name);

	RETURN(0);
}


/* Function: lvm_delete_container
 *
 *	Remove the specified container. Take all segments from the container
 *	and put back in the unassigned group. Free all memory used by the
 *	specified volume group.
 */
static int lvm_delete_container(storage_container_t	* container,
				dlist_t			objects )
{
	lvm_volume_group_t	* group = container->private_data;
	lvm_physical_volume_t	* pv_entry;
	storage_object_t	* object;
	int			i,rc;

	LOG_ENTRY;

	// Does this group contain any volumes?
	if ( group->volume_count ) {
		LOG_ERROR("Container %s contains %d regions\n",
			container->name, group->volume_count);
		RETURN(EINVAL);
	}

	// Move each PV in the group to the unassigned group. This array
	// traversal goes backwards, because when PVs are removed from the
	// group, all remaining PVs are shifted down to fill in the slots.
	for ( i = MAX_PV; i > 0; i-- ) {
		if ( ! group->pv_list[i] ) {
			continue;
		}
		pv_entry = group->pv_list[i];

		LOG_DETAILS("Removing object %s\n", pv_entry->segment->name);

		// Take the PV out of the group.
		rc = lvm_remove_pv_from_group(pv_entry);
		if (rc) {
			LOG_SERIOUS("Error removing object %s from container %s\n",
				pv_entry->segment->name, container->name);
		}

		// Erase the PV metadata and deallocate.
		object = pv_entry->segment;
		lvm_erase_pv(object);
		lvm_deallocate_physical_volume(pv_entry);

		// Put the object on the output list
		lvm_add_object_to_list(object, objects);
	}

	// Update the freespace region. This is done to free up the
	// parent/child relationships between the freespace and the PVs.
	lvm_update_freespace_volume(group);

	// Release memory for the group
	lvm_deallocate_volume_group(group);

	LOG("Successfully removed container\n");

	RETURN(0);
}


/* Function: lvm_commit_container_changes
 *
 *	Write all of the metadata for this container to the PVs.
 */
static int lvm_commit_container_changes( storage_container_t	* container,
					unsigned int		phase )
{
	lvm_volume_group_t	* group;
	lvm_physical_volume_t	* pv_entry;
	int			i;

	LOG_ENTRY;

	// Only have one set of metadata.
	if ( phase != 1 ) {
		RETURN(0);
	}

	group = container->private_data;

	// Make sure the container is dirty.
	if ( ! (container->flags & SCFLAG_DIRTY) ) {
		RETURN(0);
	}

	// Run the list of PVs that need to be removed in the kernel.
	lvm_remove_pvs_from_kernel();

	// Do the LV metadata endian-conversion here, so it only needs to
	// be done once, instead of once for every PV.
	lvm_endian_convert_lvs(group);

	// For each PV in the group, we need to write the PV structure, the
	// VG structure, the UUID list, the LV array, and the PE map.
	for ( i = 1; i <= MAX_PV; i++ ) {
		pv_entry = group->pv_list[i];
		if ( pv_entry ) {
			lvm_write_pv(pv_entry);
			lvm_write_vg(pv_entry);
			lvm_write_uuid_list(pv_entry);
			lvm_write_lv_array(pv_entry);
			lvm_write_pe_map(pv_entry);
		}
	}

	// Have to convert the LVs back.
	lvm_endian_convert_lvs(group);

	container->flags &= ~SCFLAG_DIRTY;
	LOG_DETAILS("Wrote container %s to disk\n", container->name);
	RETURN(0);
}


/* Function: lvm_get_container_info
 *
 *	Return an information array for the specified container. If the name
 *	field is set, only return the "extra" information pertaining to that
 *	name.
 */
static int lvm_get_container_info(	storage_container_t	* container,
					char			* name,
					extended_info_array_t	** info_array )
{
	lvm_volume_group_t	* group = container->private_data;
	char			* buf = NULL;
	int			pv_number;
	int			rc;

	LOG_ENTRY;

	if ( ! name ) {
		// Default case. Return all basic info about the container.
		rc = lvm_get_group_info(group, info_array);
	}
	else if ( ! strcmp(name, "Current_PVs") ) {
		// Return a list of names of all PVs
		rc = lvm_get_group_pv_list_info(group, info_array);
	}
	else if ( ! strcmp(name, "Current_LVs") ) {
		// Return a list of names of all LVs
		rc = lvm_get_group_lv_list_info(group, info_array);
	}
	else if ( (buf = strstr(name, "PV")) == name ) {
		// PV-specific information. This is what we will
		// use for pvdisplay.
		pv_number = atoi(&buf[2]);
		rc = lvm_get_pv_info(group->pv_list[pv_number], info_array);
	}
	else if ( (buf = strstr(name, "PEMapPV")) == name ) {
		// PV-specific extent information. This will be used
		// for pvdisplay -v
		pv_number = atoi(&buf[7]);
		rc = lvm_get_pv_extent_info(group->pv_list[pv_number], info_array);
	}
	else {
		LOG_ERROR("No support for extra information about \"%s\"\n", name);
		RETURN(EINVAL);
	}

	RETURN(rc);
}


/* Function tables for the LVM Region Manager */
static plugin_functions_t lvm_functions = {
	setup_evms_plugin		: lvm_setup_evms_plugin,
	cleanup_evms_plugin		: lvm_cleanup_evms_plugin,
	can_delete			: lvm_can_delete,
	can_expand			: lvm_can_expand,
	can_expand_by			: lvm_can_expand_by,
	can_shrink			: lvm_can_shrink,
	can_shrink_by			: lvm_can_shrink_by,
	can_move			: lvm_can_move,
	can_set_volume			: lvm_can_set_volume,
	discover			: lvm_discover,
	create				: lvm_create,
	delete				: lvm_delete,
	expand				: lvm_expand,
	shrink				: lvm_shrink,
	move				: lvm_move,
	set_volume			: lvm_set_volume,
	add_sectors_to_kill_list	: lvm_add_sectors_to_kill_list,
	commit_changes			: lvm_commit_changes,
	get_option_count		: lvm_get_option_count,
	init_task			: lvm_init_task,
	set_option			: lvm_set_option,
	set_objects			: lvm_set_objects,
	get_info			: lvm_get_info,
	set_info			: lvm_set_info,
	get_plugin_info			: lvm_get_plugin_info,
	read				: lvm_read,
	write				: lvm_write,
	direct_plugin_communication	: lvm_direct_plugin_communication
};


static container_functions_t lvm_container_functions = {
	can_create_container		: lvm_can_create_container,
	can_delete_container		: lvm_can_delete_container,
	can_add_object			: lvm_can_add_object,
	can_remove_object		: lvm_can_remove_object,
	create_container		: lvm_create_container,
	add_object			: lvm_add_object,
	transfer_object			: lvm_transfer_object,
	remove_object			: lvm_remove_object,
	delete_container		: lvm_delete_container,
	commit_container_changes	: lvm_commit_container_changes,
	get_container_info		: lvm_get_container_info
};


plugin_record_t LVM_Plugin = {
	id				: SetPluginID( EVMS_OEM_IBM, EVMS_REGION_MANAGER, 0x01),
	version				: {
		major			: MAJOR_VERSION,
		minor			: MINOR_VERSION,
		patchlevel		: PATCH_LEVEL
	},
	required_api_version		: {
		major			: 3,
		minor			: 0,
		patchlevel		: 0
	},
	short_name			: "LvmRegMgr",
	long_name			: "LVM Region Manager",
	oem_name			: "IBM",
	functions			: {
		plugin			: &lvm_functions
	},
	container_functions 		: &lvm_container_functions
};

plugin_record_t * evms_plugin_records[] = {
	&LVM_Plugin,
	NULL
};

