/*
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: mdregmgr
 * File: linear_discover.c
 *
 * Description: This file contains all functions related to the initial
 *              discovery of linear MD physical volumes and logical
 *              volumes.
 */ 

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>

#define MY_PLUGIN linear_plugin
#include "md.h"

int linear_create_region(md_volume_t * volume, dlist_t output_list, boolean final_call){
	int rc = 0;
	storage_object_t * region;
	int found = 0;
	int i, j = -1;

	LOG_ENTRY;
       
       	if ((!volume->super_block || (volume->nr_disks !=  volume->super_block->nr_disks)) &&
       	    !final_call) {
       		LOG_DETAILS("Region %s. missing members, delaying discovery\n",volume->name);
       		RETURN(0);
       	}

	LOG_DETAILS("Discovered region %s.\n",volume->name);
	if ((rc = EngFncs->allocate_region(volume->name, &region))){
		for (j = MAX_MD_MINORS -1;(rc != 0) && (j >=0) ; j--) {
			sprintf(volume->name, "md/md%d",j);
			rc = EngFncs->allocate_evms_object(volume->name, &region);
		}
		if (j<0) {
			LOG_ERROR("No more names for MD ");
			RETURN(ENOMEM);
		}
	}
	for (i = 0; (i < MAX_MD_DEVICES) && (found < volume->nr_disks); i++ ) {
		// check for null object, if missing, skip and set corrupt flag
		if (volume->child_object[i]) {
			// if name registration failed and we changed the name, fix up all the minor numbers
			if (j >= 0) {
				volume->super_array[i]->md_minor = j;
			}
			md_append_region_to_object(region, volume->child_object[i]);
			LOG_DETAILS("Adding Object %s.to %s\n",volume->child_object[i]->name,volume->name);
			region->size += MD_NEW_SIZE_SECTORS(volume->child_object[i]->size);
			found++;
		}else {
			MESSAGE("Error building region %s. Missing member object %d\n",volume->name,i);
			volume->flags |= MD_CORRUPT;
			region->flags |= SOFLAG_CORRUPT;
		}


	}
	// no holes in object list, but did not find enough.
	if ((volume->nr_disks != volume->super_block->nr_disks) && !(volume->flags & MD_CORRUPT)) {
		MESSAGE("Error building region %s. Missing object(s)%d thru %d\n",volume->name,i,volume->super_block->nr_disks-1);
		volume->flags |= MD_CORRUPT;
		region->flags |= SOFLAG_CORRUPT;
	}

	region->data_type = DATA_TYPE;
	region->plugin = linear_plugin;
	region->private_data = (void *)volume;
	volume->flags |= MD_DISCOVERED;
	volume->region = region;
	md_add_object_to_list(region, output_list);
	RETURN(rc);
}




/* Function: discover_regions
 *
 *	run the global list of regions and pirce them together.
 */
int linear_discover_regions( dlist_t output_list, int *count, boolean final_call )
{
	int rc = 0;
	md_volume_t * volume = volume_list_head;

	LOG_ENTRY;

	while (volume != NULL) {
		if ((!(volume->flags & MD_DISCOVERED)) && (volume->personality == LINEAR)) {
			rc = linear_create_region(volume, output_list, final_call);
			if (volume->flags & MD_DISCOVERED) {
				*count = *count + 1;
			}
		}
		volume = volume->next;
	}



	RETURN(rc);
}
