/*
 *
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: create.c
 */ 
 
#include <frontend.h>
#include <gtk/gtk.h>

#include "support.h"
#include "create.h"
#include "thing.h"
#include "logging.h"

/*
 *
 *   void on_create_volume_button_clicked (GtkButton *, gchar *, gchar *, gboolean)
 *
 *   Description:
 *      This routine initiates the create volume or create compatibility volume
 *      API call using the handle of the object associated with the last row
 *      selected. It then creates and displays the results popup which destroys
 *      both windows when dismissed.
 * 
 *   Entry:
 *      button        - address of the GtkButton widget
 *      error_msg     - string to use for additional error message on failure
 *      success_msg   - string to use as additional message on success
 *      compat_create - boolean that indicates if we are creating a 
 *                      compatability volume
 *
 *   Exit:
 *      Create volume is invoked and results window is displayed unless error is
 *      recoverable (such as EVMS name is not unique) to allow user to try again.
 *
 */
void on_create_volume_button_clicked (GtkButton *button, gchar *error_msg, gchar *success_msg, 
                                      gboolean compat_create)
{
    GtkCList       *clist;
    object_handle_t handle;
        
    clist = GTK_CLIST (lookup_widget (GTK_WIDGET (button), "selection_window_clist"));

    handle = GPOINTER_TO_UINT (get_single_select_current_row_data (clist));

    if (handle != 0)
    {
        gint rc;

        /*
         * If this is not a compatibility volume create then
         * we need to extract the name from the selection
         * window before proceeding.
         */

        if (compat_create)
            rc = evms_create_compatibility_volume (handle);
        else
        {
            gchar     *name;
            GtkWidget *entry;

            /*
             * Extract the text to use as the volume name from
             * the modified selection window text entry field.
             */
                
            entry = lookup_widget (GTK_WIDGET (button), "selection_window_entry");

            name = gtk_editable_get_chars (GTK_EDITABLE (entry), 0, -1);

            rc = evms_create_volume (handle, g_strstrip (name));

            g_free (name);
        }

        display_selection_window_results (GTK_WIDGET (button), rc, error_msg, success_msg);
    }
}

/*
 *
 *   void on_create_volume_clist_realize (GtkWidget *, gboolean)
 *   
 *   Description:
 *      This routine populates the given GtkCList with the list
 *      of objects that can be used on a create volume operation.
 * 
 *   Entry:
 *      widget        - address of the selections GtkCList widget
 *      compat_create - boolean that indicates if we are creating a 
 *                      compatability volume
 *
 *   Exit:
 *      Selection list populated with acceptable objects
 *
 */
void on_create_volume_clist_realize (GtkWidget *widget, gboolean compat_create)
{
    gint             rc;
    GtkCList        *clist = GTK_CLIST (widget);
    handle_array_t  *objects;

    rc = evms_get_object_list (0, DATA_TYPE, 0, TOPMOST, &objects);

    if (rc != SUCCESS) 
    {
        log_error ("%s: evms_get_object_list() returned error code %d.\n", __FUNCTION__, rc);
    }
    else
    {
        guint    i;
        gboolean is_selected = (objects->count == 1);
        gboolean object_acceptable;

        set_selection_window_clist_column_titles (clist, _("Size"), _("Storage Object"), NULL);
        
        for (i=0; i < objects->count; i++)
        {
            if (compat_create)
                object_acceptable = (evms_can_create_compatibility_volume (objects->handle[i]) == 0);
            else
                object_acceptable = (evms_can_create_volume (objects->handle[i]) == 0);

            if (object_acceptable)
                add_thing_to_selection_list (clist, objects->handle[i], is_selected);
        }

        evms_free (objects);
    }
}
