/*
 *
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: main_window_cb.c
 */ 

#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>

#include <frontend.h>
#include <gtk/gtk.h>

#include "support.h"
#include "about.h"
#include "resize.h"
#include "commit.h"
#include "disk_cb.h"
#include "segment_cb.h"
#include "container_cb.h"
#include "region_cb.h"
#include "object_cb.h"
#include "volume_cb.h"
#include "plugin_cb.h"
#include "views.h"
#include "setinfo.h"
#include "transfer.h"
#include "message.h"
#include "fsutils.h"
#include "help.h"
#include "main_window_cb.h"
#include "logging.h"

#define EVMS_HOWTO_URL "http://sourceforge.net/docman/display_doc.php?docid=8917&group_id=25076"
#define EVMS_FAQ_URL "http://sourceforge.net/docman/display_doc.php?docid=9078&group_id=25076"

/*
 *
 *   void on_create_evms_volume_menu_item_activate (GtkMenuItem *, gpointer)
 *   
 *   Description:
 *      This routine creates the generic selection dialog,
 *      modifies the dialog slightly and connects the 
 *      appropriate signal handlers that will handle the 
 *      actions for creating an EVMS volume.
 * 
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Modified generic selection dialog is displayed and signal 
 *      handlers have been connected for the corresponding actions.
 *
 */
void on_create_evms_volume_menu_item_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    GtkWidget *window;

    window = create_create_evms_volume_window (on_create_evms_volume_clist_realize, NULL);

    gtk_widget_show (window);
}

/*
 *
 *   void on_create_compatibility_volume_menu_item_activate (GtkMenuItem *, gpointer)
 *   
 *   Description:
 *      This routine creates the generic selection dialog
 *      and connects the appropriate signal handlers that will
 *      handle the actions for creating a compatibility volume.
 * 
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Generic selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_create_compatibility_volume_menu_item_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    GtkWidget *window;

    window = create_standard_selection_window (_("Create Compatibility Volume"),
                                               _("Create"),
                                               create_compat_volume_help_text,
                                               on_create_compat_volume_clist_realize,
                                               on_create_compat_volume_button_clicked,
                                               NULL, NULL, NULL, NULL, NULL);

    gtk_widget_show (window);
}

/*
 *
 *   void on_create_segment_menu_item_activate (GtkMenuItem *, gpointer)
 *   
 *   Description:
 *      This routine starts the process of creating a feature object
 *      by first creating the selection window for the plugin desired
 *      to supply the feature. This window will in turn create the task
 *      window once a plugin has been selected and the "Next" button
 *      has been clicked.
 * 
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Plugin selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_create_object_menu_item_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    GtkWidget *window;

    window = create_standard_selection_window (_("Create EVMS Feature Object"),
                                               NULL,
                                               create_feature_help_text,
                                               on_feature_plugin_clist_realize,
                                               on_plugin_selection_button_clicked, 
                                               NULL, NULL, NULL, NULL,
                                               GUINT_TO_POINTER (EVMS_Task_Create));

    gtk_widget_show (window);
}

/*
 *
 *   void on_create_region_menu_item_activate (GtkMenuItem *, gpointer)
 *   
 *   Description:
 *      This routine starts the process of creating a region by first
 *      creating the selection window for the plugin desired to generate
 *      the region. This window will in turn create the task window
 *      once a plugin has been selected and the "Next" button has been
 *      clicked.
 * 
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Plugin selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_create_region_menu_item_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    GtkWidget *window;

    window = create_standard_selection_window (_("Create Storage Region"),
                                               NULL,
                                               create_region_help_text,
                                               on_region_manager_clist_realize,
                                               on_plugin_selection_button_clicked, 
                                               NULL, NULL, NULL, NULL,
                                               GUINT_TO_POINTER (EVMS_Task_Create));

    gtk_widget_show (window);
}

/*
 *
 *   void on_create_container_menu_item_activate (GtkMenuItem *, gpointer)
 *   
 *   Description:
 *      This routine starts the process of creating a container
 *      by first creating the selection window for the plugin desired
 *      to generate the container. This window will in turn create the task
 *      window once a plugin has been selected and the "Next" button
 *      has been clicked.
 * 
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Plugin selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_create_container_menu_item_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    GtkWidget *window;

    window = create_standard_selection_window (_("Create Storage Container"),
                                               NULL,
                                               create_container_help_text,
                                               on_container_plugin_clist_realize,
                                               on_plugin_selection_button_clicked, 
                                               NULL, NULL, NULL, NULL,
                                               GUINT_TO_POINTER (EVMS_Task_Create_Container));

    gtk_widget_show (window);
}

/*
 *
 *   void on_create_segment_menu_item_activate (GtkMenuItem *, gpointer)
 *   
 *   Description:
 *      This routine starts the process of creating a segment
 *      by first creating the selection window for the plugin desired
 *      to manage a segment. This window will in turn create the task
 *      window once a plugin has been selected and the "Next" button
 *      has been clicked.
 * 
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Plugin selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_create_segment_menu_item_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    GtkWidget *window;

    window = create_standard_selection_window (_("Create Disk Segment"),
                                               NULL,
                                               create_segment_help_text,
                                               on_segment_manager_clist_realize,
                                               on_plugin_selection_button_clicked, 
                                               NULL, NULL, NULL, NULL,
                                               GUINT_TO_POINTER (EVMS_Task_Create));

    gtk_widget_show (window);
}

/*
 *
 *   void on_destroy_volume_menu_item_activate (GtkMenuItem *, gpointer)
 *   
 *   Description:
 *      This routine creates the generic selection dialog
 *      and connects the appropriate signal handlers that will
 *      handle the actions for destroying a volume.
 * 
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Generic selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_destroy_volume_menu_item_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    GtkWidget *window;

    window = create_standard_selection_window (_("Destroy Logical Volume"),
                                               _("Destroy"),
                                               destroy_volume_help_text,
                                               on_destroy_volume_clist_realize,
                                               on_destroy_volume_button_clicked, 
                                               NULL, NULL, NULL, NULL, NULL);

    gtk_widget_show (window);
}

/*
 *
 *   void on_destroy_object_menu_item_activate (GtkMenuItem *, gpointer)
 *   
 *   Description:
 *      This routine creates the generic selection dialog
 *      and connects the appropriate signal handlers that will
 *      handle the actions for destroying a storage object.
 * 
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Generic selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_destroy_object_menu_item_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    GtkWidget *window;

    window = create_standard_selection_window (_("Destroy Feature Object"),
                                               _("Destroy"),
                                               destroy_object_help_text,
                                               on_destroy_object_clist_realize,
                                               on_destroy_object_button_clicked, 
                                               NULL, NULL, NULL, NULL, NULL);

    gtk_widget_show (window);
}

/*
 *
 *   void on_destroy_region_menu_item_activate (GtkMenuItem *, gpointer)
 *   
 *   Description:
 *      This routine creates the generic selection dialog
 *      and connects the appropriate signal handlers that will
 *      handle the actions for destroying a storage region.
 * 
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Generic selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_destroy_region_menu_item_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    GtkWidget *window;

    window = create_standard_selection_window (_("Destroy Storage Region"),
                                               _("Destroy"),
                                               destroy_region_help_text,
                                               on_destroy_region_clist_realize,
                                               on_destroy_region_button_clicked, 
                                               NULL, NULL, NULL, NULL, NULL);

    gtk_widget_show (window);
}

/*
 *
 *   void on_destroy_container_menu_item_activate (GtkMenuItem *, gpointer)
 *   
 *   Description:
 *      This routine creates the generic selection dialog
 *      and connects the appropriate signal handlers that will
 *      handle the actions for destroying a storage container.
 * 
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Generic selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_destroy_container_menu_item_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    GtkWidget *window;

    window = create_standard_selection_window (_("Destroy Storage Container"),
                                               _("Destroy"),
                                               destroy_container_help_text,
                                               on_destroy_container_clist_realize,
                                               on_destroy_container_button_clicked, 
                                               NULL, NULL, NULL, NULL, NULL);

    gtk_widget_show (window);
}

/*
 *
 *   void on_destroy_segment_menu_item_activate (GtkMenuItem *, gpointer)
 *   
 *   Description:
 *      This routine creates the generic selection dialog
 *      and connects the appropriate signal handlers that will
 *      handle the actions for destroying a disk segment.
 * 
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Generic selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_destroy_segment_menu_item_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    GtkWidget *window;

    window = create_standard_selection_window (_("Destroy Disk Segment"),
                                               _("Destroy"),
                                               destroy_segment_help_text,
                                               on_destroy_segment_clist_realize,
                                               on_destroy_segment_button_clicked, 
                                               NULL, NULL, NULL, NULL, NULL);

    gtk_widget_show (window);
}

/*
 *
 *   void on_assign_segment_manager_to_storage_object_activate (GtkMenuItem *, gpointer)
 *   
 *   Description:
 *      This routine creates starts the process of assigning a segment
 *      manager to a storage object by first creating the selection window
 *      for the segment manager. This window will in turn create the object
 *      selection window once a plugin has been selected and the "Next" 
 *      button has been clicked. The last panel will be for options (if there
 *      are any).
 * 
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Plugin selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_assign_segment_manager_to_storage_object_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    GtkWidget *window;

    window = create_standard_selection_window (_("Assign Segment Manager to Storage Object"),
                                               NULL,
                                               assign_segment_manager_help_text,
                                               on_segment_manager_clist_realize,
                                               on_plugin_selection_button_clicked, 
                                               NULL, NULL, NULL, NULL,
                                               GUINT_TO_POINTER (EVMS_Task_Assign_Plugin));

    gtk_widget_show (window);
}

/*
 *
 *   void on_expand_volume_menu_item_activate (GtkMenuItem *, gpointer)
 *   
 *   Description:
 *      This routine creates a selection window with volumes that
 *      can be expanded. The user then selects one to be walked through
 *      the process of actually expanding a volume.
 * 
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Expandable volumes selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_expand_volume_menu_item_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    GtkWidget *window;

    window = create_standard_selection_window (_("Expand Logical Volume"),
                                               NULL,
                                               expand_volume_help_text,
                                               on_resize_volume_clist_realize,
                                               on_resize_volume_button_clicked, 
                                               NULL, NULL, NULL, NULL,
                                               GUINT_TO_POINTER (EVMS_Task_Expand));

    gtk_widget_show (window);
}

/*
 *
 *   void on_expand_object_menu_item_activate (GtkMenuItem *, gpointer)
 *   
 *   Description:
 *      This routine creates a selection window with topmost objects that
 *      can be expanded. The user then selects one to be walked through
 *      the process of actually expanding the object.
 * 
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Expandable objects selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_expand_object_menu_item_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    GtkWidget *window;

    window = create_standard_selection_window (_("Expand Storage Object"),
                                               NULL,
                                               expand_object_help_text,
                                               on_resize_object_clist_realize,
                                               on_resize_object_button_clicked, 
                                               NULL, NULL, NULL, NULL,
                                               GUINT_TO_POINTER (EVMS_Task_Expand));

    gtk_widget_show (window);
}

/*
 *
 *   void on_expand_container_menu_item_activate (GtkMenuItem *, gpointer)
 *   
 *   Description:
 *      This routine creates a selection window with containers that
 *      can be expanded. The user then selects one to be walked through
 *      the process of actually expanding the container.
 * 
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Expandable containers selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_expand_container_menu_item_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    GtkWidget *window;

    window = create_standard_selection_window (_("Expand Storage Container"),
                                               NULL,
                                               expand_container_help_text,
                                               on_expand_container_clist_realize,
                                               on_resize_point_selection_button_clicked, 
                                               NULL, NULL, NULL, NULL,
                                               GUINT_TO_POINTER (EVMS_Task_Expand_Container));

    gtk_widget_show (window);
}

/*
 *
 *   void on_shrink_volume_menu_item_activate (GtkMenuItem *, gpointer)
 *   
 *   Description:
 *      This routine creates a selection window with volumes that
 *      can be shrunk. The user then selects one to be walked through
 *      the process of actually shrinking a volume.
 * 
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Shrinkable volumes selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_shrink_volume_menu_item_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    GtkWidget *window;

    window = create_standard_selection_window (_("Shrink Logical Volume"),
                                               NULL,
                                               shrink_volume_help_text,
                                               on_resize_volume_clist_realize,
                                               on_resize_volume_button_clicked, 
                                               NULL, NULL, NULL, NULL,
                                               GUINT_TO_POINTER (EVMS_Task_Shrink));

    gtk_widget_show (window);
}

/*
 *
 *   void on_shrink_object_menu_item_activate (GtkMenuItem *, gpointer)
 *   
 *   Description:
 *      This routine creates a selection window with topmost objects that
 *      can be shrunk. The user then selects one to be walked through
 *      the process of actually shrinking the object.
 * 
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Shrinkable objects selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_shrink_object_menu_item_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    GtkWidget *window;

    window = create_standard_selection_window (_("Shrink Storage Object"),
                                               NULL,
                                               shrink_object_help_text,
                                               on_resize_object_clist_realize,
                                               on_resize_object_button_clicked, 
                                               NULL, NULL, NULL, NULL,
                                               GUINT_TO_POINTER (EVMS_Task_Shrink));

    gtk_widget_show (window);
}

/*
 *
 *   void on_remove_volume_menu_item_activate (GtkMenuItem *, gpointer)
 *   
 *   Description:
 *      This routine creates the generic selection dialog
 *      and connects the appropriate signal handlers that will
 *      handle the actions for reverting a volume.
 * 
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Generic selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_remove_volume_menu_item_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    GtkWidget *window;

    window = create_standard_selection_window (_("Remove Volume from Storage Object"),
                                               _("Remove"),
                                               remove_volume_help_text,
                                               on_revert_volume_clist_realize,
                                               on_revert_volume_button_clicked,
                                               NULL, NULL, NULL, NULL, NULL);

    gtk_widget_show (window);
}

/*
 *
 *   void on_remove_feature_menu_item_activate (GtkMenuItem *, gpointer)
 *   
 *   Description:
 *      This routine creates the generic selection dialog
 *      and connects the appropriate signal handlers that will
 *      handle the actions for removing a feature object from
 *      the top of a storage object stack.
 * 
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Generic selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_remove_feature_menu_item_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    GtkWidget *window;

    window = create_standard_selection_window (_("Remove Feature from Storage Object"),
                                               _("Remove"),
                                               remove_feature_help_text,
                                               on_revert_object_clist_realize,
                                               on_revert_object_button_clicked,
                                               NULL, NULL, NULL, NULL, NULL);

    gtk_widget_show (window);
}

/*
 *
 *   void on_remove_region_menu_item_activate (GtkMenuItem *, gpointer)
 *   
 *   Description:
 *      This routine creates the generic selection dialog
 *      and connects the appropriate signal handlers that will
 *      handle the actions for removing a storage region from
 *      the top of a storage object stack.
 * 
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Generic selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_remove_region_menu_item_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    GtkWidget *window;

    window = create_standard_selection_window (_("Remove Region from Storage Object"),
                                               _("Remove"),
                                               remove_region_help_text,
                                               on_remove_region_clist_realize,
                                               on_remove_region_button_clicked,
                                               NULL, NULL, NULL, NULL, NULL);

    gtk_widget_show (window);
}

/*
 *
 *   void on_remove_segment_menu_item_activate (GtkMenuItem *, gpointer)
 *   
 *   Description:
 *      This routine creates the generic selection dialog
 *      and connects the appropriate signal handlers that will
 *      handle the actions for removing a disk segment from
 *      the top of a storage object stack.
 * 
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Generic selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_remove_segment_menu_item_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    GtkWidget *window;

    window = create_standard_selection_window (_("Remove Segment from Storage Object"),
                                               _("Remove"),
                                               remove_segment_help_text,
                                               on_remove_segment_clist_realize,
                                               on_remove_segment_button_clicked,
                                               NULL, NULL, NULL, NULL, NULL);

    gtk_widget_show (window);
}

/*
 *
 *   void on_remove_region_from_container_menu_item_activate (GtkMenuItem *, gpointer)
 *   
 *   Description:
 *      This routine creates the generic selection dialog
 *      and connects the appropriate signal handlers that will
 *      handle the actions for transferring a region out from
 *      the container consuming it.
 * 
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Generic selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_remove_region_from_container_menu_item_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    GtkWidget *window;

    window = create_standard_selection_window (_("Remove Region from Consuming Container"),
                                               _("Remove"),
                                               remove_region_from_container_help_text,
                                               on_remove_region_from_container_clist_realize,
                                               on_remove_thing_from_container_button_clicked,
                                               NULL, NULL, NULL, NULL, NULL);

    gtk_widget_show (window);
}

/*
 *
 *   void on_remove_segment_from_container_menu_item_activate (GtkMenuItem *, gpointer)
 *   
 *   Description:
 *      This routine creates the generic selection dialog
 *      and connects the appropriate signal handlers that will
 *      handle the actions for transferring a segment out from
 *      the container consuming it.
 * 
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Generic selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_remove_segment_from_container_menu_item_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    GtkWidget *window;

    window = create_standard_selection_window (_("Remove Segment from Consuming Container"),
                                               _("Remove"),
                                               remove_segment_from_container_help_text,
                                               on_remove_segment_from_container_clist_realize,
                                               on_remove_thing_from_container_button_clicked,
                                               NULL, NULL, NULL, NULL, NULL);

    gtk_widget_show (window);    
}

/*
 *
 *   void on_remove_disk_from_container_menu_item_activate (GtkMenuItem *, gpointer)
 *   
 *   Description:
 *      This routine creates the generic selection dialog
 *      and connects the appropriate signal handlers that will
 *      handle the actions for transferring a disk out from
 *      the container consuming it.
 * 
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Generic selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_remove_disk_from_container_menu_item_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    GtkWidget *window;

    window = create_standard_selection_window (_("Remove Disk from Consuming Container"),
                                               _("Remove"),
                                               remove_disk_from_container_help_text,
                                               on_remove_disk_from_container_clist_realize,
                                               on_remove_thing_from_container_button_clicked,
                                               NULL, NULL, NULL, NULL, NULL);

    gtk_widget_show (window);
}

/*
 *
 *   void on_modify_volume_menu_item_activate (GtkMenuItem *, gpointer)
 *   
 *   Description:
 *      This routine creates the generic selection dialog
 *      and connects the appropriate signal handlers that will
 *      handle the actions for modifying a volume's properties.
 * 
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Generic selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_modify_volume_menu_item_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    GtkWidget *window;

    window = create_standard_selection_window (_("Modify Logical Volume Properties"),
                                               NULL,
                                               modify_volume_properties_help_text,
                                               on_modify_volume_clist_realize,
                                               on_modify_volume_selection_button_clicked,
                                               NULL, NULL, NULL, NULL, NULL);

    gtk_widget_show (window);    
}

/*
 *
 *   void on_modify_object_menu_item_activate (GtkMenuItem *, gpointer)
 *   
 *   Description:
 *      This routine creates the generic selection dialog
 *      and connects the appropriate signal handlers that will
 *      handle the actions for modifying a feature object's properties.
 * 
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Generic selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_modify_object_menu_item_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    GtkWidget *window;

    window = create_standard_selection_window (_("Modify Feature Object Properties"),
                                               NULL,
                                               modify_object_properties_help_text,
                                               on_set_info_clist_realize,
                                               on_set_info_button_clicked,
                                               NULL, NULL, NULL, NULL, 
                                               GUINT_TO_POINTER (EVMS_OBJECT));

    gtk_widget_show (window);    
}

/*
 *
 *   void on_modify_region_menu_item_activate (GtkMenuItem *, gpointer)
 *   
 *   Description:
 *      This routine creates the generic selection dialog
 *      and connects the appropriate signal handlers that will
 *      handle the actions for modifying a region's properties.
 * 
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Generic selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_modify_region_menu_item_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    GtkWidget *window;

    window = create_standard_selection_window (_("Modify Storage Region Properties"),
                                               NULL,
                                               modify_region_properties_help_text,
                                               on_set_info_clist_realize,
                                               on_set_info_button_clicked,
                                               NULL, NULL, NULL, NULL, 
                                               GUINT_TO_POINTER (REGION));

    gtk_widget_show (window);    
}

/*
 *
 *   void on_modify_container_menu_item_activate (GtkMenuItem *, gpointer)
 *   
 *   Description:
 *      This routine creates the generic selection dialog
 *      and connects the appropriate signal handlers that will
 *      handle the actions for modifying a container's properties.
 * 
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Generic selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_modify_container_menu_item_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    GtkWidget *window;

    window = create_standard_selection_window (_("Modify Storage Container Properties"),
                                               NULL,
                                               modify_container_properties_help_text,
                                               on_set_info_clist_realize,
                                               on_set_info_button_clicked,
                                               NULL, NULL, NULL, NULL, 
                                               GUINT_TO_POINTER (CONTAINER));

    gtk_widget_show (window);
}

/*
 *
 *   void on_modify_segment_menu_item_activate (GtkMenuItem *, gpointer)
 *   
 *   Description:
 *      This routine creates the generic selection dialog
 *      and connects the appropriate signal handlers that will
 *      handle the actions for modifying a segment's properties.
 * 
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Generic selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_modify_segment_menu_item_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    GtkWidget *window;

    window = create_standard_selection_window (_("Modify Disk Segment Properties"),
                                               NULL,
                                               modify_segment_properties_help_text,
                                               on_set_info_clist_realize,
                                               on_set_info_button_clicked,
                                               NULL, NULL, NULL, NULL, 
                                               GUINT_TO_POINTER (SEGMENT));

    gtk_widget_show (window);
}

/*
 *
 *   void on_modify_disk_menu_item_activate (GtkMenuItem *, gpointer)
 *   
 *   Description:
 *      This routine creates the generic selection dialog
 *      and connects the appropriate signal handlers that will
 *      handle the actions for modifying a volume's properties.
 * 
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Generic selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_modify_disk_menu_item_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    GtkWidget *window;

    window = create_standard_selection_window (_("Modify Logical Disk Properties"),
                                               NULL,
                                               modify_disk_properties_help_text,
                                               on_set_info_clist_realize,
                                               on_set_info_button_clicked,
                                               NULL, NULL, NULL, NULL, 
                                               GUINT_TO_POINTER (DISK));

    gtk_widget_show (window);
}

/*
 *
 *   void on_view_messages_menu_item_activate (GtkMenuItem *, gpointer)
 *   
 *   Description:
 *      This routine calls the routine that attempts to redisplay 
 *      the informational message window if it exists.
 * 
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Informational window is redisplayed if it exists.
 *
 */
void on_view_messages_menu_item_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    redisplay_info_messages_window ();
}

/*
 *
 *   void on_view_log_menu_item_activate (GtkMenuItem *, gpointer)
 *   
 *   Description:
 *      This routine initiates the display of the log entries
 *      in a new window.
 * 
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      A window containing the current log entries is displayed.
 *
 */
void on_view_log_menu_item_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    display_not_implemented_yet_popup ();    
}

/*
 *
 *   void on_mkfs_menu_item_activate (GtkMenuItem *, gpointer)
 *   
 *   Description:
 *      This routine creates the generic selection dialog
 *      and connects the appropriate signal handlers that will
 *      handle the actions for making a file system on a volume.
 * 
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Generic selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_mkfs_menu_item_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    GtkWidget *window;

    window = create_standard_selection_window (_("Make File System"),
                                               NULL,
                                               mkfs_help_text,
                                               on_fsim_clist_realize,
                                               on_plugin_selection_button_clicked,
                                               NULL, NULL, NULL, NULL,
                                               GUINT_TO_POINTER (EVMS_Task_mkfs));

    gtk_widget_show (window);
}

/*
 *
 *   void on_fsck_menu_item_activate (GtkMenuItem *, gpointer)
 *   
 *   Description:
 *      This routine creates the generic selection dialog
 *      and connects the appropriate signal handlers that will
 *      handle the actions for checking/repairing a file system
 *      on a volume.
 * 
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Generic selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_fsck_menu_item_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    GtkWidget *window;

    window = create_standard_selection_window (_("Check/Repair File System"),
                                               NULL,
                                               fsck_help_text,
                                               on_filesys_operation_clist_realize,
                                               on_initiate_filesys_operation_button_clicked,
                                               NULL, NULL, NULL, NULL, 
                                               GUINT_TO_POINTER (FSCK));

    gtk_widget_show (window);
}

/*
 *
 *   void on_defrag_menu_item_activate (GtkMenuItem *, gpointer)
 *   
 *   Description:
 *      This routine creates the generic selection dialog
 *      and connects the appropriate signal handlers that will
 *      handle the actions for defragmenting a filesystem located
 *      on a volume.
 * 
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Generic selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_defrag_menu_item_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    GtkWidget *window;

    window = create_standard_selection_window (_("Defragment File System"),
                                               NULL,
                                               defrag_help_text,
                                               on_filesys_operation_clist_realize,
                                               on_initiate_filesys_operation_button_clicked,
                                               NULL, NULL, NULL, NULL, 
                                               GUINT_TO_POINTER (DEFRAG));

    gtk_widget_show (window);
}

/*
 *
 *   void on_obliterate_filesystem_menu_item_activate (GtkMenuItem *, gpointer)
 *   
 *   Description:
 *      This routine creates the generic selection dialog
 *      and connects the appropriate signal handlers that will
 *      handle the actions for remove a file system on volume.
 * 
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Generic selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_obliterate_filesystem_menu_item_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    GtkWidget *window;

    window = create_standard_selection_window (_("Remove File System"),
                                               _("Remove"),
                                               unmkfs_help_text,
                                               on_filesys_operation_clist_realize,
                                               on_remove_filesys_button_clicked,
                                               NULL, NULL, NULL, NULL, 
                                               GUINT_TO_POINTER (UNMKFS));

    gtk_widget_show (window);
}

/*
 *
 *   void on_commit_menu_item_activate (GtkMenuItem *, gpointer)
 *   
 *   Description:
 *      This routine call the function that displays a dialog
 *      confirming that the user wants to commit changes or
 *      not. If so, the dialog will initiate the commit process.
 * 
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      If changes are pending, a confirmation dialog is displayed
 *      otherwise a popup is displayed indicating nothing to be done.
 *
 */
void on_commit_menu_item_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    if (evms_changes_pending ())
    {
        display_commit_changes_confirmation_dialog ();
    }
    else
    {
        display_nothing_to_commit_popup ();
    }
}

/*
 *
 *   void on_exit_menu_item_activate (GtkMenuItem *, gpointer)
 *   
 *   Description:
 *      This routine checks for changes pending prior to 
 *      closing the main window. If changes are pending
 *      then a popup is presented to the user to ask
 *      whether they want to continue exiting or commit
 *      changes then exit.
 * 
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      We present a popup if changes are pending or simply
 *      destroy the main window if none.
 *
 */
void on_exit_menu_item_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    if (evms_changes_pending ())
    {
        display_commit_changes_popup ();
    }
    else
    {
        gtk_widget_destroy (get_main_window_id ());
    }
}

/*
 *
 *   gboolean on_main_window_delete_event (GtkWidget *, GdkEvent *, gpointer)
 *   
 *   Description:
 *      This routine checks for changes pending prior to 
 *      closing the main window. If changes are pending
 *      and a commit is currently not already in progress
 *      then a popup is presented to the user to ask
 *      whether they want to continue exiting or commit
 *      changes then exit.
 * 
 *   Entry:
 *      widget    - widget that caused delete-event to occur
 *      event     - everything we would like to know about event
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      We present a popup if changes are pending or simply
 *      allow closure of the main window if none.
 *
 */
gboolean on_main_window_delete_event (GtkWidget *widget, GdkEvent *event, gpointer user_data)
{
    gboolean prevent_closure=FALSE;

    if (evms_changes_pending ())
    {
        if (is_commit_in_progress () == FALSE)
            display_commit_changes_popup ();

        prevent_closure = TRUE;
    }

    return prevent_closure;
}

/* BUGBUG (not implemented yet) */
void on_beginner_menu_item_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    display_not_implemented_yet_popup ();
}

/* BUGBUG (not implemented yet) */
void on_experienced_menu_item_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    display_not_implemented_yet_popup ();
}

/*
 *
 *   void on_faq_menu_item_activate (GtkMenuItem *, gpointer)
 *   
 *   Description:
 *      This routine currently invokes either the mozilla or 
 *      netscape browser to display additional the EVMS FAQ
 *      page located in the docs for the EVMS project page.
 * 
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      We try to fork a child that will exec either the mozilla or
 *      netscape browser to display the EVMS FAQ page.
 *
 */
void on_faq_menu_item_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    pid_t pid=0;
    
    pid = fork ();
    
    if (pid == 0)
    {
        execl ("/usr/bin/mozilla",  "/usr/bin/mozilla",  EVMS_FAQ_URL, NULL);
        execl ("/usr/bin/netscape", "/usr/bin/netscape", EVMS_FAQ_URL, NULL);
        execl ("/usr/local/bin/mozilla",  "/usr/local/bin/mozilla",  EVMS_FAQ_URL, NULL);
        execl ("/usr/local/bin/netscape", "/usr/local/bin/netscape", EVMS_FAQ_URL, NULL);
        
        /*
         * Unable to exec either mozilla or netscape. Use the _exit() rather than
         * exit() to avoid Xlib errors and the gui going unresponsive.
         */
        
        _exit (-1);
    }
    else if (pid == -1)
    {
        log_error ("%s: fork() call to display user's guide failed.\n", __FUNCTION__);
    }
}

/*
 *
 *   void on_user_guide_menu_item_activate (GtkMenuItem *, gpointer)
 *   
 *   Description:
 *      This routine currently invokes either the mozilla or 
 *      netscape browser to display additional general EVMS 
 *      information contained in the EVMS HOW-TO page located
 *      in the docs for the EVMS project page.
 * 
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      We try to fork a child that will exec either the mozilla or
 *      netscape browser to display the EVMS HOW-TO page.
 *
 */
void on_user_guide_menu_item_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    pid_t pid=0;
    
    pid = fork ();
    
    if (pid == 0)
    {
        execl ("/usr/bin/mozilla",  "/usr/bin/mozilla",  EVMS_HOWTO_URL, NULL);
        execl ("/usr/bin/netscape", "/usr/bin/netscape", EVMS_HOWTO_URL, NULL);
        execl ("/usr/local/bin/mozilla",  "/usr/local/bin/mozilla",  EVMS_HOWTO_URL, NULL);
        execl ("/usr/local/bin/netscape", "/usr/local/bin/netscape", EVMS_HOWTO_URL, NULL);
        
        /*
         * Unable to exec either mozilla or netscape. Use the _exit() rather than
         * exit() to avoid Xlib errors and the gui going unresponsive.
         */
        
        _exit (-1);
    }
    else if (pid == -1)
    {
        log_error ("%s: fork() call to display user's guide failed.\n", __FUNCTION__);
    }
}

/*
 *
 *   void on_about_menu_item_activate (GtkMenuItem *, gpointer)
 *   
 *   Description:
 *      This routine creates and displays the about box.
 * 
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      About box is displayed and waiting for dismissal.
 *
 */
void on_about_menu_item_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    GtkWidget * about_window;

    about_window = create_about_window ();

    gtk_widget_show (about_window);
}

/*
 *
 *   void on_browser_notebook_realize (GtkWidget *, gpointer)
 *   
 *   Description:
 *      This routine makes some modifications to the browser
 *      notebook before it gets displayed. These modifications
 *      are minor and are things not easily done through Glade.
 * 
 *   Entry:
 *      widget    - address of the GtkNotebook widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Notebook tab icons added, tooltips added, etc.
 *
 */
void on_browser_notebook_realize (GtkWidget *widget, gpointer user_data)
{
    /*
     * Given the GtkWidget of the GtkNotebook, do child
     * widget lookups to find the GtkHBox in each tab
     * to create the corresponding GtkPixmap from the
     * GdkPixmap already allocated at init time.
     */

    set_browser_notebook_tab_pixmap (widget, "volumes_view_tab_hbox", VOLUME, FALSE);
    set_browser_notebook_tab_pixmap (widget, "topmost_view_tab_hbox", EVMS_OBJECT, TRUE);
    set_browser_notebook_tab_pixmap (widget, "feature_objects_view_tab_hbox", EVMS_OBJECT, FALSE);    
    set_browser_notebook_tab_pixmap (widget, "regions_view_tab_hbox", REGION, FALSE);
    set_browser_notebook_tab_pixmap (widget, "containers_view_tab_hbox", CONTAINER, FALSE);
    set_browser_notebook_tab_pixmap (widget, "segments_view_tab_hbox", SEGMENT, FALSE);
    set_browser_notebook_tab_pixmap (widget, "disks_view_tab_hbox", DISK, FALSE);
    set_browser_notebook_tab_pixmap (widget, "plugins_view_tab_hbox", PLUGIN, FALSE);
}

/*
 *
 *   void on_plugin_columned_list_realize (GtkWidget *, gpointer)
 *   
 *   Description:
 *      This routine populates the given GtkCList with the list
 *      of plugins installed in the system.
 * 
 *   Entry:
 *      widget    - address of the plugins GtkCList widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Plugins list is populated
 *
 */
void on_plugin_columned_list_realize (GtkWidget *widget, gpointer user_data)
{
    populate_plugin_view_with_plugins (GTK_CLIST (widget));
}

/*
 *
 *   void on_view_volumes_ctree_realize (GtkWidget *, gpointer)
 *
 *   Description:
 *      This routine initializes the hiearchy found in the
 *      GtkCTree for the volumes view. 
 *
 *      Below each volume will be the tree of objects that comprise
 *      it.
 * 
 *   Entry:
 *      widget    - the address of the volumes view GtkCTree widget
 *      user_data - address of user data bound to the widget (not used)
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void on_view_volumes_ctree_realize (GtkWidget *widget, gpointer user_data)
{
    GtkCTree *ctree = GTK_CTREE (widget);        

    initialize_view (ctree, VOLUME_VIEW);
    populate_volume_view_ctree (ctree, NULL);    
}

/*
 *
 *   void on_view_objects_ctree_realize (GtkWidget *, gpointer)
 *
 *   Description:
 *      This routine initializes the hiearchy found in the
 *      GtkCTree for the objects view. The topmost objects
 *      are at the root of the hiearchy. They in turn contain
 *      the tree of storage objects that comprise them below them.
 * 
 *   Entry:
 *      widget    - the address of the topmost objects view GtkCTree widget
 *      user_data - address of user data bound to the widget (not used)
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void on_view_objects_ctree_realize (GtkWidget *widget, gpointer user_data)
{
    GtkCTree *ctree = GTK_CTREE (widget);
        
    initialize_view (ctree, OBJECT_VIEW);
    populate_object_view_ctree (ctree, NULL);
}

/*
 *
 *   void on_view_feature_objects_ctree_realize (GtkWidget *, gpointer)
 *
 *   Description:
 *      This routine initializes the hiearchy found in the
 *      GtkCTree for the feature objects view. The feature objects
 *      are at the root of the hiearchy. They in turn contain
 *      the tree of storage objects that comprise them below them.
 * 
 *   Entry:
 *      widget    - the address of the feature objects view GtkCTree widget
 *      user_data - address of user data bound to the widget (not used)
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void on_view_feature_objects_ctree_realize (GtkWidget *widget, gpointer user_data)
{
    GtkCTree *ctree = GTK_CTREE (widget);
        
    initialize_view (ctree, FEATURE_VIEW);
    populate_feature_object_view_ctree (ctree, NULL);
    
    gtk_clist_set_column_visibility (GTK_CLIST (ctree), OV_BIOS_READABLE_COLUMN, FALSE);    
}
/*
 *
 *   void on_view_regions_ctree_realize (GtkWidget *, gpointer)
 *
 *   Description:
 *      This routine initializes the hiearchy found in the
 *      GtkCTree for the regions view. The region managers
 *      are at the root of the hierachy. If the region manager
 *      supports containers, they are showed underneath with
 *      the regions below them. If the region manager does not
 *      support containers then all its regions are displayed
 *      together underneath the plugin.
 * 
 *   Entry:
 *      widget    - the address of the regions view GtkCTree widget
 *      user_data - address of user data bound to the widget (not used)
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void on_view_regions_ctree_realize (GtkWidget *widget, gpointer user_data)
{
    GtkCTree *ctree = GTK_CTREE (widget);

    initialize_view (ctree, REGION_VIEW);
    populate_region_view_ctree (ctree, NULL);
}

/*
 *
 *   void on_view_containers_ctree_realize (GtkWidget *, gpointer)
 *
 *   Description:
 *      This routine initializes the hiearchy found in the
 *      GtkCTree for the containers view. The plugins that support
 *      containers are at the root of the hierachy. They contain
 *      storage containers that they export objects from. Underneath
 *      containers is the list of objects they produce and objects
 *      they consume.
 * 
 *   Entry:
 *      widget    - the address of the containers view GtkCTree widget
 *      user_data - address of user data bound to the widget (not used)
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void on_view_containers_ctree_realize (GtkWidget *widget, gpointer user_data)
{
    GtkCTree *ctree = GTK_CTREE (widget);
    
    initialize_view (ctree, CONTAINER_VIEW);
    populate_container_view_ctree (ctree, NULL);
}

/*
 *
 *   void on_view_segments_ctree_realize (GtkWidget *, gpointer)
 *
 *   Description:
 *      This routine initializes the hiearchy found in the
 *      GtkCTree for the segments view. The segment managers
 *      are at the root of the hierachy. They contain the
 *      segments they manage and under each segment are the
 *      objects that comprise them.
 * 
 *   Entry:
 *      widget    - the address of the segment view GtkCTree widget
 *      user_data - address of user data bound to the widget
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void on_view_segments_ctree_realize (GtkWidget *widget, gpointer user_data)
{
    GtkCTree *ctree = GTK_CTREE (widget);
        
    initialize_view (ctree, SEGMENT_VIEW);
    populate_segment_view_ctree (ctree, NULL);
}

/*
 *
 *   void on_view_disks_ctree_realize (GtkWidget *, gpointer)
 *
 *   Description:
 *      This routine initializes the hiearchy found in the
 *      GtkCTree for the logical disk view. The device managers
 *      are at the root of the hierachy. They contain logical
 *      disks that they export. Under the disks is a list of
 *      parent objects such as a segment or volume.
 * 
 *   Entry:
 *      widget    - the address of the disk view GtkCTree widget
 *      user_data - address of user data bound to the widget
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void on_view_disks_ctree_realize (GtkWidget *widget, gpointer user_data)
{
    GtkCTree *ctree = GTK_CTREE (widget);
    
    initialize_view (ctree, DISK_VIEW);
    populate_disk_view_ctree (ctree, NULL);
}

/*
 *
 *   gboolean on_view_button_press_event (GtkWidget *, GdkEventButton *, gpointer)
 *
 *   Description:
 *      This routine gets invoked when there is a "button-press-event" in any
 *      of the information viewer GtkCTree or GtkCList views. We then determine
 *      if it was a right mouse button that was pressed and if over a valid row
 *      of data. If so, we call the routine that displays a context popup of 
 *      actions that can be done with the row data.
 * 
 *   Entry:
 *      widget    - the GtkCTree or GtkCList that generated the button pressed event
 *      event     - everything we want to know about the event
 *      user_data - address of user data bound to the widget
 *
 *   Exit:
 *      Creates a context sensitive popup menu populated with
 *      actions suitable for the row data the user clicked on.
 *
 */
gboolean on_view_button_press_event (GtkWidget *widget, GdkEventButton *event, gpointer user_data)
{
    gint      row;
    gint      col;
    gint      selection_in_range;
    GtkCList *clist = GTK_CLIST (widget);

    selection_in_range = gtk_clist_get_selection_info (clist, (gint) event->x, 
                                                       (gint) event->y, &row, &col);
    
    if (selection_in_range && event->button == RIGHT_MOUSE_BUTTON)
    {
        object_handle_t handle;
            
        handle = GPOINTER_TO_UINT (gtk_clist_get_row_data (clist,row));

        gtk_clist_select_row (clist, row, col);

        display_action_popup_menu_for_handle (handle, event);
    }

    return TRUE;
}

/*
 *
 *   gboolean on_refresh_pixmap_press_event (GtkWidget *, GdkEventButton *, gpointer)
 *
 *   Description:
 *      This routine gets invoked when there is a "button-press-event" in the
 *      eventbox for the refresh view pixmap. If the left mouse button was pressed
 *      we refresh the current view. If the right mouse button was pressed, we
 *      present a popup menu with refresh view choices.
 * 
 *   Entry:
 *      widget    - the widget that generated the button pressed event
 *      event     - everything we want to know about the event
 *      user_data - address of user data bound to the widget
 *
 *   Exit:
 *      Either refreshes the current view or offers a popup menu
 *      with refresh view choices.
 *
 */
gboolean on_refresh_pixmap_eventbox_button_press_event (GtkWidget *widget, GdkEventButton *event, 
                                                        gpointer user_data)
{    
    if (event->button == LEFT_MOUSE_BUTTON)
    {
        refresh_current_view ();
    }
    else if (event->button == RIGHT_MOUSE_BUTTON)
    {
        display_refresh_view_popup_menu (event);
    }

    return TRUE;
}
