/*
 *
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 * Module: message.c
 */ 
 
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <time.h>

#include <frontend.h>
#include <gtk/gtk.h>

#include "support.h"
#include "message.h"
#include "pixmap.h"
#include "main.h"

static GtkWidget *message_window = NULL;

/*
 *
 *   void on_message_combo_selection_changed (GtkEditable *, message_args_t *)
 *
 *   Description:
 *      This routine is invoked when a selection in the combo
 *      box for a multiple choice message box has been changed.
 *      We simply determine the index of the new selection from
 *      the choices array and set the answer index appropriately.
 * 
 *   Entry:
 *      editable - the text entry widget of the combo box
 *      args     - address of message_args_t struct 
 *      
 *   Exit:
 *      Updates the answer index to the index of the current selection.
 *
 */
void on_message_combo_selection_changed (GtkEditable *editable, message_args_t *args)
{
    gint    index;
    gchar **choices;
    gchar  *entry_text;
    
    index = 0;
    choices = args->choices;
    
    entry_text = gtk_editable_get_chars (editable, 0, -1);
    
    while (*choices)
    {
        if (g_strcasecmp (*choices, entry_text) == 0)
        {
            *(args->answer) = index;
            break;
        }
        else
        {    
            index++;
            choices++;
        }
    }
    
    g_free (entry_text);
}

/*
 *
 *   void on_message_window_destroy (GtkObject *, message_args_t *)
 *
 *   Description:
 *      This routine is invoked when a message window is
 *      being destroyed. We either signal a blocking thread
 *      to wakeup or exit an inner event loop.
 * 
 *   Entry:
 *      window - the object that received the destroy signal
 *      args   - address of message_args_t struct 
 *      
 *   Exit:
 *      Signals a wakeup on any thread waiting for our destruction
 *      or exits an inner or main event loop.
 *
 */
void on_message_window_destroy (GtkObject *window, message_args_t *args)
{    
    if (args->wakeup_cond)
        g_cond_signal (args->wakeup_cond);
    else if (args->hide_window == FALSE)
        gtk_main_quit ();
    
    /*
     * Handle cleanup of information only message window
     */
    
    if (args->answer == NULL)
    {    
        message_window = NULL;
        g_free (args);
    }    
}

/*
 *
 *   gboolean on_message_window_delete_event (GtkWidget *, message_args_t *args)
 *
 *   Description:
 *      This routine is invoked when a message window 
 *      received the delete event from the window
 *      manager. We ignore the delete event and hide
 *      instead if the hide window boolean in the 
 *      args is TRUE.
 * 
 *   Entry:
 *      window - the window that received the delete-event signal
 *      args   - address of message_args_t struct 
 *      
 *
 *   Exit:
 *      Allows a destroy of the window or disallows the destroy
 *      aignal and simply hides the window.
 *
 */
gboolean on_message_window_delete_event (GtkWidget *window, GdkEvent *event, message_args_t *args)
{
    gboolean prevent_window_destruction=FALSE;
    
    if (args->hide_window)
    {
        gtk_widget_hide (window);
        prevent_window_destruction = TRUE;
    }
    
    return prevent_window_destruction;
}

/*
 *
 *   void on_message_choice_button_clicked (GtkButton *, message_args_t *args)
 *
 *   Description:
 *      This routine sets the answer to the index of the
 *      choice indicate by the button. It also destroys
 *      the message window. The destroy signal handler
 *      for the message window takes care of either
 *      signaling any blocked thread that we are done
 *      or simply calls gtk_main_quit () to exit any
 *      nested GTK event loop.
 * 
 *   Entry:
 *      button - address of the GtkButton widget
 *      args   - address of message_args_t struct 
 *
 *   Exit:
 *      Sets answer and destroys parent window.
 *
 */
void on_message_choice_button_clicked (GtkButton *button, message_args_t *args)
{    
    *(args->answer) = GPOINTER_TO_INT (gtk_object_get_user_data (GTK_OBJECT (button)));
    
    on_button_clicked_destroy_window (button, NULL);
}

/*
 *
 *   void on_message_clear_button_clicked (GtkButton *, GtkText *)
 *
 *   Description:
 *      This routine 
 * 
 *   Entry:
 *      button - address of the GtkButton widget
 *      text   - address of the GtkText widget
 *
 *   Exit:
 *      Clears the text in the message window.
 *
 */
void on_message_clear_button_clicked (GtkButton *button, GtkText *text)
{
    gtk_text_backward_delete (text, gtk_text_get_length (text));
}

/*
 *
 *   void connect_info_ok_button_clicked_handler (GtkButton *, message_args_t *)
 *
 *   Description:
 *      This routine connects either gtk_widget_hide or gtk_widget_destroy
 *      as the signal handler for the "clicked" signal for the OK button
 *      located on the informational message window. It decides which to
 *      use depending on the hide_window argument.
 * 
 *   Entry:
 *      ok_button - address of info message OK button
 *      args      - address of message_args_t struct
 *
 *   Exit:
 *      Sets signal handler appropriately.
 *
 */
void connect_info_ok_button_clicked_handler (GtkButton *ok_button, message_args_t *args)
{
    GtkSignalFunc ok_button_clicked_handler;
    
    if (args->hide_window)
        ok_button_clicked_handler = gtk_widget_hide;
    else
        ok_button_clicked_handler = gtk_widget_destroy;
    
    gtk_signal_connect_object (GTK_OBJECT (ok_button), "clicked", 
                               ok_button_clicked_handler,
                               GTK_OBJECT (message_window));    
}

/*
 *
 *   gboolean is_choice_button_window_candidate (message_args_t *)
 *
 *   Description:
 *      This routine attempts to determine if the choices would
 *      work as part of a simple set of buttons message window. The
 *      criteria is that there are no more than MAX_MESSAGE_BUTTONS
 *      choices and the text length is short enough to fit in the buttons.
 * 
 *   Entry:
 *      args - address of message_args_t struct
 *
 *   Exit:
 *      Return TRUE if this choices will work in a button
 *      message window otherwise returns FALSE.
 *
 */
gboolean is_choice_button_window_candidate (message_args_t *args)
{
    gint       count=0;
    gchar    **choices;    
    gboolean   is_candidate=FALSE;
    
    choices = args->choices;    
    
    /* 
     * See if count exceeds our max
     */
    
    while (*choices && count <= MAX_MESSAGE_BUTTONS)
    {
        count++;
        choices++;
    }
    
    /*
     * If count does not exceed max then check string lengths
     * to see if they are short enough for buttons.
     */
    
    if (count > 0 && count <= MAX_MESSAGE_BUTTONS)
    {    
        gint i;
        
        is_candidate = TRUE;
        args->choice_count = count;
        choices = args->choices;
        
        for (i=0; i < count; i++)
        {
            if (strlen (*choices) > MAX_BUTTON_CHARS)
            {
                is_candidate = FALSE;
                break;
            }
        }
    }
    
    return is_candidate;
}

/*
 *
 *   GtkWidget *create_informational_message_window (gchar *)
 *
 *   Description:
 *      This routine creates the information message window
 *      if it does not already exist. If it does exist, it
 *      simply updates it with the new message text so it
 *      may be re-displayed.
 *
 *      The hide_window parameter determines whether the window
 *      should be hidden or destroyed when the user clicks on
 *      the OK button.
 *
 *   Entry:
 *      text        - the message string to display
 *      hide_window - TRUE if the window should be hidden or
 *                    FALSE if should be destroyed when dismissed
 *
 *   Exit:
 *      Returns the address of the informational message window
 *
 */
GtkWidget *create_informational_message_window (gchar *text, gboolean hide_window)
{
    gchar            *timestamped_text;
    gchar             buf[MAX_TIME_BUF_SIZE+1];
    time_t            t;
    message_args_t   *args;
    static GtkWidget *ok_button;   
    static GtkWidget *text_box;
    
    time (&t);

    strftime (buf, MAX_TIME_BUF_SIZE, "%b %d %H:%M:%S ", localtime (&t));
    
    timestamped_text = g_strdup_printf ("%s %s\n", buf, text);
    
    if (message_window == NULL)
    {
        GtkWidget   *vbox;
        GtkWidget   *hbox;
        GtkWidget   *label;
        GtkWidget   *text_box_frame;
        GtkWidget   *scrolledwindow;
        GtkWidget   *hseparator;
        GtkWidget   *hbuttonbox;
        GtkWidget   *clear_button;
        GtkWidget   *pixmap;
        GdkBitmap   *mask;
        GdkPixmap   *gdk_pixmap;                
        GtkTooltips *tooltips;

        args = g_new0 (message_args_t, 1);
        args->hide_window = hide_window;
        
        get_dialog_pixmap (WARNING_PIXMAP, &gdk_pixmap, &mask);

        tooltips       = gtk_tooltips_new ();
        message_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
        vbox           = gtk_vbox_new (FALSE, 0);
        hbox           = gtk_hbox_new (FALSE, 10);
        text_box_frame = gtk_frame_new (NULL);
        scrolledwindow = gtk_scrolled_window_new (NULL, NULL);
        text_box       = gtk_text_new (NULL, NULL);
        hseparator     = gtk_hseparator_new ();
        hbuttonbox     = gtk_hbutton_box_new ();
        ok_button      = gtk_button_new_with_label (_("OK"));
        clear_button   = gtk_button_new_with_label (_("Clear"));
        pixmap         = gtk_pixmap_new (gdk_pixmap, mask);
        label          = gtk_label_new (_("Alert messages received from plug-ins are shown below:"));
                
        gtk_window_set_title (GTK_WINDOW (message_window), _("Messages"));
        gtk_widget_show (vbox);
        gtk_widget_show (text_box_frame);
        gtk_widget_show (scrolledwindow);
        gtk_widget_show (text_box);
        gtk_widget_show (hseparator);
        gtk_widget_show (hbuttonbox);
        gtk_widget_show (ok_button);
        gtk_widget_show (clear_button);
        gtk_widget_show (pixmap);
        gtk_widget_show (label);
        gtk_widget_show (hbox);
        
        gtk_tooltips_set_tip (tooltips, ok_button, _("Hide message window"), NULL);
        gtk_tooltips_set_tip (tooltips, clear_button, _("Clear current messages"), NULL);
        
        gtk_widget_set_name (ok_button, "ok_button");
        gtk_widget_ref (ok_button);
        gtk_object_set_data_full (GTK_OBJECT (message_window), "ok_button", ok_button,
                                  (GtkDestroyNotify) gtk_widget_unref);

        gtk_container_add (GTK_CONTAINER (message_window), vbox);
        
        gtk_box_pack_start (GTK_BOX (hbox), pixmap, FALSE, FALSE, 8);        
        gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
        gtk_container_set_border_width (GTK_CONTAINER (hbox), 4);
        
        gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
        
        gtk_widget_set_usize (text_box, 586, 206);        
        gtk_box_pack_start (GTK_BOX (vbox), text_box_frame, TRUE, TRUE, 0);
        gtk_container_set_border_width (GTK_CONTAINER (text_box_frame), 6);
        
        gtk_container_add (GTK_CONTAINER (text_box_frame), scrolledwindow);
        gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow), 
                                        GTK_POLICY_NEVER, GTK_POLICY_ALWAYS);
        
        gtk_container_add (GTK_CONTAINER (scrolledwindow), text_box);
        
        gtk_box_pack_start (GTK_BOX (vbox), hseparator, FALSE, TRUE, 10);    
        gtk_box_pack_start (GTK_BOX (vbox), hbuttonbox, FALSE, TRUE, 19);
        gtk_button_box_set_layout (GTK_BUTTON_BOX (hbuttonbox), GTK_BUTTONBOX_END);
        gtk_button_box_set_spacing (GTK_BUTTON_BOX (hbuttonbox), 18);        
        gtk_button_box_set_child_size (GTK_BUTTON_BOX (hbuttonbox), 108, 46);
        
        gtk_container_add (GTK_CONTAINER (hbuttonbox), clear_button);
        gtk_container_set_border_width (GTK_CONTAINER (hbuttonbox), 8);
        
        GTK_WIDGET_SET_FLAGS (clear_button, GTK_CAN_DEFAULT);

        gtk_container_add (GTK_CONTAINER (hbuttonbox), ok_button);
        GTK_WIDGET_SET_FLAGS (ok_button, GTK_CAN_DEFAULT);

        connect_info_ok_button_clicked_handler (GTK_BUTTON (ok_button), args);
        
        gtk_signal_connect (GTK_OBJECT (clear_button), "clicked",
                            GTK_SIGNAL_FUNC (on_message_clear_button_clicked), text_box);
                                   
        gtk_signal_connect (GTK_OBJECT (message_window), "delete-event",
                            GTK_SIGNAL_FUNC (on_message_window_delete_event), args);
                            
        gtk_signal_connect (GTK_OBJECT (message_window), "destroy",
                            GTK_SIGNAL_FUNC (on_message_window_destroy), args);
                            
        gtk_object_set_user_data (GTK_OBJECT (message_window), args);
    }
    else
    {
        /*
         * Make sure that if the user has requested that the window no longer
         * be hid when the user clicks on OK that the window knows it by updating
         * the hide_window boolean in the message_args_t struct associated as
         * user_data to the window.
         */
        
        args = gtk_object_get_user_data (GTK_OBJECT (message_window));
        
        args->hide_window = hide_window;
     
        connect_info_ok_button_clicked_handler (GTK_BUTTON (ok_button), args);
    }
    
    gtk_text_insert (GTK_TEXT (text_box), NULL, NULL, NULL, timestamped_text, -1); 
    gtk_widget_grab_default (ok_button);
    
    g_free (timestamped_text);
    
    return message_window;
}

/*
 *
 *   void create_choice_button (gint, GtkContainer *, gchar *, message_args_t *, gboolean)
 *
 *   Description:
 *      This routine creates a choice button, associates its answer
 *      index with it, places it in the button box, connects the
 *      "clicked" signal handler and sets it as default if necessary.
 * 
 *   Entry:
 *      index      - choice index
 *      hbuttonbox - the button box container to add the button to
 *      text       - the label text for the button
 *      args       - address of message_args_t struct
 *      is_default - TRUE if this is the default button
 *
 *   Exit:
 *      Choice button is created.
 *
 */
void create_choice_button (gint index, GtkContainer *hbuttonbox, gchar *text, 
                           message_args_t *args, gboolean is_default)
{
    GtkWidget *button;
    
    button = gtk_button_new_with_label (text);
    gtk_widget_show (button);
    gtk_container_add (hbuttonbox, button);
    GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);

    gtk_object_set_user_data (GTK_OBJECT (button), GINT_TO_POINTER (index));
    
    gtk_signal_connect (GTK_OBJECT (button), "clicked",
                        GTK_SIGNAL_FUNC (on_message_choice_button_clicked),
                        args);

    if (is_default)
        gtk_widget_grab_default (button);        
}

/*
 *
 *   GtkWidget *create_choice_button_message_window (gchar *, message_args_t *)
 *
 *   Description:
 *      This routine creates a simple popup window that contains
 *      the text of the given message and one to three buttons labeled
 *      with the short text from the plugin. The focus is placed on the 
 *      button corresponding to the initial value of the "answer" parameter.
 *      It essentially for boolean conditions but the plugin supplies
 *      the button text.
 *
 *      Depending on if the window creation was initiated as part
 *      of a callback on the main event loop or a secondary thread,
 *      we set our signal handlers for the buttons to either
 *      issue a gtk_main_quit() or a gtk_widget_destroy().
 *
 *   Entry:
 *      text - the message string to display
 *      args - contains additional arguments typically passed as user
 *             data to the message window signal handlers
 *
 *   Exit:
 *      Returns address of window created
 *
 */
GtkWidget *create_choice_button_message_window (gchar *text, message_args_t *args)
{
    gint       i;
    GtkWidget *window;
    GtkWidget *vbox;
    GtkWidget *message_label;
    GtkWidget *hseparator;
    GtkWidget *hbuttonbox;
    GtkWidget *pixmap;
    GdkBitmap *mask;
    GdkPixmap *gdk_pixmap;        
    
    get_dialog_pixmap (QUESTION_PIXMAP, &gdk_pixmap, &mask);
    
    window        = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    vbox          = gtk_vbox_new (FALSE, 0);
    message_label = gtk_label_new (text);
    hseparator    = gtk_hseparator_new ();
    hbuttonbox    = gtk_hbutton_box_new ();
    pixmap        = gtk_pixmap_new (gdk_pixmap, mask);
            
    gtk_window_set_title (GTK_WINDOW (window), _("Response Required"));
    gtk_window_set_default_size (GTK_WINDOW (window), 515, 339);
    
    gtk_widget_show (vbox);
    gtk_widget_show (message_label);
    gtk_widget_show (hseparator);
    gtk_widget_show (hbuttonbox);
    gtk_widget_show (pixmap);
    
    gtk_container_add (GTK_CONTAINER (window), vbox);
    
    gtk_box_pack_start (GTK_BOX (vbox), pixmap, FALSE, FALSE, 0);
    gtk_misc_set_alignment (GTK_MISC (pixmap), 0.10, 1);
    gtk_misc_set_padding (GTK_MISC (pixmap), 0, 5);
    
    gtk_box_pack_start (GTK_BOX (vbox), message_label, TRUE, FALSE, 10);
    gtk_label_set_justify (GTK_LABEL (message_label), GTK_JUSTIFY_LEFT);
    gtk_label_set_line_wrap (GTK_LABEL (message_label), TRUE);
    
    gtk_box_pack_start (GTK_BOX (vbox), hseparator, TRUE, FALSE, 10);
    
    gtk_box_pack_start (GTK_BOX (vbox), hbuttonbox, FALSE, FALSE, 19);
    gtk_button_box_set_layout (GTK_BUTTON_BOX (hbuttonbox), GTK_BUTTONBOX_END);
    gtk_button_box_set_spacing (GTK_BUTTON_BOX (hbuttonbox), 18);        
    gtk_button_box_set_child_size (GTK_BUTTON_BOX (hbuttonbox), 108, 46);
    gtk_container_set_border_width (GTK_CONTAINER (hbuttonbox), 8);

    /*
     * Add the button(s) to the button box. Each button will have its
     * index bound with the button as user_data. The default button
     * will be set as well.
     */
     
    for (i=0; i < args->choice_count; i++)
        create_choice_button (i, GTK_CONTAINER (hbuttonbox), 
                              args->choices[i], args, (i == *(args->answer)));

    gtk_signal_connect (GTK_OBJECT (window), "delete-event",
                        GTK_SIGNAL_FUNC (on_message_window_delete_event), args);
                        
    gtk_signal_connect (GTK_OBJECT (window), "destroy",
                        GTK_SIGNAL_FUNC (on_message_window_destroy), args);                        
    
    return window;
}

/*
 *
 *   GtkWidget *create_multiple_choice_message_window (gchar *, message_args_t *)
 *
 *   Description:
 *      This routine creates a window containing the provided message
 *      along with a combo entry widget that provides a list of choices
 *      the user can select with the default selection already selected.
 *
 *   Entry:
 *      text - the message string to display
 *      args - contains additional arguments typically passed as user
 *             data to the message window signal handlers
 *
 *   Exit:
 *      Returns address of window created
 *
 */
GtkWidget *create_multiple_choice_message_window (gchar *text, message_args_t *args)
{
    gchar    **choices;
    GList     *glist=NULL;
    GtkWidget *window;
    GtkWidget *vbox;
    GtkWidget *message_label;
    GtkWidget *hseparator;
    GtkWidget *combo_box;
    GtkWidget *combo_entry;
    GtkWidget *hseparator1;
    GtkWidget *hbuttonbox;
    GtkWidget *ok_button;
    GtkWidget *pixmap;
    GdkBitmap *mask;
    GdkPixmap *gdk_pixmap;        
        
    choices = args->choices;
    
    /*
     * First, populate the choices for the drop-down list from
     * the choices supplied.
     */
    
    while (*choices)
    {
        glist = g_list_append (glist, *choices);
        choices++;
    }
    
    get_dialog_pixmap (QUESTION_PIXMAP, &gdk_pixmap, &mask);
    
    window        = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    vbox          = gtk_vbox_new (FALSE, 0);
    message_label = gtk_label_new (text);
    hseparator    = gtk_hseparator_new ();
    combo_box     = gtk_combo_new ();
    combo_entry   = GTK_COMBO (combo_box)->entry;
    hseparator1   = gtk_hseparator_new ();
    hbuttonbox    = gtk_hbutton_box_new ();
    ok_button     = gtk_button_new_with_label (_("OK"));
    pixmap        = gtk_pixmap_new (gdk_pixmap, mask);
    
    gtk_window_set_title (GTK_WINDOW (window), _("Response Required"));
    gtk_window_set_default_size (GTK_WINDOW (window), 515, 339);
    gtk_combo_set_popdown_strings (GTK_COMBO (combo_box), glist);
    gtk_entry_set_text (GTK_ENTRY (combo_entry), g_list_nth_data (glist, *(args->answer)));
        
    gtk_widget_show (vbox);
    gtk_widget_show (message_label);
    gtk_widget_show (hseparator);
    gtk_widget_show (combo_box);
    gtk_widget_show (combo_entry);
    gtk_widget_show (hseparator1);
    gtk_widget_show (hbuttonbox);
    gtk_widget_show (ok_button);
    gtk_widget_show (pixmap);
            
    gtk_container_add (GTK_CONTAINER (window), vbox);

    gtk_box_pack_start (GTK_BOX (vbox), pixmap, FALSE, TRUE, 0);
    gtk_misc_set_alignment (GTK_MISC (pixmap), 0.10, 1);
    gtk_misc_set_padding (GTK_MISC (pixmap), 0, 5);

    gtk_box_pack_start (GTK_BOX (vbox), message_label, TRUE, TRUE, 0);
    gtk_label_set_justify (GTK_LABEL (message_label), GTK_JUSTIFY_LEFT);
    gtk_label_set_line_wrap (GTK_LABEL (message_label), TRUE);
    
    gtk_box_pack_start (GTK_BOX (vbox), hseparator, FALSE, TRUE, 3);
    
    gtk_box_pack_start (GTK_BOX (vbox), combo_box, TRUE, FALSE, 19);
    gtk_container_set_border_width (GTK_CONTAINER (combo_box), 25);
    gtk_combo_set_value_in_list (GTK_COMBO (combo_box), TRUE, TRUE);
    
    gtk_entry_set_editable (GTK_ENTRY (combo_entry), FALSE);
    
    gtk_box_pack_start (GTK_BOX (vbox), hseparator1, TRUE, FALSE, 10);
    
    gtk_box_pack_start (GTK_BOX (vbox), hbuttonbox, FALSE, FALSE, 19);
    gtk_button_box_set_layout (GTK_BUTTON_BOX (hbuttonbox), GTK_BUTTONBOX_END);
    gtk_button_box_set_child_size (GTK_BUTTON_BOX (hbuttonbox), 108, 46);
    
    gtk_container_add (GTK_CONTAINER (hbuttonbox), ok_button);
    gtk_container_set_border_width (GTK_CONTAINER (hbuttonbox), 8);
    
    GTK_WIDGET_SET_FLAGS (ok_button, GTK_CAN_DEFAULT);
    gtk_widget_grab_default (ok_button);
    
    gtk_signal_connect (GTK_OBJECT (ok_button), "clicked",
                        GTK_SIGNAL_FUNC (on_button_clicked_destroy_window), args);

    gtk_signal_connect (GTK_OBJECT (combo_entry), "changed",
                        GTK_SIGNAL_FUNC (on_message_combo_selection_changed), args);

    gtk_signal_connect (GTK_OBJECT (window), "delete-event",
                        GTK_SIGNAL_FUNC (on_message_window_delete_event), args);
                        
    gtk_signal_connect (GTK_OBJECT (window), "destroy",
                        GTK_SIGNAL_FUNC (on_message_window_destroy), args);                        
    
    return window;
}

/*
 *
 *   void display_message_window (message_args_t *)
 *
 *   Description:
 *      This routine creates a window to display the appropriate
 *      alert/message window.
 *
 *   Entry:
 *      args - contains required data to be passed to the message window
 *             signal handlers
 *
 *   Exit:
 *      Appropriate message window is created and displayed
 *
 */
void display_message_window (message_args_t *args)
{
    GtkWidget *window;
    
    if (args->answer == NULL)
        window = create_informational_message_window (args->text, TRUE);
    else if (is_choice_button_window_candidate (args))
        window = create_choice_button_message_window (args->text, args);
    else
        window = create_multiple_choice_message_window (args->text, args);
    
    gdk_beep ();    
    gtk_widget_show (window);
}

/*
 *
 *   gboolean display_message_window_from_main_thread (message_args_t *)
 *
 *   Description:
 *      This routine is an a one-time execution idle function that calls
 *      the routine that displays the alert/message window while running
 *      under the control of the main event loop. Doing it this way avoids
 *      a nasty hang encountered in the original implementation.
 *
 *   Entry:
 *      args - contains required data to be passed to the message window
 *             signal handlers
 *
 *   Exit:
 *      Appropriate message window is created and displayed
 *
 */
gboolean display_message_window_from_main_thread (message_args_t *args)
{
    gdk_threads_enter ();
    display_message_window (args);
    gdk_threads_leave ();
    
    /*
     * The cleanup of the message_args_t had to be delayed
     * for an informational message until we had made the
     * gtk_ calls. So, now should be safe to cleanup.
     */
    
    if (args->answer == NULL)
    {
        g_free (args->text);
        g_free (args);
    }
    
    return FALSE;
}

/*
 *
 *   gint user_message_cb (gchar *, gint *, gchar **)
 *
 *   Description:
 *      This routine is called by the engine for having
 *      us display simple messages or get simple responses
 *      from a set of choices (strings). 
 *
 *      If this callback is running on a thread other than
 *      the main thread, we schedule an idle function to
 *      display the window on the main thread. The window
 *      destroy signal handler will signal a wakeup condition
 *      to allow this thread to unblock for a one-to-three
 *      button window and multiple choice type messages.
 *
 *      Informational-only message are always dispatched
 *      asynchronously though. That is, we create/redisplay
 *      the window and return from this callback to the 
 *      engine immediately.
 *
 *   Entry:
 *      text    - the message string to display
 *      answer  - the index of the choice selected 
 *                (initialized with the default selection)
 *      choices - an optional array of choices the user
 *                can select from
 *
 *   Exit:
 *      Always returns 0 (success)
 *
 */
gint user_message_cb (gchar *text, gint *answer, gchar **choices)
{
    gboolean        is_main_thread;
    gboolean        is_info_message;
    message_args_t *args;
    
    is_info_message = answer == NULL;
    is_main_thread  = getpid () == get_main_thread_id ();
    
    args = g_new0 (message_args_t, 1);
    args->text    = text;
    args->answer  = answer;
    args->choices = choices;
        
    if (!is_main_thread)
    {
        /*
         * Since this is not the main thread and it is
         * a message that must get a response before
         * returning, we allocate a condition semaphore
         * to allow us to block and allow the message
         * window to signal us to continue when the
         * window is being destroyed.
         */
        
        if (!is_info_message)
        {
            args->wakeup_cond  = g_cond_new (); 
            args->wakeup_mutex = g_mutex_new ();
        }
        else
        {
            /*
             * Since we will return on an informational before the
             * idle function may get a chance to run, we should
             * dup the text string to be on the safe side.
             */
            args->text = g_strdup (text);
        }
        
        /*
         * Schedule a one time idle function to create and display the message
         * window under the control of the main thread...
         */
        
        gtk_idle_add ((GtkFunction) display_message_window_from_main_thread, args);
        
        /* 
         * If this is not an information message then we
         * will be blocking until one of the window 
         * message handlers clears a condition to let
         * us continue to return control to the engine.
         */
        
        if (!is_info_message)
        {    
            g_cond_wait (args->wakeup_cond, args->wakeup_mutex);
            
            g_cond_free (args->wakeup_cond);
            g_mutex_unlock (args->wakeup_mutex);
            g_mutex_free (args->wakeup_mutex);
            g_free (args);
        }
    }
    else
    {
        display_message_window (args);
        
        /*
         * If this is an informational message then run the
         * main event loop until it has processed the queued
         * events.
         *
         * If this is one of the message types that needs to
         * wait for a response before returning then enter
         * a nested message loop to process the events. We
         * break out of the nested loop when we get a response.
         */
        
        if (is_info_message)
            while (gtk_events_pending ())
                gtk_main_iteration_do (FALSE);
        else
            gtk_main ();
        
        g_free (args);
    }
        
    return 0;
}

/*
 * Function table given to evms_open_engine()
 */

static engine_message_functions_t msg_func_table = {
    user_message       : user_message_cb,
    user_communication : NULL                      /* BUGBUG: Not implemented yet */
};

/*
 * Getter function for message function/callback table
 */

engine_message_functions_t* get_message_function_table (void)
{
    return &msg_func_table;
}

/*
 *
 *   void redisplay_info_messages_window (void)
 *
 *   Description:
 *      This routine attempts to redisplay the messages window
 *      if it exists. Otherwise, it displays a popup window
 *      that tells the users that no messages exist to be shown.
 * 
 *   Entry:
 *      None
 *      
 *   Exit:
 *      Re-displays the messages window or displays an no messages
 *      window.
 *
 */
void redisplay_info_messages_window (void)
{
    if (message_window)
        gtk_widget_show (message_window);
    else
        display_popup_window (_("View Messages"), 
                              _("There are no informational messages available."));
}

/*
 *
 *   void set_message_window_transient_for (GtkWindow *)
 *
 *   Description:
 *      This routine attempts to make the message window a transient
 *      window of the given window. It hopefully allows the suggests
 *      to the window manager to place it on top of the given window.
 * 
 *   Entry:
 *      parent - the window id of the parent for the message window
 *      
 *   Exit:
 *      The message window hopefully is placed on top of the parent
 *      window.
 *
 */
void set_message_window_transient_for (GtkWindow *parent)
{
    if (message_window && GTK_WIDGET_MAPPED (message_window))
        gtk_window_set_transient_for (GTK_WINDOW (message_window), parent);
}
