# (C) Copyright IBM Corp. 2004
#
# This program is free software;  you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY;  without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
# the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program;  if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
#
# Perl module with common subroutines for managing LVM2 containers and regions.

package Evms::Lvm2;

use strict;
use warnings;

use Evms::Common;
use Evms::Object;

BEGIN {
	use Exporter();
	our (@ISA, @EXPORT);
	@ISA = qw(Exporter);
	@EXPORT = qw(&create_lvm2_container
		     &expand_lvm2_container
		     &shrink_lvm2_container
		     &delete_lvm2_container
		     &query_lvm2_container
		     &extended_query_lvm2_container
		     &get_lvm2_container_pe_size
		     &get_lvm2_container_pvs
		     &rename_lvm2_container
		     &create_lvm2_region
		     &expand_lvm2_region
		     &shrink_lvm2_region
		     &delete_lvm2_region
		     &query_lvm2_region
		     &rename_lvm2_region
		     &move_lvm2_extent
		     &move_lvm2_pv
		    );
}

# create_lvm2_container: Create a new LVM2 container.
#
# Arguments:
#    options: Reference to a hash of options for creating the container.
#             Allowable hash keys and values are:
#                name: Desired name for new container.
#                extent_size: Desired physical-extent size. Must include
#                             an appropriate size suffix.
#    objects: Array of object names.
sub create_lvm2_container($ @)
{
	my ($options_ref, @objects) = @_;
	my %options = %{$options_ref};
	my ($object, $objects);
	my (@keys, $key, $option_args);
	my ($command, $rc);

	$objects = "";
	foreach $object (@objects) {
		$objects .= "$object,";
	}

	$option_args = "";
	@keys = ("name", "extent_size");
	foreach $key (@keys) {
		if (defined($options{$key})) {
			$option_args .= "$key=$options{$key},";
		}
	}

	# Strip trailing commas.
	$objects =~ s/^(.*)\,$/$1/;
	$option_args =~ s/^(.*)\,$/$1/;

	$command = "create:container,LVM2={$option_args},$objects";
	$rc = run_evms_command($command);

	return $rc;
}

# expand_lvm2_container: Add additional objects to an LVM2 container.
#
# Arguments:
#    container: Name of container to expand.
#    objects: Array of objects to add to the container.
sub expand_lvm2_container($ @)
{
	my ($container, @objects) = @_;
	my ($rc, $object, $objects, $command);

	$objects = "";
	foreach $object (@objects) {
		$objects .= ",$object";
	}

	$command = "expand:lvm2/$container$objects";
	$rc = run_evms_command($command);

	return $rc;
}

# shrink_lvm2_container: Remove objects from an LVM2 container.
#
# Arguments:
#    container: Name of container to shrink.
#    objects: Array of objects to remove from the container.
sub shrink_lvm2_container($ @)
{
	my ($container, @objects) = @_;
	my ($rc, $object, $objects, $command);

	$objects = "";
	foreach $object (@objects) {
		$objects .= ",$object";
	}

	$command = "shrink:lvm2/$container$objects";
	$rc = run_evms_command($command);

	return $rc;
}

# delete_lvm2_container: Delete an LVM2 container
#
# Arguments:
#    container: Name of container to delete.
sub delete_lvm2_container($)
{
	my $container = $_[0];
	my $rc;

	$rc = delete_thing("lvm2/" . $container);

	return $rc;
}

# query_lvm2_container
#
# Arguments:
#    container: Name of container to query.
#    output: Reference of array to pass back query output.
# Returns:
#    Array of query information.
sub query_lvm2_container($)
{
	my $container = $_[0];
	my $command = "query:containers,container=lvm2/$container";
	my @output;
	my $rc;

	$rc = run_evms_command($command, \@output);
	if ($rc) {
		return;
	}

	return @output;
}

# extended_query_lvm2_container
#
# Arguments:
#    container: Name of container to query.
# Returns:
#    Array of extended info.
sub extended_query_lvm2_container($)
{
	my $container = $_[0];
	my $command = "query:ei,lvm2/$container";
	my @output;
	my $rc;

	$rc = run_evms_command($command, \@output);
	if ($rc) {
		return;
	}

	return @output;
}

# get_lvm2_container_pe_size
# Return the PE size of the specified container.
#
# Arguments:
#    container: Name of container.
sub get_lvm2_container_pe_size($)
{
	my $container = $_[0];
	my @output;
	my $pe_size;
	my $rc = -1;

	@output = extended_query_lvm2_container($container);
	if (@output) {
		$pe_size = get_extended_info_value("Extent_Size", \@output);
		if (defined($pe_size) &&
		    $pe_size =~ /^(\d+)\s+sectors/) {
			$rc = $1;
		}
	}

	return $rc;
}

# get_lvm2_container_pvs
#
# Arguments:
#    container: Name of container to query.
# Returns:
#    Array of names of PVs that belong to this container.
sub get_lvm2_container_pvs($)
{
	my $container = $_[0];
	my @output;
	my ($line, @lines);
	my @pvs;
	my $command = "query:children,lvm2/$container";
	my $rc;

	$rc = run_evms_command($command, \@output);
	if ($rc) {
		log_error("Error getting list of children for " .
			  "container $container.\n");
		return;
	}

	@lines = grep(/Name/, @output);
	foreach $line (@lines) {
		if ($line =~ /Name:\s+(.*)$/) {
			push(@pvs, $1);
		}
	}

	return @pvs;
}

# rename_lvm2_container
#
# Arguments:
#    container: Name of container to rename.
#    new_name: New name for the container.
sub rename_lvm2_container($ $)
{
	my $container = $_[0];
	my $new_name = $_[1];
	my ($command, $rc);

	$command = "set:lvm2/$container,name=$new_name";

	$rc = run_evms_command($command);

	return $rc;
}

# create_lvm2_region
#
# Arguments:
#    container: Name of container to create the new region in.
#    options: Reference to a hash of options for creating the region.
#             Allowable keys and values are:
#               name: Name for the new region.
#               size: Size of new region. Must include an appropriate suffix.
#               stripes: Number of stripes in the region.
#               stripe_size: Size of each stripe.
#               pvs: Name of PV to limit the new region to.
#            The "name" is required. All other are optional.
sub create_lvm2_region($ $)
{
	my $container = $_[0];
	my $options_ref = $_[1];
	my %options = %{$options_ref};
	my $freespace = "lvm2/" . $container . "/Freespace";
	my ($key, @keys);
	my ($command, $rc);

	$command = "allocate:$freespace";
	@keys = ("name", "size", "stripes", "stripe_size", "pvs");

	foreach $key (@keys) {
		if (defined($options{$key})) {
			$command .= ",$key=$options{$key}";
		}
	}

	$rc = run_evms_command($command);

	return $rc;
}

# expand_lvm2_region
#
# Arguments:
#    region: Name of region to expand ("container_name/region_name").
#    options: Reference to a hash of options for expanding the region.
#             Allowable keys and values are:
#               size: Size to expand the region by. Must include an
#                     appropriate size suffix.
#               stripes: Number of stripes for the new portion of the region.
#               stripe_size: Stripe-size for the new portion of the region.
#                            Must include an appropriate size suffix.
#               pvs: Name of PV to limit the expand to.
sub expand_lvm2_region($ $)
{
	my $region = $_[0];
	my $options_ref = $_[1];
	my %options = %{$options_ref};
	my ($key, @keys);
	my ($command, $rc);

	$command = "expand:lvm2/$region";
	@keys = ("size", "stripes", "stripe_size", "pvs");

	foreach $key (@keys) {
		if (defined($options{$key})) {
			$command .= ",$key=$options{$key}";
		}
	}

	$rc = run_evms_command($command);

	return $rc;
}

# shrink_lvm2_region
#
# Arguments:
#    region: Name of region to shrink ("container_name/region_name").
#    options: Reference to a hash of options for shrinking the region.
#             Allowable keys and values are:
#               size: Size to shrink the region by. Must include an
#                     appropriate size suffix.
sub shrink_lvm2_region($ $)
{
	my $region = $_[0];
	my $options_ref = $_[1];
	my %options = %{$options_ref};
	my ($key, @keys);
	my ($command, $rc);

	$command = "shrink:lvm2/$region";
	@keys = ("size");

	foreach $key (@keys) {
		if (defined($options{$key})) {
			$command .= ",$key=$options{$key}";
		}
	}

	$rc = run_evms_command($command);

	return $rc;
}

# delete_lvm2_region
#
# Arguments:
#    region: Name of region to delete ("container_name/region_name").
sub delete_lvm2_region($)
{
	my $region = $_[0];
	my $rc;

	$rc = delete_thing("lvm2/" . $region);

	return $rc;
}

# query_lvm2_region
#
# Arguments:
#    region: Name of region to query ("container_name/region_name").
sub query_lvm2_region
{
	my $region = $_[0];
	my @output;

	@output = query_object("lvm2/" . $region);

	return @output;
}

# rename_lvm2_region
#
# Arguments:
#    region: Name of region to rename.
#    new_name: New name for the region.
sub rename_lvm2_region($ $)
{
	my $region = $_[0];
	my $new_name = $_[1];
	my ($command, $rc);

	$command = "set:lvm2/$region,name=$new_name";

	$rc = run_evms_command($command);

	return $rc;
}

# move_lvm2_extent
# Move one logical extent to a different physical extent.
#
# Arguments:
#    region: Name of region to move ("container_name/region_name").
#    le: Index of logical extent to move.
#    pv: Name of PV to move the extent to.
#    pe: Index of physical extent to move to.
sub move_lvm2_extent($ $ $ $)
{
	my $region = $_[0];
	my $le = $_[1];
	my $pv = $_[2];
	my $pe = $_[3];
	my $command = "task:move_extent,lvm2/$region,le=$le,pv=$pv,pe=$pe";
	my $rc;

#	$rc = run_evms_command($command);
# FIXME: Not implemented yet.
	$rc = 1;

	return $rc;
}

# move_lvm2_pv
# Move all the in-use physical extents from the specified PV to other PVs in
# the container.
#
# Arguments:
#    container: Container owning the PV to be moved.
#    pv: Name of PV to be moved.
#    options: Reference to a hash of options for the move. Allowable hash
#             keys and values are:
#               target_pvs: Name of PV to limit the move to.
#               maintain_stripes: "none", "loose", or "strict".
sub move_lvm2_pv($ $ $)
{
	my $container = $_[0];
	my $pv = $_[1];
	my $options_ref = $_[2];
	my %options = %{$options_ref};
	my ($key, @keys);
	my ($command, $rc);

	@keys = ("target_pvs", "maintain_stripes");
	$command = "task:move_pv,lvm2/$container";
	foreach $key (@keys) {
		if (defined($options{$key})) {
			$command .= ",$key=$options{$key}";
		}
	}
	$command .= ",$pv";

#	$rc = run_evms_command($command);
# FIXME: Not implemented yet.
	$rc = 1;

	return $rc;
}

1;

