/*
 *
 *   (C) Copyright IBM Corp. 2002, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * 	Module: evmsccm.c
 */


/*
 * this program ensures that evms commands do not run before
 * ccm has recovered. It is called in evms_failover script
 */

#include <stdlib.h>
#include <stdio.h>
#include <sys/time.h>
#include <sys/types.h>
#include <string.h>
#include <errno.h>
#include <getopt.h>
#include <libgen.h>
#include <pthread.h>
#include <ocf/oc_event.h>
#include <heartbeat/hb_api.h>


char * prog_name;				
oc_ev_t *ev_token;
int list = FALSE;
int verbose = FALSE;
int got_membership = FALSE;
oc_ev_membership_t * membership = NULL;

extern void oc_ev_special(const oc_ev_t *, oc_ev_class_t , int );


static void
show_help(void) {
	printf("Usage: %s [options]\n", prog_name);
	printf("Query CCM membership.\n\n"
	       "  [-l] list the membership\n"
	       "  [-v] verbose mode\n"
	       "  [-h | -? | --help]\n"
	       "     display this help\n");
}


static int
parse_options(int argc, char * * argv) {

	int    rc = 0;
	int    c;
	char * short_opts = "lvh?";
	struct option long_opts[] = {
		{ "list",    no_argument, NULL, 'l'},
		{ "verbose", no_argument, NULL, 'v'},
		{ "help",    no_argument, NULL, 'h'},
		{ NULL,      0,           NULL,   0}
	};

	while ((c = getopt_long(argc, argv,
				short_opts, long_opts, NULL)) != EOF) {
		switch (c) {
		case 'l':
			list = TRUE;
			/* Fall though and set verbose. */

		case 'v':
			verbose = TRUE;
			break;

		case 'h':
		case '?':
			show_help();
			rc = EINVAL;
			break;

		default:
			printf("%s -- unrecognized option \"%c\"\n\n", prog_name, c);
			show_help();
			rc = EINVAL;
			break;
		}
	}

	return rc;
}

static void
list_membership(const oc_ev_membership_t *oc)
{

	int rc;
	ll_cluster_t *hb_handle=NULL;
	const char *hname;
	int num_nodes = 2;
	const char * node;
	const char **node_names = calloc(num_nodes, sizeof(char*));
	int node_idx = 0;

	if (node_names == NULL) {
		if (verbose) {
			printf("Could not get memory for the node name array.\n");
		}
		return;
	}

	hb_handle = ll_cluster_new("heartbeat");

	if(!hb_handle) {
		if (verbose) {
			printf("Could not get a handle for the cluster.\n");
		}
		return;
	}

	rc = hb_handle->llc_ops->signon(hb_handle, "evms");
	if (rc != HA_OK) {
		if (verbose) {
			printf("Failed to signon to the cluster.\n");
		}
		return;
	}

	hname = hb_handle->llc_ops->get_mynodeid(hb_handle);
	if(hname == NULL) {
		if (verbose) {
			printf("Failed to get my nodeid.\n");
		}
		goto signoff;
	}

	printf("This node is \"%s\".\n", hname);

        rc = hb_handle->llc_ops->init_nodewalk(hb_handle);
	if (rc != HA_OK) {
		if (verbose) {
			printf("Failed to initialize for a node walk.\n");
		}
		goto signoff;
	}
	
	while((node = hb_handle->llc_ops->nextnode(hb_handle)) != NULL) {
		/* ignore non normal nodes */
		if(strcmp(hb_handle->llc_ops->node_type(hb_handle, node),
				"normal") != 0) {
			if(strcmp(node,hname) == 0) {
				if (verbose) {
					printf("heartbeat inconsistency running on a ping node\n");
				}
				break;
			}
		       	continue;
		}

		if (node_idx >= num_nodes) {
			num_nodes <<= 1;
			node_names = realloc(node_names, num_nodes * sizeof(char*));
			if (node_names == NULL) {
				if (verbose) {
					printf("Could not reallocate memory for the node name array.\n");
				}
				goto end_node_walk;
			}
		}

		node_names[node_idx] = node;
		node_idx++;
	}

	printf("Membership:\n");
	for (node_idx = oc->m_memb_idx; node_idx < oc->m_memb_idx + oc->m_n_member; node_idx++) {
		printf("  Node %d: %s\n", node_idx, node_names[node_idx]);
	}

end_node_walk:
	rc = hb_handle->llc_ops->end_nodewalk(hb_handle);
	if (rc != HA_OK) {
		if (verbose) {
			printf("Failed to end the node walk.\n");
		}
	}

signoff:
#if (LLC_PROTOCOL_VERSION <= 1)
	hb_handle->llc_ops->signoff(hb_handle);
#else
	hb_handle->llc_ops->signoff(hb_handle, TRUE);
#endif
}


pthread_mutex_t event_mutex = PTHREAD_MUTEX_INITIALIZER;
static void
my_ms_events(oc_ed_t event, void *cookie,
		size_t size, const void *data)
{
	const oc_ev_membership_t *oc = (const oc_ev_membership_t *)data;
	pthread_mutex_lock(&event_mutex);
	/* exit only when ccm has a new membership 	*/
	/* NOTE: OC_EV_MS_INVALID is a new membership   */
	/* event with no quorum 			*/
	if(event == OC_EV_MS_NEW_MEMBERSHIP ||
			event == OC_EV_MS_INVALID) {

		got_membership = TRUE;

		if (verbose) {
			printf("CCM has a membership.\n");
			
			if (event == OC_EV_MS_NEW_MEMBERSHIP) {
				printf("The membership has quorum.\n");
			} else {
				printf("The membership does NOT have quorum.\n");
			}
		}

		if (list) {
			int size = sizeof(oc_ev_membership_t) + sizeof(oc_node_t) * (oc->m_n_member + oc->m_n_in + oc->m_n_out - 1);
			
			membership = malloc(size);
			if (membership != NULL) {
				memcpy(membership, oc, size);
			}
		}
	}
	pthread_mutex_unlock(&event_mutex);
	oc_ev_callback_done(cookie);
}

int
main(int argc, char *argv[])
{
	int ret;
	fd_set rset;
	int	my_ev_fd;
	int max_event_num = 10;
	int  i;

	prog_name = basename(argv[0]);

	parse_options(argc, argv);

	oc_ev_register(&ev_token);

	oc_ev_set_callback(ev_token, OC_EV_MEMB_CLASS, my_ms_events, NULL);
	oc_ev_special(ev_token, OC_EV_MEMB_CLASS, 0/*don't care*/);

	ret = oc_ev_activate(ev_token, &my_ev_fd);
	if(ret){
		if (verbose) {
			printf("Failed to activate callbacks.  Error code was %d: %s\n", ret, strerror(ret));
		}
		oc_ev_unregister(ev_token);
		exit(1);
	}

	for (i=0; i<max_event_num; i++) {
		struct timeval tv;
		 	
		FD_ZERO(&rset);
		FD_SET(my_ev_fd, &rset);
           	tv.tv_sec = 5;
              	tv.tv_usec = 0;

		if(select(my_ev_fd + 1, &rset, NULL, NULL, &tv) == -1){
			if (verbose) {
				printf("Event file descriptor not ready within %ld sec timeout.\n", tv.tv_sec);
			}
			exit(1);
		}
		ret = oc_ev_handle_event(ev_token);
		if(ret){
			if (verbose) {
				printf("Error handling event.  Error code was %d: %s\n", ret, strerror(ret));
			}
			exit(1);
		}

		if (got_membership) {
			if (list && membership != NULL) {
				list_membership(membership);
				free(membership);
			}
			exit(0);
                }
	}
	/* return 2. Logically I should exit with 1. But as it
	* stand there is another bug in ipc library, fails
	* the receives with EGAIN, even when the CCM daemon is
	* running. So we are returning 2. When that bug is
	* fixed, this ugly patch shall be removed.
	*/
	if (verbose) {
		printf("None of the first %d events was a membership event.\n", max_event_num);
	}
	exit(2);
}
