/*
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: snap_options.c
 *
 * Functions for manipulating user options and information.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>
#include "snapshot.h"

/**
 * validate_snapshot_name
 *
 * Verify that the desired name is not already in use.
 **/
static int validate_snapshot_name(char * snap_name,
				  storage_object_t * snap_child)
{
	storage_container_t * disk_group = snap_child->disk_group;
	char test_name[EVMS_NAME_SIZE];
	int rc;

	LOG_ENTRY();

	test_name[0] = '\0';
	if (disk_group) {
		strncat(test_name, disk_group->name, EVMS_NAME_SIZE);
		strncat(test_name, "/", EVMS_NAME_SIZE-strlen(test_name));
	}
	strncat(test_name, snap_name, EVMS_NAME_SIZE-strlen(test_name));

	rc = EngFncs->validate_name(test_name);

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * parse_creation_options
 *
 * Valid options for creating snapshots:
 *  origin volume name
 *  snapshot object name
 *  chunk size (default = 64k)
 *  writeable (default = no)
 **/
void parse_creation_options(option_array_t * options,
			    char ** org_vol_name,
			    char ** snap_name,
			    u_int32_t * chunk_size,
			    int * writeable)
{
	int i;

	LOG_ENTRY();

	/* Set option defaults. */
	*org_vol_name = NULL;
	*snap_name = NULL;
	*chunk_size = SNAPSHOT_DEFAULT_CHUNK_SIZE;
	*writeable = TRUE;

	for (i = 0; i < options->count; i++) {

		/* If only the name-based index is specified, get the number. */
		if (!options->option[i].is_number_based) {
			if (!strcmp(options->option[i].name, SNAP_OPTION_ORG_VOLUME_NAME)) {
				options->option[i].number = SNAP_OPTION_ORG_VOLUME_INDEX;
			} else if (!strcmp(options->option[i].name, SNAP_OPTION_SNAPSHOT_NAME)) {
				options->option[i].number = SNAP_OPTION_SNAPSHOT_INDEX;
			} else if (!strcmp(options->option[i].name, SNAP_OPTION_CHUNKSIZE_NAME)) {
				options->option[i].number = SNAP_OPTION_CHUNKSIZE_INDEX;
			} else if (!strcmp(options->option[i].name, SNAP_OPTION_WRITEABLE_NAME)) {
				options->option[i].number = SNAP_OPTION_WRITEABLE_INDEX;
			} else {
				continue;
			}
		}

		LOG_DEBUG("Parsing option %d\n", options->option[i].number);

		/* Use the number-based index to record the option. */
		switch (options->option[i].number) {
		case SNAP_OPTION_ORG_VOLUME_INDEX:
			*org_vol_name = options->option[i].value.s;
			break;
		case SNAP_OPTION_SNAPSHOT_INDEX:
			*snap_name = options->option[i].value.s;
			break;
		case SNAP_OPTION_CHUNKSIZE_INDEX:
			*chunk_size = options->option[i].value.ui32;
			break;
		case SNAP_OPTION_WRITEABLE_INDEX:
			*writeable = options->option[i].value.b;
			break;
		default:
			continue;
		}
	}

	LOG_EXIT_VOID();
}

/**
 * verify_creation_options
 *
 * Verify that the specified options are valid for creating a new snapshot.
 * Must have valid names, and chunk size must be a power-of-2.
 **/
int verify_creation_options(char * org_vol_name,
			    char * snap_name,
			    u_int32_t chunk_size,
			    storage_object_t * snap_child)
{
	int rc = EINVAL;

	LOG_ENTRY();

	if (!org_vol_name) {
		LOG_ERROR("Did not specify origin volume name.\n");
	} else if (!snap_name ||
		   strlen(snap_name) < 1 ||
		   strlen(snap_name) > EVMS_VOLUME_NAME_SIZE) {
		LOG_ERROR("Did not specify snapshot object name.\n");
	} else if (validate_snapshot_name(snap_name, snap_child)) {
		LOG_ERROR("Snapshot object name (%s) is already in use.\n",
			  snap_name);
	} else if ((chunk_size & (chunk_size-1)) != 0) {
		LOG_ERROR("Chunk size (%u) is not a power-of-2.\n", chunk_size);
	} else {
		rc = 0;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * get_acceptable_volumes
 *
 * Get a list of all volumes in the system which can be snapshotted. The list
 * of volumes must come from the same CSM disk-group as the snapshot object.
 **/
static int get_volume_list(value_list_t ** value_list,
			   storage_object_t * snap_child)
{
	value_list_t * list = NULL;
	snapshot_volume_t * snap_volume;
	logical_volume_t * volume;
	list_anchor_t volumes;
	list_element_t itr;
	int rc, count=0;

	LOG_ENTRY();

	/* Get the list of all volumes from the engine. */
	rc = EngFncs->get_volume_list(NULL, snap_child->disk_group, 0, &volumes);
	if (rc) {
		goto done;
	}

	count = EngFncs->list_count(volumes);
	list = EngFncs->engine_alloc(sizeof(value_list_t) +
				     count * sizeof(value_t));
	if (!list) {
		rc = ENOMEM;
		goto done;
	}

	count = 0;
	LIST_FOR_EACH(volumes, itr, volume) {
		snap_volume = volume->object->private_data;
		if (volume->object->plugin != my_plugin_record ||
		    snap_volume->flags & SNAPSHOT_ORIGIN) {
			rc = verify_origin(volume->object, snap_child);
			if (!rc) {
				list->value[count].s = EngFncs->engine_strdup(volume->name);
				if (!list->value[count].s) {
					rc = ENOMEM;
					goto done;
				}
				count++;
			}
		}
	}

	rc = 0;
	list->count = count;
	*value_list = list;

done:
	if (rc && list) {
		while (count--) {
			EngFncs->engine_free(list->value[count].s);
		}
		EngFncs->engine_free(list);
	}
	EngFncs->destroy_list(volumes);
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * init_option_descriptor_create
 *
 * Initialize an option descriptor for a create task.
 **/
static int init_option_descriptor_create(task_context_t * context)
{
	option_desc_array_t * od = context->option_descriptors;
	int index;

	LOG_ENTRY();

	index = SNAP_OPTION_ORG_VOLUME_INDEX;
	od->option[index].name = EngFncs->engine_strdup(SNAP_OPTION_ORG_VOLUME_NAME);
	od->option[index].title = EngFncs->engine_strdup(_("Volume to be snapshotted"));
	od->option[index].tip = EngFncs->engine_strdup(_("The volume you wish to take a snapshot of."));
	od->option[index].type = EVMS_Type_String;
	od->option[index].max_len = EVMS_VOLUME_NAME_SIZE;
	od->option[index].flags |= EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;
	od->option[index].constraint_type = EVMS_Collection_List;
	od->option[index].constraint.list = EngFncs->engine_alloc(sizeof(value_list_t));
	od->option[index].value.s = EngFncs->engine_alloc(EVMS_VOLUME_NAME_SIZE+1);

	/* Must wait until the set-object call to initialize the constraint
	 * list for the origin volumes, so that the list will only contain
	 * volumes from the same CSM disk-group as the selected object.
	 */

	index = SNAP_OPTION_SNAPSHOT_INDEX;
	od->option[index].name = EngFncs->engine_strdup(SNAP_OPTION_SNAPSHOT_NAME);
	od->option[index].title = EngFncs->engine_strdup(_("Snapshot object name"));
	od->option[index].tip = EngFncs->engine_strdup(_("The name you wish to assign to the object being created."));
	od->option[index].type = EVMS_Type_String;
	od->option[index].max_len = EVMS_VOLUME_NAME_SIZE;
	od->option[index].min_len = 1;
	od->option[index].flags |= EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;
	od->option[index].value.s = EngFncs->engine_alloc(EVMS_VOLUME_NAME_SIZE+1);

	index = SNAP_OPTION_CHUNKSIZE_INDEX;
	od->option[index].name = EngFncs->engine_strdup(SNAP_OPTION_CHUNKSIZE_NAME);
	od->option[index].title = EngFncs->engine_strdup(_("Chunk size"));
	od->option[index].tip = EngFncs->engine_strdup(_("Chunksize is the amount of data that will be copied at a time to the snapshot."));
	od->option[index].type = EVMS_Type_Unsigned_Int32;
	od->option[index].unit = EVMS_Unit_Sectors;
	od->option[index].flags |= EVMS_OPTION_FLAGS_NOT_REQUIRED;
	od->option[index].constraint_type = EVMS_Collection_List;
	SET_POWER2_LIST(od->option[index].constraint.list, SNAPSHOT_MIN_CHUNK_SIZE, SNAPSHOT_MAX_CHUNK_SIZE);
	od->option[index].value.ui32 = SNAPSHOT_DEFAULT_CHUNK_SIZE;

	index = SNAP_OPTION_WRITEABLE_INDEX;
	od->option[index].name = EngFncs->engine_strdup(SNAP_OPTION_WRITEABLE_NAME);
	od->option[index].title = EngFncs->engine_strdup(_("Writeable"));
	od->option[index].tip = EngFncs->engine_strdup(_("Enableing this option allows writes to the snapshot."));
	od->option[index].type = EVMS_Type_Boolean;
	od->option[index].flags |= EVMS_OPTION_FLAGS_NOT_REQUIRED;
	od->option[index].value.b = TRUE;

	index++;
	od->count = index;

	LOG_EXIT_INT(0);
	return 0;
}

/**
 * init_task_create
 *
 * Initialize the task context record for a create task.
 **/
int init_task_create(task_context_t * context)
{
	storage_object_t * object;
	list_anchor_t objects;
	list_element_t itr;
	int rc;

	LOG_ENTRY();

	/* A create task must select exactly one object. */
	context->min_selected_objects = 1;
	context->max_selected_objects = 1;

	/* Get a list of all valid input objects from the engine. Any object
	 * that does not already belong to the snapshot plugin can be used as
	 * a new snapshot child object.
	 */
	EngFncs->get_object_list(0, DATA_TYPE, NULL,
				 NULL, VALID_INPUT_OBJECT, &objects);

	LIST_FOR_EACH(objects, itr, object) {
		if (object->plugin != my_plugin_record) {
			EngFncs->insert_thing(context->acceptable_objects,
					      object, INSERT_AFTER, NULL);
		}
	}

	/* Initialize the option descriptor and list
	 * of possible volumes to snapshot.
	 */
	rc = init_option_descriptor_create(context);

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * set_option_create
 *
 * Set an option for a create task.
 **/
int set_option_create(task_context_t * context,
		      u_int32_t index,
		      value_t * value,
		      task_effect_t * effect)
{
	option_desc_array_t * od = context->option_descriptors;
	storage_object_t * snap_child;
	int i, rc = EINVAL;

	LOG_ENTRY();

	switch (index) {

	case SNAP_OPTION_ORG_VOLUME_INDEX:
		if (strlen(value->s) > EVMS_VOLUME_NAME_SIZE) {
			break;
		}

		/* Compare the input with the list of volumes
		 * from the constraint list.
		 */
		for (i = 0; i < od->option[index].constraint.list->count; i++) {
			rc = strcmp(value->s,
				    od->option[index].constraint.list->value[i].s);
			if (!rc) {
				strcpy(od->option[index].value.s, value->s);
				od->option[index].flags &= ~EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;
				break;
			}
		}
		break;

	case SNAP_OPTION_SNAPSHOT_INDEX:
		rc = strlen(value->s);
		if (rc < 1 || rc > EVMS_VOLUME_NAME_SIZE) {
			rc = EINVAL;
			break;
		}
		snap_child = EngFncs->first_thing(context->selected_objects, NULL);
		if (!snap_child) {
			rc = EINVAL;
			break;
		}
		rc = validate_snapshot_name(value->s, snap_child);
		if (!rc) {
			strcpy(od->option[index].value.s, value->s);
			od->option[index].flags &= ~EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;
		} else {
			MESSAGE(_("The name \"%s\" is already in use.\n"), value->s);
		}
		break;

	case SNAP_OPTION_CHUNKSIZE_INDEX:
		if (value->ui32 >= SNAPSHOT_MIN_CHUNK_SIZE &&
		    value->ui32 <= SNAPSHOT_MAX_CHUNK_SIZE &&
		    (value->ui32 & (value->ui32 - 1)) == 0) {
			od->option[index].value.ui32 = value->ui32;
			rc = 0;
		}
		break;

	case SNAP_OPTION_WRITEABLE_INDEX:
		od->option[index].value.b = value->b;
		rc = 0;
		break;

	default:
		break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * set_objects_create
 *
 * Validate the selected objects for a create task.
 **/
int set_objects_create(task_context_t * context,
		       list_anchor_t declined_objects,
		       task_effect_t * effect)
{
	option_desc_array_t * oda = context->option_descriptors;
	option_descriptor_t * od = &(oda->option[SNAP_OPTION_ORG_VOLUME_INDEX]);
	storage_object_t * object;
	int rc = 0;

	LOG_ENTRY();

	object = EngFncs->first_thing(context->selected_objects, NULL);
	if (!object) {
		rc = EINVAL;
		goto out;
	}

	/* Can't create a snapshot object on another snapshot object. */
	if (object->plugin == my_plugin_record) {
		rc = EINVAL;
		goto out;
	}

	/* Now that the object is selected, the list of origin volumes
	 * can be initialized.
	 */
	EngFncs->engine_free(od->constraint.list);
	od->constraint.list = NULL;
	rc = get_volume_list(&(od->constraint.list), object);

out:
	LOG_EXIT_INT(rc);
	return rc;
}

