/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: backup.c
 */

#include <unistd.h>
#include <sys/types.h>
#include <pthread.h>
#include <frontend.h>
#include <gtk/gtk.h>

#include "support.h"
#include "backup.h"
#include "thing.h"
#include "logging.h"
#include "pixmap.h"
#include "views.h"
#include "main.h"
#include "readable.h"
#include "message.h"

/*
 *
 *   void populate_clist_with_changed_objects (GtkCList *, change_record_array_t *)
 *
 *   Description:
 *      This routine populates the given changed objects clist with
 *      the name of the volumes, containers that have changed and
 *      why.
 *
 *   Entry:
 *      changes - array containing changed object names and reasons
 *
 *   Exit:
 *     The clist is populated with information on object changed
 *
 */
static void populate_clist_with_changed_objects(GtkCList * clist, change_record_array_t * changes)
{
	gint i;
	gint row;
	gchar *text[MAX_BU_COLUMNS];

	for (i = 0; i < changes->count; i++) {
		/* Don't show things that only need activation. */
		if (!(changes->changes_pending[i].changes & ~(CHANGE_ACTIVATE |
							      CHANGE_DEACTIVATE |
							      CHANGE_REACTIVATE))) {
			continue;
		}

		text[BU_ICON_COLUMN] = "";
		text[BU_NAME_COLUMN] = changes->changes_pending[i].name;
		text[BU_REASON_COLUMN] =
		    changes_flag_to_string(changes->changes_pending[i].changes);

		row = clist_append_row(clist, text);

		if (row != -1)
			set_clist_row_pixmap(clist, row, changes->changes_pending[i].type);

		g_free(text[BU_REASON_COLUMN]);
	}
}

/*
 *
 *   void on_details_button_clicked (GtkButton *, gpointer *)
 *
 *   Description:
 *      This routine displays a dialog listing the objects
 *      that indicate they have changes pending.
 *
 *   Entry:
 *      details_button - the details button clicked
 *      user_data - not used
 *
 *   Exit:
 *      A dialog is created and displayed populate with a list
 *      of objects that report changes.
 *
 */
static void on_details_button_clicked(GtkButton * details_button, gpointer * user_data)
{
	boolean changes_pending;
	change_record_array_t *changes = NULL;
	GtkWidget *window;
	GtkWidget *vbox;
	GtkWidget *text_box_frame;
	GtkWidget *scrolledwindow;
	GtkWidget *clist;
	GtkWidget *icon_col_label;
	GtkWidget *object_col_label;
	GtkWidget *reason_col_label;
	GtkWidget *hseparator;
	GtkWidget *hbuttonbox;
	GtkWidget *button;
	GdkPixmap *pixmap;
	GdkBitmap *mask;
	GtkAccelGroup *accel_group;
	GtkTooltips *tooltips;

	tooltips = gtk_tooltips_new();
	accel_group = gtk_accel_group_new();
	window = gtk_window_new(GTK_WINDOW_DIALOG);
	vbox = gtk_vbox_new(FALSE, 0);
	text_box_frame = gtk_frame_new(NULL);
	scrolledwindow = gtk_scrolled_window_new(NULL, NULL);
	clist = gtk_clist_new(MAX_BU_COLUMNS);
	icon_col_label = gtk_label_new("");
	object_col_label = gtk_label_new(_("Name"));
	reason_col_label = gtk_label_new(_("Reason"));
	hseparator = gtk_hseparator_new();
	hbuttonbox = gtk_hbutton_box_new();
	button = gtk_button_new();

	gtk_accel_group_attach(accel_group, GTK_OBJECT(window));

	get_ok_pixmap(&pixmap, &mask);
	add_pixmap_label_to_button(button, pixmap, mask, _("_OK"), accel_group);

	gtk_widget_show(vbox);
	gtk_widget_show(text_box_frame);
	gtk_widget_show(scrolledwindow);
	gtk_widget_show(clist);
	gtk_widget_show(icon_col_label);
	gtk_widget_show(object_col_label);
	gtk_widget_show(reason_col_label);
	gtk_widget_show(hseparator);
	gtk_widget_show(hbuttonbox);
	gtk_widget_show(button);

	gtk_window_set_title(GTK_WINDOW(window), _("Objects with Pending Changes"));
	gtk_window_set_default_size(GTK_WINDOW(window), 475, -1);

	gtk_container_add(GTK_CONTAINER(window), vbox);

	gtk_box_pack_start(GTK_BOX(vbox), text_box_frame, TRUE, TRUE, 0);
	gtk_widget_set_usize(text_box_frame, 213, 300);
	gtk_container_set_border_width(GTK_CONTAINER(text_box_frame), 6);

	gtk_container_add(GTK_CONTAINER(text_box_frame), scrolledwindow);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolledwindow),
				       GTK_POLICY_NEVER, GTK_POLICY_ALWAYS);

	gtk_container_add(GTK_CONTAINER(scrolledwindow), clist);
	gtk_clist_set_column_width(GTK_CLIST(clist), BU_ICON_COLUMN, 24);
	gtk_clist_set_column_width(GTK_CLIST(clist), BU_NAME_COLUMN, 151);
	gtk_clist_set_column_width(GTK_CLIST(clist), BU_REASON_COLUMN, 80);
	gtk_clist_column_titles_show(GTK_CLIST(clist));

	gtk_clist_set_column_widget(GTK_CLIST(clist), BU_ICON_COLUMN, icon_col_label);
	gtk_clist_set_column_widget(GTK_CLIST(clist), BU_NAME_COLUMN, object_col_label);
	gtk_clist_set_column_widget(GTK_CLIST(clist), BU_REASON_COLUMN, reason_col_label);

	gtk_box_pack_start(GTK_BOX(vbox), hseparator, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbuttonbox, FALSE, FALSE, 0);
	gtk_button_box_set_layout(GTK_BUTTON_BOX(hbuttonbox), GTK_BUTTONBOX_END);
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(hbuttonbox), 18);
	gtk_button_box_set_child_size(GTK_BUTTON_BOX(hbuttonbox), 108, 46);

	gtk_container_add(GTK_CONTAINER(hbuttonbox), button);
	GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
	gtk_widget_grab_default(button);

	if (evms_changes_pending(&changes_pending, &changes) == 0 && changes != NULL) {
		populate_clist_with_changed_objects(GTK_CLIST(clist), changes);
		evms_free(changes);
	}

	gtk_tooltips_set_tip(tooltips, button, _("Close this window"), NULL);

	gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
				  gtk_widget_destroy, GTK_OBJECT(window));

	gtk_signal_connect(GTK_OBJECT(clist), "select_row",
			   GTK_SIGNAL_FUNC(on_readonly_clist_select_row), NULL);

	gtk_widget_show(window);
}

/*
 *
 *   void display_cannot_backup_metadata_dialog (void)
 *
 *   Description:
 *      This routine creates and displays a dialog that
 *      tells the user that metadata cannot be backed up
 *      at this time.due to metadata changes pending.  A
 *      "details" button shows the user what metadata
 *      changes are pending.
 *
 *   Entry:
 *      Nothing
 *
 *   Exit:
 *      We present a dialog that tells the user that
 *      metadata cannot be backed up at this time.
 *
 */
void display_cannot_backup_metadata_dialog(void)
{
	GtkWidget *cannot_backup_window;
	GtkWidget *vbox;
	GtkWidget *question_label;
	GtkWidget *hbuttonbox;
	GtkWidget *ok_button;
	GtkWidget *details_button;
	GtkWidget *pixmap;
	GdkBitmap *mask;
	GdkPixmap *gdk_pixmap;
	GtkAccelGroup *accel_group;
	GtkTooltips *tooltips;

	tooltips = gtk_tooltips_new();
	accel_group = gtk_accel_group_new();

	get_dialog_pixmap(QUESTION_PIXMAP, &gdk_pixmap, &mask);

	cannot_backup_window = gtk_window_new(GTK_WINDOW_DIALOG);
	gtk_window_set_title(GTK_WINDOW(cannot_backup_window),
			     _("Cannot Backup Metadata"));
	gtk_window_set_position(GTK_WINDOW(cannot_backup_window), GTK_WIN_POS_CENTER);
	gtk_accel_group_attach(accel_group, GTK_OBJECT(cannot_backup_window));

	vbox = gtk_vbox_new(FALSE, 0);
	question_label = gtk_label_new(_("The metadata cannot be backed up because metadata are scheduled to be written."));
	hbuttonbox = gtk_hbutton_box_new();
	ok_button = gtk_button_new();
	details_button = gtk_button_new();
	pixmap = gtk_pixmap_new(gdk_pixmap, mask);

	get_ok_pixmap(&gdk_pixmap, &mask);
	add_pixmap_label_to_button(ok_button, gdk_pixmap, mask, _("_OK"), accel_group);

	get_search_16_pixmap(&gdk_pixmap, &mask);
	add_pixmap_label_to_button(details_button, gdk_pixmap, mask, _("_Details"), accel_group);

	gtk_container_add(GTK_CONTAINER(cannot_backup_window), vbox);

	gtk_box_pack_start(GTK_BOX(vbox), pixmap, FALSE, FALSE, 0);
	gtk_misc_set_alignment(GTK_MISC(pixmap), 0.10, 1);
	gtk_misc_set_padding(GTK_MISC(pixmap), 0, 5);

	gtk_box_pack_start(GTK_BOX(vbox), question_label, FALSE, FALSE, 0);
	gtk_label_set_line_wrap(GTK_LABEL(question_label), TRUE);
	gtk_misc_set_padding(GTK_MISC(question_label), 63, 20);

	gtk_box_pack_start(GTK_BOX(vbox), hbuttonbox, TRUE, TRUE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(hbuttonbox), 10);
	gtk_button_box_set_layout(GTK_BUTTON_BOX(hbuttonbox), GTK_BUTTONBOX_END);
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(hbuttonbox), 14);
	gtk_button_box_set_child_size(GTK_BUTTON_BOX(hbuttonbox), -1, 50);

	gtk_container_add(GTK_CONTAINER(hbuttonbox), details_button);
	GTK_WIDGET_SET_FLAGS(details_button, GTK_CAN_DEFAULT);
	gtk_tooltips_set_tip(tooltips, details_button,
			     _("List objects that report changes pending"), NULL);

	gtk_container_add(GTK_CONTAINER(hbuttonbox), ok_button);
	GTK_WIDGET_SET_FLAGS(ok_button, GTK_CAN_DEFAULT);

	gtk_signal_connect_object(GTK_OBJECT(ok_button), "clicked",
				  gtk_widget_destroy, GTK_OBJECT(cannot_backup_window));
	gtk_signal_connect(GTK_OBJECT(details_button), "clicked", on_details_button_clicked, NULL);

	gtk_widget_grab_default(ok_button);
	gtk_window_set_transient_for(GTK_WINDOW(cannot_backup_window),
				     GTK_WINDOW(get_main_window_id()));

	gtk_widget_show_all(cannot_backup_window);
}

/*
 *
 *   void on_backup_button_clicked (GtkButton *, gpointer *)
 *
 *   Description:
 *      This routine calls the function that initiates the
 *      backup of metadata. It also dismisses the dialog
 *      containing the button that caused this callback
 *      to get invoked.
 *
 *   Entry:
 *      button    - the backup button clicked
 *      user_data - contains either a gboolean which
 *                  determines whether an exit should
 *                  occur after a successful backup
 *
 *   Exit:
 *      Backup metadata is invoked and the button's parent dialog
 *      is destroyed.
 *
 */
void on_backup_button_clicked(GtkButton * button, gpointer * user_data)
{
	int rc;

	gtk_widget_destroy(gtk_widget_get_toplevel(GTK_WIDGET(button)));

	rc = evms_metadata_backup(NULL);

	if (rc == SUCCESS) {
		display_results_window(rc, _("Backup Metadata"),
				       _("Metadata backed up successfully.\n"),
				       NULL, TRUE, get_main_window_id());
	} else {
		display_results_window(rc, _("Backup Metadata"),
				       _("An error occured while backing up the metadata.\n"),
				       NULL, TRUE, get_main_window_id());
	}
}

/*
 *
 *   void display_backup_metadata_confirmation_dialog (void)
 *
 *   Description:
 *      This routine creates and displays a dialog to
 *      allow the user to confirm they want to backup
 *      metadata or cancel and return to the main window.
 *
 *   Entry:
 *      Nothing
 *
 *   Exit:
 *      We present a dialog that asks the user whether they
 *      would like to backup changes or cancel and return
 *      to the main window.
 *
 */
void display_backup_metadata_confirmation_dialog(void)
{
	GtkWidget *backup_confirmation_window;
	GtkWidget *vbox;
	GtkWidget *question_label;
	GtkWidget *hbuttonbox;
	GtkWidget *backup_button;
	GtkWidget *cancel_button;
	GtkWidget *pixmap;
	GdkBitmap *mask;
	GdkPixmap *gdk_pixmap;
	GtkAccelGroup *accel_group;
	GtkTooltips *tooltips;

	tooltips = gtk_tooltips_new();
	accel_group = gtk_accel_group_new();

	get_dialog_pixmap(QUESTION_PIXMAP, &gdk_pixmap, &mask);

	backup_confirmation_window = gtk_window_new(GTK_WINDOW_DIALOG);
	gtk_window_set_title(GTK_WINDOW(backup_confirmation_window),
			     _("Backup Metadata Confirmation"));
	gtk_window_set_position(GTK_WINDOW(backup_confirmation_window), GTK_WIN_POS_CENTER);
	gtk_accel_group_attach(accel_group, GTK_OBJECT(backup_confirmation_window));

	vbox = gtk_vbox_new(FALSE, 0);
	question_label = gtk_label_new(_("Do you wish to backup the metadata now or cancel?"));
	hbuttonbox = gtk_hbutton_box_new();
	backup_button = gtk_button_new();
	cancel_button = gtk_button_new();
	pixmap = gtk_pixmap_new(gdk_pixmap, mask);

	get_commit_16_pixmap(&gdk_pixmap, &mask);
	add_pixmap_label_to_button(backup_button, gdk_pixmap, mask, _("_Backup"), accel_group);

	get_cancel_pixmap(&gdk_pixmap, &mask);
	add_pixmap_label_to_button(cancel_button, gdk_pixmap, mask, _("_Cancel"), accel_group);

	gtk_container_add(GTK_CONTAINER(backup_confirmation_window), vbox);

	gtk_box_pack_start(GTK_BOX(vbox), pixmap, FALSE, FALSE, 0);
	gtk_misc_set_alignment(GTK_MISC(pixmap), 0.10, 1);
	gtk_misc_set_padding(GTK_MISC(pixmap), 0, 5);

	gtk_box_pack_start(GTK_BOX(vbox), question_label, FALSE, FALSE, 0);
	gtk_label_set_line_wrap(GTK_LABEL(question_label), TRUE);
	gtk_misc_set_padding(GTK_MISC(question_label), 63, 20);

	gtk_box_pack_start(GTK_BOX(vbox), hbuttonbox, TRUE, TRUE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(hbuttonbox), 10);
	gtk_button_box_set_layout(GTK_BUTTON_BOX(hbuttonbox), GTK_BUTTONBOX_END);
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(hbuttonbox), 14);
	gtk_button_box_set_child_size(GTK_BUTTON_BOX(hbuttonbox), -1, 50);

	gtk_container_add(GTK_CONTAINER(hbuttonbox), cancel_button);
	GTK_WIDGET_SET_FLAGS(cancel_button, GTK_CAN_DEFAULT);
	gtk_tooltips_set_tip(tooltips, cancel_button, _("Cancel save and dismiss this dialog"),
			     NULL);

	gtk_container_add(GTK_CONTAINER(hbuttonbox), backup_button);
	GTK_WIDGET_SET_FLAGS(backup_button, GTK_CAN_DEFAULT);
	gtk_tooltips_set_tip(tooltips, backup_button, _("Backup metadata"), NULL);

	gtk_signal_connect(GTK_OBJECT(backup_button), "clicked", on_backup_button_clicked,
			   GINT_TO_POINTER(FALSE));
	gtk_signal_connect_object(GTK_OBJECT(cancel_button), "clicked", gtk_widget_destroy,
				  GTK_OBJECT(backup_confirmation_window));

	gtk_widget_grab_default(backup_button);
	gtk_window_set_transient_for(GTK_WINDOW(backup_confirmation_window),
				     GTK_WINDOW(get_main_window_id()));

	gtk_widget_show_all(backup_confirmation_window);
}

