/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@susqu.edu                 *
*************************************************************/

/**********************************************************************
*
*  File: symtable.c
*
*  Purpose: Handles symbol table functions for queries.
*/

#include "include.h"

/* symbol table variables */
#define SYMMAX 200
struct sym symtable[SYMMAX];    /* symbol table */
#define MAXSYMDEPTH 10
int symsave[MAXSYMDEPTH];  /* start of scope pointers */
int symdepth;  /* depth of symbol table */
int symtop;  /* current stack top, index of first empty spot */

/**********************************************************************
*
*  function: begin_scope()
*
*  purpose:  enter scope, save symbol stack position
*/

int begin_scope()
{
  if ( symdepth >= MAXSYMDEPTH - 1 )
     kb_error(1666,"Scopes nested too deep.\n",RECOVERABLE);

  symsave[symdepth++] = symtop;
  return symdepth;
}


/**********************************************************************
*
*  function: end_scope()
*
*  purpose:  leave scope, pop symbol stack 
*/

void end_scope()
{ int oldsymtop = symtop;
  if ( symdepth >= 1 ) /* clear stack */
      symtop = symsave[--symdepth];
  if ( oldsymtop > symtop )
     memset((char*)(symtable+symtop),0,(oldsymtop-symtop)*sizeof(struct sym));
}

/**********************************************************************
*
*  function: set_scope()
*
*  purpose:  chop symbol stack after error
*/

void set_scope(depth)
int depth;
{ int oldsymtop = symtop;
  symdepth = depth-1;
  if ( symdepth > 0 )
    symtop = symsave[symdepth-1];
  else symtop = 0;
  if ( oldsymtop > symtop )
     memset((char*)(symtable+symtop),0,(oldsymtop-symtop)*sizeof(struct sym));
}



/**********************************************************************
*
*  function: clear_symtable()
*
*  purpose:  resets symbol table after command error
*/

void clear_symtable()
{ 
  
  symdepth = 0;
  symtop = 0;
  memset((char*)symtable,0,SYMMAX*sizeof(struct sym));
}

/**********************************************************************
*
*  function: symbol_add()
*
*  purpose: add symbol to symbol table
*
*/
#ifdef __WIN32__
struct sym * symbol_add(char * name,int type)
#else
struct sym * symbol_add(name,type)
char *name;
int type;
#endif

{ int i;

  /* check for duplication in scope */
  for ( i = symsave[symdepth-1] ; i < symtop ; i++ )
     if ( strncmp(symtable[i].name,name,SYMNAMESIZE) == 0 )
     { sprintf(errmsg,"Duplicate name '%s' in scope.\n",name);
        kb_error(1667,errmsg,RECOVERABLE);
     }
     
  if ( symtop >= SYMMAX )
     kb_error(1668,"Internal error: Symbol table overflow.\n",RECOVERABLE);

  strncpy(symtable[symtop].name,name,SYMNAMESIZE);
  symtable[symtop].type = type;
  return symtable + symtop++;
}

/**********************************************************************
*
*  function: symbol_lookup()
*
*  purpose: find symbol in symbol table.  Returns pointer to
*              symbol structure if found, NULL if not.
*/

#ifdef __WIN32__
struct sym *symbol_lookup(char *name)
#else
struct sym *symbol_lookup(name)
char *name;
#endif
{
  int i;

  for ( i = symtop-1 ; i >= 0 ; i-- )
     if ( strncmp(symtable[i].name,name,SYMNAMESIZE) == 0 )
        return symtable + i;  /* success */

  return NULL; /* failure */
}


/**********************************************************************
*
*  function: add_global()
*
*  purpose:  add new global variable
*
*  return:    index number in global list
*                -index - 1 if already there
*/

int add_global(name)
char *name;
{ int k;
  int slot = -1;

  if ( web.global_count == 0 )
    { web.maxglobals = 100;
      dy_globals = dy_calloc(web.maxglobals,sizeof(struct global));
      Globals = globals;  /* handy for debugging */
    }

  /* search for duplicate name or empty slot */
  for ( k = 0 ; k < web.global_count ; k++ )
     if ( stricmp(name,globals[k].name) == 0 )
        { if ( globals[k].flags & LEFTOVER )
             { memset((char*)(globals+k),0,sizeof(struct global));
                slot = k;
                break;
             }
          else return -k-1;
        }
     else if ( (globals[k].flags == 0) && (slot < 0) )
        { slot = k; break; }

  if ( slot < 0 )
  { if ( web.global_count >= web.maxglobals )
     { web.maxglobals *= 2;
        dy_globals = dy_realloc(dy_globals,web.maxglobals,sizeof(struct global));
        Globals = globals;  /* handy for debugging */
     }
     slot = web.global_count++;
  }

  strncpy(globals[slot].name,name,GLOBAL_NAME_SIZE);
  globals[slot].flags |= GLOB_USED;
  globals[slot].delta = OPTPARAM_DELTA;
  globals[slot].scale = 1.0;
  if ( strcmp(name,globals[slot].name) != 0 )
    { sprintf(errmsg,"Name too long. Truncated to %s.\n",globals[slot].name);
      kb_error(1669,errmsg,WARNING);

    }
  return slot;
}


/**********************************************************************
*
*  function: lookup_global()
*
*  purpose:  searches for global variable
*
*  return:    index number in global list, -1 if not found
*/

int lookup_global(name)
char *name;
{ int k;
  /* search for name */
  for ( k = 0 ; k < web.global_count ; k++ )
     if ( stricmp(name,globals[k].name) == 0 ) 
         return k;
     
  return -1;

}

/*****************************************************************
*
* function: clear_globals()
*
* purpose: strip non-permanent definitions from global variable list.
*             Called for every new surface.  Does not change position
*             of permanent globals in list.
*/

void clear_globals()
{ int i;

  for ( i = 0 ; i < web.global_count ; i++ )
  { if ( globals[i].flags & SUBROUTINE ) 
      free_expr(&globals[i].value.proc);
    if ( globals[i].flags & STRINGVAL ) 
      myfree(globals[i].value.string);
    if ( !(globals[i].flags & PERMANENT) )
          memset((char*)(globals+i),0,sizeof(struct global));
    else
      { globals[i].flags &=
               (ORDINARY_PARAM|SUBROUTINE|PERMANENT|GLOB_USED|STRINGVAL);
        globals[i].flags |= LEFTOVER;
      }
    if ( globals[i].gradhess ) myfree((char*)globals[i].gradhess);
    globals[i].gradhess = NULL;
  }

  /* cut global count down to top of permanent globals */
  while ( (web.global_count > 0) && !(globals[web.global_count-1].flags & PERMANENT) )
     web.global_count--;
}

