/* $Id: cim_cardbus.c,v 1.43 2009-11-12 12:05:20 vrsieh Exp $ 
 *
 * Copyright (C) 2007-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */
#include "config.h"

#include <assert.h>
#include <sched.h>
#include <stdio.h>
#include <string.h>

#ifdef DARWIN
#include <string.h>
#endif

#include "glue-io.h"

#include "sig_cardbus.h"
#include "cim_cardbus.h"


static void
cim_cardbus_p5V_set(void *_css, unsigned int val)
{
	struct cim_cardbus *css = (struct cim_cardbus *) _css;
	struct cim_cardbus_msg msg;

	msg.msg_type = CIM_CARDBUS_POWER;
	msg.ack = 0;
	msg.p5V = val;

	cim_send(&css->bridge, &msg, sizeof(msg));
}

static void
cim_cardbus_n_reset_set(void *_css, unsigned int val)
{
	struct cim_cardbus *css = (struct cim_cardbus *) _css;
	struct cim_cardbus_msg msg;

	msg.msg_type = CIM_CARDBUS_N_RESET;
	msg.ack = 0;
	msg.n_reset = val;

	cim_send(&css->bridge, &msg, sizeof(msg));
}

static int
cim_cardbus__in(
	void *_css,
	unsigned int type,
	uint32_t *valp,
	uint32_t addr,
	unsigned int bs
)
{
	struct cim_cardbus *css = (struct cim_cardbus *) _css;
	struct cim_cardbus_msg req;
	struct cim_cardbus_msg *msg;

	req.msg_type = type;
	req.ack = 0;
	req.addr = addr;
	req.bs = bs;

	cim_send(&css->bridge, &req, sizeof(req));

	do {
		sched_yield();
		io_do();
	} while (css->ans_available == 0);
	css->ans_available = 0;

	msg = &css->ans;
	assert(msg->msg_type == type);

	*valp = msg->val;

	return 0;
}

static int
cim_cardbus__out(
	void *_css,
	unsigned int type,
	uint32_t val,
	uint32_t addr,
	unsigned int bs
)
{
	struct cim_cardbus *css = (struct cim_cardbus *) _css;
	struct cim_cardbus_msg msg;

	msg.msg_type = type;
	msg.ack = 0;
	msg.addr = addr;
	msg.bs = bs;
	msg.val = val;

	cim_send(&css->bridge, &msg, sizeof(msg));
	return 0;
}

static int
cim_cardbus_type_addr(
	void *css,
	unsigned int type,
	uint32_t addr
)
{
	return cim_cardbus__out(css, CIM_CARDBUS_TYPE_ADDR, 0, addr, type);
}

static int
cim_cardbus_read_data(
	void *css,
	unsigned int bs,
	uint32_t *valp
)
{
	return cim_cardbus__in(css, CIM_CARDBUS_READ_DATA, valp, 0, bs);
}

static int
cim_cardbus_write_data(
	void *css,
	unsigned int bs,
	uint32_t val
)
{
	return cim_cardbus__out(css, CIM_CARDBUS_WRITE_DATA, val, 0, bs);
}

static int
cim_cardbus_map(
	void *_css,
	unsigned long pa,
	char **haddr_p
)
{
	/* FIXME */
	return -1;
}

static void
cim_cardbus_irq_set(void *_css, unsigned int val)
{
	struct cim_cardbus *css = (struct cim_cardbus *) _css;
	struct cim_cardbus_msg msg;

	msg.msg_type = CIM_CARDBUS_IRQ;
	msg.ack = 0;
	msg.val = val;

	cim_send(&css->bridge, &msg, sizeof(msg));
}

static void
cim_cardbus_interrupt(void *_css, void *_buf, unsigned int bufsize)
{
	struct cim_cardbus *css = (struct cim_cardbus *) _css;
	struct cim_cardbus_msg *msg = (struct cim_cardbus_msg *) _buf;

	assert(bufsize == sizeof(*msg));

	sig_boolean_set(css->sig_conn, css, cim_get_peer_count(&css->bridge));

	if (msg->ack) {
		assert(! css->ans_available);
		memcpy(&css->ans, msg, sizeof(*msg));
		css->ans_available = 1;

	} else switch (msg->msg_type) {
	case CIM_CARDBUS_POWER:
		sig_boolean_set(css->sig_p5V, css, msg->p5V);
		break;

	case CIM_CARDBUS_N_RESET:
		sig_boolean_set(css->sig_n_reset, css, msg->n_reset);
		break;

	case CIM_CARDBUS_TYPE_ADDR:
		if (sig_pci_bus_main_type_addr(css->sig_bus, css, msg->bs, msg->addr) == 0) {
			css->type = -1;
		} else {
			css->type = msg->bs;
			css->addr = msg->addr;
		}
		break;

	case CIM_CARDBUS_READ_DATA:
		if (css->type == -1) {
			sig_pci_bus_main_read_data(css->sig_bus, css, msg->bs, &msg->val);
		} else switch (css->type) {
		case SIG_PCI_BUS_C0R:
			sig_pci_bus_main_c0r(css->sig_bus, css,
					css->addr, msg->bs, &msg->val);
			break;
		case SIG_PCI_BUS_C1R:
			sig_pci_bus_c1r(css->sig_bus, css,
					css->addr, msg->bs, &msg->val);
			break;
		case SIG_PCI_BUS_IOR:
			sig_pci_bus_ior(css->sig_bus, css,
					css->addr, msg->bs, &msg->val);
			break;
		case SIG_PCI_BUS_MR:
			sig_pci_bus_mr(css->sig_bus, css,
					css->addr, msg->bs, &msg->val);
			break;
		default:
			assert(0); /* Mustn't happen. */
		}
		msg->ack = 1;
		cim_send(&css->bridge, msg, sizeof(*msg));
		break;

	case CIM_CARDBUS_WRITE_DATA:
		if (css->type == -1) {
			sig_pci_bus_main_write_data(css->sig_bus, css, msg->bs, msg->val);
		} else switch (css->type) {
		case SIG_PCI_BUS_C0W:
			sig_pci_bus_main_c0w(css->sig_bus, css,
					css->addr, msg->bs, msg->val);
			break;
		case SIG_PCI_BUS_C1W:
			sig_pci_bus_c1w(css->sig_bus, css,
					css->addr, msg->bs, msg->val);
			break;
		case SIG_PCI_BUS_IOW:
			sig_pci_bus_iow(css->sig_bus, css,
					css->addr, msg->bs, msg->val);
			break;
		case SIG_PCI_BUS_MW:
			sig_pci_bus_mw(css->sig_bus, css,
					css->addr, msg->bs, msg->val);
			break;
		default:
			assert(0); /* Mustn't happen. */
		}
		break;

	case CIM_CARDBUS_IRQ:
		sig_boolean_or_set(css->sig_irq, css, msg->val);
		break;

	default:
		assert(0); /* Mustn't happen. */
	}
}

void
cim_cardbus_reconfig(struct cim_cardbus *css)
{
	cim_reconfig(&css->bridge);
	sig_boolean_set(css->sig_conn, css, cim_get_peer_count(&css->bridge));
}

void
cim_cardbus_init(
	struct cim_cardbus *css,
	struct sig_cardbus *sig
)
{
	static const struct sig_boolean_funcs p5V_funcs = {
		.set =		cim_cardbus_p5V_set,
	};
	static const struct sig_boolean_funcs n_reset_funcs = {
		.set =		cim_cardbus_n_reset_set,
	};
	static const struct sig_pci_bus_main_funcs bus_funcs = {
		.type_addr =	cim_cardbus_type_addr,
		.read_data =	cim_cardbus_read_data,
		.write_data =	cim_cardbus_write_data,

		.map_r =	cim_cardbus_map,
		.map_w =	cim_cardbus_map,
	};
	static const struct sig_boolean_or_funcs irq_funcs = {
		.set_ext =	cim_cardbus_irq_set,
	};

	css->sig_conn = sig->conn;

	css->sig_p5V = sig->p5V;
	sig_boolean_connect_out(sig->p5V, css, 0);
	sig_boolean_connect_in(sig->p5V, css, &p5V_funcs);

	css->sig_n_reset = sig->n_reset;
	sig_boolean_connect_out(sig->n_reset, css, 0);
	sig_boolean_connect_in(sig->n_reset, css, &n_reset_funcs);

	css->sig_bus = sig->bus;
	sig_pci_bus_main_connect(sig->bus, css, &bus_funcs);

	css->sig_irq = sig->irq;
	sig_boolean_or_connect_in(sig->irq, css, &irq_funcs);

	cim_connect(&css->bridge, cim_cardbus_interrupt, css);
}

void
cim_cardbus_create(struct cim_cardbus *css)
{
	cim_create(&css->bridge);
}

void
cim_cardbus_destroy(struct cim_cardbus *css)
{
	cim_destroy(&css->bridge);
}
