/*
 * $Id: patternm.c,v 1.157 2009-06-03 11:34:00 vrsieh Exp $ 
 *
 * Copyright (C) 2005-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>
#include <errno.h>
#include <inttypes.h>
#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "glue-log.h"
#include "glue-png.h"
#include "glue-ppm.h"

#include "patternm.h"

/** a point in the 2d landscape has an x and y. */
struct point {
	uint16_t x;	/**< x coordinate */
	uint16_t y;	/**< y coordinate */
};

/** an area in the 2d landscape has a width and a height */
struct area {
	uint16_t w;	/**< width */
	uint16_t h;	/**< height */
};

/** a rectangular area */
struct rectangle {
	struct point origin;	/**< upper left point (x1/y1) */
	struct area dimensions;	/**< width and heigth */
};

static bool
rect_fits_into(const struct rectangle inner, const struct rectangle outer);

static bool
point_inside(const struct point p, const struct rectangle r);

static bool
area_smaller_or_same(const struct area a, const struct area reference);

/* include subcomponentent gfx */

#define DEFINITIONS
#define NAME 		patternm_gfx
#define NAME_(x)	patternm_gfx_ ## x
#include "patternm_gfx.c"
#undef NAME_
#undef NAME
#undef DEFINITIONS

#define DEFINITIONS
#define NAME 		patternm_text
#define NAME_(x)	patternm_text_ ## x
#include "patternm_text.c"
#undef NAME_
#undef NAME
#undef DEFINITIONS

#define DEFINITIONS
#define NAME 		patternm_asc
#define NAME_(x)	patternm_asc_ ## x
#include "patternm_asc.c"
#undef NAME_
#undef NAME
#undef DEFINITIONS

struct cpssp {
	/* Config */

	/* Signals */
	struct sig_match *port_patternm_gfx[4];

	struct {
		struct sig_integer *x;
		struct sig_integer *y;
		struct sig_integer *w;
		struct sig_integer *h;
	} port_match_rectangle[4];

	/* State */

	/** contents of screen */
	uint32_t current_screen[MAX_HEIGHT][MAX_WIDTH];
	/** dimensions of the screen */
	struct area screen_dimensions;
	/** number of updates until matching is done */
	unsigned int sync_count;

	/** patternm_gfx subcomponent */
#define STATE
#define NAME 		patternm_gfx
#define NAME_(x)	patternm_gfx_ ## x
#include "patternm_gfx.c"
#undef NAME_
#undef NAME
#undef STATE

#define STATE
#define NAME 		patternm_text
#define NAME_(x)	patternm_text_ ## x
#include "patternm_text.c"
#undef NAME_
#undef NAME
#undef STATE

#define STATE
#define NAME		patternm_asc
#define NAME_(x)	patternm_asc_ ## x
#include "patternm_asc.c"
#undef NAME_
#undef NAME
#undef STATE
};

#define BEHAVIOR
#define NAME 		patternm_gfx
#define NAME_(x)	patternm_gfx_ ## x
#include "patternm_gfx.c"
#undef NAME_
#undef NAME
#undef BEHAVIOR

#define BEHAVIOR
#define NAME 		patternm_text
#define NAME_(x)	patternm_text_ ## x
#include "patternm_text.c"
#undef NAME_
#undef NAME
#undef BEHAVIOR

#define BEHAVIOR
#define NAME		patternm_asc
#define NAME_(x)	patternm_asc_ ## x
#include "patternm_asc.c"
#undef NAME_
#undef NAME
#undef BEHAVIOR

/* =============== 2D functions =========================================== */

/** return true if p is inside r
 *  @param p point to check
 *  @param r rectangle.
 *  @return true if p is inside r (including the borders)
 */
static bool
point_inside(const struct point p, const struct rectangle r)
{
	return (r.origin.x <= p.x) 
	    && (r.origin.y <= p.y)
	    && (p.x < r.origin.x + r.dimensions.w) 
	    && (p.y < r.origin.y + r.dimensions.h);
}

/** return true if inner fits into outer
 *  @param inner inner rectangle.
 *  @param outer outer rectangel.
 *  @return true, if inner fits (or is equal sized as) outer.
 */
static bool
rect_fits_into(const struct rectangle inner, const struct rectangle outer)
{
	struct point lower_right = {
		.x = inner.origin.x + inner.dimensions.w - 1,
		.y = inner.origin.y + inner.dimensions.h - 1
	};

	return point_inside(inner.origin, outer) 
	    && point_inside(lower_right, outer);
}

/** return true if a is smaller or of equal size as reference
 *  @param a check area.
 *  @param reference reference area
 *  @return true, if a <= reference.
 */
static bool
area_smaller_or_same(const struct area a, const struct area reference)
{
	return (a.w <= reference.w)
	    && (a.h <= reference.h);
}

/* ======================================================================== */

static void
patternm_sync(void *s)
{
	struct cpssp* cpssp = (struct cpssp*)s;
	cpssp->sync_count++;

	patternm_asc_sync(cpssp);

	if (2 <= cpssp->sync_count) {
		cpssp->sync_count = 0;
		patternm_gfx_sync(cpssp);
		patternm_text_sync(cpssp);
	}
}

static void
patternm_pixel_set(void *s, unsigned int x, unsigned int y,
		uint8_t r, uint8_t g, uint8_t b)
{
	struct cpssp* cpssp = (struct cpssp*)s;

	cpssp->current_screen[y][x] = (r << 16) | (g << 8) | (b << 0);

	patternm_asc_pixel_set(cpssp, x, y, r, g, b);
	patternm_gfx_buffer_event(cpssp);
	patternm_text_buffer_event(cpssp);
}

static void
patternm_size_set(void* s, unsigned int w, unsigned int h)
{
	struct cpssp* cpssp = (struct cpssp*)s;

	assert(0 <= w && w < MAX_WIDTH);
	assert(0 <= h && h < MAX_HEIGHT);

	cpssp->screen_dimensions.h = h;
	cpssp->screen_dimensions.w = w;
	/* FIXME blank screen? */

	patternm_asc_size_set(cpssp, w, h);
	patternm_gfx_buffer_event(cpssp);
	patternm_text_buffer_event(cpssp);
}

static void
patternm_slot_event(
	void *_cpssp,
	unsigned int nr,
        bool visible,
	uint16_t x,
	uint16_t y,
	uint16_t w,
	uint16_t h
)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (! visible) {
		/* WARNING: Send x first! */
		sig_integer_set(cpssp->port_match_rectangle[nr].x,
				cpssp, -1);
		sig_integer_set(cpssp->port_match_rectangle[nr].y,
				cpssp, -1);
		sig_integer_set(cpssp->port_match_rectangle[nr].w,
				cpssp, -1);
		sig_integer_set(cpssp->port_match_rectangle[nr].h,
				cpssp, -1);
	} else {
		sig_integer_set(cpssp->port_match_rectangle[nr].y,
				cpssp, y);
		sig_integer_set(cpssp->port_match_rectangle[nr].w,
				cpssp, w);
		sig_integer_set(cpssp->port_match_rectangle[nr].h,
				cpssp, h);
		/* WARNING: Send x last! */
		sig_integer_set(cpssp->port_match_rectangle[nr].x,
				cpssp, x);
	}
}

static void
patternm_slot0_event(
	void *_cpssp,
        bool visible,
	uint16_t x,
	uint16_t y,
	uint16_t w,
	uint16_t h
)
{
	patternm_slot_event(_cpssp, 0, visible, x, y, w, h);
}

static void
patternm_slot1_event(
	void *_cpssp,
        bool visible,
	uint16_t x,
	uint16_t y,
	uint16_t w,
	uint16_t h
)
{
	patternm_slot_event(_cpssp, 1, visible, x, y, w, h);
}

static void
patternm_slot2_event(
	void *_cpssp,
        bool visible,
	uint16_t x,
	uint16_t y,
	uint16_t w,
	uint16_t h
)
{
	patternm_slot_event(_cpssp, 2, visible, x, y, w, h);
}

static void
patternm_slot3_event(
	void *_cpssp,
        bool visible,
	uint16_t x,
	uint16_t y,
	uint16_t w,
	uint16_t h
)
{
	patternm_slot_event(_cpssp, 3, visible, x, y, w, h);
}

static void
pattern_pattern_rectangle0_set(void *_cpssp, const char *pattern)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (pattern[0]) {
		sig_match_add_match(cpssp->port_patternm_gfx[0], cpssp, pattern);
	} else {
		sig_integer_set(cpssp->port_match_rectangle[0].x, cpssp, -2);
		sig_integer_set(cpssp->port_match_rectangle[0].y, cpssp, -2);
		sig_integer_set(cpssp->port_match_rectangle[0].w, cpssp, -2);
		sig_integer_set(cpssp->port_match_rectangle[0].h, cpssp, -2);
		sig_match_remove_match(cpssp->port_patternm_gfx[0], cpssp);
	}
}

static void
pattern_pattern_rectangle1_set(void *_cpssp, const char *pattern)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (pattern[1]) {
		sig_match_add_match(cpssp->port_patternm_gfx[1], cpssp, pattern);
	} else {
		sig_integer_set(cpssp->port_match_rectangle[1].x, cpssp, -2);
		sig_integer_set(cpssp->port_match_rectangle[1].y, cpssp, -2);
		sig_integer_set(cpssp->port_match_rectangle[1].w, cpssp, -2);
		sig_integer_set(cpssp->port_match_rectangle[1].h, cpssp, -2);
		sig_match_remove_match(cpssp->port_patternm_gfx[1], cpssp);
	}
}

static void
pattern_pattern_rectangle2_set(void *_cpssp, const char *pattern)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (pattern[2]) {
		sig_match_add_match(cpssp->port_patternm_gfx[2], cpssp, pattern);
	} else {
		sig_integer_set(cpssp->port_match_rectangle[2].x, cpssp, -2);
		sig_integer_set(cpssp->port_match_rectangle[2].y, cpssp, -2);
		sig_integer_set(cpssp->port_match_rectangle[2].w, cpssp, -2);
		sig_integer_set(cpssp->port_match_rectangle[2].h, cpssp, -2);
		sig_match_remove_match(cpssp->port_patternm_gfx[2], cpssp);
	}
}

static void
pattern_pattern_rectangle3_set(void *_cpssp, const char *pattern)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (pattern[3]) {
		sig_match_add_match(cpssp->port_patternm_gfx[3], cpssp, pattern);
	} else {
		sig_integer_set(cpssp->port_match_rectangle[3].x, cpssp, -2);
		sig_integer_set(cpssp->port_match_rectangle[3].y, cpssp, -2);
		sig_integer_set(cpssp->port_match_rectangle[3].w, cpssp, -2);
		sig_integer_set(cpssp->port_match_rectangle[3].h, cpssp, -2);
		sig_match_remove_match(cpssp->port_patternm_gfx[3], cpssp);
	}
}

static void
patternm_init_members(struct cpssp* cpssp)
{
	int i;

	/* Set default resolution. */
	patternm_size_set(cpssp, 640, 400);
	cpssp->sync_count = 0;
	
	/* clear screen */
	for (i = 0; i < MAX_HEIGHT; i++) {
		memset(cpssp->current_screen[i], 0, 
				sizeof(cpssp->current_screen[0]));
	}
}

void *
patternm_create(
	const char *name,
	struct sig_manage *port_manage,
	struct sig_opt_rgb *port_video,
	struct sig_match *port_slot0,
	struct sig_match *port_slot1,
	struct sig_match *port_slot2,
	struct sig_match *port_slot3,
	struct sig_match *port_slot4,
	struct sig_match *port_slot5,
	struct sig_match *port_slot6,
	struct sig_match *port_slot7,
	struct sig_match *port_slot8,
	struct sig_match *port_slot9,
	struct sig_match *port_slot10,
	struct sig_match *port_slot11,
	struct sig_match *port_slot12,
	struct sig_match *port_slot13,
	struct sig_match *port_slot14,
	struct sig_match *port_slot15,

        struct sig_string *port_text0,
        struct sig_boolean *port_text0_state,
        struct sig_string *port_text1,
        struct sig_boolean *port_text1_state,
        struct sig_string *port_text2,
        struct sig_boolean *port_text2_state,
        struct sig_string *port_text3,
        struct sig_boolean *port_text3_state,

	struct sig_string *port_asc_text0,
	struct sig_boolean *port_asc_text0_state,
	struct sig_string *port_asc_text1,
	struct sig_boolean *port_asc_text1_state,
	struct sig_string *port_asc_text2,
	struct sig_boolean *port_asc_text2_state,
	struct sig_string *port_asc_text3,
	struct sig_boolean *port_asc_text3_state,

	struct sig_string *port_pattern_rectangle0,
	struct sig_integer *port_match_rectangle0_x,
	struct sig_integer *port_match_rectangle0_y,
	struct sig_integer *port_match_rectangle0_w,
	struct sig_integer *port_match_rectangle0_h,
	struct sig_string *port_pattern_rectangle1,
	struct sig_integer *port_match_rectangle1_x,
	struct sig_integer *port_match_rectangle1_y,
	struct sig_integer *port_match_rectangle1_w,
	struct sig_integer *port_match_rectangle1_h,
	struct sig_string *port_pattern_rectangle2,
	struct sig_integer *port_match_rectangle2_x,
	struct sig_integer *port_match_rectangle2_y,
	struct sig_integer *port_match_rectangle2_w,
	struct sig_integer *port_match_rectangle2_h,
	struct sig_string *port_pattern_rectangle3,
	struct sig_integer *port_match_rectangle3_x,
	struct sig_integer *port_match_rectangle3_y,
	struct sig_integer *port_match_rectangle3_w,
	struct sig_integer *port_match_rectangle3_h
)
{
	static const struct sig_opt_rgb_funcs vga_f = {
		.pixel_set = patternm_pixel_set,
		.size_set = patternm_size_set,
		.sync = patternm_sync
	};
	static const struct sig_match_funcs slot0_funcs = {
		.event = patternm_slot0_event,
	};
	static const struct sig_match_funcs slot1_funcs = {
		.event = patternm_slot1_event,
	};
	static const struct sig_match_funcs slot2_funcs = {
		.event = patternm_slot2_event,
	};
	static const struct sig_match_funcs slot3_funcs = {
		.event = patternm_slot3_event,
	};
	static const struct sig_string_funcs pattern_rectangle0_funcs = {
		.set = pattern_pattern_rectangle0_set,
	};
	static const struct sig_string_funcs pattern_rectangle1_funcs = {
		.set = pattern_pattern_rectangle1_set,
	};
	static const struct sig_string_funcs pattern_rectangle2_funcs = {
		.set = pattern_pattern_rectangle2_set,
	};
	static const struct sig_string_funcs pattern_rectangle3_funcs = {
		.set = pattern_pattern_rectangle3_set,
	};
	struct cpssp *cpssp;

	cpssp = malloc(sizeof(*cpssp));
	assert(cpssp);

	/*
	 * We store pixel in ARGB format in an uint32_t.
	 * sig_video does the same via defining MAX_DEPTH? (# bits).
	 *
	 * Hence assert that any change in sig_video will break early and 
	 * hard, and not in patternm_pixel_set callback.
	 */
	assert(sizeof(uint32_t) == MAX_DEPTH / 8);

	patternm_gfx_init(
		port_slot0,
		port_slot1,
		port_slot2,
		port_slot3,
		port_slot4,
		port_slot5,
		port_slot6,
		port_slot7,
		port_slot8,
		port_slot9,
		port_slot10,
		port_slot11,
		port_slot12,
		port_slot13,
		port_slot14,
		port_slot15,
		cpssp);

	patternm_init_members(cpssp);

	patternm_text_init(
		port_text0, port_text0_state,
		port_text1, port_text1_state,
		port_text2, port_text2_state,
		port_text3, port_text3_state,
		cpssp);

	patternm_asc_init(
		port_asc_text0,
		port_asc_text0_state,
		port_asc_text1,
		port_asc_text1_state,
		port_asc_text2,
		port_asc_text2_state,
		port_asc_text3,
		port_asc_text3_state,
		cpssp
	);

	cpssp->port_patternm_gfx[0] = port_slot0;
	cpssp->port_patternm_gfx[1] = port_slot1;
	cpssp->port_patternm_gfx[2] = port_slot2;
	cpssp->port_patternm_gfx[3] = port_slot3;

	cpssp->port_match_rectangle[0].x = port_match_rectangle0_x;
	cpssp->port_match_rectangle[0].y = port_match_rectangle0_y;
	cpssp->port_match_rectangle[0].w = port_match_rectangle0_w;
	cpssp->port_match_rectangle[0].h = port_match_rectangle0_h;
	cpssp->port_match_rectangle[1].x = port_match_rectangle1_x;
	cpssp->port_match_rectangle[1].y = port_match_rectangle1_y;
	cpssp->port_match_rectangle[1].w = port_match_rectangle1_w;
	cpssp->port_match_rectangle[1].h = port_match_rectangle1_h;
	cpssp->port_match_rectangle[2].x = port_match_rectangle2_x;
	cpssp->port_match_rectangle[2].y = port_match_rectangle2_y;
	cpssp->port_match_rectangle[2].w = port_match_rectangle2_w;
	cpssp->port_match_rectangle[2].h = port_match_rectangle2_h;
	cpssp->port_match_rectangle[3].x = port_match_rectangle3_x;
	cpssp->port_match_rectangle[3].y = port_match_rectangle3_y;
	cpssp->port_match_rectangle[3].w = port_match_rectangle3_w;
	cpssp->port_match_rectangle[3].h = port_match_rectangle3_h;

	/* Call */
	sig_match_connect(port_slot0, cpssp, &slot0_funcs);
	sig_match_connect(port_slot1, cpssp, &slot1_funcs);
	sig_match_connect(port_slot2, cpssp, &slot2_funcs);
	sig_match_connect(port_slot3, cpssp, &slot3_funcs);

	/* Out */
	sig_integer_connect_out(port_match_rectangle0_x, cpssp, -1);
	sig_integer_connect_out(port_match_rectangle0_y, cpssp, -1);
	sig_integer_connect_out(port_match_rectangle0_w, cpssp, -1);
	sig_integer_connect_out(port_match_rectangle0_h, cpssp, -1);
	sig_integer_connect_out(port_match_rectangle1_x, cpssp, -1);
	sig_integer_connect_out(port_match_rectangle1_y, cpssp, -1);
	sig_integer_connect_out(port_match_rectangle1_w, cpssp, -1);
	sig_integer_connect_out(port_match_rectangle1_h, cpssp, -1);
	sig_integer_connect_out(port_match_rectangle2_x, cpssp, -1);
	sig_integer_connect_out(port_match_rectangle2_y, cpssp, -1);
	sig_integer_connect_out(port_match_rectangle2_w, cpssp, -1);
	sig_integer_connect_out(port_match_rectangle2_h, cpssp, -1);
	sig_integer_connect_out(port_match_rectangle3_x, cpssp, -1);
	sig_integer_connect_out(port_match_rectangle3_y, cpssp, -1);
	sig_integer_connect_out(port_match_rectangle3_w, cpssp, -1);
	sig_integer_connect_out(port_match_rectangle3_h, cpssp, -1);

	/* In */
	sig_opt_rgb_connect(port_video, cpssp, &vga_f);

	sig_string_connect(port_pattern_rectangle0, cpssp, &pattern_rectangle0_funcs);
	sig_string_connect(port_pattern_rectangle1, cpssp, &pattern_rectangle1_funcs);
	sig_string_connect(port_pattern_rectangle2, cpssp, &pattern_rectangle2_funcs);
	sig_string_connect(port_pattern_rectangle3, cpssp, &pattern_rectangle3_funcs);

	return cpssp;
}

void
patternm_destroy(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	free(cpssp);
}
