/*
 * $Id: sig_magneto_optical.c,v 1.13 2009-10-02 13:43:13 vrsieh Exp $
 *
 * Copyright (C) 2004-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>

#include "fixme.h"

#include "sig_magneto_optical.h"

int
sig_magneto_optical_atip(
	struct sig_magneto_optical *b,
	void *s,
	struct cd_atip *atip
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		int (*func)(void *, struct cd_atip *);

		if (nr == b->member_count) {
			return -1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->atip;
		if (func) {
			return func(b->member[nr].s, atip);
		}
	}
}

int
sig_magneto_optical_read(
	struct sig_magneto_optical *b,
	void *s,
	int32_t blk,
	uint8_t *data,
	uint8_t *psub,
	uint8_t *qsub
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		int (*func)(void *, int32_t, uint8_t *, uint8_t *, uint8_t *);

		if (nr == b->member_count) {
			return -1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->read;
		if (func) {
			return func(b->member[nr].s, blk, data, psub, qsub);
		}
	}
}

int
sig_magneto_optical_write(
	struct sig_magneto_optical *b,
	void *s,
	int32_t blk,
	const uint8_t *data,
	const uint8_t *psub,
	const uint8_t *qsub
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		int (*func)(void *, int32_t,
				const uint8_t *, const uint8_t *, const uint8_t *);

		if (nr == b->member_count) {
			return -1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->write;
		if (func) {
			return func(b->member[nr].s, blk, data, psub, qsub);
		}
	}
}

void
sig_magneto_optical_connect(
	struct sig_magneto_optical *b,
	void *s,
	const struct sig_magneto_optical_funcs *f
)
{
	assert(b);
	assert(b->type == SIG_GEN_MAGNETO_OPTICAL);
	assert(b->member_count < sizeof(b->member) / sizeof(b->member[0]));

	b->member[b->member_count].s = s;
	b->member[b->member_count].f = f;
	b->member_count++;
}

void
sig_magneto_optical_disconnect(
	struct sig_magneto_optical *b,
	void *s
)
{
	unsigned int nr;

	assert(b);
	assert(b->type == SIG_GEN_MAGNETO_OPTICAL);

	for (nr = 0; ; nr++) {
		assert(nr < sizeof(b->member) / sizeof(b->member[0]));
		if (b->member[nr].s == s) {
			break;
		}
	}
	b->member_count--;
	b->member[nr].s = b->member[b->member_count].s;
	b->member[nr].f = b->member[b->member_count].f;
}

static int
sig_magneto_optical_s0_atip(
	void *_f,
	struct cd_atip *atip
)
{
	struct sig_magneto_optical_merge *f = _f;

	return sig_magneto_optical_atip(f->s1, f, atip);
}

static int
sig_magneto_optical_s0_read(
	void *_f,
	int32_t blk,
	uint8_t *data,
	uint8_t *psub,
	uint8_t *qsub
)
{
	struct sig_magneto_optical_merge *f = _f;

	return sig_magneto_optical_read(f->s1, f, blk, data, psub, qsub);
}

static int
sig_magneto_optical_s0_write(
	void *_f,
	int32_t blk,
	const uint8_t *data,
	const uint8_t *psub,
	const uint8_t *qsub
)
{
	struct sig_magneto_optical_merge *f = _f;

	return sig_magneto_optical_write(f->s1, f, blk, data, psub, qsub);
}

static int
sig_magneto_optical_s1_atip(
	void *_f,
	struct cd_atip *atip
)
{
	struct sig_magneto_optical_merge *f = _f;

	return sig_magneto_optical_atip(f->s0, f, atip);
}

static int
sig_magneto_optical_s1_read(
	void *_f,
	int32_t blk,
	uint8_t *data,
	uint8_t *psub,
	uint8_t *qsub
)
{
	struct sig_magneto_optical_merge *f = _f;

	return sig_magneto_optical_read(f->s0, f, blk, data, psub, qsub);
}

static int
sig_magneto_optical_s1_write(
	void *_f,
	int32_t blk,
	const uint8_t *data,
	const uint8_t *psub,
	const uint8_t *qsub
)
{
	struct sig_magneto_optical_merge *f = _f;

	return sig_magneto_optical_write(f->s0, f, blk, data, psub, qsub);
}

struct sig_magneto_optical_merge *
sig_magneto_optical_merge(
	struct sig_magneto_optical *s0,
	struct sig_magneto_optical *s1
)
{
	static const struct sig_magneto_optical_funcs s0_funcs = {
		.atip = sig_magneto_optical_s0_atip,
		.read = sig_magneto_optical_s0_read,
		.write = sig_magneto_optical_s0_write,
	};
	static const struct sig_magneto_optical_funcs s1_funcs = {
		.atip = sig_magneto_optical_s1_atip,
		.read = sig_magneto_optical_s1_read,
		.write = sig_magneto_optical_s1_write,
	};
	struct sig_magneto_optical_merge *m;

	m = malloc(sizeof(*m));
	assert(m);

	/* Out */
	m->s0 = s0;
	m->s1 = s1;

	/* In */
	sig_magneto_optical_connect(s0, m, &s0_funcs);
	sig_magneto_optical_connect(s1, m, &s1_funcs);

	return m;
}

void
sig_magneto_optical_split(struct sig_magneto_optical_merge *m)
{
	sig_magneto_optical_disconnect(m->s0, m);
	sig_magneto_optical_disconnect(m->s1, m);
}

struct sig_magneto_optical *
sig_magneto_optical_create(const char *name)
{
	struct sig_magneto_optical *sig;

	sig = malloc(sizeof(*sig));
	assert(sig);

	sig->type = SIG_GEN_MAGNETO_OPTICAL;
	sig->member_count = 0;

	return sig;
}

void
sig_magneto_optical_destroy(struct sig_magneto_optical *sig)
{
	assert(sig);
	assert(sig->type == SIG_GEN_MAGNETO_OPTICAL);

	free(sig);
}
