/*
 * $Id: system.h,v 1.68 2009-10-13 07:30:54 vrsieh Exp $
 *
 * Copyright (C) 2003-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#ifndef __SYSTEM_H_INCLUDED
#define __SYSTEM_H_INCLUDED

enum port_type {
	PORT_NONE = 0,

	PORT_CDDVD_DRIVE = 2,
	PORT_CDDVD_MEDIA,

	PORT_FLOPPY_DRIVE,
	PORT_FLOPPY_MEDIA,

	PORT_RJ45_FEMALE,
	PORT_RJ45_MALE,

	PORT_USB_FEMALE,
	PORT_USB_MALE,
};

extern void
system_name_push(const char *name);
extern void
system_name_pop(void);
extern const char *
system_path(void);

extern int
system_page_create(const char *name);
extern int
system_page_destroy(unsigned int page_id);
extern int
system_page_lookup(const char *name);

extern int
system_sig_create(const char *type, const char *name);
extern void *
system_sig_get(unsigned int sig_id);
extern int
system_sig_unget(void *ptr);
extern void
system_sig_destroy(unsigned int sig_id);

extern int
system_sig_info(unsigned int sigid, char *type, char *name);

/** create a component.
 *  @param type component type.
 *  @param name path name of the component.
 *  @param node desired node id of the component.
 *  @param page desired page id of the component.
 *  @return component id or -1 on error.
 */
extern int
system_comp_create(
	const char *type,
	const char *name,
	unsigned int node,
	unsigned int page
);

/** set a generic of a component
 *  @param comp component id (as from system_comp_create)
 *  @param type VHDL type name.
 *  @param generic name of the generic
 *  @param value string representation of value.
 *  @return currently always 0
 */
extern int
system_comp_generic_set(unsigned int comp,
		const char *type, const char *name, const char *value);
extern int
system_comp_port_connect(unsigned int comp, const char *port, unsigned int sig);
extern int
system_comp_init(unsigned int comp);
extern int
system_comp_exit(unsigned int comp);
extern int
system_comp_destroy(unsigned int comp);

extern int
system_comp_info(unsigned int compid, char *type, char *name,
		unsigned int *nodep, unsigned int *pagep);
extern int
system_comp_generic_info(unsigned int compid, unsigned int genericid,
		char *type, char *generic, char *value);
extern int
system_comp_port_info(unsigned int compid, unsigned int portid,
		char *port, unsigned int *sigidp);

/** create an architecture.
 *  @param type architecture type.
 *  @param name path name of the architecture.
 *  @param node desired node id of the architecture.
 *  @param page desired page id of the architecture.
 *  @return architecture id or -1 on error.
 */
extern int
system_arch_create(
	const char *type,
	const char *name,
	unsigned int node,
	unsigned int page
);

/** set a generic of an architecture
 *  @param comp architecture id (as from system_arch_create)
 *  @param type VHDL type name.
 *  @param generic name of the generic
 *  @param value string representation of value.
 *  @return currently always 0
 */
extern int
system_arch_generic_set(unsigned int arch,
		const char *type, const char *name, const char *value);
extern int
system_arch_port_connect(unsigned int arch, const char *port, unsigned int sig);
extern int
system_arch_init(unsigned int arch);
extern int
system_arch_exit(unsigned int arch);
extern int
system_arch_destroy(unsigned int arch);

extern void
system_comp_connect(unsigned int comp, const char *port, unsigned int sig);
extern void
system_comp_disconnect(unsigned int comp, const char *port);

/** lookup a component by path name
 *  @param name path name of component (same as name of system_comp_create)
 *  @return component id, or -1 if not found.
 */
extern int
system_comp_lookup(const char *name);

/** lookup a signal by path name
 *  @param name path name of signal (same as name of system_sig_create)
 *  @return signal id, or -1 if not found.
 */
extern int
system_sig_lookup(const char *name);

enum system_gender {
	SYSTEM_GENDER_NEUTRAL = 0,
	SYSTEM_GENDER_FEMALE = 1,
	SYSTEM_GENDER_MALE = 2,
};

extern int
system_port_lookup(const char *type, enum system_gender gender, unsigned int n,
		const char **comp_name, const char **port_name);

extern int
system_port_connect(const char *comp_name0, const char *port_name0,
		const char *comp_name1, const char *port_name1);
extern int
system_port_disconnect(const char *comp_name0, const char *port_name0);

#endif /* __SYSTEM_H_INCLUDED */
