/*
 * Copyright (C) 2017 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "lib/system_node_tree.h"

#include "lib/string_tools.h"

static system_node system_node_root;
static tree_node *system_tree_root;


void system_node_tree_init(void)
{
	system_tree_root = tree_new();
	assert(system_tree_root != NULL);

	system_node_root.name = "ROOT";
	system_node_root.type = NODE_SYSTEM;
	system_node_root.opaque = NULL;
	system_node_root.has_active_breakpoints = 0;
	system_node_root.num_fields = 0;

	tree_set_data(system_tree_root, &system_node_root);
}

tree_node* system_get_node_tree(void)
{
	return system_tree_root;
}

system_node* system_find_tree_node(const char *name, int len)
{
	tree_node *current = system_get_node_tree();

	if (name == NULL || len == 0) {
		return NULL;
	}
	if (len == -1) {
		len = strlen(name);
	}

	len--;
	name++;

	while (len > 0) {
		const char *name_end = str_nchr(name, '/', len);
		if (name_end == NULL || name_end == name + len) {
			/* last node found */
			lst_dnode *node_iter;
			lst *children;

			children = tree_get_children(current);
			lst_Foreach(children, node_iter) {
				tree_node *node = (tree_node *)node_iter->data;
				if (node == NULL) {
					continue;
				}
				system_node *sysnode = (system_node *)tree_get_data(node);
				if (sysnode == NULL) {
					continue;
				}
				if (0 == strncmp(name, sysnode->name, len)
				    && strlen(sysnode->name) == len) {
					/* found node */
					return sysnode;
				}
			}
			/* no such node */
			return NULL;
		} else {
			int current_len = name_end - name;
			lst_dnode *node_iter;
			lst *children;

			children = tree_get_children(current);
			lst_Foreach(children, node_iter) {
				tree_node *node = (tree_node *)node_iter->data;
				if (node == NULL) {
					continue;
				}
				system_node *sysnode = (system_node *)tree_get_data(node);
				if (sysnode == NULL) {
					continue;
				}
				if (0 == strncmp(name, sysnode->name, current_len)
				    && strlen(sysnode->name) == current_len) {
					/* found intermediate node */
					current = node;
					name = name_end + 1;
					len -= current_len +1;
					break;
				}
			}
		}
	}
	return NULL;
}

typedef struct {
	const char *name;
	size_t len;
	tree_node *found;
} sysnode_compare_data;

static int sysnode_compare_names(tree_node *node, void *opaque)
{
	sysnode_compare_data *cb_data = (sysnode_compare_data *)opaque;

	system_node *sysnode = (system_node *)tree_get_data(node);
	if (sysnode == NULL) {
		return 0;
	}
	if (strncmp(cb_data->name, sysnode->name, cb_data->len) == 0) {
		/* found */
		cb_data->found = node;
		return 1;
	}
	return 0;
}

static tree_node* find_child_node(tree_node *parent, const char *name, size_t len)
{
	sysnode_compare_data cb_data;
	cb_data.name = name;
	cb_data.len = len;
	cb_data.found = NULL;

	tree_visit_children(parent, &sysnode_compare_names, &cb_data);

	return cb_data.found;
}

system_node*
system_comp_register(const char *name, void* opaque)
{
	static const char separator = ':';
	assert(name != NULL && name[0] == separator);
	/* iterate over name tokens to insert/lookup child nodes */

	const char *begin = strchr(name, separator);
	const char *end = strchr(begin + 1, separator);
	tree_node *current_node = system_get_node_tree();

	while (begin != NULL) {
		if (end == NULL) {
			/* leaf node */
			tree_node *child = find_child_node(current_node, begin+1, -1);
			if (child != NULL) {
				/* this component is already registred */
				return (system_node*)tree_get_data(child);
			}
			system_node *new_node = (system_node *)malloc(sizeof(system_node));
			assert(new_node != NULL);
			new_node->name = strdup(begin+1);
			assert(new_node->name != NULL);
			new_node->opaque = opaque;
			new_node->set_inspection_cb = NULL;
			new_node->has_active_breakpoints = 0;
			new_node->type = NODE_COMPONENT;
			new_node->num_fields = 0;

			current_node = tree_new_child(current_node, new_node);
			assert(current_node != NULL);
			return new_node;

		}
		/* intermediate node */
		tree_node *child = find_child_node(current_node, begin+1, end-begin-1);
		if (child != NULL) {
			current_node = child;
		} else {
			/* did not find intermediate node */
			system_node *new_node = (system_node *)malloc(sizeof(system_node));
			assert(new_node != NULL);
			new_node->name = strndup(begin+1, end-begin-1);
			assert(new_node->name != NULL);
			new_node->opaque = NULL;
			new_node->set_inspection_cb = NULL;
			new_node->has_active_breakpoints = 0;
			new_node->type = NODE_UNDEFINED;
			new_node->num_fields = 0;

			current_node = tree_new_child(current_node, new_node);
			assert(current_node != NULL);
		}

		begin = end;
		end = strchr(begin + 1, separator);
	}
	return NULL;
}


void
system_comp_register_inspection_cb(system_node *node, set_inspection_cb_t cb)
{
	node->set_inspection_cb = cb;
}

void
system_comp_register_step_cb(system_node *node, single_step_cb_t cb)
{
	node->single_step_cb = cb;
}

void system_node_register_field(system_node *node,
                                const char *name,
                                size_t offset,
                                data_type_t type)
{
	system_node_field *field = NULL;

	assert(node && node->num_fields < MAX_FIELDS);

	field = &node->fields[node->num_fields];
	node->num_fields++;

	field->name = name;
	field->offset = offset;
	field->type = type;
	field->is_used_in_breakpoints = 0;
}

system_node_field*
system_find_field(system_node *current,
                  const char *name, int len,
                  system_node **out_node)
{
	system_node *sysnode = NULL;
	unsigned i;
	const char *field_name = NULL;
	int field_len = 0;

	if (name == NULL || len == 0) {
		return NULL;
	}
	if (len == -1) {
		len = strlen(name);
	}

	if (name[0] == '/') {
		/* absolute path */
		const char *last_separator = str_nrchr(name, '/', len);
		if (last_separator == NULL) {
			return NULL;
		}
		field_name = last_separator + 1;
		field_len = len - (field_name - name);
		if (field_len < 1) {
			return NULL;
		}
		sysnode = system_find_tree_node(name, last_separator-name);
		if (sysnode == NULL) {
			return NULL;
		}
	} else {
		/* current node */
		field_name = name;
		field_len = len;
		sysnode = current;
	}
	for (i=0; i<sysnode->num_fields; i++) {
		system_node_field *field = &sysnode->fields[i];
		if (0 == strncmp(field_name, field->name, field_len)
		    && strlen(field->name) == field_len) {
			if (out_node != NULL) {
				*out_node = sysnode;
			}
			return field;
		}
	}
	return NULL;
}
