/*
 * Copyright (C) 2000-2001 Chris Ross and Evan Webb
 * Copyright (C) 1999-2000 Chris Ross
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies of the Software, its documentation and marketing & publicity
 * materials, and acknowledgment shall be given in the documentation, materials
 * and software packages that this Software was used.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include <ferite.h>
#include "builder_compile.h"
#include <sys/stat.h>
#include <sys/types.h>
#include "builder.h"

struct
{
   char *scriptname;
   int verbose;
   int showargs;
   int estring;
}
opt;

FeriteStack   *class_stack;
char           module_init_name[8096];
char           file_list[8096];
char           internal_file_list[8096];
BuilderModule *current_module;
FILE          *module_current_c_file = NULL;
int            in_class = 0;
int            want_deinit = 1;
int            want_makefile = 1;
int            want_configm4 = 1;
int            force_makefile = 0;

void show_help()
{
   fprintf( stderr, "This is builder, [%s] (v %s)\n", FERITE_NAME, FERITE_VERSION );
   printf( "\nCopyright 1999-2001, Chris Ross\n" );
   printf( "\nUsage: builder [OPTIONS] [FILE]\n\n" );
   printf( "\t--help          \t-h\tShow this help screen.\n" );
   printf( "\t--verbose       \t-v\tBe verbose when running.\n" );
   printf( "\t--showargs      \t-s\tShow all the arguments passed to ferite.\n" );
   printf( "\t--execute       \t-e\tExecute the script supplied on the command line\n" );
   printf( "\t--module-name   \t-m\tName of the module to be generated (default: modulename)\n" );
   printf( "\t--add-file      \t-a\tAdd an external file to the generated makefile\n" );
   printf( "\t--init-name     \t-i\tChange the name of the default module init function (default: module_init)\n" );
   printf( "\t--no-deinit     \t-d\tStop builder generating a module_deinit function\n" );
   printf( "\t--no-makefile   \t-f\tStop builder generating a 'Makfile.am'\n" );
   printf( "\t--no-configm4   \t-c\tStop builder generating a 'config.m4'\n" );
   printf( "\t--force-makefile\t-k\tForce builder to generate a 'Makefile.am'\n" );
   printf( "\n" );
   exit(0);
}

void parse_args( int argc, char **argv )
{
   int i, j;
   char *buf;

   opt.verbose = 0;
   opt.showargs = 0;
   opt.estring = 0;
   strcpy( current_module->name, "modulename" );
   strcpy( module_init_name, "modulename_init" );
   for(i = 1; i < argc; i++)
   {
      if(argv[i][0] == '-')
      {
		 if((!strcmp(argv[i], "--verbose")) || (!strcmp(argv[i], "-v")))
		   opt.verbose = 1;
		 if((!strcmp(argv[i], "--showargs")) || (!strcmp(argv[i], "-s")))
		   opt.showargs = 1;
		 if((!strcmp(argv[i], "--help")) || (!strcmp(argv[i], "-h")))
		 {
			show_help();
			exit(0);
		 }
		 if((!strcmp(argv[i], "--execute")) || (!strcmp(argv[i], "-e")))
		 {
			opt.scriptname = strdup( argv[++i] );
			opt.estring = 1;
		 }
		 if((!strcmp(argv[i], "--module-name")) || (!strcmp(argv[i], "-m")))
		 {
			strcpy( current_module->name, argv[++i] );
			if( strcmp( module_init_name, "modulename_init" ) == 0 ){
			   sprintf( module_init_name, "%s_init", current_module->name );
			}
		 }
         if((!strcmp(argv[i], "--add-file")) || (!strcmp(argv[i], "-a")))
		 {
			strcat( file_list, argv[++i] ); strcat( file_list, " " );
		 }
         if((!strcmp(argv[i], "--init-name")) || (!strcmp(argv[i], "-i")))
		 {
			strcpy( module_init_name, argv[++i] );
		 }
         if((!strcmp(argv[i], "--no-deinit")) || (!strcmp(argv[i], "-d")))
		 {
			want_deinit = 0;
		 }
         if((!strcmp(argv[i], "--no-makefile")) || (!strcmp(argv[i], "-f")))
		 {
			want_makefile = 0;
		 }
         if((!strcmp(argv[i], "--no-configm4")) || (!strcmp(argv[i], "-c")))
		 {
			want_configm4 = 0;
		 }
         if((!strcmp(argv[i], "--force-makefile")) || (!strcmp(argv[i], "-k")))
		 {
			force_makefile = 1;
		 }
      }
      else
      {
		 buf = (char *)malloc(1024);
		 memset(buf, '\0', 1024);
		 if(argv[i][0] != '/')
		 {
		/* relative path */
			getcwd(buf, 1024);
			strcat(buf, "/");
			strcat(buf, argv[i]);
		 }
		 else
		 {
		/* absolute path */
			strcpy(buf, argv[i]);
		 }
		 opt.scriptname = buf;
      }
   }

   if(!opt.scriptname)
   {
      fprintf( stderr, "This is builder, [%s] (v %s)\n", FERITE_NAME, FERITE_VERSION );
	  printf( "\nCopyright 1999-2001, Chris Ross\n" );
      printf( "\nUsage: builder [OPTIONS] [FILE]\n\n" );
      printf( "\tYou need to supply a file name. For help try builder --help\n\n" );
      exit( -1 );
   }
}

char *builder_generate_current_name()
{
   int i;
   static char nameBuffer[1024];

   FE_ENTER_FUNCTION;
   nameBuffer[0] = '\0';
   for( i = 1; i <= current_module->name_stack->stack_ptr; i++ )
   {
      strcat( nameBuffer, current_module->name_stack->stack[i] );
      strcat( nameBuffer, "_" );
   }
   nameBuffer[strlen(nameBuffer)-1] = '\0';
   FE_LEAVE_FUNCTION( nameBuffer );
}

void builder_process_variable( FeriteScript *script, FeriteVariable *var, char *parent )
{
   int i;
   
   FE_ENTER_FUNCTION;
   if( strcmp( var->name, "'module_init" ) == 0 || 
	   strcmp( var->name, "'module_deinit" ) == 0 ||
	   strcmp( var->name, "'module_header" ) == 0 || 
      strcmp( var->name, "'module_flags" ) == 0 ){
	      FE_LEAVE_FUNCTION(NOWT);
   }
   
   __ferite_stack_push( current_module->name_stack, var->name );
   if( opt.verbose )
     printf( "%s - processing variable\n", builder_generate_current_name() );
   fprintf( current_module->core, "      %s( script, %s%s, ",
			(in_class == 1 ? "ferite_register_class_variable" : "__ferite_register_ns_variable" ),
			current_module->name_stack->stack[current_module->name_stack->stack_ptr-1],
			(in_class == 1 ? "_class" : "_namespace") );
   
#define DO_STATIC  ( in_class == 1 ? (var->flags.is_static ? ", 1" : ", 0" ) : "" )

   switch( var->type )
   {
    case F_VAR_LONG:
    case F_VAR_DOUBLE:
      fprintf( current_module->core, "fe_new_lng( \"%s\", 0 )%s );", var->name, DO_STATIC );
      break;
    case F_VAR_STR:
      fprintf( current_module->core, "fe_new_str( \"%s\", \"\" )%s );", var->name, DO_STATIC );
      break;
    case F_VAR_OBJ:
      fprintf( current_module->core, "fe_new_obj( \"%s\" )%s );", var->name, DO_STATIC );
      break;
    case F_VAR_UARRAY:
      fprintf( current_module->core, "fe_new_array( \"%s\", 10 )%s );", var->name, DO_STATIC );
      break;
    case F_VAR_VOID:
      fprintf( current_module->core, "fe_new_void( \"%s\" )%s );", var->name, DO_STATIC );
      break;
   }
   fprintf( current_module->core, "\n" );
   __ferite_stack_pop( current_module->name_stack );
   FE_LEAVE_FUNCTION( NOWT );
}

void builder_process_function( FeriteScript *script, FeriteFunction *fnc, char *parent )
{
   int i, is_ptr = 0, do_getvars = 1;
   int decrementArgCount = 0;
   char c_variable_type[64], *buf, c_variable_init[512];
   FeriteStack *stk;

   FE_ENTER_FUNCTION;
   
   __ferite_stack_push( current_module->name_stack, fnc->name );
   if( opt.verbose )
     printf( "  Writing function %s\n", parent );
   stk = __ferite_create_stack( 30 );
   fprintf( current_module->header, "FE_NS_FUNCTION( %s );\n", builder_generate_current_name() );
   fprintf( current_module->core, "      %s( script, %s%s, \"%s\", %s, \"",
			(in_class == 1 ? "fe_create_cls_fnc" : "fe_create_ns_fnc"),
			current_module->name_stack->stack[current_module->name_stack->stack_ptr-1],
			(in_class == 1 ? "_class" : "_namespace"),
			fnc->name,
			builder_generate_current_name() );
   fprintf( module_current_c_file, "FE_NS_FUNCTION( %s )\n{\n", builder_generate_current_name() );
   for( i = 0; i < fnc->arg_count; i++ )
   {
      switch( fnc->signature[i]->variable->type )
      {
       case F_VAR_LONG:
       case F_VAR_DOUBLE:
		 strcpy( c_variable_type, "double" );
		 sprintf( c_variable_init, "" );
		 fprintf( current_module->core, "n" );
		 is_ptr = 0;
		 break;
       case F_VAR_STR:
		 strcpy( c_variable_type, "char" );
		 sprintf( c_variable_init, " = fcalloc( strlen(VAS(params[%d]))+1, sizeof(char) )", i, i );
		 fprintf( current_module->core, "s" );
		 is_ptr = 1;
		 break;
       case F_VAR_OBJ:
		 strcpy( c_variable_type, "FeriteObject" );
		 sprintf( c_variable_init, "" );
		 if( i < (in_class == 0 ? fnc->arg_count : fnc->arg_count - 2) )
		   fprintf( current_module->core, "o" );
		 is_ptr = 1;
		 break;
       case F_VAR_UARRAY:
		 strcpy( c_variable_type, "FeriteUnifiedArray" );
		 sprintf( c_variable_init, "" );
		 fprintf( current_module->core, "a" );
		 is_ptr = 1;
		 break;
       case F_VAR_VOID:
		 if( strcmp( fnc->signature[i]->variable->name, "..." ) == 0 )
		 {
			i = fnc->arg_count;
			if( in_class ){
			   fprintf( module_current_c_file, "   FeriteObject *self = VAO(params[ferite_get_parameter_count(params) - 1]);\n" );
			}
			fprintf( current_module->core, "." );
			decrementArgCount = 1;
			continue;
		 } 
		 else 
		 {
			strcpy( c_variable_type, "FeriteVariable" );
			sprintf( c_variable_init, " = params[%d]", i );
			fprintf( current_module->core, "v" );
			is_ptr = 1;
			break;
		 }
      }
      if( i < (in_class == 0 ? fnc->arg_count -1 : fnc->arg_count - 3) )
		fprintf( current_module->core, "," );
      fprintf( module_current_c_file, "   %s %s%s%s;\n", c_variable_type, (is_ptr ? "*" : "" ), 
			   ( fnc->signature[i]->variable->name[0] == '.' ?
				 "var_args" : fnc->signature[i]->variable->name ), 
			   c_variable_init );
      if( fnc->signature[i]->variable->type != F_VAR_VOID )
		sprintf( c_variable_type, ", %s%s", (fnc->signature[i]->variable->type == F_VAR_STR ? "" : "&"), fnc->signature[i]->variable->name );
      else
		sprintf( c_variable_type, ", NULL" );
      __ferite_stack_push( stk, fstrdup( c_variable_type ) );
   }
   if( !in_class ){
	  fprintf( current_module->core, "\" );\n" );
   } else {
	  fprintf( current_module->core, "\", %s );\n", (fnc->is_static ? "1" : "0") );
   }
   if( fnc->arg_count > 0 )
   {
      fprintf( module_current_c_file, "\n   ferite_get_parameters( params, %d", (decrementArgCount 
																				 ? fnc->arg_count - (in_class ? 3 : 1)
																				 : fnc->arg_count) );
      for( i = 1; i <= stk->stack_ptr; i++ )
      {
		 buf = stk->stack[i];
		 fprintf( module_current_c_file, "%s", buf );
		 ffree( buf );
      }
      fprintf( module_current_c_file, " );\n" );
   }
   __ferite_delete_stack( stk );
   fprintf( module_current_c_file, "\n   { /* Main function body. */\n %s\n   }\n   FE_RETURN_VOID;\n}\n\n", (fnc->fncPtr != NULL ? (char *)(fnc->fncPtr) : "") );
   if( fnc->fncPtr != NULL )
     ffree( fnc->fncPtr );
   __ferite_stack_pop( current_module->name_stack );
   FE_LEAVE_FUNCTION( NOWT );
}

int builder_class_in_stack( char *name )
{
   int i;
   
   FE_ENTER_FUNCTION;
   for( i = 1; i <= class_stack->stack_ptr; i++ ){
	  if( strcmp( (char*)class_stack->stack[i], name ) == 0 ){
		 FE_LEAVE_FUNCTION(1);
	  }
   }
   FE_LEAVE_FUNCTION(0);
}

void builder_process_class( FeriteScript *script, FeriteClass *cls, char *parent )
{
   char buf[1024];

   FE_ENTER_FUNCTION;
   
   if( (char *)cls->parent != NULL ) /* we have a inheirted class */
   {
	  if( !builder_class_in_stack( (char *)cls->parent ) ) /* it's not in the class stack */
	  {
		 FeriteNamespaceBucket *nsb = __ferite_find_namespace( script, script->mainns, (char *)cls->parent, FENS_CLS );
		 if( nsb != NULL )
		 {
			builder_process_class( script, nsb->data, parent );
			__ferite_stack_push( class_stack, fstrdup((char *)cls->parent) );
		 }
		 else
		 {
			ferite_warning( script, "Class '%s' extends '%s' which does not exist in this module - assuming it is external\n", cls->name, (char*)cls->parent );
		 }
	  }
   }
   if( builder_class_in_stack( cls->name ) ){ /* we return if we are already in the list */
	  FE_LEAVE_FUNCTION(NOWT);
   }
   
   __ferite_stack_push( current_module->name_stack, cls->name );
   sprintf( buf, "%s.c", builder_generate_current_name() );
   strcat( internal_file_list, buf ); strcat( internal_file_list, " " );
   if( opt.verbose )
     printf( "Generating file %s for class %s\n", buf, parent );
   module_current_c_file = fopen( buf, "w" );

   fprintf( module_current_c_file,
			"/* This file has been automatically generated by builder part of the ferite distribution */\n" \
                         "/* file:  %s */\n" \
                         "/* class: %s */\n\n" \
                         "#include <ferite.h>       /* we need this without a doubt */\n" \
                         "#include \"%s_header.h\"  /* this is the module header */\n\n", buf, cls->name, current_module->name );

   fprintf( current_module->core, "   if( __ferite_namespace_element_exists( script, ns, \"%s\" ) == NULL )\n" \
	                 "   {\n" \
			 "      FeriteClass *%s_class = ferite_register_inherited_class( script, ns, \"%s\", %s%s%s );\n", cls->name, cls->name, cls->name,
			( cls->parent == NULL ? "" : "\"" ),
			( cls->parent == NULL ? "NULL" : (char *)cls->parent ),
			( cls->parent == NULL ? "" : "\"" )
			);
   if( cls->parent != NULL )
   {
      ffree( cls->parent );
   }
   in_class = 1;
   __ferite_process_hash( script, cls->variables, (void (*)(FeriteScript*,void *,char*))builder_process_variable );
   __ferite_process_hash( script, cls->functions, (void (*)(FeriteScript*,void *,char*))builder_process_function );
   in_class = 0;
   fprintf( current_module->core, "   }\n\n" );

   fclose( module_current_c_file );
   __ferite_stack_pop( current_module->name_stack );
   FE_LEAVE_FUNCTION( NOWT );
}

void builder_process_namespace_element( FeriteScript *script, FeriteNamespaceBucket *nsb, char *parent )
{
   char buf[1024];

   FE_ENTER_FUNCTION;
   if( nsb != NULL )
   {
      switch( nsb->type )
      {
       case FENS_NS:
		 __ferite_stack_push( current_module->name_stack, parent );
		 sprintf( buf, "%s.c", builder_generate_current_name() );
		 strcat( internal_file_list, buf ); strcat( internal_file_list, " " );
		 if( opt.verbose )
		   printf( "Generating file %s for namespace %s\n", buf, parent );
		 module_current_c_file = fopen( buf, "w" );
		 fprintf( module_current_c_file, "/* This file has been automatically generated by builder part of the ferite distribution */\n" \
"/* file: %s */\n" \
"/* namespace: %s */\n\n" \
"#include <ferite.h>       /* we need this without a doubt */\n" \
"#include \"%s_header.h\"  /* this is the module header */\n\n", buf, parent, current_module->name );
		 fprintf( current_module->core, "   if( __ferite_namespace_element_exists( script, ns, \"%s\" ) == NULL )\n" \
"   {\n" \
"      FeriteNamespace *%s_namespace = __ferite_register_namespace( script, \"%s\", ns );\n", parent, parent, parent );
		 in_class = 0;
		 __ferite_process_hash( script, ((FeriteNamespace*)nsb->data)->space, (void (*)(FeriteScript *,void *,char*))builder_process_namespace_element );
		 in_class = 0;
		 fprintf( current_module->core, "   }\n\n" );
		 fclose( module_current_c_file );
		 __ferite_stack_pop( current_module->name_stack );
		 break;
       case FENS_VAR:
		 builder_process_variable( script, nsb->data, parent );
		 break;
       case FENS_FNC:
		 builder_process_function( script, nsb->data, parent );
		 break;
       case FENS_CLS:
		 builder_process_class( script, nsb->data, parent );		 
		 break;
       default:
      }
   }
   FE_LEAVE_FUNCTION( NOWT );
}

void builder_dump_plan( FeriteScript *script )
{
   char buf[1024];
   struct stat sbuf;
   int i = 0;
   FeriteNamespaceBucket *nsb;
   FeriteVariable *var;
   
   FE_ENTER_FUNCTION;
   current_module->name_stack = __ferite_create_stack( 30 );
   __ferite_stack_push( current_module->name_stack, current_module->name );

   sprintf( buf, "%s_core.c", current_module->name );
   strcat( internal_file_list, buf ); strcat( internal_file_list, " " );
   if( opt.verbose )
     printf( "Generating file %s for the module core\n", buf );
   current_module->core = fopen( buf, "w" );
   fprintf( current_module->core, "/* This file has been automatically generated by builder part of the ferite distribution */\n" \
                 "#include <ferite.h>       /* we need this without a doubt */\n" \
                 "#include \"%s_header.h\"  /* this is the module header */\n\n", current_module->name );
   fprintf( current_module->core, "void %s( FeriteScript *script, FeriteNamespace *ns )\n{\n\n", module_init_name );

   /******************************************************************
	* MODULE FLAGS
	******************************************************************/
   nsb = __ferite_namespace_element_exists(script, script->mainns, "'module_flags");
   if(nsb != NULL && nsb->data != NULL) {
	  sprintf(buf, "%s_flags.sh", current_module->name );
	  if( opt.verbose )
		printf("Generating file %s for the module flags\n", buf);
	  current_module->flags = fopen(buf,"w");
	  fprintf(current_module->flags, "# This file has been automatically generated by builder part of the ferite distribution\n\n");
      var = nsb->data;
      fprintf( current_module->flags,"%s\n", VAS(var));
	  fclose(current_module->flags);
   }

   /******************************************************************
	* MODULE HEADER
	******************************************************************/
   sprintf( buf, "%s_header.h", current_module->name );
   if( opt.verbose )
     printf( "Generating file %s for the module header\n", buf );
   current_module->header = fopen( buf, "w" );
   fprintf( current_module->header, "/* This file has been automatically generated by builder part of the ferite distribution */\n" \
                 "#ifndef __%s_HEADER__\n" \
                 "#define __%s_HEADER__\n\n", current_module->name, current_module->name );


   nsb = __ferite_namespace_element_exists( script, script->mainns, "'module_header" );
   if( nsb != NULL && nsb->data != NULL ){
	  var = nsb->data;
	  fprintf( current_module->header, "    /* Start user defined header code */\n" );
	  fprintf( current_module->header, "%s\n", VAS(var) );
	  fprintf( current_module->header, "    /* End user defined header code */\n" );	  
   }

   /******************************************************************
	* MODULE INITIALISER
	******************************************************************/
   __ferite_process_hash( script, script->mainns->space, (void (*)(FeriteScript*,void *,char*))builder_process_namespace_element );
   /* what we do is look for a function called 'module_init and then dump the ->fncPtr as a char * */
   nsb = __ferite_namespace_element_exists( script, script->mainns, "'module_init" );
   if( nsb != NULL && nsb->data != NULL ){
	  var = nsb->data;
	  fprintf( current_module->core, "    /* Start user defined initialisation code */\n" );
	  fprintf( current_module->core, "%s\n", VAS(var) );
	  fprintf( current_module->core, "    /* End user defined initialisation code */\n" );	  
   }
   
   fprintf( current_module->core, "}\n\n" );
   
   /******************************************************************
	* MODULE DEINITIALISER
	******************************************************************/
   if( want_deinit ){
	  fprintf( current_module->core, "void %s_deinit()\n{\n", current_module->name );
	  /* what we do is look for a function called 'module_init and then dump the ->fncPtr as a char * */
	  nsb = __ferite_namespace_element_exists( script, script->mainns, "'module_deinit" );
	  if( nsb != NULL && nsb->data != NULL ){
		 var = nsb->data;
		 fprintf( current_module->core, "    /* Start user defined deinitialisation code */\n" );
		 fprintf( current_module->core, "%s\n", VAS(var) );
		 fprintf( current_module->core, "    /* End user defined deinitialisation code */\n" );	  
	  }
      fprintf( current_module->core, "}\n\n" );
   }
   fprintf( current_module->header, "\n#endif /* __%s_HEADER__ */\n", current_module->name );
   if( current_module->core )
     fclose( current_module->core );
   if( current_module->header )
     fclose( current_module->header );
   __ferite_delete_stack( current_module->name_stack );
   
   /******************************************************************
	* MODULE Makefile.am
	******************************************************************/
   if( (stat( "Makefile.am", &sbuf ) != 0 && want_makefile) || force_makefile == 1 )
   {
      if( opt.verbose )
		printf( "Generating Makefile.am\n" );
      current_module->Makefile = fopen( "Makefile.am", "w" );
	  i = strlen( opt.scriptname ) - 1;
	  strcat( internal_file_list, current_module->name ); strcat( internal_file_list, "_header.h " );
	  while( opt.scriptname[i] != '/' ) i--;
	  i++;
	  memmove( opt.scriptname, opt.scriptname + i, strlen( opt.scriptname + i ) + 1 );
      fprintf( current_module->Makefile, MakefileTemplate, 
			   current_module->name, current_module->name, current_module->name, 
			   current_module->name, 
			   current_module->name,
			   current_module->name,
			   current_module->name, 
			   current_module->name, 
			   internal_file_list, file_list, 
			   current_module->name, current_module->name, current_module->name, current_module->name, current_module->name, opt.scriptname, 
			   current_module->name, current_module->name, opt.scriptname );
      fclose( current_module->Makefile );
   }
   
   /******************************************************************
	* MODULE Config.m4
	******************************************************************/
   if( stat( "config.m4", &sbuf ) != 0 && want_configm4)
   {
      if( opt.verbose )
		printf( "Generating config.m4\n" );
      current_module->Makefile = fopen( "config.m4", "w" );
      fprintf( current_module->Makefile, ConfigM4Template,
			   current_module->name,
			   current_module->name,
			   current_module->name,
			   current_module->name,
			   current_module->name );
      fclose( current_module->Makefile );
   }
   FE_LEAVE_FUNCTION( NOWT );
}

int main( int argc, char **argv )
{
   FeriteEnvironment *env;
   FeriteScript *script;

   int i = 0;

   if ( argc < 2 )
   {
      fprintf( stderr, "%s (v %s)\n", FERITE_NAME, FERITE_VERSION );
      printf( "\nUsage: builder [OPTIONS] [FILE]\n\n" );
      printf( "\tYou need to supply a file name. For help try builder --help\n\n" );
      exit( -1 );
   }
   else
   {

      if( opt.showargs )
      {
		 printf( "Argument Count: %d\n", argc );
		 for( i = 0; i < argc; i++ )
		   printf( "Argument #%d: %s\n", i+1, argv[i] );
      }

      if( opt.verbose )
		printf( "--> initialising builder\n");

      memset( file_list, '\0', 8096 );
      memset( internal_file_list, '\0', 8096 );

      env = ferite_init( argc, argv );

      current_module = fmalloc( sizeof(BuilderModule) );
      memset( current_module, '\0', sizeof(BuilderModule) );
      parse_args( argc, argv );
	  
      if( opt.verbose )
		printf( "--> creating script object\n" );

      if( opt.verbose )
      {
		 printf( "--> loading script from file\n" );
		 printf( "-->   `-> script file: %s\n", opt.scriptname );
      }

      if( opt.verbose )
		printf( "--> compiling script\n" );

      if( !opt.estring )
      {
		 script = builder_script_compile( opt.scriptname );
      }
      else
      {
		 script = __builder_compile_string( opt.scriptname );
      }

      if( script )
      {
		 class_stack = __ferite_create_stack( 30 );

		 if( opt.verbose )
		   printf( "--> dumping script\n" );
		 builder_dump_plan( script );
		 
		 if( opt.verbose )
		   printf( "--> destroying script object\n" );
		 ferite_script_delete( script );
		 
		 for( i = 1; i <= class_stack->stack_ptr; i++ ){
			if( class_stack->stack[i] != NULL ){
			   ffree( class_stack->stack[i] );
			}
		 }
		 __ferite_delete_stack( class_stack );
      }
      free( opt.scriptname );
      ffree( current_module );

      ferite_deinit();
      if( opt.verbose )
		printf( "--> Quiting Application\n" );

      exit(0);
   }
}
