/* 

                          Firewall Builder

                 Copyright (C) 2000 Vadim Kurland

  Author:  Vadim Kurland     vadim@vk.crocodile.org

  $Id: BackgroundOpWidget.cc,v 1.20 2001/12/26 08:40:07 vkurland Exp $


  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "config.h"

#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>

#include <gtk--/packer.h>

#include "BackgroundOpWidget.hh"
#include "StockButton.hh"

using namespace libfwbuilder;


BackgroundOpWidget::BackgroundOpWidget(BackgroundOp *b,bool m) : Gtk::Packer()
{
    Gtk::ScrolledWindow *sw;

    modal=m;

    set_border_width(10);
    set_default_pad(0, 10);

    text = manage(new class Gtk::Text());
    sw   = manage(new class Gtk::ScrolledWindow());

    text->set_editable(false);
    text->set_word_wrap(false);
    text->set_point(0);
    sw->set_policy(GTK_POLICY_NEVER, GTK_POLICY_ALWAYS);
    sw->add(*text);

    button1=NULL;
    button2=NULL;
    status=NULL;

    add(*sw,GTK_SIDE_TOP, GTK_ANCHOR_CENTER, GTK_EXPAND|GTK_FILL_X|GTK_FILL_Y);
    sw->show();

    show_all();

    can_focus();

    bio=b;
    logger=NULL;
#ifndef STL_HAS_OFSTREAM_FD
    fbuf  =NULL;
#endif
    ConnectSignals();

    showBusy(false);
}

BackgroundOpWidget::~BackgroundOpWidget()
{
//    if (bio && bio->get_running_flag())
//        bio->stop_operation();

    delete logger;
    logger=NULL;
#ifndef STL_HAS_OFSTREAM_FD
    delete fbuf;
#endif
}

void BackgroundOpWidget::ConnectSignals()
{
//    delete_event.connect(SigC::slot(this,&BackgroundOpWidget::on_delete));
    hide.connect(SigC::slot(this,&BackgroundOpWidget::on_hide));
    if(bio)  
        bio->completed.connect(SigC::slot(this,
				  &BackgroundOpWidget::on_completed));
}

void BackgroundOpWidget::on_hide()
{
    if (bio && bio->get_running_flag())
        bio->stop_operation();
}

void BackgroundOpWidget::showBusy(bool busy)
{
    if (button1) {
	button1->hide();
	remove(*button1);
	button1=NULL;
    }

    if (busy) {

	button1=manage(new StockButton(StockButton::STOP));
    } else {
	if (modal){
	    button1=manage(new StockButton(StockButton::OK));
	}
    }
    if (button1) {
	button1->set_name("background_op_widget_button");
	button1->set_usize(90,30);
	add(*button1, GTK_SIDE_TOP, GTK_ANCHOR_CENTER, 0);

	button1->show();
	button1->grab_focus();

	button1->clicked.connect(SigC::slot(this, 
			       &BackgroundOpWidget::on_button_clicked));
    }
}

/**
 *
 *  This method shows text message in the place of "OK"/"Cancel" buttons
 *  Convenient for non-modal modes when widget should show short status
 *  summary in the end of operation
 */
void BackgroundOpWidget::showStatusMessage(const string &msg)
{
    if (button1) {
	button1->hide();
	remove(*button1);
	button1=NULL;
    }

    status  = manage(new Gtk::Label(""));
    status->set_name("status_text");
    status->set_text(msg);
    status->set_alignment(0.0,0.5);
    status->set_line_wrap(true);
    status->set_justify(GTK_JUSTIFY_LEFT);
    status->show();
    add(*status , GTK_SIDE_TOP, GTK_ANCHOR_CENTER, 0);

}

void BackgroundOpWidget::AppendLine(char *str)
{
    text->insert( text->get_context() , str );
}

void BackgroundOpWidget::AppendLine(int i)
{
    char  str[64];
    sprintf(str,"%d",i);
    text->insert( text->get_context() , str );
}

void BackgroundOpWidget::run()
{   
    Gtk::Main::grab_add(*this);
    Gtk::Main::run();
    Gtk::Main::grab_remove(*this);
}


BackgroundOpWidget& BackgroundOpWidget::operator<< (char  *str)
{
    text->insert( text->get_context() , str );
    return (*((BackgroundOpWidget*)this));
}

BackgroundOpWidget& BackgroundOpWidget::operator<< (string str)
{
    text->insert( text->get_context() , str );
    return (*((BackgroundOpWidget*)this));
}

BackgroundOpWidget& BackgroundOpWidget::operator<< (int    i)
{
    char  str[64];
    sprintf(str,"%d",i);
    text->insert( text->get_context() , str );
    return (*((BackgroundOpWidget*)this));
}

void BackgroundOpWidget::on_button_clicked()
{
    if(bio && bio->get_running_flag()) 
    {
        button1->set_sensitive(false);
        *this << "Terminating task. Please wait...\n";
	bio->stop_operation();
        
        // When operatrion in cancelled, BackgroundOp
        // will not send 'completed' signal, so we
        // must to it oursevs.
        on_completed(1);
    } else 
    {
	if(modal)  
        {
	    Gtk::Main::quit();
	    hide();
	}
    }
}

/**
 *  Callback for "completed" signal
 *
 *  BackgroundOp sends signal "completed" when background operation
 *  finished. BackgroundOpWidget connects its own callback to that
 *  signal, so that it can react. BackgroundOpWidget changes button in
 *  this callback and sends the same signal to whatever widget
 *  connected its callback to it - so higher level widget can also do
 *  something when operation finishes. If BackgroundOpWidget is used
 *  in modal mode (typically as a part of a dialog) then most probably
 *  there will be no use for this signal as modal dialog will simply
 *  stay there waiting for user to click the "OK" button to close
 *  it. On the other hand, if BackgroundOpWidget is used in non-modal
 *  mode then higher level widget will probably connect its callback
 *  to "completed" signal to react. This is used in Druid running DNS
 *  queries.
 *
 *  This callback function gets on argument, which is a flag indicating
 *  whether background operation encountered an error. Detailed error
 *  explanation can be obtained using a call to bio->get_latest_error()
 *  BackgroundOpWidget passes this code up to the higher level widget
 *  the same way - as an argument to the "completed" signal
 *
 *  BackgroundOpWidget sends "completed" signal from within idle
 *  callback because it is very likely that higher level widget will
 *  want to destroy BackgroundOp object in the callback
 *  function. Since original "completed" signal was initiated by that
 *  object itself, there may be interesting side effects because this
 *  object gets destroyed while emmitting signal. To avoid this, we  
 *  schedule emmitting "completed" signal on the next idle cycle
 *
 *  (vk)
 */
int BackgroundOpWidget::on_completed(int arg)
{
    completed_op_arg=arg;

    showBusy(false);

    if (arg!=0) {
	FWException *ex=bio->get_latest_error();
	if (ex) 
	    *this << ex->toString();
    }

    Gtk::Main::idle.connect(slot(this,
			     &BackgroundOpWidget::signal_completed_when_idle));

    return 0;
}

int  BackgroundOpWidget::signal_completed_when_idle()
{
    completed(completed_op_arg);   // this sends signal to another widget
    return false;
}

void BackgroundOpWidget::get_line(int fd , GdkInputCondition c)
{
    if(c==GDK_INPUT_READ)
    {
        char buf[2048];
        int  n;
        while((n=read(fd,buf,sizeof(buf)))>0) 
        {
            buf[n]='\0';
            *this << buf;
        }
    }
}

void BackgroundOpWidget::stop()
{
    if(bio && bio->get_running_flag())     
        bio->stop_operation();
    
    if (modal) 
	Gtk::Main::quit();
}


void BackgroundOpWidget::execute()  throw(FWException)
{
    assert(bio!=NULL);
    
    show();

    try 
    {
	showBusy(true);
        
        pipe(pipe_fd);
        fcntl(pipe_fd[0], F_SETFL, O_NONBLOCK);
        
        Gtk::Main::input.connect(SigC::slot(this,&BackgroundOpWidget::get_line),
                      pipe_fd[0],
                      (GdkInputCondition)(GDK_INPUT_READ|GDK_INPUT_EXCEPTION));
#ifdef STL_HAS_OFSTREAM_FD
        logger=new ofstream(pipe_fd[1]);
        logger->setbuf(NULL,0);
#else
	FILE *fff=fdopen(pipe_fd[1],"w");
	if(!fff)
		throw FWException("Error opening descriptor");
	fbuf=new filebuf(fff,ios_base::out,0);
        logger=new ostream(fbuf);
#endif
	bio->start_operation( logger );
        Gtk::Main::idle.connect(SigC::slot(bio,&BackgroundOp::monitor_operation));

    } catch(FWException &ex)
    {
	*this << "\n* " << ex.toString();
        throw;
    }
    
//    if(modal && bio->get_latest_error())
//        throw *bio->get_latest_error();
    
    return;
}

