/* 

                          Firewall Builder

                 Copyright (C) 2000 Vadim Kurland

  Author:  Vadim Kurland     vadim@vk.crocodile.org

  $Id: HostDialog.cc,v 1.31 2001/12/27 06:48:47 vkurland Exp $


  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/


#include "config.h"

#include "HostDialog.hh"
#include "fwbuilder/Host.hh"
#include "fwbuilder/Interface.hh"

#include "GenericBackgroundOpDialog.hh"
#include "MessageDialog.hh"
#include "helpers.hh"
#include "fwbuilder/dns.hh"
#include "IPAddressWidget.hh"
#include "InterfaceDialog.hh"

#include "fwbuilder/snmp.hh"


using namespace libfwbuilder;

#ifdef HAVE_LIBSNMP

class SNMPHostSysdescQueryPool: public Pool<SNMP_sysdesc_query>
{
    private:
    
    static SNMPHostSysdescQueryPool *instance;
    static Mutex                 sysdesc_mutex;
    
    public:
    
    SNMPHostSysdescQueryPool(size_t initial, size_t max_):Pool<SNMP_sysdesc_query>(initial, max_) {}
    
    static SNMPHostSysdescQueryPool *getInstance()
    {
        sysdesc_mutex.lock();
        if(instance==NULL)
            instance=new SNMPHostSysdescQueryPool(0, 32);
        sysdesc_mutex.unlock();
        return instance;
    }
};
 
SNMPHostSysdescQueryPool *SNMPHostSysdescQueryPool::instance=NULL;
Mutex SNMPHostSysdescQueryPool::sysdesc_mutex;

class SNMPHostInterfaceQueryPool: public Pool<SNMP_interface_query>
{
    private:
    
    static SNMPHostInterfaceQueryPool *instance;
    static Mutex                       interface_mutex;
    
    public:
    
    SNMPHostInterfaceQueryPool(size_t initial, size_t max_):Pool<SNMP_interface_query>(initial, max_) {}
    
    static SNMPHostInterfaceQueryPool *getInstance()
    {
        interface_mutex.lock();
        if(instance==NULL)
            instance=new SNMPHostInterfaceQueryPool(0, 32);
        interface_mutex.unlock();
        return instance;
    }
};
 
SNMPHostInterfaceQueryPool *SNMPHostInterfaceQueryPool::instance=NULL;
Mutex SNMPHostInterfaceQueryPool::interface_mutex;

#endif


HostDialog::HostDialog(FWObject *obj) 
{
    object=obj; 

    obj_addr=manage(new IPAddressWidget());
    table19->attach(*obj_addr, 1, 2, 1, 2, GTK_FILL, 0, 0, 0);
    obj_addr->changed.connect(SigC::slot(this,&HostDialog::on_changed));

    if_list->select_row.connect(SigC::slot(this, 
				  &HostDialog::on_if_list_select_row));

    object_parameters.push_back( DialogOption( obj_name         , "name"                 ) );
    object_parameters.push_back( DialogOption( comment          , "comment"              ) );
    object_parameters.push_back( DialogOption( obj_addr         , "address"              ) );
    object_parameters.push_back( DialogOption( snmp_r_community , "snmp_read_community"  ) );
    object_parameters.push_back( DialogOption( snmp_w_community , "snmp_write_community" ) );

    common_options.push_back(    DialogOption( use_mac_addr     , "use_mac_addr_filter"  ) );
    common_options.push_back(    DialogOption( snmp_description , "snmp_description"     ) );
    common_options.push_back(    DialogOption( snmp_location    , "snmp_location"        ) );
    common_options.push_back(    DialogOption( snmp_contact     , "snmp_contact"         ) );
}


void HostDialog::wrk2dlg()
{
    options=(Host::cast(object))->getOptionsObject();
    assert(options!=NULL);

    loadPageOptions(object,  object_parameters );
    loadPageOptions(options, common_options );

    showInterfaces();

    obj_name->grab_focus();
}

bool HostDialog::dlg2wrk()
{

/*
 *  check if name is valid
 */
    if ( ! checkObjectName(obj_name->get_text()) )      return(false);
    if ( ! checkIPaddress( obj_addr->getAsString() ) )  return(false);

    savePageOptions(object,  object_parameters );
    savePageOptions(options, common_options );

    return(true);
}

/**
 *  this method runs DNS lookup on the host name and enters address in obj_addr
 */
void HostDialog::on_dns_lookup()
{
    vector<IPAddress> addr;

    try {
	addr=DNS::getHostByName( obj_name->get_text() );
    } catch (FWException &e)
    {
	MessageDialog::Error( e.toString() );
	return;
    }

    obj_addr->setAddress( addr.front() );
    data_changed_flag(true);
}

void HostDialog::on_changed()
{
    data_changed_flag(true);
}

void HostDialog::showInterfaces()
{
    if_list->freeze();
    if_list->clear();

    vector<FWObject*>::iterator m;
    for (m=((Host*)object)->begin(); 
	 m!=((Host*)object)->end(); 
	 ++m) {

	if(Interface::isA((*m))) 
        {
	    if_list->addInterface( Interface::cast(*m) );
	}
    }
    if_list->thaw();
    if_list->show_all();
}

Interface* HostDialog::getInterfaceByRow(int row)
{
    Interface *intf=NULL;
    string id=if_list->getIdByRow(row);
    if(!id.empty()) 
	intf = Interface::cast( object->getById( id ) );

    return intf;
}

Interface* HostDialog::getSelectedInterface()
{
    Interface *intf=NULL;

    string id=if_list->getSelectedInterfaceId();
    if(!id.empty()) 
	intf = Interface::cast( object->getById( id ) );

    return intf;
}


void HostDialog::on_if_list_select_row(gint row, 
				       gint column, 
				       GdkEvent *event)
{   

    string id=if_list->getIdByRow(row);
    if(!id.empty()) 
    {
/*
	Interface *intf;
	intf = Interface::cast( object->getById( id ) );
	if(intf) 
            showInterfacePolicy( intf );
*/
    }
}


void HostDialog::on_new_if_clicked()
{ 
    bool res;

    options=(Host::cast(object))->getOptionsObject();
    assert(options!=NULL);

    Interface *intf=new Interface();

    InterfaceDialog *id = new InterfaceDialog(intf);
    res=id->run();
    delete id;
    if (res) {
	if ( (Host::cast(object))->Appropriate(intf) ) 
	    object->add(intf);

	showInterfaces();
	data_changed_flag(true);
    } else
	delete intf;
}

void HostDialog::on_del_if_clicked()
{   
    Interface *intf=getSelectedInterface();
    if (intf) {
	object->remove(intf);
	showInterfaces();
	data_changed_flag(true);
    }
}


void HostDialog::on_edit_clicked()
{   
    bool res;

    Interface *intf=getSelectedInterface();
    if(intf) 
    {
	InterfaceDialog *id = new InterfaceDialog( intf );
	res=id->run();
	delete id;
	if(res) 
        {
	    showInterfaces();
	    data_changed_flag(true);
	}
    }
}


void HostDialog::on_snmp_get_clicked()
{
}

void HostDialog::on_snmp_get_released()
{   
#ifdef HAVE_LIBSNMP
    string addr;
    string rcomm=snmp_r_community->get_text();

    options=(Host::cast(object))->getOptionsObject();
    assert(options!=NULL);


    if ( ! checkIPaddress(obj_addr->getAsString()) ) return;
    addr=obj_addr->getAsString();

    if ( rcomm.empty() ) {
	MessageDialog::Error("No SNMP community string defined");
	return;
    }


    FWObject *o;
    while ( (o=object->getFirstByType(Interface::TYPENAME))!=NULL )
	object->remove(o);


    int t=Preferences::global_prefs->getOptInt("/FWBuilderPreferences/Network/SNMPTimeout", -1);
    SNMP_interface_query *q=SNMPHostInterfaceQueryPool::getInstance()->lease();
    q->init(addr,  rcomm,
	    Preferences::global_prefs->getOptInt("/FWBuilderPreferences/Network/SNMPRetries",  SNMP_DEFAULT_RETRIES),
	    t==-1?SNMP_DEFAULT_TIMEOUT:(1000000L*t)
    );

    GenericBackgroundOpDialog bdisp(q);
    
    try
    {
        bdisp.execute();
        
        const map<int, Interface> &intf = q->getInterfaces();
        for(map<int, Interface>::const_iterator i=intf.begin();i!=intf.end(); ++i)
        {
            Interface *ifs=new Interface((*i).second);
            if((Host::cast(object))->Appropriate(ifs))
            {
                object->add( ifs );
            }
        }
        showInterfaces();
        data_changed_flag(true);
    } catch(const FWException &ex)
    {
        //do nothing
    }
#endif
}

void HostDialog::on_snmp_get_descr_released()
{
#ifdef HAVE_LIBSNMP

    string addr;
    string rcomm=snmp_r_community->get_text();

    if ( ! checkIPaddress(obj_addr->getAsString()) ) return;
    addr=obj_addr->getAsString();

    if(rcomm.empty()) 
    {
	MessageDialog::Error("No SNMP community string defined");
	return;
    }

    int t=Preferences::global_prefs->getOptInt("/FWBuilderPreferences/Network/SNMPTimeout", -1);
    SNMP_sysdesc_query *q=SNMPHostSysdescQueryPool::getInstance()->lease();
    q->init(addr , rcomm,
	    Preferences::global_prefs->getOptInt("/FWBuilderPreferences/Network/SNMPRetries",  SNMP_DEFAULT_RETRIES),
	    t==-1?SNMP_DEFAULT_TIMEOUT:(1000000L*t)
    );


    GenericBackgroundOpDialog bdisp(q);
    
    try
    {
        bdisp.execute();
        snmp_description->set_point(0);
        snmp_description->forward_delete( snmp_description->get_length() );
        snmp_description->insert( q->getDescr());
        
        snmp_location->set_text(q->getLocation());
        snmp_contact->set_text(q->getContact());

        data_changed_flag(true);
    } catch(const FWException &ex)
    {
        //do nothing
    }
#endif
}


