/* 

                          Firewall Builder

                 Copyright (C) 2003 NetCitadel, LLC

  Author:  Vadim Kurland     vadim@fwbuilder.org

  $Id: ObjConflictResolutionDialog.cpp,v 1.17 2004/07/11 06:43:10 vkurland Exp $

  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/


#include "config.h"
#include "global.h"
#include "utils.h"

#include "fwbuilder/FWObject.h"
#include "fwbuilder/FWObjectDatabase.h"

#include "ObjConflictResolutionDialog.h"
#include "FWObjectPropertiesFactory.h"
#include "FWBSettings.h"


#include <qdialog.h>
#include <qlabel.h>
#include <qmessagebox.h>
#include <qpixmap.h>
#include <qtextbrowser.h>
#include <qcheckbox.h>
#include <qpushbutton.h>

#include <iostream>

using namespace std;
using namespace libfwbuilder;


ObjConflictResolutionDialog::ObjConflictResolutionDialog(QWidget *parent) :
    ObjConflictResolutionDialog_q(parent,"ObjConflictResolutionDialog")
{
    alwaysCurrent=false;
    alwaysNew    =false;
    dlgIcon->setPixmap( QMessageBox::standardIcon( QMessageBox::Warning ) );

    if (st->haveGeometry(this)) st->restoreGeometry(this);
}

ObjConflictResolutionDialog::~ObjConflictResolutionDialog()
{
    saveGeometry();
}

int ObjConflictResolutionDialog::run(    FWObject *o1,
                                         FWObject *o2)
{
    if (alwaysCurrent) return QDialog::Rejected;
    if (alwaysNew)     return QDialog::Accepted;

    QString leftBtnTxt, rightBtnTxt;
    bool    leftCB,  rightCB, leftBtn, rightBtn;

    leftBtnTxt  = tr("Keep current object");
    rightBtnTxt = tr("Replace with this object");
    leftCB=rightCB=leftBtn=rightBtn=true;

    QString p1, p2;

    if (o1->getParent()->getId()!=FWObjectDatabase::getDeletedObjectsId())
        p1= FWObjectPropertiesFactory::getObjectPropertiesDetailed(o1,
                                                                   true,
                                                                   false,
                                                                   false);
    else
    {
        p1=tr("Object <b>'%1'</b> has been deleted").arg(o1->getName().c_str());
        rightBtnTxt = tr("Delete");
        leftCB  = rightCB = leftBtn = false;
    }

    if (o2->getParent()->getId()!=FWObjectDatabase::getDeletedObjectsId())
        p2= FWObjectPropertiesFactory::getObjectPropertiesDetailed(o2,
                                                                   true,
                                                                   false,
                                                                   false);
    else
    {
        p2=tr("Object <b>'%1'</b> has been deleted").arg(o2->getName().c_str());
        leftBtnTxt = tr("Delete");
        leftCB  = rightCB = rightBtn = false;
    }

    useCurrentObj->setText(leftBtnTxt);
    useNewObj->setText(rightBtnTxt);

    useNewObj->setEnabled(rightBtn);
    newAll->setEnabled(rightCB);
    useCurrentObj->setEnabled(leftBtn);
    currentAll->setEnabled(leftCB);

    if (leftBtn) useCurrentObj->setFocus();
    else         useNewObj->setFocus();

    QString f1= FWObjectDatabase::cast(o1->getRoot())->getFileName().c_str();
    QString f2= FWObjectDatabase::cast(o2->getRoot())->getFileName().c_str();

    if (f1.isEmpty()) f1=tr("Object <b>'%1'</b> in the objects tree")
                          .arg(o1->getName().c_str());
    else f1=tr("Object <b>'%1'</b> in file %2").arg(o1->getName().c_str()).arg(f1);

    f2=tr("Object <b>'%1'</b> in file %2").arg(o2->getName().c_str()).arg(f2);

    currentObjLbl->setText(f1);
    newObjLbl->setText(f2);
    
    currentObj->clear();
    newObj->clear();

    QString s;
    s="<a name=\"top\">\n";
    s+=p1;
    s+="<hr height=\"0\">";
    s+=o1->getComment().c_str();

    currentObj->moveCursor( QTextEdit::MoveHome , false );
    currentObj->append( s );
    currentObj->scrollToAnchor("top");

    s="<a name=\"top\">\n";
    s+=p2;
    s+="<hr height=\"0\">";
    s+=o2->getComment().c_str();

    newObj->moveCursor( QTextEdit::MoveHome , false );
    newObj->append( s );
    newObj->scrollToAnchor("top");

    return QDialog::exec();
}

void ObjConflictResolutionDialog::saveGeometry()
{
    st->saveGeometry(this);
}

/*
 * Important
 *
 * normally close event is sent when user clicks "close window" button
 * on the window titlebar. When this event is processed in this method,
 * the window is still visible so it is safe to retrieve and use its
 * geometry (it is bad to get geometry of the window when it is hidden
 * because at that time window manager decorations do not exist
 * anymore, so window's position on the screen is shiften up and to
 * the left). 
 *
 * It seems under certain window manager (at this time it is unknown
 * which one) in Gnome "close event" is generated after the window is
 * closed by clicking one of the buttons at the bottom (choosing which
 * objects to keep). We call saveGeometry from accept and reject to
 * get size and position when user clicks those buttons. Window is
 * then closed and (it seems) window manager sends "close" event to
 * it. By the time when we get control in this method, the window is
 * already closed and geometry returned for it would be incorrect.
 *
 * Finally, I decided to make it so the user can not close conflict
 * resolution dialog using titlebar button. The user is suppposed to
 * make a decision, and although closing dialog was equivalent to
 * clicking one of the choice buttons, it wasn't obvious. So it is
 * better to disable this completely and make it obvious for the user
 * that they must make a choice.
 */
void ObjConflictResolutionDialog::closeEvent(QCloseEvent *e)
{
    if (fwbdebug) qDebug("ObjConflictResolutionDialog::closeEvent");

    e->ignore();
}

void ObjConflictResolutionDialog::setFlags()
{
    alwaysCurrent = currentAll->isChecked();
    alwaysNew     = newAll->isChecked();
}

void ObjConflictResolutionDialog::accept()
{
  if (fwbdebug)
    qDebug("ObjConflictResolutionDialog::accept(): isVisible=%d",
	   isVisible());

    QDialog::accept();
}

void ObjConflictResolutionDialog::reject()
{
  if (fwbdebug)
    qDebug("ObjConflictResolutionDialog::reject(): isVisible=%d",
	   isVisible());

    QDialog::reject();
}
