///////
   //    _Url.cc
   //    _Url Class definitions
   //
   //    Class to interface with Url table of mysql Database
   //    This inherits from the Url class.
   //
   //    Copyright (c) 1999-2004 Comune di Prato - Prato - Italy
   //    Author: Gabriele Bartolini - Prato - Italy <angusgb@users.sourceforge.net>
   //
   //    For copyright details, see the file COPYING in your distribution
   //    or the GNU General Public License version 2 or later 
   //    <http://www.gnu.org/copyleft/gpl.html>
   //
   //    $Id: _Url.cc,v 1.23 2004/05/26 08:34:10 angusgb Exp $
   //
   //    G.Bartolini
   //    started: 05.07.1999
///////

#ifdef HAVE_CONFIG_H
#include "htconfig.h"
#endif /* HAVE_CONFIG_H */

#include <ctype.h>
#include "_Url.h"

///////
   //    Static variables
///////

   unsigned int _Url::TotUrls = 0;
   

///////
   //    Construction
///////

_Url::_Url ()
   : URL (),
   IDUrl(0),
   IDServer(0),
   ContentType(0),
   TransferEncoding(0),
   LastModified(0),
   LastAccess(0),
   Size(0),
   StatusCode(0),
   ReasonPhrase(0),
   Location(0),
   pServer(0),
   Title(0),
   ConnStatus(Url_OtherError),
   ContentLanguage(0),
   Contents(0),
   Charset(0),
   DocType(Url_Undefined),
   Description(0),
   Keywords(0),
   _HideLastModified(0)
{
}


_Url::_Url (const String &url)
   : URL (url),
   IDUrl(0),
   IDServer(0),
   ContentType(0),
   TransferEncoding(0),
   LastModified(0),
   LastAccess(0),
   Size(0),
   StatusCode(0),
   ReasonPhrase(0),
   Location(0),
   pServer(0),
   Title(0),
   ConnStatus(Url_OtherError),
   ContentLanguage(0),
   Contents(0),
   Charset(0),
   DocType(Url_Undefined),
   Description(0),
   Keywords(0),
   _HideLastModified(0)
{
}


_Url::_Url (const _Url &rhs)
   : URL (rhs),
   IDUrl(rhs.IDUrl),
   IDServer(rhs.IDServer),
   ContentType(rhs.ContentType),
   TransferEncoding(rhs.TransferEncoding),
   LastModified(rhs.LastModified),
   LastAccess(rhs.LastAccess),
   Size(rhs.Size),
   StatusCode(rhs.StatusCode),
   ReasonPhrase(rhs.ReasonPhrase),
   Location(rhs.Location),
   pServer(rhs.pServer),
   Title(rhs.Title),
   ConnStatus(rhs.ConnStatus),
   ContentLanguage(rhs.ContentLanguage),
   Contents(rhs.Contents),
   Charset(rhs.Charset),
   DocType(rhs.DocType),
   Description(rhs.Description),
   Keywords(rhs.Keywords),
   _HideLastModified(rhs._HideLastModified)
{

}


_Url::_Url (const String &ref, _Url &parent)
   : URL (ref, parent),
   IDUrl(0),
   IDServer(parent.IDServer),
   ContentType(0),
   TransferEncoding(0),
   LastModified(0),
   LastAccess(0),
   Size(0),
   StatusCode(0),
   ReasonPhrase(0),
   Location(0),
   pServer(parent.pServer),
   Title(0),
   ConnStatus(Url_OtherError),
   ContentLanguage(0),
   Contents(0),
   Charset(0),
   DocType(Url_Undefined),
   Description(0),
   Keywords(0),
   _HideLastModified(0)
{
}


///////
   //    Destruction
///////

_Url::~_Url ()
{
   
}


///////
   //    Reset
///////

void _Url::Reset()
{
    IDUrl = 0;
    IDServer = 0;
    ContentType.trunc();
    TransferEncoding.trunc();
    Size = 0;
    StatusCode = 0;
    ReasonPhrase.trunc();
    Location.trunc();
    pServer = 0;
    Title.trunc();
    ContentLanguage.trunc();
    Charset.trunc();
   
    if (LastModified)
        delete LastModified;

    if (LastAccess)
        delete LastAccess;
      
    ConnStatus = Url_OtherError;
    Contents = 0;
    _HideLastModified = false;

}


///////
   //    Managing the Connection Status of the Url
///////

void _Url::SetLastModified (HtDateTime *d)
{

   // If we had a previous value, let's delete it
   //   if (LastModified)
   //      delete LastModified;

   LastModified = d;  // just change it

}


///////
   //    Gives back the last modified time value for a URL
   //    depending also on the settings of the HideLastModified
   //    variable (in some cases we don't wanna show a last modified
   //    value - not found URLs, redirected ones, etc.)
///////

const HtDateTime *_Url::GetLastModified () const
{

   if (_HideLastModified)
      return 0;
   else return LastModified;

}


///////
   //    Managing the Connection Status of the Url
///////

///////
   //    Converts the Url_ConnStatus value into the corresponding
   //    String value
///////

void _Url::RetrieveConnStatus(String &Status) const
{

   switch(GetConnStatus())
   {
      case (Url_OK):
         Status="OK";
         break;
      case (Url_NoHeader):
         Status="NoHeader";
         break;
      case (Url_NoHost):
         Status="NoHost";
         break;
      case (Url_NoPort):
         Status="NoPort";
         break;
      case (Url_ConnectionDown):
         Status="ConnectionDown";
         break;
      case (Url_NoConnection):
         Status="NoConnection";
         break;
      case (Url_ServiceNotValid):
         Status="ServiceNotValid";
         break;
      case (Url_OtherError):
         Status="OtherError";
         break;
      case (Url_ServerError):
         Status="ServerError";
         break;
   }

}


///////
   //    Converts the Status string value
   //    into the corresponding Schedule_Status value 
   //    Returns 0 if an error occurs, 1 if OK.
///////

int _Url::SetConnStatus(const String &Status)
{

   if (!strcmp(Status, "OK"))
      SetConnStatus (Url_OK);
   if (!strcmp(Status, "NoHeader"))
      SetConnStatus (Url_NoHeader);
   else if (!strcmp(Status, "NoHost"))
      SetConnStatus (Url_NoHost);
   else if (!strcmp(Status, "NoPort"))
      SetConnStatus (Url_NoPort);
   else if (!strcmp(Status, "NoConnection"))
      SetConnStatus (Url_NoConnection);
   else if (!strcmp(Status, "ConnectionDown"))
      SetConnStatus (Url_ConnectionDown);
   else if (!strcmp(Status, "ServiceNotValid"))
      SetConnStatus (Url_ServiceNotValid);
   else if (!strcmp(Status, "OtherError"))
      SetConnStatus (Url_OtherError);
   else if (!strcmp(Status, "ServerError"))
      SetConnStatus (Url_ServerError);
   else return 0;
   
   return 1;
}


///////
   //    Converts the Url_DocType value into the corresponding
   //    String value
///////

void _Url::RetrieveDocType(String &DocType) const
{

   switch(GetDocType())
   {
      case (Url_Undefined):
         DocType="";
         break;
      case (Url_XHtml_11_Strict):
         DocType="xhtml-11";
		 break;
      case (Url_XHtml_10_Strict):
         DocType="xhtml-10";
		 break;
      case (Url_XHtml_10_Transitional):
         DocType="xhtml-10-transitional";
		 break;
      case (Url_XHtml_10_Frameset):
         DocType="xhtml-10-frameset";
		 break;
      case (Url_Html_401_Strict):
         DocType="html-401";
		 break;
      case (Url_Html_401_Transitional):
         DocType="html-401-transitional";
		 break;
      case (Url_Html_401_Frameset):
         DocType="html-401-frameset";
		 break;
      case (Url_Html_40_Strict):
         DocType="html-40";
		 break;
      case (Url_Html_40_Transitional):
         DocType="html-40-transitional";
		 break;
      case (Url_Html_40_Frameset):
         DocType="html-40-frameset";
		 break;
      case (Url_Html_ISO_IEC_15445_2000):
         DocType="html-iso-iec-15445-2000";
		 break;
      case (Url_Html_32):
         DocType="html-32";
		 break;
      case (Url_Html_20):
         DocType="html-20";
		 break;
      case (Url_Html_20_Level2):
         DocType="html-20-level2";
		 break;
      case (Url_Html_20_Level1):
         DocType="html-20-level1";
		 break;
      case (Url_Html_20_Strict):
         DocType="html-20-strict";
		 break;
      case (Url_Html_20_Strict_Level1):
         DocType="html-20-strict-level1";
		 break;
      case (Url_Not_Public):
         DocType="not-public";
         break;
      case (Url_Not_Html):
         DocType="not-html";
         break;
      case (Url_Unknown):
         DocType="unknown";
         break;
   }

}


///////
   //    Set the content type and, in case the charset is specified,
   //    grabs it and puts it into the Charset field.
   //    For instance: "text/html; iso-8859-1" now set the ContentType to text/html
   //    and the charset to iso-8859-1.
///////

void _Url::SetContentType (const String &Ct)
{
	const char *p = (const char *)Ct;
    ContentType.trunc();

	// Append the content-type 
	while (p && *p && (*p != ';' && !isspace(*p)))
	{
			ContentType.append(*p);
			++p;
	}

	while (p && *p && (*p == ';' || isspace(*p)))
			++p;

	if (!mystrncasecmp(p, "charset", 7))
	{
			// Found charset (go after that)
			while (p && *p && *p != '=')
					++p;

			while (p && *p && (*p == '=' || isspace(*p)))
					++p;

    		Charset.trunc(); // Clean the charset
			while (p && *p && !isspace(*p))
			{
					// Append the charset
					Charset.append(*p);
					++p;
			}
	}

}

///////
   //    Converts the DocType string value
   //    into the corresponding Url_DocType value 
   //    Returns 0 if an error occurs, 1 if OK.
///////

int _Url::SetDocType(const String &DocType)
{

	if (DocType.length() > 0 )
	{
		const char *p = (const char *)DocType;

   		if (!mystrncasecmp(p, "html", 4))
		{
			for (p += 4; *p && isspace(*p); ++p);

			if (!mystrncasecmp(p, "public", 6))
			{
				for (p += 6; *p && (isspace(*p) || *p != '"'); ++p);

				if (*p && *p == '"')
				{
					char doctype[128];
					char *p2 = doctype;
					*p2 = '\0';
					for (++p; *p && *p != '"'; ++p)
					{
						// Skip consecutive spaces or initial spaces
						if (isspace(*p))
						{
						   if (p2 > doctype && isspace(* (p2-1)))
						      continue;

						   // Skip a space if the preceding character is a '/' or a '-'
						   if (p2 > doctype && (*(p2-1) == '/' || *(p2-1) == '-'))
						      continue;

						   // Skip a space if the following character is a '/' or a '-'
						   if (*(p+1) && (*(p+1) == '/' || *(p+1) == '-'))
						      continue;
						}
						*p2++ = *p;
					}

					*p2 = '\0'; // closes the string
					// Let's detect the DOCTYPE declaration
					if (!mystrcasecmp(doctype, "-//W3C//DTD XHTML 1.1//EN"))
						SetDocType(Url_XHtml_11_Strict);
					else if (!mystrcasecmp(doctype, "-//W3C//DTD XHTML 1.0 Strict//EN"))
						SetDocType(Url_XHtml_10_Strict);
					else if (!mystrcasecmp(doctype, "-//W3C//DTD XHTML 1.0 Transitional//EN"))
						SetDocType(Url_XHtml_10_Transitional);
					else if (!mystrcasecmp(doctype, "-//W3C//DTD XHTML 1.0 Frameset//EN"))
						SetDocType(Url_XHtml_10_Frameset);
					else if (!mystrcasecmp(doctype, "-//W3C//DTD HTML 4.01//EN"))
						SetDocType(Url_Html_401_Strict);
					else if (!mystrcasecmp(doctype, "-//W3C//DTD HTML 4.01 Transitional//EN"))
						SetDocType(Url_Html_401_Transitional);
					else if (!mystrcasecmp(doctype, "-//W3C//DTD HTML 4.01 Frameset//EN"))
						SetDocType(Url_Html_401_Frameset);
					else if (!mystrcasecmp(doctype, "-//W3C//DTD HTML 4.0//EN"))
						SetDocType(Url_Html_40_Strict);
					else if (!mystrcasecmp(doctype, "-//W3C//DTD HTML 4.0 Transitional//EN"))
						SetDocType(Url_Html_40_Transitional);
					else if (!mystrcasecmp(doctype, "-//W3C//DTD HTML 4.0 Frameset//EN"))
						SetDocType(Url_Html_40_Frameset);
					else if (!mystrcasecmp(doctype, "ISO/IEC 15445:2000//DTD HyperText Markup Language//EN") ||
						!mystrcasecmp(doctype, "ISO/IEC 15445:2000//DTD HTML//EN"))
							SetDocType(Url_Html_ISO_IEC_15445_2000);
					else if (!mystrcasecmp(doctype, "-//W3C//DTD HTML 3.2//EN") ||
						!mystrcasecmp(doctype, "-//W3C//DTD HTML 3.2 Final//EN"))
							SetDocType(Url_Html_32);
					else if (!mystrcasecmp(doctype, "-//IETF//DTD HTML//EN") ||
						!mystrcasecmp(doctype, "-//IETF//DTD HTML 2.0//EN"))
							SetDocType(Url_Html_20);
					else if (!mystrcasecmp(doctype, "-//IETF//DTD HTML 2.0 Level 2//EN"))
						SetDocType(Url_Html_20_Level2);
					else if (!mystrcasecmp(doctype, "-//IETF//DTD HTML 2.0 Level 1//EN"))
						SetDocType(Url_Html_20_Level1);
					else if (!mystrcasecmp(doctype, "-//IETF//DTD HTML 2.0 Strict//EN"))
						SetDocType(Url_Html_20_Strict);
					else if (!mystrcasecmp(doctype, "-//IETF//DTD HTML 2.0 Strict Level 1//EN"))
						SetDocType(Url_Html_20_Strict_Level1);
					else
						SetDocType(Url_Unknown);
				}
				else
					SetDocType(Url_Unknown);
			}
			else
				SetDocType(Url_Not_Public);

		}
		else
			SetDocType(Url_Not_Html);
	}
	else
		SetDocType(Url_Undefined);

	return 1;
}

