/* Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements. 
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * @file
 *
 * @Author christian liesch <liesch@gmx.ch>
 *
 * Implementation of the HTTP Test Tool worker.
 */

/************************************************************************
 * Includes
 ***********************************************************************/
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <openssl/ssl.h>

#include <apr.h>
#include <apr_lib.h>
#include <apr_errno.h>
#include <apr_strings.h>
#include <apr_network_io.h>
#include <apr_thread_proc.h>
#include <apr_thread_cond.h>
#include <apr_thread_mutex.h>
#include <apr_portable.h>
#include <apr_hash.h>
#include <apr_base64.h>

#include <pcre.h>
#if APR_HAVE_UNISTD_H
#include <unistd.h> /* for getpid() */
#endif

#include "defines.h"
#include "util.h"
#include "regex.h"
#include "file.h"
#include "socket.h"
#include "ssl.h"
#include "worker.h"


/************************************************************************
 * Definitions 
 ***********************************************************************/

typedef struct url_escape_seq {
  char c;
  char *esc;
  apr_size_t len;
} url_escape_seq_t;

typedef struct write_buf_to_file_s {
  char *buf;
  apr_size_t len;
  apr_file_t *fp;
} write_buf_to_file_t;

typedef struct tunnel_s {
  apr_pool_t *pool;
  sockreader_t *sockreader;
  socket_t *sendto;
} tunnel_t;

/************************************************************************
 * Globals 
 ***********************************************************************/

url_escape_seq_t url_escape_seq[] = {
  { '?', "%3F", 3 },
  { '\n', "%0D", 3 },
  { ' ', "+", 1 },
  { '/', "%2F", 3 },
  { ';', "%3B", 3 },
  { '%', "%25", 3 },
  { '=', "%3D", 3 },
  { '"', "%22", 3 },
  { '\'', "%2C", 3 },
  { '.', "%2E", 3 },
  { ':', "%3A", 3 },
  { '@', "%40", 3 },
  { '\\', "%5C", 3 },
  { '&', "%26", 3 },
  { '+', "%2B", 3 },
};


/************************************************************************
 * Implementation
 ***********************************************************************/

static apr_status_t socket_send(socket_t *socket, char *buf, apr_size_t len);

/**
 * a simple log mechanisme
 *
 * @param self IN thread data object
 * @param log_mode IN log mode
 *                    LOG_DEBUG for a lot of infos
 *                    LOG_INFO for much infos
 *                    LOG_ERR for only very few infos
 * @param fmt IN printf format string
 * @param ... IN params for format strings
 */
void worker_log(worker_t * self, int log_mode, char *fmt, ...) {
  char *tmp;
  va_list va;

  va_start(va, fmt);
  if (self->log_mode >= log_mode) {
    if (log_mode == LOG_ERR) {
      tmp = apr_pvsprintf(self->ptmp, fmt, va);
      //tmp = apr_psprintf(self->ptmp, "%s: error: %s", self->file_and_line, tmp);
      fprintf(stderr, "\n%-88s", tmp);
      fflush(stderr);
    }
    else {
      fprintf(stdout, "\n%s", self->prefix);
      vfprintf(stdout, fmt, va);
      fflush(stdout);
    }
  }
  va_end(va);
}

/**
 * a simple error log mechanisme
 *
 * @param self IN thread data object
 * @param fmt IN printf format string
 * @param ... IN params for format strings
 */
void worker_log_error(worker_t * self, char *fmt, ...) {
  char *tmp;
  va_list va;

  va_start(va, fmt);
  if (self->log_mode >= LOG_ERR) {
    tmp = apr_pvsprintf(self->ptmp, fmt, va);
    tmp = apr_psprintf(self->ptmp, "%s: error: %s", self->file_and_line,
	               tmp);
    fprintf(stderr, "\n%-88s", tmp);
    fflush(stderr);
  }
}

/**
 * a simple log buf mechanisme
 *
 * @param self IN thread data object
 * @param log_mode IN log mode
 *                    LOG_DEBUG for a lot of infos
 *                    LOG_INFO for much infos
 *                    LOG_ERR for only very few infos
 * @param buf IN buf to print (binary data allowed)
 * @param prefix IN prefix before buf
 * @param len IN buf len
 */
void worker_log_buf(worker_t * self, int log_mode, char *buf,
                    char *prefix, int len) {
  int i;
  char *null="<null>";

  FILE *fd = stdout;

  if (!buf) {
    buf = null;
    len = strlen(buf);
  }
  
  if (log_mode == LOG_ERR) {
    fd = stderr;
  }
  if (self->log_mode >= log_mode) {
    i = 0;
    fprintf(fd, "\n%s%s", self->prefix, prefix);
    while (i < len) {
      while (i < len && buf[i] != '\r' && buf[i] != '\n') {
	if (buf[i] >= 0x20) {
	  fprintf(fd, "%c", buf[i]);
	}
	else {
	  fprintf(fd, "0x%02x ", (unsigned char)buf[i]);
	}
        i++;
      }
      while (buf[i] == '\r' || buf[i] == '\n') {
	fprintf(fd, "%c", buf[i]);
	if (buf[i] == '\n') {
	  fprintf(fd, "%s%s", self->prefix, prefix);
	}
        i++;
      }
      fflush(fd);
    }
  }
}

/**
 * client thread
 *
 * @param thread IN thread object
 * @param selfv IN void pointer to tunnel object
 *
 * @return NULL 
 */
static void * APR_THREAD_FUNC streamer(apr_thread_t * thread, void *selfv) {
  apr_status_t status;
  char buf[BLOCK_MAX];
  apr_size_t len;

  tunnel_t *tunnel = selfv;

  do {
    /* read polled from and send to */
    len = BLOCK_MAX - 1;
    status = sockreader_read_block(tunnel->sockreader, buf, &len);
    if (APR_STATUS_IS_EOF(status) && len > 0) {
      status = APR_SUCCESS;
    }
    else if (APR_STATUS_IS_TIMEUP(status)) {
      status = APR_SUCCESS;
    }
    if (status == APR_SUCCESS) {
      status = socket_send(tunnel->sendto, buf, len);
    }
  } while (status == APR_SUCCESS);

  if (APR_STATUS_IS_EOF(status)) {
    status = APR_SUCCESS;
  }
  apr_thread_exit(thread, APR_SUCCESS);
  return NULL;
}

/**
 * Buffer converter depends on the worker->flags
 *
 * @param worker IN thread data object
 * @param buf INOUT buffer to rewrite
 * @param len INOUT buffer len 
 */
void worker_buf_convert(worker_t *self, char **buf, apr_size_t *len) {
  int j;
  char *hexbuf;
  apr_pool_t *pool;
  
  if (!(*buf)) {
    return;
  }
  
  if (self->flags & FLAGS_ONLY_PRINTABLE) {
    for (j = 0; j < *len; j++) {
      if ((*buf)[j] < 32) {
	(*buf)[j] = ' ';
      }
    }
  }
  
  if (self->flags & FLAGS_PRINT_HEX) {
    apr_pool_create(&pool, NULL);
    hexbuf = NULL;
    for (j = 0; j < *len; j++) {
      if (hexbuf == NULL) {
	 hexbuf = apr_psprintf(pool, "%02X", (*buf)[j]);
      }
      else {
	 hexbuf = apr_psprintf(pool, "%s%02X", hexbuf, (*buf)[j]);
      }
    }
    *buf = apr_pstrdup(self->ptmp, hexbuf);
    *len = strlen(*buf);
    apr_pool_destroy(pool);
  }
}


#ifdef USE_SSL
/**
 * worker ssl handshake client site
 *
 * @param worker IN thread data object
 *
 * @return apr status
 */
apr_status_t worker_ssl_handshake(worker_t * worker) {
  apr_status_t status;
  char *error;
  
  if ((status = ssl_handshake(worker->socket->ssl, &error, worker->ptmp)) 
      != APR_SUCCESS) {
    worker_log(worker, LOG_ERR, "%s", error);
  }
  return status;
}

/**
 * Get server ctx with loaded cert and key file
 *
 * @param self IN thread object data
 *
 * @return APR_SUCCESS or APR_ECONNABORTED
 */
apr_status_t worker_ssl_ctx(worker_t * self, char *certfile, char *keyfile, int check) {
  if (self->is_ssl && !self->ssl_ctx) {
    if (!(self->ssl_ctx = SSL_CTX_new(self->meth))) {
      worker_log(self, LOG_ERR, "Could not initialize SSL Context.");
      return APR_ECONNABORTED;
    }
  }
  if (self->is_ssl && self->ssl_ctx) {
    if (SSL_CTX_use_certificate_file(self->ssl_ctx, certfile, 
	SSL_FILETYPE_PEM) <= 0 && check) { 
      worker_log(self, LOG_ERR, "Could not load RSA server certifacte \"%s\"",
	         certfile);
      return APR_ECONNABORTED;
    }
    if (SSL_CTX_use_PrivateKey_file(self->ssl_ctx, keyfile, 
	SSL_FILETYPE_PEM) <= 0 && check) {
      worker_log(self, LOG_ERR, "Could not load RSA server private key \"%s\"",
	         keyfile);
      return APR_ECONNABORTED;
    }
    if (!SSL_CTX_check_private_key(self->ssl_ctx) && check) {
      worker_log(self, LOG_ERR, "Private key does not match the certificate public key");
      return APR_ECONNABORTED;
    }
  }
  return APR_SUCCESS;
}

/**
 * Do a ssl accept
 *
 * @param worker IN thread data object
 *
 * @return APR_SUCCESS
 */
apr_status_t worker_ssl_accept(worker_t * worker) {
  apr_status_t status;
  char *error;

  if (worker->socket->is_ssl) {
    if (!worker->socket->ssl) {
      BIO *bio;
      apr_os_sock_t fd;

      if ((worker->socket->ssl = SSL_new(worker->ssl_ctx)) == NULL) {
	worker_log(worker, LOG_ERR, "SSL_new failed.");
	status = APR_ECONNREFUSED;
      }
      SSL_set_ssl_method(worker->socket->ssl, worker->meth);
      ssl_rand_seed();
      apr_os_sock_get(&fd, worker->socket->socket);
      bio = BIO_new_socket(fd, BIO_NOCLOSE);
      SSL_set_bio(worker->socket->ssl, bio, bio);
    }
    else {
      return APR_SUCCESS;
    }
  }
  else {
    return APR_SUCCESS;
  }

  if ((status = ssl_accept(worker->socket->ssl, &error, worker->ptmp)) 
      != APR_SUCCESS) {
    worker_log(worker, LOG_ERR, "%s", error);
  }
  return status;
}
#endif

/**
 * Test socket state
 *
 * @param worker IN thread data object
 *
 * @return APR_SUCCESS or APR_ECONNABORTED
 */
apr_status_t worker_sockstate(worker_t * worker) {
  apr_status_t status = APR_SUCCESS;
  apr_size_t len = 1;

  if (!worker->socket || !worker->socket->socket) {
    return APR_ENOSOCKET;
  }
  
  /* give the other side a little time to finish closing 10 ms */
  apr_sleep(10000);
  
  if ((status = apr_socket_timeout_set(worker->socket->socket, 1000)) 
      != APR_SUCCESS) {
    return status;
  }

#ifdef USE_SSL
  if (worker->socket->ssl) {
    status = SSL_read(worker->socket->ssl, &worker->socket->peek[worker->socket->peeklen], len);
    if (status <= 0) {
      int scode = SSL_get_error(worker->socket->ssl, status);

      if (scode == SSL_ERROR_ZERO_RETURN) {
        status = APR_ECONNABORTED; 
	goto go_out;
      }
      else if (scode != SSL_ERROR_WANT_WRITE && scode != SSL_ERROR_WANT_READ) {
        status = APR_ECONNABORTED; 
	goto go_out;
      }
      else {
	status = APR_SUCCESS;
	goto go_out;
      }
    }
    else {
      worker->socket->peeklen += len;
      status = APR_SUCCESS;
      goto go_out;
    }
  }
  else
#endif
  {
    status = apr_socket_recv(worker->socket->socket, 
	                     &worker->socket->peek[worker->socket->peeklen], &len);
    if (APR_STATUS_IS_TIMEUP(status)) {
      status = APR_SUCCESS;
    }

    if (APR_STATUS_IS_EOF(status)) {
      status = APR_ECONNABORTED; 
      goto go_out;
    }
    else if (status != APR_SUCCESS) {
      status = APR_ECONNABORTED; 
      goto go_out;
    }
    else {
      worker->socket->peeklen += len;
      status = APR_SUCCESS;
      goto go_out;
    }
  }

go_out:
  apr_socket_timeout_set(worker->socket->socket, worker->socktmo);

  return status;
}

/**
 * replace variables in a line
 *
 * @param worker IN thread data object
 * @param line IN line to replace in
 *
 * @return new line 
 */
char * worker_replace_vars(worker_t * worker, char *line) {
  char *new_line;

  /* replace all parameters first */
  new_line = my_replace_vars(worker->ptmp, line, worker->params); 
  /* replace all vars */
  new_line = my_replace_vars(worker->ptmp, new_line, worker->vars); 

  return new_line;
}

/**
 * gets values from data and store it in the variable table
 *
 * @param worker IN thread data object
 * @param regexs IN table of regular expressions to get the values from data
 * @param data IN data to match
 *
 * @return APR_SUCCESS
 */
apr_status_t worker_match(worker_t * worker, apr_table_t * regexs, 
                          const char *data, apr_size_t len) {
  int rc;
  apr_table_entry_t *e;
  apr_table_entry_t *v;
  regmatch_t regmatch[11];
  int i;
  int j;
  char *val;
  char *last;
  char *var;
  char *tmp;
  apr_table_t *vtbl;
  int n;

  if (!data) {
    return APR_SUCCESS;
  }

  vtbl = apr_table_make(worker->ptmp, 2);
  
  e = (apr_table_entry_t *) apr_table_elts(regexs)->elts;
  for (i = 0; i < apr_table_elts(regexs)->nelts; ++i) {
    /* prepare vars if multiple */
    apr_table_clear(vtbl);
    tmp = apr_pstrdup(worker->ptmp, e[i].key);
    var = apr_strtok(tmp, " ", &last);
    while (var) {
      apr_table_set(vtbl, var, var);
      var = apr_strtok(NULL, " ", &last);
    }

    n = apr_table_elts(vtbl)->nelts;
    if (n > 10) {
      worker_log(worker, LOG_ERR, "Too many vars defined for _MATCH statement, max 10 vars allowed");
      return APR_EINVAL;
    }
    
    if (e[i].val
        && (rc =
            regexec((regex_t *) e[i].val, data, len, n + 1, regmatch,
                    PCRE_MULTILINE)) == 0) {
      v = (apr_table_entry_t *) apr_table_elts(vtbl)->elts;
      for (j = 0; j < n; j++) {
	val =
	  apr_pstrndup(worker->pool, &data[regmatch[j + 1].rm_so],
		       regmatch[j + 1].rm_eo - regmatch[j + 1].rm_so);
	apr_table_set(worker->vars, v[j].key, val);
      }
    }
  }

  return APR_SUCCESS;
}

/**
 * checks if data contains a given pattern
 *
 * @param self IN thread data object
 * @param regexs IN table of regular expressions
 * @param data IN data to check
 *
 * @return APR_SUCCESS
 */
apr_status_t worker_expect(worker_t * self, apr_table_t * regexs, 
                           const char *data, apr_size_t len) {
  int rc;
  apr_table_entry_t *e;
  int i;

  if (!data) {
    return APR_SUCCESS;
  }

  e = (apr_table_entry_t *) apr_table_elts(regexs)->elts;
  for (i = 0; i < apr_table_elts(regexs)->nelts; ++i) {
    if (e[i].val
        && (rc =
            regexec((regex_t *) e[i].val, data, len, 0, NULL,
                    PCRE_MULTILINE)) == 0) {
    }
  }

  return APR_SUCCESS;
}

apr_status_t worker_validate_match(worker_t * self, apr_table_t *match, 
                                   char *error_prefix, apr_status_t status) {
  apr_table_entry_t *e;
  int i;

  e = (apr_table_entry_t *) apr_table_elts(match)->elts;
  for (i = 0; i < apr_table_elts(match)->nelts; ++i) {
    if (!regdidmatch((regex_t *) e[i].val)) {
      worker_log(self, LOG_ERR, "%s: Did expect %s", error_prefix, e[i].key);
      if (status == APR_SUCCESS) {
	status = APR_EINVAL;
      }
    }
  }
  apr_table_clear(match);
  return status;
}

apr_status_t worker_validate_expect(worker_t * self, apr_table_t *expect, 
                                    char *error_prefix, apr_status_t status) {
  apr_table_entry_t *e;
  int i;

  e = (apr_table_entry_t *) apr_table_elts(expect)->elts;
  for (i = 0; i < apr_table_elts(expect)->nelts; ++i) {
    if (e[i].key[0] != '!' && !regdidmatch((regex_t *) e[i].val)) {
      worker_log(self, LOG_ERR, "%s: Did expect \"%s\"", error_prefix, 
	         e[i].key);
      if (status == APR_SUCCESS) {
	status = APR_EINVAL;
      }
    }
    if (e[i].key[0] == '!' && regdidmatch((regex_t *) e[i].val)) {
      worker_log(self, LOG_ERR, "%s: Did not expect \"%s\"", error_prefix, 
	         &e[i].key[1]);
      if (status == APR_SUCCESS) {
	status = APR_EINVAL;
      }
    }
  }
  apr_table_clear(expect);
  return status;
}

/**
 * Do check for if all defined expects are handled 
 *
 * @param self IN worker thread object
 * @param status IN current status
 *
 * @return current status or APR_EINVAL if there are unhandled expects
 */
apr_status_t worker_check_expect(worker_t * self, apr_status_t status) {
  status = worker_validate_match(self, self->match_headers, "MATCH headers", 
                                 status);
  status = worker_validate_match(self, self->match_body, "MATCH body", 
                                 status);
  status = worker_validate_expect(self, self->expect_dot, "EXPECT", 
                                 status);
  return status;
}

/**
 * Check for error expects handling
 *
 * @param self IN worker thread object
 * @param status IN current status
 *
 * @return current status or APR_INVAL
 */
apr_status_t worker_check_error(worker_t *self, apr_status_t status) {
  char *error;
  apr_table_entry_t *e;
  int i;

  /* nothing to do in this case */
  if (status == APR_SUCCESS) {
    return status;
  }
  
  /* handle special case (break loop) */
  if (status == -1) {
    return status;
  }

  error = apr_psprintf(self->pool, "%s(%d)",
		     my_status_str(self->pool, status), status);

  worker_match(self, self->match_error, error, strlen(error));
  worker_expect(self, self->expect_error, error, strlen(error));

  if (apr_table_elts(self->expect_error)->nelts) {
    status = APR_SUCCESS;
    e = (apr_table_entry_t *) apr_table_elts(self->expect_error)->elts;
    for (i = 0; i < apr_table_elts(self->expect_error)->nelts; ++i) {
      if (e[i].key[0] != '!' && !regdidmatch((regex_t *) e[i].val)) {
	worker_log(self, LOG_ERR, "EXPECT: Did expect error \"%s\"", e[i].key);
	status = APR_EINVAL;
	goto error;
      }
      if (e[i].key[0] == '!' && regdidmatch((regex_t *) e[i].val)) {
	worker_log(self, LOG_ERR, "EXPECT: Did not expect error \"%s\"", &e[i].key[1]);
	status = APR_EINVAL;
	goto error;
      }
    }
    apr_table_clear(self->expect_error);
  }
 
  if (apr_table_elts(self->match_error)->nelts) {
    status = APR_SUCCESS;
    e = (apr_table_entry_t *) apr_table_elts(self->match_error)->elts;
    for (i = 0; i < apr_table_elts(self->match_error)->nelts; ++i) {
      if (!regdidmatch((regex_t *) e[i].val)) {
	worker_log(self, LOG_ERR, "MATCH headers: Did expect %s", e[i].key);
	status = APR_EINVAL;
      }
    }
    apr_table_clear(self->match_error);
  }

error:
  if (status == APR_SUCCESS) {
    worker_log(self, LOG_INFO, "%s %s", self->name, error);
  }
  else {
    worker_log_error(self, "%s %s", self->name, error);
  }
  return status;
}

/**
 * Test for unused expects and matchs
 *
 * @param self IN thread data object
 *
 * @return APR_SUCCESS or APR_EGENERAL
 */
apr_status_t worker_test_unused(worker_t * self) {
  if (apr_table_elts(self->match_headers)->nelts) {
    worker_log(self, LOG_ERR, "There are unused MATCH headers");
    return APR_EGENERAL;
  }
  if (apr_table_elts(self->match_body)->nelts) {
    worker_log(self, LOG_ERR, "There are unused MATCH body");
    return APR_EGENERAL;
  }
  if (apr_table_elts(self->match_exec)->nelts) {
    worker_log(self, LOG_ERR, "There are unused MATCH exec");
    return APR_EGENERAL;
  }
  if (apr_table_elts(self->expect_dot)->nelts) {
    worker_log(self, LOG_ERR, "There are unused EXPECT .");
    return APR_EGENERAL;
  }

  return APR_SUCCESS;
}

/**
 * Test for unused expects errors and matchs
 *
 * @param self IN thread data object
 *
 * @return APR_SUCCESS or APR_EGENERAL
 */
apr_status_t worker_test_unused_errors(worker_t * self) {
  if (apr_table_elts(self->expect_error)->nelts) { 
    worker_log(self, LOG_ERR, "There are unused EXPECT ERROR");
    return APR_EGENERAL;
  }

  if (apr_table_elts(self->match_error)->nelts) {
    worker_log(self, LOG_ERR, "There are unused MATCH ERROR");
    return APR_EGENERAL;
  }
 
  apr_pool_clear(self->ptmp);

  return APR_SUCCESS;
}

/**
 * Close current socket
 *
 * @param self IN thread data object
 *
 * @return apr status
 */
apr_status_t worker_conn_close(worker_t * self) {
  apr_status_t status;
#ifdef USE_SSL
  int i;
#endif

  if (!self->socket || !self->socket->socket) {
    return APR_ENOSOCKET;
  }

  if (self->socket->socket_state == SOCKET_CLOSED) {
    return APR_SUCCESS;
  }
  
#ifdef USE_SSL
  if (self->socket->is_ssl) {
    if (self->socket->ssl) {
      for (i = 0; i < 4; i++) {
	if (SSL_shutdown(self->socket->ssl) != 0) {
	  break;
	}
      }
      SSL_free(self->socket->ssl);
      self->socket->ssl = NULL;
    }
    self->socket->is_ssl = 0;
  }
#endif
  if ((status = apr_socket_close(self->socket->socket)) != APR_SUCCESS) {
    return status;
  }
  self->socket->socket_state = SOCKET_CLOSED;
  self->socket->socket = NULL;

  return APR_SUCCESS;
}

/**
 * Close all sockets for this worker
 *
 * @param self IN thread data object
 *
 * @return apr status
 */
void worker_conn_close_all(worker_t *self) {
  apr_hash_index_t *hi;
  void *s;
  
  socket_t *cur = self->socket;

  for (hi = apr_hash_first(self->ptmp, self->sockets); hi; hi = apr_hash_next(hi)) {
    apr_hash_this(hi, NULL, NULL, &s);
    self->socket = s;
    worker_conn_close(self);
  }
  self->socket = cur;
  if (self->listener) {
    apr_socket_close(self->listener);
  }
}

/**
 * local file write 
 *
 * @param sockett IN socket 
 * @param buf IN buffer to send
 * @param len IN no bytes of buffer to send
 *
 * @return apr status
 */
static apr_status_t file_write(apr_file_t *file, char *buf,
                               apr_size_t len) {
  apr_status_t status = APR_SUCCESS;
  apr_size_t total = len;
  apr_size_t count = 0;

  while (total != count) {
    len = total - count;
    if ((status = apr_file_write(file, &buf[count], &len)) 
	!= APR_SUCCESS) {
      goto error;
    }
    count += len;
  }
error:
  return status;
}

/**
 * write buf to file pointer
 *
 * @param thread IN thread pointer
 * @param selfv IN void pointer of type write_buf_to_file_t
 *
 * @return NULL
 */
static void * APR_THREAD_FUNC worker_write_buf_to_file(apr_thread_t * thread, void *selfv) {
  write_buf_to_file_t *wbtf = selfv;
  apr_size_t len;

  len = wbtf->len;
  file_write(wbtf->fp, wbtf->buf, len);
  apr_file_close(wbtf->fp);

  apr_thread_exit(thread, APR_SUCCESS);
  return NULL;
}

static apr_status_t worker_handle_buf(worker_t *worker, apr_pool_t *pool, 
                                      char *buf, apr_size_t len) {
  apr_status_t status = APR_SUCCESS;
  apr_size_t inlen;
  apr_exit_why_e exitwhy;
  int exitcode;
  write_buf_to_file_t write_buf_to_file;
  apr_threadattr_t *tattr;
  apr_thread_t *thread;
  bufreader_t *br;
  char *line;
  apr_status_t tmp_status;

  if (buf) {
    worker_buf_convert(worker, &buf, &len);
    if (worker->flags & FLAGS_PIPE_IN) {
      worker->flags &= ~FLAGS_PIPE_IN;
      inlen = len;
      if ((status = file_write(worker->proc.in, buf, inlen))
	  != APR_SUCCESS) {
	goto out_err;
      }
      apr_file_close(worker->proc.in);
      apr_proc_wait(&worker->proc, &exitcode, &exitwhy, APR_WAIT);
      if (exitcode != 0) {
	status = APR_EGENERAL;
	goto out_err;
      }
    }
    else if (worker->flags & FLAGS_FILTER) {
      worker_log(worker, LOG_DEBUG, "write to stdin, read from stdout");
      worker->flags &= ~FLAGS_FILTER;
      /* start write thread */
      write_buf_to_file.buf = buf;
      write_buf_to_file.len = len;
      write_buf_to_file.fp = worker->proc.in;
      if ((status = apr_threadattr_create(&tattr, pool)) != APR_SUCCESS) {
	goto out_err;
      }
      if ((status = apr_threadattr_stacksize_set(tattr, DEFAULT_THREAD_STACKSIZE))
	  != APR_SUCCESS) {
	goto out_err;
      }
      if ((status = apr_threadattr_detach_set(tattr, 1)) != APR_SUCCESS) {
	goto out_err;
      }
      if ((status =
	   apr_thread_create(&thread, tattr, worker_write_buf_to_file,
			     &write_buf_to_file, pool)) != APR_SUCCESS) {
	goto out_err;
      }
      /* read from worker->proc.out to buf */
      if ((status = bufreader_new(&br, worker->proc.out, worker->ptmp)) == APR_SUCCESS) {
	apr_size_t len;

	bufreader_read_eof(br, &line, &len);
	if (line) {
	  worker_log_buf(worker, LOG_INFO, line, "<", len);
	  worker_match(worker, worker->match_body, line, len);
	  worker_expect(worker, worker->expect_dot, line, len);
	}
      }
      if (status == APR_EOF) {
	status = APR_SUCCESS;
      }
      apr_thread_join(&tmp_status, thread);
      apr_proc_wait(&worker->proc, &exitcode, &exitwhy, APR_WAIT);
      if (exitcode != 0) {
	status = APR_EGENERAL;
	goto out_err;
      }
      /* tag it "done" */
      buf = NULL;
    }
    if (buf) {
      worker_log_buf(worker, LOG_INFO, buf, "<", len);
      worker_match(worker, worker->match_body, buf, len);
      worker_expect(worker, worker->expect_dot, buf, len);
    }
  }
out_err:
  return status;
}

/**
 * Wait for data (same as command_recv)
 *
 * @param self IN command object
 * @param worker IN thread data object
 * @param data IN unused 
 *
 * @return an apr status
 */
apr_status_t command_WAIT(command_t * self, worker_t * worker,
                                 char *data) {
  char *copy;
  int matches;
  int expects;
  char *line;
  char *buf;
  apr_status_t status;
  sockreader_t *sockreader;
  apr_pool_t *pool;
  char *last;
  char *key;
  const char *val = "";
  apr_size_t len;
  apr_ssize_t recv_len = -1;
  apr_size_t peeklen;

  buf = NULL;
  len = 0;
  matches = 0;
  expects = 0;

  COMMAND_OPTIONAL_ARG;

  if ((status = worker_flush(worker)) != APR_SUCCESS) {
    return status;
  }

  if (apr_isdigit(copy[0])) {
    recv_len = apr_atoi64(copy);
  }
  else {
    recv_len = -1;
  }

  apr_pool_create(&pool, NULL);

  if (worker->sockreader == NULL) {
    peeklen = worker->socket->peeklen;
    worker->socket->peeklen = 0;
    if ((status = sockreader_new(&sockreader, worker->socket->socket,
#ifdef USE_SSL
				 worker->socket->is_ssl ? worker->socket->ssl : NULL,
#endif
				 worker->socket->peek, peeklen, pool)) != APR_SUCCESS) {
      goto out_err;
    }
  }
  else {
    sockreader = worker->sockreader;
  }

  if (worker->headers) {
    apr_table_clear(worker->headers);
  }
  else {
    worker->headers = apr_table_make(worker->pool, 5);
  }
  
  if (worker->headers_add) {
    int i;
    apr_table_entry_t *e;

    e = (apr_table_entry_t *) apr_table_elts(worker->headers_add)->elts;
    for (i = 0; i < apr_table_elts(worker->headers_add)->nelts; ++i) {
      apr_table_add(worker->headers, e[i].key, e[i].val);
    }
    apr_table_clear(worker->headers_add);
  }

  /** Status line */
  if ((status = sockreader_read_line(sockreader, &line)) == APR_SUCCESS && 
      line[0] != 0) {

    worker_log(worker, LOG_INFO, "<%s", line);
    worker_match(worker, worker->match_headers, line, strlen(line));
    worker_expect(worker, worker->expect_dot, line, strlen(line));

    if (!strstr(line, "HTTP/") && !strstr(line, "ICAP/")) {
      apr_table_add(worker->headers, "Connection", "close");
      status = sockreader_push_line(sockreader, line);
      goto http_0_9;
    }
  }
  else {
    if (line[0] == 0) {
      worker_log(worker, LOG_INFO, "<%s", line);
      worker_log_error(worker, "No status line received");
      status = APR_EINVAL;
      goto out_err;
    }
    else {
      worker_log(worker, LOG_INFO, "<%s", line);
      worker_log_error(worker, "Network error");
      goto out_err;
    }
  }
 
  /** get headers */
  while ((status = sockreader_read_line(sockreader, &line)) == APR_SUCCESS && 
         line[0] != 0) {
    worker_log(worker, LOG_INFO, "<%s", line);
    /* before splitting do a match */
    worker_match(worker, worker->match_headers, line, strlen(line));
    worker_expect(worker, worker->expect_dot, line, strlen(line));

    /* headers */
    key = apr_strtok(line, ":", &last);
    val = last;
    while (*val == ' ') ++val;
    if (worker->headers_allow) {
      if (!apr_table_get(worker->headers_allow, key)) {
	worker_log(worker, LOG_ERR, "%s header not allowed", key);
	status = APR_EGENERAL;
	goto out_err;
      }
    }
    if (worker->headers_filter) {
      if (!apr_table_get(worker->headers_filter, key)) {
	apr_table_add(worker->headers, key, val);
      }
    }
    else {
      apr_table_add(worker->headers, key, val);
    }
  }
  if (line[0] == 0) {
    worker_log(worker, LOG_INFO, "<");
  }

http_0_9:
  if (status == APR_SUCCESS) {
    /* if recv len is specified use this */
    if (recv_len > 0) {
      len = recv_len;
      if ((status = worker_check_error(worker, 
	   content_length_reader(sockreader, &buf, &len, val))) 
	  != APR_SUCCESS) {
	goto out_err;
      }
    }
    else if (recv_len == 0) {
      buf = NULL; 
    }
    /* else get transfer type */
    else if ((val = apr_table_get(worker->headers, "Content-Length"))) {
      len = apr_atoi64(val);
      if ((status = worker_check_error(worker, 
	   content_length_reader(sockreader, &buf, &len, val))) 
	  != APR_SUCCESS) {
	goto out_err;
      }
    }
    else if ((val = apr_table_get(worker->headers, "Transfer-Encoding"))) {
      if ((status = worker_check_error(worker, 
	   transfer_enc_reader(sockreader, &buf, &len, val))) != APR_SUCCESS) {
	goto out_err;
      }
    }
    else if ((val = apr_table_get(worker->headers, "Encapsulated"))) {
      if ((status = worker_check_error(worker,
	   encapsulated_reader(sockreader, &buf, &len, val,
	                       apr_table_get(worker->headers, "Preview"))))
	  != APR_SUCCESS) {
	goto out_err;
      }
    }
    else if (worker->flags & FLAGS_CLIENT && 
	     (val = apr_table_get(worker->headers, "Connection"))) {
      if ((status = worker_check_error(worker,
	   eof_reader(sockreader, &buf, &len, val))) != APR_SUCCESS) {
	goto out_err;
      }
    }
    if ((status = worker_handle_buf(worker, pool, buf, len)) != APR_SUCCESS) {
      goto out_err;
    }
  }

out_err:
  status = worker_check_expect(worker, status);

  apr_pool_destroy(pool);
  return status;
}

/****
 * Scriptable commands 
 ****/

/**
 * Get socket from hash or add a new one
 *
 * @param self IN thread data object
 * @param hostname IN host name
 * @param portname IN port as ascii string
 *
 */
void worker_get_socket(worker_t *self, const char *hostname, 
                       const char *portname) {
  socket_t *socket;

  socket = 
    apr_hash_get(self->sockets, apr_pstrcat(self->ptmp, hostname, portname, 
	                                    NULL),
	         APR_HASH_KEY_STRING);

  if (!socket) {
    socket = apr_pcalloc(self->pool, sizeof(*socket));
    socket->socket_state = SOCKET_CLOSED;
    apr_hash_set(self->sockets, apr_pstrcat(self->pool, hostname, portname,
	                                    NULL),
	         APR_HASH_KEY_STRING, socket);
  }

  self->socket = socket;
}

/**
 * Setup a connection to host
 *
 * @param self IN command object
 * @param worker IN thread data object
 * @param data IN aditional data
 *
 * @return an apr status
 */
apr_status_t command_REQ(command_t * self, worker_t * worker,
                                char *data) {
  apr_status_t status;
  apr_sockaddr_t *remote_addr;
  char *portname;
  char *portstr;
  char *hostname;
  char *filename;
  char *tag;
  char *last;
  int port;
  char *copy;
  int is_ssl;

  COMMAND_NEED_ARG("Need hostname and port");

  if ((status = worker_flush(worker)) != APR_SUCCESS) {
    return status;
  }

  if ((status = worker_test_unused(worker)) != APR_SUCCESS) {
    return status;
  }

  hostname = apr_strtok(copy, " ", &last);
  portname = apr_strtok(NULL, " ", &last);
  portstr = apr_pstrdup(worker->pool, portname);

  if (!hostname) {
    worker_log(worker, LOG_ERR, "no host name specified");
    return APR_EGENERAL;
  }
  
  if (!portname) {
    worker_log(worker, LOG_ERR, "no portname name specified");
    return APR_EGENERAL;
  }

#ifdef USE_SSL
  is_ssl = 0;
  if (strncmp(portstr, "SSL:", 4) == 0) {
    is_ssl = 1;
    worker->meth = SSLv23_client_method();
    portstr += 4;
  }
  else if (strncmp(portstr, "SSL2:", 4) == 0) {
    is_ssl = 1;
    worker->meth = SSLv2_client_method();
    portstr += 5;
  }
  else if (strncmp(portstr, "SSL3:", 4) == 0) {
    is_ssl = 1;
    worker->meth = SSLv3_client_method();
    portstr += 5;
  }
  else if (strncmp(portstr, "TLS1:", 4) == 0) {
    is_ssl = 1;
    worker->meth = TLSv1_client_method();
    portstr += 5;
  }
  
  if (!portstr[0]) {
    worker_log(worker, LOG_ERR, "no SSL port specified");
    return APR_EGENERAL;
  }
#endif
  portstr = apr_strtok(portstr, ":", &tag);
  port = apr_atoi64(portstr);

  worker_get_socket(worker, hostname, portname);
  
  worker->socket->is_ssl = is_ssl;

  if (worker->socket->socket_state == SOCKET_CLOSED) {
#ifdef USE_SSL
    if (worker->socket->is_ssl) {
      if (!worker->ssl_ctx && !(worker->ssl_ctx = SSL_CTX_new(worker->meth))) {
        worker_log(worker, LOG_ERR, "Could not initialize SSL Context.");
      }
      SSL_CTX_set_options(worker->ssl_ctx, SSL_OP_ALL);
      SSL_CTX_set_options(worker->ssl_ctx, SSL_OP_SINGLE_DH_USE);
      /* get cert file if any is specified */
      filename = apr_strtok(NULL, " ", &last);
      if (filename && SSL_CTX_use_certificate_file(worker->ssl_ctx, filename, 
				       SSL_FILETYPE_PEM) <= 0) { 
	worker_log(worker, LOG_ERR, "Could not load RSA certifacte \"%s\"", 
	           filename);
	return APR_EINVAL;
      }

      /* get key file if any is specified */
      filename = apr_strtok(NULL, " ", &last);
      if (filename && SSL_CTX_use_PrivateKey_file(worker->ssl_ctx, filename, 
	                              SSL_FILETYPE_PEM) <= 0) {
	worker_log(worker, LOG_ERR, "Could not load RSA private key \"%s\"", 
	           filename);
	return APR_EINVAL;
      }

      /* get ca file if any is specified */
      filename = apr_strtok(NULL, " ", &last);
      if (filename && !SSL_CTX_load_verify_locations(worker->ssl_ctx, filename,
	                                             NULL)) {
	worker_log(worker, LOG_ERR, "Could not load CA file \"%s\"", filename);
	return APR_EINVAL;
      }
#if (OPENSSL_VERSION_NUMBER < 0x00905100L)
      SSL_CTX_set_verify_depth(worker->ssl_ctx,1);
#endif

    }
#endif

    if ((status = apr_socket_create(&worker->socket->socket, APR_INET, 
	                            SOCK_STREAM, APR_PROTO_TCP,
                                    worker->pool)) != APR_SUCCESS) {
      worker->socket->socket = NULL;
      return status;
    }

    if ((status =
         apr_socket_opt_set(worker->socket->socket, APR_TCP_NODELAY,
                            1)) != APR_SUCCESS) {
      return status;
    }

    if ((status =
         apr_socket_timeout_set(worker->socket->socket, worker->socktmo)) != APR_SUCCESS) {
      return status;
    }

#ifdef USE_SSL
    if (worker->socket->is_ssl) {
      BIO *bio;
      apr_os_sock_t fd;

      if ((worker->socket->ssl = SSL_new(worker->ssl_ctx)) == NULL) {
        worker_log(worker, LOG_ERR, "SSL_new failed.");
        status = APR_ECONNREFUSED;
      }
      SSL_set_ssl_method(worker->socket->ssl, worker->meth);
      ssl_rand_seed();
      apr_os_sock_get(&fd, worker->socket->socket);
      bio = BIO_new_socket(fd, BIO_NOCLOSE);
      SSL_set_bio(worker->socket->ssl, bio, bio);
      SSL_set_connect_state(worker->socket->ssl);
    }
#endif

    if ((status =
         apr_sockaddr_info_get(&remote_addr, hostname, AF_UNSPEC, port,
                               APR_IPV4_ADDR_OK, worker->pool))
        != APR_SUCCESS) {
      return status;
    }

    if ((status =
         apr_socket_connect(worker->socket->socket, remote_addr)) 
	!= APR_SUCCESS) {
      return status;
    }

    if ((status =
         apr_socket_opt_set(worker->socket->socket, APR_SO_KEEPALIVE,
                            1)) != APR_SUCCESS) {
      return status;
    }

    worker->socket->socket_state = SOCKET_CONNECTED;
#ifdef USE_SSL
    if (worker->socket->is_ssl) {
      if ((status = worker_ssl_handshake(worker)) != APR_SUCCESS) {
	return status;
      }
    }
#endif
  }

  /* reset the matcher tables */
  apr_table_clear(worker->match_headers);
  apr_table_clear(worker->match_body);
  apr_table_clear(worker->expect_dot);
  apr_table_clear(worker->expect_error);
  apr_table_clear(worker->match_error);

  return APR_SUCCESS;
}

/**
 * Setup a connection to host
 *
 * @param self IN command object
 * @param worker IN thread data object
 * @param data IN unused 
 *
 * @return an apr status
 */
apr_status_t command_RES(command_t * self, worker_t * worker,
                                char *data) {
  apr_status_t status;

  COMMAND_NO_ARG;

  if ((status = worker_flush(worker)) != APR_SUCCESS) {
    return status;
  }

  if ((status = worker_test_unused(worker)) != APR_SUCCESS) {
    return status;
  }

  worker_get_socket(worker, "Default", "0");
  worker->socket->is_ssl = worker->is_ssl;

  if (worker->socket->socket_state == SOCKET_CLOSED) {
    worker_log(worker, LOG_DEBUG, "--- accept");
    if (!worker->listener) {
      worker_log_error(worker, "Server down");
      return APR_EGENERAL;
    }

    if ((status =
         apr_socket_accept(&worker->socket->socket, worker->listener,
                           worker->pool)) != APR_SUCCESS) {
      worker->socket->socket = NULL;
      return status;
    }
    if ((status =
           apr_socket_timeout_set(worker->socket->socket, worker->socktmo)) 
	!= APR_SUCCESS) {
      return status;
    }
#ifdef USE_SSL
    if ((status = worker_ssl_accept(worker)) != APR_SUCCESS) {
      return status;
    }
#endif
    worker->socket->socket_state = SOCKET_CONNECTED;
  }

  apr_table_clear(worker->match_headers);
  apr_table_clear(worker->match_body);
  apr_table_clear(worker->expect_dot);
  apr_table_clear(worker->expect_error);
  apr_table_clear(worker->match_error);

  return APR_SUCCESS;
}

/**
 * Sleep for a given time (ms)
 *
 * @param self IN command object
 * @param worker IN thread data object
 * @param data IN time to wait in ms
 *
 * @return an apr status
 */
apr_status_t command_SLEEP(command_t * self, worker_t * worker,
                                  char *data) {
  apr_status_t status;
  char *copy;

  if ((status = worker_flush(worker)) != APR_SUCCESS) {
    return status;
  }

  COMMAND_NEED_ARG("Time not specified");
 
  apr_sleep(apr_atoi64(copy) * 1000);
  return APR_SUCCESS;
}

/**
 * Define an expect
 *
 * @param self IN command object
 * @param worker IN thread data object
 * @param data IN "%s %s" type match 
 *
 * @return an apr status
 */
apr_status_t command_EXPECT(command_t * self, worker_t * worker,
                                   char *data) {
  char *last;
  char *type;
  char *match;
  regex_t *compiled;
  const char *err;
  int off;
  char *copy;
  char *interm;

  COMMAND_NEED_ARG("Type and regex not specified");

  type = apr_strtok(copy, " ", &last);
  
  match = my_unescape(last, &last);

  if (!type) {
    worker_log(worker, LOG_ERR, "Type not specified");
    return APR_EGENERAL;
  }
  
  if (!match) {
    worker_log(worker, LOG_ERR, "Regex not specified");
    return APR_EGENERAL;
  }

  interm = apr_pstrdup(worker->ptmp, match);

  if (interm[0] == '!') {
    ++interm;
  }

  if (!(compiled = pregcomp(worker->ptmp, interm, &err, &off))) {
    worker_log(worker, LOG_ERR, "EXPECT regcomp failed: \"%s\"", last);
    return APR_EINVAL;
  }

  if (strcmp(type, ".") == 0) {
    apr_table_addn(worker->expect_dot, match, (char *) compiled);
  }
  else if (strcasecmp(type, "Exec") == 0) {
    apr_table_addn(worker->expect_exec, match, (char *) compiled);
  }
  else if (strcasecmp(type, "Error") == 0) {
    apr_table_addn(worker->expect_error, match, (char *) compiled);
  }
  else if (strncasecmp(type, "Var(", 4) == 0) {
    const char *val;
    char *var;
    
    var= apr_strtok(type, "(", &last);
    var = apr_strtok(NULL, ")", &last);
    val = apr_table_get(worker->vars, var);
    if (val) {
      if (!worker->tmp_table) {
	worker->tmp_table = apr_table_make(worker->pool, 1);
      }
      apr_table_clear(worker->tmp_table);
      apr_table_addn(worker->tmp_table, match, (char *) compiled);
      worker_expect(worker, worker->tmp_table, val, strlen(val));
      return worker_validate_expect(worker, worker->tmp_table, "EXPECT var", 
	                            APR_SUCCESS);
    }
    else {
      worker_log(worker, LOG_ERR, "Variable \"%s\" do not exist", var);
      return APR_EINVAL;
    }
  }
  else {
    worker_log(worker, LOG_ERR, "EXPECT type \"%s\" unknown", type);
    return APR_EINVAL;
  }

  return APR_SUCCESS;
}

/**
 * Close socket
 *
 * @param self IN command object
 * @param worker IN thread data object
 * @param data IN unused
 *
 * @return an apr status
 */
apr_status_t command_CLOSE(command_t * self, worker_t * worker,
                                  char *data) {
  apr_status_t status;

  COMMAND_NO_ARG;

  if ((status = worker_flush(worker)) != APR_SUCCESS) {
    worker_conn_close(worker);
    return status;
  }

  if ((status = worker_test_unused(worker)) != APR_SUCCESS) {
    worker_conn_close(worker);
    return status;
  }

  if ((status = worker_conn_close(worker)) != APR_SUCCESS) {
    return status;
  }

  return APR_SUCCESS;
}

/**
 * Specify a timeout for socket operations (ms) 
 *
 * @param self IN command object
 * @param worker IN thread data object
 * @param data IN time in ms 
 *
 * @return an apr status
 */
apr_status_t command_TIMEOUT(command_t * self, worker_t * worker,
                                    char *data) {
  apr_time_t tmo;
  char *copy;

  COMMAND_NEED_ARG("Time not specified");

  tmo = apr_atoi64(copy);
  worker->socktmo = tmo * 1000;

  return APR_SUCCESS;
}

/**
 * Define an expect
 *
 * @param self IN command object
 * @param worker IN thread data object
 * @param data IN "%s %s %s" type match variable
 *
 * @return an apr status
 */
apr_status_t command_MATCH(command_t * self, worker_t * worker,
                                  char *data) {
  char *last;
  char *type;
  char *match;
  char *vars;
  regex_t *compiled;
  const char *err;
  int off;
  char *copy;

  COMMAND_NEED_ARG("Type, regex and variable not specified");

  type = apr_strtok(copy, " ", &last);
  
  match = my_unescape(last, &last);
  
  vars = apr_strtok(NULL, "", &last);

  if (!type) {
    worker_log(worker, LOG_ERR, "Type not specified");
    return APR_EGENERAL;
  }

  if (!match) {
    worker_log(worker, LOG_ERR, "Regex not specified");
    return APR_EGENERAL;
  }

  if (!vars) {
    worker_log(worker, LOG_ERR, "Variable not specified");
    return APR_EGENERAL;
  }

  if (vars) {
    ++vars;
  }

  if (!vars) {
    return APR_EINVAL;
  }

  if (!(compiled = pregcomp(worker->ptmp, match, &err, &off))) {
    worker_log(worker, LOG_ERR, "MATCH regcomp failed: %s", last);
    return APR_EINVAL;
  }
  if (strcasecmp(type, "Headers") == 0) {
    apr_table_addn(worker->match_headers, vars, (char *) compiled);
  }
  else if (strcasecmp(type, "Body") == 0) {
    apr_table_addn(worker->match_body, vars, (char *) compiled);
  }
  else if (strcasecmp(type, "Error") == 0) {
    apr_table_addn(worker->match_error, vars, (char *) compiled);
  }
  else if (strcasecmp(type, "Exec") == 0) {
    apr_table_addn(worker->match_exec, vars, (char *) compiled);
  }
  else if (strncasecmp(type, "Var(", 4) == 0) {
    const char *val;
    char *var;
    
    var= apr_strtok(type, "(", &last);
    var = apr_strtok(NULL, ")", &last);
    val = apr_table_get(worker->vars, var);
    if (val) {
      if (!worker->tmp_table) {
	worker->tmp_table = apr_table_make(worker->pool, 1);
      }
      apr_table_clear(worker->tmp_table);
      apr_table_addn(worker->tmp_table, vars, (char *) compiled);
      worker_match(worker, worker->tmp_table, val, strlen(val));
      return worker_validate_match(worker, worker->tmp_table, "MATCH var", 
	                           APR_SUCCESS);
    }
    else {
      /* this should cause an error? */
    }
  }
  else {
    worker_log(worker, LOG_ERR, "Match type %s do not exist", type);
    return APR_ENOENT;
  }

  return APR_SUCCESS;
}

/**
 * set command
 *
 * @param self IN command object
 * @param worker IN thread data object
 * @param data IN key=value 
 *
 * @return an apr status
 */
apr_status_t command_SET(command_t * self, worker_t * worker,
                                char *data) {
  char *vars_last;
  const char *vars_key;
  const char *vars_val;
  char *copy;
  int i;

  COMMAND_NEED_ARG("Variable and value not specified");
  
  vars_key = apr_strtok(copy, "=", &vars_last);
  for (i = 0; vars_key[i] != 0 && strchr(VAR_ALLOWED_CHARS, vars_key[i]); i++); 
  if (vars_key[i] != 0) {
    worker_log(worker, LOG_ERR, "Char '%c' is not allowed in \"%s\"", vars_key[i], vars_key);
    return APR_EINVAL;
  }

  vars_val = apr_strtok(NULL, "", &vars_last);

  if (!vars_key) {
    worker_log(worker, LOG_ERR, "Key not specified");
    return APR_EGENERAL;
  }

  if (!vars_val) {
    worker_log(worker, LOG_ERR, "Value not specified");
    return APR_EGENERAL;
  }
  
  apr_table_set(worker->vars, vars_key, vars_val);

  return APR_SUCCESS;
}

/**
 * Send data 
 *
 * @param self IN command object
 * @param worker IN thread data object
 * @param data IN data to send
 *
 * @return an apr status
 */
apr_status_t command_DATA(command_t * self, worker_t * worker,
                                 char *data) {
  char *copy;

  if (!worker->socket || !worker->socket->socket) {
    return APR_ENOSOCKET;
  }
    
  copy = apr_pstrdup(worker->ptmp, data); 
  copy = worker_replace_vars(worker, copy);
  worker_log(worker, LOG_INFO, "%s%s", self->name, copy); 

  if (strncasecmp(copy, "Content-Length: AUTO", 20) == 0) {
    apr_table_add(worker->cache, "Content-Length", "Content-Length");
  }
  else {
    apr_table_addn(worker->cache, "TRUE", copy);
  }

  return APR_SUCCESS;
}

/**
 * Flush data 
 *
 * @param self IN command object
 * @param worker IN thread data object
 * @param data IN unused
 *
 * @return an apr status
 */
apr_status_t command_FLUSH(command_t * self, worker_t * worker,
                                  char *data) {
  apr_status_t status;

  COMMAND_NO_ARG;

  if ((status = worker_flush(worker)) != APR_SUCCESS) {
    return status;
  }

  return APR_SUCCESS;
}

/**
 * Chunk info 
 *
 * @param self IN command object
 * @param worker IN thread data object
 * @param data IN unused
 *
 * @return an apr status
 */
apr_status_t command_CHUNK(command_t * self, worker_t * worker,
                                  char *data) {
  apr_status_t status;

  COMMAND_NO_ARG;

  apr_table_add(worker->cache, "CHUNKED", "CHUNKED");

  if ((status = worker_flush(worker)) != APR_SUCCESS) {
    return status;
  }

  return APR_SUCCESS;
}

/**
 * read from file descriptor and write it to the HTTP stream
 * 
 * @param worker IN thread data object
 * @param file IN open file descriptor for reading
 * @param flags IN FLAGS_CHUNKED or FLAGS_NONE
 *
 * @return APR_SUCCESS or an apr error status
 */
apr_status_t worker_file_to_http(worker_t *self, apr_file_t *file, int flags) {
  apr_status_t status;
  apr_size_t len;
  char *buf;

  while (1) {
    if (flags & FLAGS_CHUNKED) {
      len = self->chunksize;
    }
    else {
      len = BLOCK_MAX;
    }
    buf = apr_pcalloc(self->ptmp, len + 1);
    if ((status = apr_file_read(file, buf, &len)) != APR_SUCCESS) {
      break;
    }
    apr_table_addn(self->cache, 
		   apr_psprintf(self->ptmp, "NOCRLF:%d", len), buf);
    if (flags & FLAGS_CHUNKED) {
      worker_log(self, LOG_DEBUG, "--- chunk size: %d", len);
      apr_table_add(self->cache, "CHUNKED", "CHUNKED");
      if ((status = worker_flush(self)) != APR_SUCCESS) {
	return status;
      }
    }
  }

  if (APR_STATUS_IS_EOF(status)) {
    return APR_SUCCESS;
  }
  else {
    return status;
  }
}

/**
 * Execute an external program 
 *
 * @param self IN command object
 * @param worker IN thread data object
 * @param data IN external program call with arguments 
 *
 * @return an apr status
 */
apr_status_t command_EXEC(command_t * self, worker_t * worker,
                                 char *data) {
  char *copy;
  apr_status_t status;
  apr_procattr_t *attr;
  apr_table_t *table;
  apr_table_entry_t *e;
  bufreader_t *br;
  const char *progname;
  const char **args;
  apr_exit_why_e exitwhy;
  int exitcode;
  char *last;
  char *val;
  int i;
  int flags;

  COMMAND_NEED_ARG("Need a shell command");

  flags = worker->flags;
  worker->flags &= ~FLAGS_PIPE;
  worker->flags &= ~FLAGS_CHUNKED;
  worker->flags &= ~FLAGS_PIPE_IN;
  worker->flags &= ~FLAGS_FILTER;
  /* pipe http to shell */
  if (copy[0] == '|') {
    ++copy;
    worker->flags |= FLAGS_PIPE_IN;
  }
  /* filter http */
  else if (copy[0] == '<') {
    ++copy;
    worker->flags |= FLAGS_FILTER;
  }
  
  table = apr_table_make(worker->ptmp, 5);
  progname = apr_strtok(copy, " ", &last);

  if (!progname) {
    worker_log(worker, LOG_ERR, "No program name specified");
    return APR_EGENERAL;
  }
  
  apr_table_addn(table, progname, "TRUE");

  while ((val = apr_strtok(NULL, " ", &last))) {
    apr_table_addn(table, val, "TRUE");
  }

  args = apr_pcalloc(worker->ptmp,
                     (apr_table_elts(table)->nelts + 1) * sizeof(const char *));

  e = (apr_table_entry_t *) apr_table_elts(table)->elts;
  for (i = 0; i < apr_table_elts(table)->nelts; i++) {
    args[i] = e[i].key;
  }
  args[i] = NULL;

  if ((status = apr_procattr_create(&attr, worker->ptmp)) != APR_SUCCESS) {
    return status;
  }

  if ((status = apr_procattr_cmdtype_set(attr, APR_SHELLCMD_ENV)) != APR_SUCCESS) {
    return status;
  }

  if ((status = apr_procattr_detach_set(attr, 0)) != APR_SUCCESS) {
    return status;
  }

  if (worker->flags & FLAGS_FILTER) {
    if ((status = apr_procattr_io_set(attr, APR_FULL_BLOCK, APR_FULL_BLOCK,
				      APR_FULL_BLOCK))
	!= APR_SUCCESS) {
      return status;
    }
  }
  else if (worker->flags & FLAGS_PIPE_IN) {
    if ((status = apr_procattr_io_set(attr, APR_FULL_BLOCK, APR_FULL_BLOCK,
				      APR_NO_PIPE))
	!= APR_SUCCESS) {
      return status;
    }
  }
  else {
    if ((status = apr_procattr_io_set(attr,  APR_NO_PIPE, APR_FULL_BLOCK,
				      APR_NO_PIPE))
	!= APR_SUCCESS) {
      return status;
    }
  }

  if ((status = apr_proc_create(&worker->proc, progname, args, NULL, attr,
                                worker->ptmp)) != APR_SUCCESS) {
    return status;
  }

  if (flags & FLAGS_PIPE) {
    worker_log(worker, LOG_DEBUG, "write stdout to http: %s", progname);
    if ((status = worker_file_to_http(worker, worker->proc.out, flags)) != APR_SUCCESS) {
      return status;
    }
  }
  else if (!(worker->flags & FLAGS_PIPE_IN) && !(worker->flags & FLAGS_FILTER)) {
    apr_size_t len = 0;
    char *buf = NULL;

    worker_log(worker, LOG_DEBUG, "read stdin: %s", progname);
    if ((status = bufreader_new(&br, worker->proc.out, worker->ptmp)) == APR_SUCCESS) {

      bufreader_read_eof(br, &buf, &len);
      if (buf) {
	worker_log(worker, LOG_INFO, "<%s", buf);
	worker_match(worker, worker->match_exec, buf, len);
	worker_expect(worker, worker->expect_exec, buf, len);
      }
    }

    if (buf && status == APR_EOF) {
	worker_log(worker, LOG_INFO, "<%s", buf);
	worker_match(worker, worker->match_exec, buf, len);
	worker_expect(worker, worker->expect_exec, buf, len);
    }
    
    status = APR_SUCCESS;
    status = worker_validate_match(worker, worker->match_exec, "MATCH exec", 
	                           status);
    status = worker_validate_expect(worker, worker->expect_exec, "EXPECT exec", 
	                            status);
  }

  if (!(worker->flags & FLAGS_PIPE_IN) && !(worker->flags & FLAGS_FILTER)) {
    worker_log(worker, LOG_DEBUG, "wait for: %s", progname);
    apr_proc_wait(&worker->proc, &exitcode, &exitwhy, APR_WAIT);

    apr_file_close(worker->proc.out);

    if (exitcode != 0) {
      status = APR_EGENERAL;
    }
  }

  return status;
}

/**
 * Send file
 *
 * @param self IN command object
 * @param worker IN thread data object
 * @param data IN file
 *
 * @return an apr status
 */
apr_status_t command_SENDFILE(command_t * self, worker_t * worker,
                                     char *data) {
  char *copy;
  char *last;
  char *filename;
  apr_status_t status;
  int flags;
  apr_file_t *fp;

  COMMAND_NEED_ARG("Need a file name");

  filename = apr_strtok(copy, " ", &last);

  flags = worker->flags;
  worker->flags &= ~FLAGS_PIPE;
  worker->flags &= ~FLAGS_CHUNKED;
  
  if ((status =
       apr_file_open(&fp, filename, APR_READ, APR_OS_DEFAULT,
		     worker->ptmp)) != APR_SUCCESS) {
    fprintf(stderr, "\nCan not send file: File \"%s\" not found", copy);
    return APR_ENOENT;
  }
  
  if (flags & FLAGS_PIPE) {
    if ((status = worker_file_to_http(worker, fp, flags)) != APR_SUCCESS) {
      return status;
    }
  }

  return APR_SUCCESS;
}

/**
 * Declare a pipe
 *
 * @param self IN command object
 * @param worker IN thread data object
 * @param data IN not used
 *
 * @return an apr status
 */
apr_status_t command_PIPE(command_t * self, worker_t * worker,
                                 char *data) {
  char *copy;
  char *last;
  char *add;
  char *val;

  COMMAND_OPTIONAL_ARG;

  add = apr_strtok(copy, " ", &last);
  if (add) {
    val = apr_strtok(NULL, " ", &last);
  }
  else {
    val = NULL;
  }
  
  worker_log(worker, LOG_DEBUG, "additional: %s, value: %s", add, val);
  
  if (add && strncasecmp(add, "chunked", 7) == 0) {
    worker->chunksize = val ? apr_atoi64(val) : BLOCK_MAX;
    worker->flags |= FLAGS_CHUNKED;
  }
  
  worker->flags |= FLAGS_PIPE;

  return APR_SUCCESS;
}

/**
 * Send data without a CRLF
 *
 * @param self IN command object
 * @param worker IN thread data object
 * @param data IN data to send
 *
 * @return an apr status
 */
apr_status_t command_NOCRLF(command_t * self, worker_t * worker,
                                   char *data) {
  char *copy;

  copy = apr_pstrdup(worker->ptmp, data); 
  copy = worker_replace_vars(worker, copy);
  worker_log(worker, LOG_INFO, "%s%s", self->name, copy); 

  apr_table_addn(worker->cache, "NOCRLF", copy);

  return APR_SUCCESS;
}

/**
 * Send data without a CRLF
 *
 * @param self IN command object
 * @param worker IN thread data object
 * @param data IN data to send
 *
 * @return an apr status
 */
apr_status_t command_SOCKSTATE(command_t * self, worker_t * worker,
                                      char *data) {
  char *copy;

  COMMAND_NEED_ARG("Need a variable name");

  if (worker_sockstate(worker) == APR_SUCCESS) {
    apr_table_set(worker->vars, copy, "CONNECTED");
  }
  else {
    apr_table_set(worker->vars, copy, "CLOSED");
  }

  return APR_SUCCESS;
}

/**
 * Ignores errors specified.
 *
 * @param self IN command object
 * @param worker IN thread data object
 * @param data IN regex 
 *
 * @return an apr status
 */
apr_status_t command_IGNORE_ERR(command_t * self, worker_t * worker,
                                       char *data) {
  char *copy;

  COMMAND_NEED_ARG("Need a regex");

  worker_log(worker, LOG_WARN, "Do not use this command it is marked DEPRECIATED");
  
  /* TODO: only .* is implemented */
  worker->flags |= FLAGS_IGNORE_ERRORS;
  
  return APR_SUCCESS;
}

/**
 * HEADER command
 *
 * @param self IN command
 * @param worker IN thread data object
 * @param data IN header name (spaces are possible) 
 *
 * @return APR_SUCCESS
 */
apr_status_t command_HEADER(command_t *self, worker_t *worker, char *data) {
  char *copy;
  char *method;
  char *header;
  char *last;

  COMMAND_NEED_ARG("Need method ALLOW or FILTER and a header name");

  method = apr_strtok(copy, " ", &last);
  header = apr_strtok(NULL, " ", &last);
  
  if (strcasecmp(method, "ALLOW") == 0) {
    if (!worker->headers_allow) {
      worker->headers_allow = apr_table_make(worker->pool, 10);
    }
    apr_table_add(worker->headers_allow, header, method);
  }
  else if (strcasecmp(method, "FILTER") == 0) {
    if (!worker->headers_filter) {
      worker->headers_filter = apr_table_make(worker->pool, 5);
    }
    apr_table_add(worker->headers_filter, header, method);
  }
  else {
    return APR_ENOTIMPL;
  }

  return APR_SUCCESS;
}

/**
 * RAND command
 *
 * @param self IN command
 * @param worker IN thread data object
 * @param data IN header name (spaces are possible) 
 *
 * @return APR_SUCCESS
 */
apr_status_t command_RAND(command_t *self, worker_t *worker, char *data) {
  char *copy;
  char *val;
  char *last;
  int start;
  int end;
  int result;

  COMMAND_NEED_ARG("Need a start and end number and a variable"); 
  
  val = apr_strtok(copy, " ", &last);
  start = apr_atoi64(val);
  val = apr_strtok(NULL, " ", &last);
  end = apr_atoi64(val);
  val = apr_strtok(NULL, " ", &last);

  if (val == NULL) {
    worker_log(worker, LOG_ERR, "No variable name specified");
    return APR_EINVAL;
  }
  
  result = start + (rand() % (end - start)); 

  apr_table_set(worker->vars, val, apr_itoa(worker->ptmp, result));

  return APR_SUCCESS;
}

/**
 * DEBUG command
 *
 * @param self IN command
 * @param worker IN thread data object
 * @param data IN string to print on stderr
 *
 * @return APR_SUCCESS
 */
apr_status_t command_DEBUG(command_t *self, worker_t *worker, char *data) {
  char *copy;
  
  COMMAND_OPTIONAL_ARG;

  worker_log(worker, LOG_ERR, "%s", copy);

  return APR_SUCCESS;
}

/**
 * Setup listener
 *
 * @param worker IN thread data object
 *
 * @return APR_SUCCESS
 */
apr_status_t worker_listener_up(worker_t *worker) {
  apr_sockaddr_t *local_addr;

  apr_status_t status = APR_SUCCESS;

  worker_get_socket(worker, "Default", "0");
  
  if (worker->listener) {
    worker_log_error(worker, "Server allready up");
    return APR_EGENERAL;
  }
  
  if ((status = apr_sockaddr_info_get(&local_addr, worker->listener_addr, APR_UNSPEC,
                                      worker->listener_port, APR_IPV4_ADDR_OK, worker->pool))
      != APR_SUCCESS) {
    goto error;
  }

  if ((status = apr_socket_create(&worker->listener, APR_INET, SOCK_STREAM,
                                  APR_PROTO_TCP, worker->pool)) != APR_SUCCESS)
  {
    worker->listener = NULL;
    goto error;
  }

  status = apr_socket_opt_set(worker->listener, APR_SO_REUSEADDR, 1);
  if (status != APR_SUCCESS && status != APR_ENOTIMPL) {
    goto error;
  }
  
  worker_log(worker, LOG_DEBUG, "--- bind");
  if ((status = apr_socket_bind(worker->listener, local_addr)) != APR_SUCCESS) {
    goto error;
  }

  worker_log(worker, LOG_DEBUG, "--- listen");
  if ((status = apr_socket_listen(worker->listener, LISTENBACKLOG_DEFAULT)) != APR_SUCCESS) {
    goto error;
  }

  worker->socket->socket_state = SOCKET_CLOSED;

error:
  return status;
}

/**
 * UP command bind a listener socket
 *
 * @param self IN command
 * @param worker IN thread data object
 * @param data IN unused
 *
 * @return APR_SUCCESS
 */
apr_status_t command_UP(command_t *self, worker_t *worker, char *data) {
  COMMAND_NO_ARG;

  return worker_listener_up(worker);
}

/**
 * DOWN command shuts down listener
 *
 * @param self IN command
 * @param worker IN thread data object
 * @param data IN unused
 *
 * @return APR_SUCCESS
 */
apr_status_t command_DOWN(command_t *self, worker_t *worker, char *data) {
  apr_status_t status;

  COMMAND_NO_ARG;

  if (!worker->listener) {
    worker_log_error(worker, "Server allready down", self->name);
    return APR_EGENERAL;
  }
  
  if ((status = apr_socket_close(worker->listener)) != APR_SUCCESS) {
    return status;
  }
  worker->listener = NULL;
  return status;
}

/**
 * TIME command stores time in a variable [ms]
 *
 * @param self IN command
 * @param worker IN thread data object
 * @param data IN variable name 
 *
 * @return APR_SUCCESS
 */
apr_status_t command_TIME(command_t *self, worker_t *worker, char *data) {
  char *copy;

  COMMAND_NEED_ARG("Need a variable name to store time");
  
  apr_table_set(worker->vars, copy, apr_off_t_toa(worker->ptmp, apr_time_as_msec(apr_time_now())));

  return APR_SUCCESS;
}

/**
 * LOG_LEVEL command sets log level 
 *
 * @param self IN command
 * @param worker IN thread data object
 * @param data IN number 0-4 
 *
 * @return APR_SUCCESS
 */
apr_status_t command_LOG_LEVEL(command_t *self, worker_t *worker, char *data) {
  char *copy;

  COMMAND_NEED_ARG("Need a number between 0 and 4");

  worker->log_mode = apr_atoi64(copy);

  return APR_SUCCESS;
}

/**
 * SYNC command
 *
 * @param self IN command
 * @param worker IN thread data object
 * @param data IN unused
 *
 * @return APR_SUCCESS
 */
apr_status_t command_SYNC(command_t *self, worker_t *worker, char *data) {
  apr_time_t sec;
  apr_time_t nxt_sec;
  apr_time_t now;

  /* get current time */
  now = apr_time_now();
  /* get next second */
  sec = apr_time_sec(now) + 1;
  /* next second in us */
  nxt_sec = apr_time_from_sec(sec);
  /* sleep until next sec */
  apr_sleep(nxt_sec - now);
  
  return APR_SUCCESS;
}

/**
 * RECV command
 *
 * @param self IN command
 * @param worker IN thread data object
 * @param data IN either POLL or number of bytes
 *
 * @return APR_SUCCESS
 */
apr_status_t command_RECV(command_t *self, worker_t *worker, char *data) {
  char *copy;
  apr_pool_t *pool;
  apr_status_t status;
  apr_size_t recv_len;
  apr_size_t peeklen;
  sockreader_t *sockreader;
  char *buf;
  char *last;
  char *val;

  int poll = 0;

  COMMAND_NEED_ARG("Need a number or POLL");

  /* get first value, can be either POLL or a number */
  val = apr_strtok(copy, " ", &last);
  if (strcasecmp(val, "POLL") == 0) {
    poll = 1;
    /* recv_len to max and timeout to min */
    recv_len = BLOCK_MAX;
    /* set timout to specified socket tmo */
    if ((status =
           apr_socket_timeout_set(worker->socket->socket, worker->socktmo)) 
	!= APR_SUCCESS) {
      return status;
    }
  }
  else if (strcasecmp(val, "CHUNKED") == 0) {
    recv_len = 0;
  }
  else if (strcasecmp(val, "CLOSE") == 0) {
    recv_len = 0;
  }
  else {
    /* must be a number */
    recv_len = apr_atoi64(val);
  }

  apr_pool_create(&pool, NULL);

  if (worker->sockreader == NULL) {
    peeklen = worker->socket->peeklen;
    worker->socket->peeklen = 0;
    if ((status = sockreader_new(&sockreader, worker->socket->socket,
#ifdef USE_SSL
				 worker->socket->is_ssl ? worker->socket->ssl : NULL,
#endif
				 worker->socket->peek, peeklen, pool)) != APR_SUCCESS) {
      goto out_err;
    }
  }
  else {
    sockreader = worker->sockreader;
  }

  if (strcasecmp(val, "CHUNKED") == 0) {
    if ((status = transfer_enc_reader(sockreader, &buf, &recv_len, "chunked")) != APR_SUCCESS) {
      goto out_err;
    }
  }
  else if (strcasecmp(val, "CLOSE") == 0) {
    if ((status = eof_reader(sockreader, &buf, &recv_len, "close")) != APR_SUCCESS) {
      goto out_err;
    }
  }
  else {
    if ((status = content_length_reader(sockreader, &buf, &recv_len, "")) != APR_SUCCESS) {
      if (poll && APR_STATUS_IS_INCOMPLETE(status)) {
	status = APR_SUCCESS;
      }
      else {
	goto out_err;
      }
    }
  }

  if ((status = worker_handle_buf(worker, pool, buf, recv_len)) 
      != APR_SUCCESS) {
    goto out_err;
  }

out_err:
  status = worker_check_expect(worker, status);
  apr_pool_destroy(pool);

  return status;
}

/**
 * READLINE command
 *
 * @param self IN command
 * @param worker IN thread data object
 * @param data IN unused
 *
 * @return APR_SUCCESS
 */
apr_status_t command_READLINE(command_t *self, worker_t *worker, char *data) {
  apr_pool_t *pool;
  apr_status_t status;
  apr_size_t peeklen;
  apr_size_t len;
  sockreader_t *sockreader;
  char *buf;

  COMMAND_NO_ARG;

  apr_pool_create(&pool, NULL);

  if (worker->sockreader == NULL) {
    peeklen = worker->socket->peeklen;
    worker->socket->peeklen = 0;
    if ((status = sockreader_new(&sockreader, worker->socket->socket,
#ifdef USE_SSL
				 worker->socket->is_ssl ? worker->socket->ssl : NULL,
#endif
				 worker->socket->peek, peeklen, pool)) != APR_SUCCESS) {
      goto out_err;
    }
  }
  else {
    sockreader = worker->sockreader;
  }

  if ((status = sockreader_read_line(sockreader, &buf)) != APR_SUCCESS) {
    goto out_err;
  }

  if (buf) {
    len = strlen(buf);
    if ((status = worker_handle_buf(worker, pool, buf, len)) 
	!= APR_SUCCESS) {
      goto out_err;
    }
  }

out_err:
  status = worker_check_expect(worker, status);
  apr_pool_destroy(pool);

  return status;
}

/**
 * OP command
 *
 * @param self IN command
 * @param worker IN thread data object
 * @param data IN left op right var
 *
 * @return APR_SUCCESS or apr error code
 */
apr_status_t command_OP(command_t *self, worker_t *worker, char *data) {
  char *copy;
  char *last;
  char *left;
  char *op;
  char *right;
  char *var;
  apr_int64_t ileft;
  apr_int64_t iright;
  apr_int64_t result;

  COMMAND_NEED_ARG("<left> ADD|SUB|MUL|DIV <right> <variable> expected");

  /* split into left, op, right, var */
  left = apr_strtok(copy, " ", &last);
  op = apr_strtok(NULL, " ", &last);
  right = apr_strtok(NULL, " ", &last);
  var = apr_strtok(NULL, " ", &last);

  /* do checks */
  if (!left || !op || !right || !var) {
    worker_log(worker, LOG_ERR, "<left> ADD|SUB|MUL|DIV <right> <variable> expected", copy);
    return APR_EINVAL;
  }

  /* get integer value */
  ileft = apr_atoi64(left);
  iright = apr_atoi64(right);

  /* do operation */
  if (strcasecmp(op, "ADD") == 0) {
    result = ileft + iright;
  }
  else if (strcasecmp(op, "SUB") == 0) {
    result = ileft - iright;
  }
  else if (strcasecmp(op, "MUL") == 0) {
    result = ileft * iright;
  }
  else if (strcasecmp(op, "DIV") == 0) {
    if (iright == 0) {
      worker_log(worker, LOG_ERR, "Division by zero");
      return APR_EINVAL;
    }
    result = ileft / iright;
  }
  else {
    worker_log(worker, LOG_ERR, "Unknown operant %s", op);
    return APR_ENOTIMPL;
  }

  /* store it do var */
  apr_table_set(worker->vars, var, apr_off_t_toa(worker->ptmp, result));
  
  return APR_SUCCESS;
}

/**
 * WHICH command
 *
 * @param self IN command
 * @param worker IN thread data object
 * @param data IN varname
 *
 * @return APR_SUCCESS or apr error code
 */
apr_status_t command_WHICH(command_t *self, worker_t *worker, char *data) {
  char *copy;
  char *result;

  COMMAND_NEED_ARG("<variable> expected");
 
  result  = apr_psprintf(worker->ptmp, "%d", worker->which);
  apr_table_set(worker->vars, copy, result);
  
  return APR_SUCCESS;
}

/**
 * CERT command
 *
 * @param self IN command
 * @param worker IN thread data object
 * @param data IN cert, key and optional ca file
 *
 * @return APR_SUCCESS or apr error code
 */
apr_status_t command_CERT(command_t *self, worker_t *worker, char *data) {
  char *copy;
  char *last;
  char *cert;
  char *key;
  char *ca;

  COMMAND_NEED_ARG("<cert-file> <key-file>");
  
  if (worker->ssl_ctx) {
    cert = apr_strtok(copy, " ", &last);
    key = apr_strtok(NULL, " ", &last);
    ca = apr_strtok(NULL, " ", &last);
    worker_ssl_ctx(worker, cert, key, 1);
    if (ca && !SSL_CTX_load_verify_locations(worker->ssl_ctx, ca, NULL)) {
	worker_log(worker, LOG_ERR, "Could not load CA file");
      return APR_EINVAL;
    }
  }
  else {
    worker_log(worker, LOG_ERR, "Can not set cert, ssl not enabled in CLIENT/SERVER");
    return APR_EINVAL;
  }
  
  return APR_SUCCESS;
}

/**
 * verify callback for peer cert verification for debugging purpose
 * @param cur_ok IN current ok state
 * @param ctx IN X509 store context
 */
int debug_verify_callback(int cur_ok, X509_STORE_CTX *ctx) {
  char buf[256];
  X509 *err_cert;
  int err, depth;

  err_cert = X509_STORE_CTX_get_current_cert(ctx);
  err = X509_STORE_CTX_get_error(ctx);
  depth = X509_STORE_CTX_get_error_depth(ctx);

  X509_NAME_oneline(X509_get_subject_name(err_cert), buf, 256);
  fprintf(stdout, "\nverify error:num=%d:%s:depth=%d:%s",
	  err, X509_verify_cert_error_string(err), depth, buf);
  return cur_ok;
}

/**
 * VERIFY_PEER command
 *
 * @param self IN command
 * @param worker IN thread data object
 * @param data IN 
 *
 * @return APR_SUCCESS or apr error code
 */
apr_status_t command_VERIFY_PEER(command_t *self, worker_t *worker, char *data) {
  return command_RENEG(self, worker, "verify");
}

/**
 * RENEG command
 *
 * @param self IN command
 * @param worker IN thread data object
 * @param data IN 
 *
 * @return APR_SUCCESS or apr error code
 */
apr_status_t command_RENEG(command_t *self, worker_t *worker, char *data) {
#ifdef USE_SSL
  int rc;
  char *copy;

  X509 *peer_cert = NULL;

  COMMAND_OPTIONAL_ARG;

  if (!worker->socket->is_ssl || !worker->socket->ssl) {
    worker_log(worker, LOG_ERR, 
	       "No ssl connection established can not verify peer");
    return APR_ENOSOCKET;
  }

  if (worker->flags & FLAGS_SERVER) {
    if (strcasecmp(copy, "verify") == 0) {
      /* if we are server request the peer cert */
      if (worker->log_mode >= LOG_DEBUG) {
	SSL_set_verify(worker->socket->ssl,
		       SSL_VERIFY_PEER | SSL_VERIFY_FAIL_IF_NO_PEER_CERT,
		       debug_verify_callback);
      }
      else {
	SSL_set_verify(worker->socket->ssl,
		       SSL_VERIFY_PEER | SSL_VERIFY_FAIL_IF_NO_PEER_CERT,
		       NULL);
      }
    }

    if((rc = SSL_renegotiate(worker->socket->ssl) <= 0)) {
      worker_log(worker, LOG_ERR, "SSL renegotiation a error: %d", rc);
      return APR_EACCES;
    }
    worker_ssl_handshake(worker);
    worker->socket->ssl->state=SSL_ST_ACCEPT;
    worker_ssl_handshake(worker);

    if (strcasecmp(copy, "verify") == 0) {
      peer_cert = SSL_get_peer_certificate(worker->socket->ssl);
      if (!peer_cert) {
	worker_log(worker, LOG_ERR, "No peer certificate");
	return APR_EACCES;
      }
    }
  }
  else {
    if (strcasecmp(copy, "verify") == 0) {
      peer_cert = SSL_get_peer_certificate(worker->socket->ssl);
      if (!peer_cert) {
	worker_log(worker, LOG_ERR, "No peer certificate");
	return APR_EACCES;
      }

      if((rc = SSL_get_verify_result(worker->socket->ssl))!=X509_V_OK) {
	worker_log(worker, LOG_ERR, "SSL peer verify failed: %s(%d)",
	X509_verify_cert_error_string(rc), rc);
	return APR_EACCES;
      }
    }
  }
#endif

  return APR_SUCCESS;
}

/**
 * ONLY_PRINTABLE command
 *
 * @param self IN command
 * @param worker IN thread data object
 * @param data IN on|off
 *
 * @return APR_SUCCESS or apr error code
 */
apr_status_t command_ONLY_PRINTABLE(command_t *self, worker_t *worker, char *data) {
  char *copy;

  COMMAND_NEED_ARG("Need on|off");
 
  if (strcasecmp(copy, "on") == 0) {
    worker->flags |= FLAGS_ONLY_PRINTABLE;
  }
  else {
    worker->flags &= ~FLAGS_ONLY_PRINTABLE;
  }
  return APR_SUCCESS;
}

/**
 * PRINT_HEX command
 * @param self IN command
 * @param worker IN thread data object
 * @param data IN on|off
 *
 * @return APR_SUCCESS or apr error code
 */
apr_status_t command_PRINT_HEX(command_t *self, worker_t *worker, char *data) {
  char *copy;

  COMMAND_NEED_ARG("Need on|off");
 
  if (strcasecmp(copy, "on") == 0) {
    worker->flags |= FLAGS_PRINT_HEX;
  }
  else {
    worker->flags &= ~FLAGS_PRINT_HEX;
  }
  return APR_SUCCESS;
}

/**
 * SH command
 *
 * @param self IN command
 * @param worker IN thread data object
 * @param data IN 
 *
 * @return APR_SUCCESS or apr error code
 */
apr_status_t command_SH(command_t *self, worker_t *worker, char *data) {
  char *copy;
  apr_size_t len;
  char *name;
  char *old;

  apr_status_t status = APR_SUCCESS;
  
  COMMAND_NEED_ARG("Either shell commands or END");

  if (strcasecmp(copy, "END")== 0) {
    if (worker->tmpf) {
      /* get file name */
      if ((status = apr_file_name_get((const char **)&name, worker->tmpf)) != APR_SUCCESS) {
	return status;
      }

      if ((status = apr_file_perms_set(name, 0x700)) != APR_SUCCESS) {
	return status;
      }

      /* close file */
      apr_file_close(worker->tmpf);
      worker->tmpf = NULL;

      /* exec file */
      old = self->name;
      self->name = apr_pstrdup(worker->ptmp, "_EXEC"); 
      status = command_EXEC(self, worker, apr_pstrcat(worker->ptmp, "./", name, NULL));
      self->name = old;
      
      apr_file_remove(name, worker->ptmp);
    }
  }
  else {
    if (!worker->tmpf) {
      name = apr_pstrdup(worker->ptmp, "httXXXXXX");
      if ((status = apr_file_mktemp(&worker->tmpf, name, 
	                            APR_CREATE | APR_READ | APR_WRITE | 
				    APR_EXCL, 
				    worker->ptmp))
	  != APR_SUCCESS) {
	worker_log(worker, LOG_ERR, "Could not mk temp file %s(%d)", 
	           my_status_str(worker->pool, status), status);
	return status;
      }
    }
    
    len = strlen(copy);
    if ((status = file_write(worker->tmpf, copy, len)) != APR_SUCCESS) {
      worker_log(worker, LOG_ERR, "Could not write to temp file");
      return status;
    }
    len = 1;
    if ((status = file_write(worker->tmpf, "\n", len)) != APR_SUCCESS) {
      worker_log(worker, LOG_ERR, "Could not write to temp file");
      return status;
    }
  }

  return status;
}

/**
 * ADD_HEADER command
 *
 * @param self IN command
 * @param worker IN thread data object
 * @param data IN header value
 *
 * @return APR_SUCCESS or apr error code
 */
apr_status_t command_ADD_HEADER(command_t *self, worker_t *worker, char *data) {
  char *copy;
  char *header;
  char *value;

  COMMAND_NEED_ARG("<header> <value>");

  if (!worker->headers_add) {
    worker->headers_add = apr_table_make(worker->ptmp, 12);
  }

  header = apr_strtok(copy, " ", &value);
  apr_table_add(worker->headers_add, header, value);

  return APR_SUCCESS;
}

/**
 * DETACH command to run process in background
 *
 * @param self IN command
 * @param worker IN thread data object
 * @param data IN 
 *
 * @return APR_SUCCESS or apr error code
 */
apr_status_t command_DETACH(command_t *self, worker_t *worker, char *data) {
  return apr_proc_detach(1);
}

/**
 * PID command
 *
 * @param self IN command
 * @param worker IN thread data object
 * @param data IN variable
 *
 * @return APR_SUCCESS or apr error code
 */
apr_status_t command_PID(command_t *self, worker_t *worker, char *data) {
  char *copy;

  COMMAND_NEED_ARG("<variable>");

  apr_table_set(worker->vars, copy, apr_psprintf(worker->ptmp, "%u", getpid()));
  
  return APR_SUCCESS;
}

/**
 * Get index of url escape sequenze array 
 *
 * @param c IN char for lookup
 *
 * @return index of url escape sequenz array
 */
static int get_url_escape_index(char c) {
  int i;
  for (i = 0; i < sizeof(url_escape_seq)/sizeof(url_escape_seq_t); i++) {
    if (url_escape_seq[i].c == c) {
      return i;
    }
  }
  return -1;
}

/**
 * URLENC command
 *
 * @param self IN command
 * @param worker IN thread data object
 * @param data IN string and variable name
 *
 * @return APR_SUCCESS or APR_EGENERAL on wrong parameters
 */
apr_status_t command_URLENC(command_t *self, worker_t *worker, char *data) {
  char *copy;
  char *string;
  char *result;
  char *last;
  char *var;
  int i;
  int j;
  int k;
  apr_size_t len;

  COMMAND_NEED_ARG("\"<string>\" <variable>");

  string = my_unescape(copy, &last);
  var = apr_strtok(NULL, "", &last);
  if (!var) {
    worker_log(worker, LOG_ERR, "Variable not specified");
    return APR_EGENERAL;
  }
  apr_collapse_spaces(var, var);

  len = strlen(string);
  /* allocate worste case -> every char enc with pattern %XX */
  result = apr_pcalloc(worker->ptmp, 3 * len + 1);

  /** do the simple stuff */
  for (j = 0, i = 0; string[i]; i++) {
    k = get_url_escape_index(string[i]);
    if (k != -1) {
      strncpy(&result[j], url_escape_seq[k].esc, url_escape_seq[k].len);
      j += url_escape_seq[k].len;
    }
    else {
      result[j++] = string[i];
    }
  }

  apr_table_set(worker->vars, var, result);

  return APR_SUCCESS;
}

/**
 * URLDEC command
 *
 * @param self IN command
 * @param worker IN thread data object
 * @param data IN string and variable name
 *
 * @return APR_SUCCESS or APR_EGENERAL on wrong parameters
 */
apr_status_t command_URLDEC(command_t *self, worker_t *worker, char *data) {
  char *copy;
  char *string;
  char *last;
  char *var;
  char c;
  int i;
  int j;
  apr_size_t len;
  char *inplace;

  COMMAND_NEED_ARG("\"<string>\" <variable>");

  string = my_unescape(copy, &last);
  var = apr_strtok(NULL, "", &last);
  if (!var) {
    worker_log(worker, LOG_ERR, "Variable not specified");
    return APR_EGENERAL;
  }
  apr_collapse_spaces(var, var);

  inplace = string;
  len = strlen(string);
  for (i = 0, j = 0; i < len; i++, j++) {
    c = string[i];
    if (string[i] == '+') {
      c = 32;
    }
    else if (string[i] == '%') {
      if (i + 2 < len) {
        c = x2c(&string[i + 1]);
	i += 2;
      }
    }
    else if (string[i] == '\\' && i + 1 < len && string[i + 1] == 'x') {
      if (i + 3 < len) {
        c = x2c(&string[i + 2]);
	i += 3;
      }
    }
    inplace[j] = c;
  }
  inplace[j] = 0;

  apr_table_set(worker->vars, var, inplace);

  return APR_SUCCESS;
}

/**
 * HTMLDEC command
 *
 * @param self IN command
 * @param worker IN thread data object
 * @param data IN string and variable name
 *
 * @return APR_SUCCESS or APR_EGENERAL on wrong parameters
 */
apr_status_t command_HTMLDEC(command_t *self, worker_t *worker, char *data) {
  char *copy;
  char *string;
  char *last;
  char *var;
  char c;
  int i;
  int j;
  apr_size_t len;
  char *inplace;

  COMMAND_NEED_ARG("\"<string>\" <variable>");

  string = my_unescape(copy, &last);
  var = apr_strtok(NULL, "", &last);
  if (!var) {
      worker_log(worker, LOG_ERR, "Variable not specified");
      return APR_EGENERAL;
    }
  apr_collapse_spaces(var, var);

  inplace = string;
  len = strlen(string);
  for (i = 0, j = 0; i < len; i++, j++) {
      c = string[i];
      if (string[i] == '&' && i + 2 < len && string[i + 1] == '#' && 
	  string[i + 2] == 'x') {
	/* hex */
      }
      else if (string[i] == '&' && i + 1 < len && string[i + 1] == '#') {
	/* decimal */
      }
      inplace[j] = c;
    }
  inplace[j] = 0;

  apr_table_set(worker->vars, var, inplace);

  return APR_SUCCESS;
}

/**
 * B64ENC command
 *
 * @param self IN command
 * @param worker IN thread data object
 * @param data IN string and variable name
 *
 * @return APR_SUCCESS or APR_EGENERAL on wrong parameters
 */
apr_status_t command_B64ENC(command_t *self, worker_t *worker, char *data) {
  char *copy;
  char *string;
  char *last;
  char *var;
  apr_size_t len;
  char *base64;

  COMMAND_NEED_ARG("\"<string>\" <variable>");

  string = my_unescape(copy, &last);
  var = apr_strtok(NULL, "", &last);
  if (!var) {
      worker_log(worker, LOG_ERR, "Variable not specified");
      return APR_EGENERAL;
    }
  apr_collapse_spaces(var, var);

  len = apr_base64_encode_len(strlen(string));
  base64 = apr_pcalloc(worker->ptmp, len + 1);
  apr_base64_encode(base64, string, strlen(string));
  
  apr_table_set(worker->vars, var, base64);

  return APR_SUCCESS;
}

/**
 * BASE64DEC command
 *
 * @param self IN command
 * @param worker IN thread data object
 * @param data IN string and variable name
 *
 * @return APR_SUCCESS or APR_EGENERAL on wrong parameters
 */
apr_status_t command_B64DEC(command_t *self, worker_t *worker, char *data) {
  char *copy;
  char *string;
  char *last;
  char *var;
  apr_size_t len;
  char *plain;

  COMMAND_NEED_ARG("\"<string>\" <variable>");

  string = my_unescape(copy, &last);
  var = apr_strtok(NULL, "", &last);
  if (!var) {
      worker_log(worker, LOG_ERR, "Variable not specified");
      return APR_EGENERAL;
    }
  apr_collapse_spaces(var, var);

  len = apr_base64_decode_len(string);
  plain = apr_pcalloc(worker->ptmp, len + 1);
  apr_base64_decode(plain, string);
  
  apr_table_set(worker->vars, var, plain);

  return APR_SUCCESS;
}

/**
 * STRFTIME command
 *
 * @param self IN command
 * @param worker IN thread data object
 * @param data IN time [ms] "format" variable
 *
 * @return APR_SUCCESS or APR_EGENERAL on wrong parameters
 */
apr_status_t command_STRFTIME(command_t *self, worker_t *worker, char *data) {
  apr_status_t status;
  char *copy;
  char *time;
  char *fmt;
  char *var;
  char *type;
  char *last;
  char *timefmt;
  apr_size_t len;
  apr_time_exp_t  tm;
  apr_time_t timems;

  COMMAND_NEED_ARG("<time> \"<format>\" <variable> [Local|GMT]");

  time = apr_strtok(copy, " ", &last);
  fmt = my_unescape(last, &last);
  var = apr_strtok(NULL, " ", &last);
  type = apr_strtok(NULL, " ", &last);

  if (!time) {
    worker_log(worker, LOG_ERR, "Time not specified");
    return APR_EGENERAL;
  }
  if (!fmt) {
    worker_log(worker, LOG_ERR, "Format not specified");
    return APR_EGENERAL;
  }
  if (!var) {
    worker_log(worker, LOG_ERR, "Variable not specified");
    return APR_EGENERAL;
  }
  apr_collapse_spaces(var, var);

  timems = apr_atoi64(time);
  
  timefmt = apr_pcalloc(worker->ptmp, 255);
  
  if (type && strncasecmp(type, "Local", 5) == 0) {
    if ((status = apr_time_exp_lt(&tm, timems * 1000)) != APR_SUCCESS) { 
      return status;
    }
  }
  else {
    if ((status = apr_time_exp_gmt(&tm, timems * 1000)) != APR_SUCCESS) { 
      return status;
    }
  }
  
  if ((status = apr_strftime(timefmt, &len, 254, fmt, &tm)) != APR_SUCCESS) {
    return status;
  }

  apr_table_set(worker->vars, var, timefmt);
  
  return APR_SUCCESS;
}

/**
 * TUNNEL command
 *
 * @param self IN command
 * @param worker IN thread data object
 * @param data IN <host> [SSL:]<port>
 *
 * @return APR_SUCCESS or APR_EGENERAL on wrong parameters
 */
apr_status_t command_TUNNEL(command_t *self, worker_t *worker, char *data) {
  apr_status_t status;
  apr_status_t rc;
  apr_threadattr_t *tattr;
  apr_thread_t *client_thread;
  apr_thread_t *backend_thread;
  tunnel_t client;
  tunnel_t backend;
  apr_size_t peeklen;

  if (!worker->flags & FLAGS_SERVER) {
    worker_log_error(worker, "This command is only valid in a SERVER");
    return APR_EGENERAL;
  }

  if (worker->socket->socket_state == SOCKET_CLOSED) {
    worker_log_error(worker, "Socket to client is closed\n");
    return APR_ECONNREFUSED;
  }

  worker_log(worker, LOG_DEBUG, "--- tunnel\n");

  apr_pool_create(&client.pool, NULL);
  apr_pool_create(&backend.pool, NULL);

  /* client side */
  if ((status = apr_socket_timeout_set(worker->socket->socket, 100000)) 
      != APR_SUCCESS) {
    goto error1;
  }
  if (worker->sockreader == NULL) {
    peeklen = worker->socket->peeklen;
    worker->socket->peeklen = 0;
    sockreader_new(&client.sockreader, worker->socket->socket,
#ifdef USE_SSL
		   worker->socket->is_ssl ? worker->socket->ssl : NULL,
#endif
		   worker->socket->peek, peeklen, client.pool);
    if (status != APR_SUCCESS && !APR_STATUS_IS_TIMEUP(status)) {
      goto error1;
    }
  }
  else {
    client.sockreader = worker->sockreader;
  }
  backend.sendto = worker->socket;

  /* backend side */
  if ((status = command_REQ(self, worker, data)) != APR_SUCCESS) {
    goto error1;
  }
  if ((status = apr_socket_timeout_set(worker->socket->socket, 100000)) 
      != APR_SUCCESS) {
    goto error2;
  }
  sockreader_new(&backend.sockreader, worker->socket->socket,
#ifdef USE_SSL
		 worker->socket->is_ssl ? worker->socket->ssl : NULL,
#endif
		 NULL, 0, backend.pool);
  if (status != APR_SUCCESS && !APR_STATUS_IS_TIMEUP(status)) {
    goto error2;
  }
  client.sendto = worker->socket;

  /* need two threads reading/writing from/to backend */
  if ((status = apr_threadattr_create(&tattr, worker->pool)) != APR_SUCCESS) {
    goto error2;
  }

  if ((status = apr_threadattr_stacksize_set(tattr, DEFAULT_THREAD_STACKSIZE))
      != APR_SUCCESS) {
    goto error2;
  }

  if ((status = apr_threadattr_detach_set(tattr, 0)) != APR_SUCCESS) {
    goto error2;
  }

  if ((status = apr_thread_create(&client_thread, tattr, streamer, 
	                          &client, worker->pool)) != APR_SUCCESS) {
    goto error2;
  }

  if ((status = apr_thread_create(&backend_thread, tattr, streamer, 
	                          &backend, worker->pool)) != APR_SUCCESS) {
    goto error2;
  }

  rc = apr_thread_join(&status, client_thread);
  if (status != APR_SUCCESS) {
    goto error2;
  }
  rc = apr_thread_join(&status, backend_thread);
  if (status != APR_SUCCESS) {
    goto error2;
  }

error2:
  command_CLOSE(self, worker, "");
error1:
  worker_get_socket(worker, "Default", "0");
  apr_pool_destroy(client.pool);
  apr_pool_destroy(backend.pool);
  worker_log(worker, LOG_DEBUG, "--- tunnel end\n");
  return status;
}

/**
 * BREAK command
 *
 * @param self IN command
 * @param worker IN thread data object
 * @param data IN unused
 *
 * @return APR_SUCCESS or APR_EGENERAL on wrong parameters
 */
apr_status_t command_BREAK(command_t *self, worker_t *worker, char *data) {
  /* singal break for loop */
  COMMAND_NO_ARG;
  return -1;
}

/**
 * DURATION command calls a defined block
 *
 * @param self IN command
 * @param worker IN thread data object
 * @param data IN name of calling block 
 *
 * @return APR_SUCCESS
 */
apr_status_t command_TIMER(command_t *self, worker_t *worker, char *data) {
  char *copy;
  char *last;
  char *cmd;
  char *var;

  apr_time_t cur = apr_time_now();

  COMMAND_NEED_ARG("<variable> expected");
  cmd = apr_strtok(copy, " ", &last);
  var = apr_strtok(NULL, " ", &last);
 
  if (strcasecmp(cmd, "GET") == 0) {
    /* nothing special to do */
  }
  else if (strcasecmp(cmd, "RESET") == 0) {
    worker->start_time = apr_time_now();
  }
  else {
    worker_log_error(worker, "Timer command %s not implemented", cmd);
  }

  if (var && var[0] != 0) {
    apr_table_set(worker->vars, var, 
		  apr_off_t_toa(worker->ptmp, apr_time_as_msec(cur - worker->start_time)));
  }
  return APR_SUCCESS;
}

/**
 * Object thread data
 */

/**
 * New thread data object 
 *
 * @param self OUT thread data object
 * @param log_mode IN log mode  
 *
 * @return an apr status
 */
apr_status_t worker_new(worker_t ** self, char *additional,
                        int log_mode, char *prefix,
                        apr_thread_cond_t * sync_cond,
                        apr_thread_mutex_t * sync_mutex,
                        apr_thread_mutex_t * mutex,
			apr_time_t socktmo, apr_hash_t *blocks) {
  apr_pool_t *p;
  apr_pool_t *ptmp;

  apr_pool_create(&p, NULL);
  (*self) = apr_pcalloc(p, sizeof(worker_t));
  (*self)->heartbeat = p;
  apr_pool_create(&p, (*self)->heartbeat);
  (*self)->pool = p;
  apr_pool_create(&ptmp, p);
  (*self)->ptmp = ptmp;
  (*self)->filename = apr_pstrdup(p, "<none>");
  (*self)->socktmo = socktmo;
  (*self)->prefix = apr_pstrdup(p, prefix);
  (*self)->additional = apr_pstrdup(p, additional);
  (*self)->sync_cond = sync_cond;
  (*self)->sync_mutex = sync_mutex;
  (*self)->mutex = mutex;
  (*self)->lines = apr_table_make(p, 20);
  (*self)->cache = apr_table_make(p, 20);
  (*self)->expect_dot = apr_table_make(p, 2);
  (*self)->expect_exec= apr_table_make(p, 2);
  (*self)->expect_error = apr_table_make(p, 2);
  (*self)->match_headers = apr_table_make(p, 2);
  (*self)->match_body = apr_table_make(p, 2);
  (*self)->match_error = apr_table_make(p, 2);
  (*self)->match_exec = apr_table_make(p, 2);
  (*self)->sockets = apr_hash_make(p);
  (*self)->headers_allow = NULL;
  (*self)->headers_filter = NULL;
  (*self)->params = apr_table_make(p, 4);
  (*self)->vars = apr_table_make(p, 4);
  (*self)->blocks = blocks;
  (*self)->start_time = apr_time_now();
  (*self)->log_mode = log_mode;
#ifdef USE_SSL
  (*self)->bio_out = BIO_new_fp(stdout, BIO_NOCLOSE);
  (*self)->bio_err = BIO_new_fp(stderr, BIO_NOCLOSE);
#endif
  (*self)->listener_addr = apr_pstrdup(p, APR_ANYADDR);

  worker_log(*self, LOG_DEBUG, "worker_new: pool: %p, ptmp: %p\n", (*self)->pool, (*self)->ptmp);
  return APR_SUCCESS;
}

/**
 * Clone thread data object 
 *
 * @param self OUT thread data object
 * @param orig IN thread data object to copy from 
 *
 * @return an apr status
 */
apr_status_t worker_clone(worker_t ** self, worker_t * orig) {
  apr_pool_t *p;
  apr_pool_t *ptmp;

  apr_pool_create(&p, NULL);
  (*self) = apr_pcalloc(p, sizeof(worker_t));
  memcpy(*self, orig, sizeof(worker_t));
  (*self)->heartbeat = p;
  apr_pool_create(&p, (*self)->heartbeat);
  (*self)->pool = p;
  apr_pool_create(&ptmp, p);
  (*self)->ptmp = ptmp;
  (*self)->flags = orig->flags;
  (*self)->prefix = apr_pstrdup(p, orig->prefix);
  (*self)->additional = apr_pstrdup(p, orig->additional);
  (*self)->lines = my_table_deep_copy(p, orig->lines);
  (*self)->cache = my_table_deep_copy(p, orig->cache);
  (*self)->expect_dot = my_table_swallow_copy(p, orig->expect_dot);
  (*self)->expect_exec = my_table_swallow_copy(p, orig->expect_exec);
  (*self)->expect_error = my_table_swallow_copy(p, orig->expect_error);
  (*self)->match_headers = my_table_swallow_copy(p, orig->match_headers);
  (*self)->match_body = my_table_swallow_copy(p, orig->match_body);
  (*self)->match_error = my_table_swallow_copy(p, orig->match_error);
  (*self)->match_exec = my_table_swallow_copy(p, orig->match_exec);
  (*self)->start_time = orig->start_time;
  (*self)->listener = NULL;
  (*self)->sockets = apr_hash_make(p);
  if (orig->headers_allow) {
    (*self)->headers_allow = my_table_deep_copy(p, orig->headers_allow);
  }
  if (orig->headers_filter) {
    (*self)->headers_filter = my_table_deep_copy(p, orig->headers_filter);
  }
  (*self)->params = apr_table_make(p, 4);
  (*self)->vars = my_table_deep_copy(p, orig->vars);
#ifdef USE_SSL
  (*self)->bio_out = BIO_new_fp(stdout, BIO_NOCLOSE);
  (*self)->bio_err = BIO_new_fp(stderr, BIO_NOCLOSE);
#endif
  (*self)->listener_addr = apr_pstrdup(p, orig->listener_addr);

  worker_log(*self, LOG_DEBUG, "worker_clone: pool: %p, ptmp: %p\n", (*self)->pool, (*self)->ptmp);
  return APR_SUCCESS;
}
 
/**
 * Clone and copy a body of lines
 *
 * @param body OUT body which has been copied
 * @param worker IN  worker from which we copy the lines for body
 * @param end IN this bodys terminate string
 *
 * @return APR_SUCCESS
 */
apr_status_t worker_body(worker_t **body, worker_t *worker, char *command) {
  char *file_and_line;
  char *line = "";
  apr_table_entry_t *e; 
  apr_pool_t *p;
  char *end;
  char *kind;
  int ends;
  int end_len;
  int kind_len;

  /* create body */
  apr_pool_create(&p, NULL);
  end = apr_pstrcat(p, "_END ", command, NULL);
  end_len = strlen(end);
  kind = apr_pstrcat(p, "_", command, NULL);
  kind_len = strlen(kind);
  ends = 1;
  (*body) = apr_pcalloc(p, sizeof(worker_t));
  memcpy(*body, worker, sizeof(worker_t));
  /* give it an own heartbeat :) */
  (*body)->heartbeat = p;

  /* fill lines */
  (*body)->lines = apr_table_make(p, 20);
  e = (apr_table_entry_t *) apr_table_elts(worker->lines)->elts;
  for (worker->cmd += 1; worker->cmd < apr_table_elts(worker->lines)->nelts; worker->cmd++) {
    file_and_line = e[worker->cmd].key;
    line = e[worker->cmd].val;
    /* count numbers of same kinds to include all their ends */
    if (strlen(line) >= kind_len && strncmp(line, kind, kind_len) == 0) {
      ++ends;
      worker_log(worker, LOG_DEBUG, "Increment loops: %d for line %s", ends, line);
    }
    /* check end and if it is our end */
    if (strlen(line) >= end_len && strncmp(line, end, end_len) == 0 && ends == 1) {
      break;
    }
    /* no is not our end, decrement ends */
    else if (strlen(line) >= end_len && strncmp(line, end, end_len) == 0) {
      --ends;
      worker_log(worker, LOG_DEBUG, "Decrement loops: %d for line %s", ends, line);
    }
    apr_table_addn((*body)->lines, file_and_line, line);
  }
  /* check for end */
  if (strlen(line) < end_len || strncmp(line, end, end_len) != 0) {
    worker_log(worker, LOG_ERR, "Compilation failed: no %s found", end);
    return APR_EGENERAL;
  }

  return APR_SUCCESS;
}
 
/**
 * Close a body 
 *
 * @param body IN body which has been copied
 * @param worker IN  worker from which we copy the lines for body
 */
void worker_body_end(worker_t *body, worker_t *worker) {
  worker->flags = body->flags;
  /* write back sockets and state */
  worker->socket = body->socket;
  worker->listener = body->listener;

  /* destroy body */
  worker_destroy(body);
}

/**
 * Destroy thread data object
 *
 * @param self IN thread data object
 */
void worker_destroy(worker_t * self) {
  worker_log(self, LOG_DEBUG, "worker_destroy: %p, ptmp: %p", self->pool, self->ptmp);
  apr_pool_destroy(self->heartbeat);
}

/**
 * Clone thread data object 
 *
 * @param self IN thread data object
 * @param line IN command line
 *
 * @return an apr status
 */
apr_status_t worker_add_line(worker_t * self, const char *file_and_line,
                             char *line) {
  apr_table_add(self->lines, file_and_line, line);
  return APR_SUCCESS;
}

/**
 * local socket send
 *
 * @param sockett IN socket 
 * @param buf IN buffer to send
 * @param len IN no bytes of buffer to send
 *
 * @return apr status
 */
static apr_status_t socket_send(socket_t *socket, char *buf,
                                apr_size_t len) {
  apr_status_t status = APR_SUCCESS;
  apr_size_t total = len;
  apr_size_t count = 0;

#ifdef USE_SSL
  if (socket->is_ssl) {
    apr_size_t e_ssl;
  tryagain:
    apr_sleep(1);
    e_ssl = SSL_write(socket->ssl, buf, len);
    if (e_ssl != len) {
      int scode = SSL_get_error(socket->ssl, e_ssl);
      if (scode == SSL_ERROR_WANT_WRITE) {
	goto tryagain;
      }
      status = APR_ECONNABORTED;
      goto error;
    }
  }
  else
#endif
  {
    while (total != count) {
      len = total - count;
      if ((status = apr_socket_send(socket->socket, &buf[count], &len)) 
	  != APR_SUCCESS) {
	goto error;
      }
      count += len;
    }
  }

error:
  return status;
}

/**
 * Send buf with len
 * 
 * @param self IN thread data object
 * @param buf IN buffer to send
 * @param len IN no bytes of buffer to send
 *
 * @return apr status
 */
apr_status_t worker_socket_send(worker_t *self, char *buf, 
                                apr_size_t len) {
  return socket_send(self->socket, buf, len);
}

/**
 * flush data 
 *
 * @param self IN thread data object
 *
 * @return an apr status
 */
apr_status_t worker_flush(worker_t * self) {
  apr_size_t len;
  int i;
  int start;
  char *chunked;

  apr_status_t status = APR_SUCCESS;
  apr_table_entry_t *e =
    (apr_table_entry_t *) apr_table_elts(self->cache)->elts;

  if (!self->socket || !self->socket->socket) {
    goto error;
  }
  
  chunked = NULL;
  if (apr_table_get(self->cache, "Content-Length")) {
    /* calculate body len */
    start = 0;
    len = 0;
    for (i = 0; i < apr_table_elts(self->cache)->nelts; ++i) {
      if (!start && !e[i].val[0]) {
        /* start body len */
        start = 1;
      }
      else if (start) {
        /* do not forget the \r\n */
	if (strncasecmp(e[i].key, "NOCRLF", 6) != 0) {
	  len += 2;
	}
	if (strncasecmp(e[i].key, "NOCRLF:", 7) == 0) { 
	  len += apr_atoi64(&e[i].key[7]);
	}
	else {
          len += strlen(e[i].val);
	}
      }
    }

    apr_table_setn(self->cache, "Content-Length",
                   apr_psprintf(self->pool, "Content-Length: %d", len));
  }
  else if (apr_table_get(self->cache, "CHUNKED")) {
    apr_table_unset(self->cache, "CHUNKED");
    len = 0;
    for (i = 0; i < apr_table_elts(self->cache)->nelts; ++i) {
      /* do not forget the \r\n */
      if (strncasecmp(e[i].key, "NOCRLF", 6) != 0) {
	len += 2;
      }
      if (strncasecmp(e[i].key, "NOCRLF:", 7) == 0) { 
	len += apr_atoi64(&e[i].key[7]);
      }
      else {
	len += strlen(e[i].val);
      }
    }
    chunked = apr_psprintf(self->pool, "\r\n%x\r\n", len);
  }

  if (chunked) {
    worker_log(self, LOG_INFO, ">");
    worker_log(self, LOG_INFO, ">%x", len);
    worker_log(self, LOG_INFO, ">");
  }
  if (chunked) {
    len = strlen(chunked);
    if ((status = worker_socket_send(self, chunked, len)) != APR_SUCCESS) {
      goto error;
    }
    self->sent += len;
  }
  /* iterate through all cached lines and send them */
  for (i = 0; i < apr_table_elts(self->cache)->nelts; ++i) {
    if (strncasecmp(e[i].key, "NOCRLF:", 7) == 0) { 
      len = apr_atoi64(&e[i].key[7]);
      worker_log_buf(self, LOG_INFO, e[i].val, ">", len);
    }
    else {
      len = strlen(e[i].val);
      worker_log(self, LOG_INFO, ">%s %s", e[i].val, 
	         strcasecmp(e[i].key, "NOCRLF") ? "" : e[i].key);
    }

    if ((status = worker_socket_send(self, e[i].val, len)) 
	!= APR_SUCCESS) {
      goto error;
    }
    self->sent += len;
    if (strncasecmp(e[i].key, "NOCRLF", 6) != 0) {
      len = 2;
      if ((status = worker_socket_send(self, "\r\n", len)) != APR_SUCCESS) {
	goto error;
      }
      self->sent += len;
    }
  }

error:
  apr_table_clear(self->cache);

  return status;
}

/**
 * write worker data to file with worker->name
 *
 * @param self IN thread data object
 *
 * @return an apr status
 */
apr_status_t worker_to_file(worker_t * self) {
  apr_status_t status;
  apr_file_t *fp;
  apr_table_entry_t *e;
  char *line;
  int i;

  if ((status =
       apr_file_open(&fp, self->name, APR_CREATE | APR_WRITE, APR_OS_DEFAULT,
		     self->ptmp)) != APR_SUCCESS) {
    return status;
  }

  e = (apr_table_entry_t *) apr_table_elts(self->lines)->elts;
  for (i = 0; i < apr_table_elts(self->lines)->nelts; i++) {
    line = e[i].val;
    apr_file_printf(fp, "%s\n", &line[1]);
  }

  apr_file_close(fp);

  return APR_SUCCESS;
}
