/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2011 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "ifieldviewsubject.h"

#include "icolorbars.h"
#include "idata.h"
#include "idatareader.h"
#include "idatalimits.h"
#include "imergedatafilter.h"
#include "ireplicatedactor.h"
#include "iviewmodule.h"

//#include <vtkStructuredPoints.h>

//
//  Templates (needed for some compilers)
//
#include "iarraytemplate.h"


using namespace iParameter;


IVIEWSUBJECT_DEFINE_TYPE(iFieldViewSubject,FieldViewSubject,-fvo);

IOBJECT_DEFINE_KEY(iFieldViewSubject,ColorBy,cb,OffsetInt,1);
IOBJECT_DEFINE_KEY(iFieldViewSubject,GlyphSampleRate,gr,Int,1);
IOBJECT_DEFINE_KEY(iFieldViewSubject,ConnectedToScalars,cs,Bool,1);
IOBJECT_DEFINE_KEY(iFieldViewSubject,DataOffset,-off,OffsetInt,1);

IOBJECT_DEFINE_DISTANCE_KEY(iFieldViewSubject,GlyphSize,gs);


//
// iFieldViewSubject class
//
iFieldViewSubject::iFieldViewSubject(iViewModule *vm, const iDataType &type, const iDataType &scalarType, const iString &name, int numActors, unsigned int flags) : iSolidViewSubject(vm,type,name,numActors,flags), mGlyphSize(vm,true), mScalarDataType(scalarType)
{
	mColorBy = -1;
	mGlyphSize = 0.1;
	mGlyphSampleRate = 16;
	mColorByOffset = 0;

	this->AddSecondaryDataType(mScalarDataType);

	//
	//  Do VTK stuff
	//	
	mMergeDataFilter = iMergeDataFilter::New(this); IERROR_ASSERT(mMergeDataFilter);
}


iFieldViewSubject::~iFieldViewSubject()
{
	mMergeDataFilter->Delete();
}


void iFieldViewSubject::SetGlyphSize(const iDistance &v)
{ 
	if(v < 1.0e5)
	{
		mGlyphSize = v;
		this->UpdateGlyphSize();
		this->ClearCache();
	}
}


void iFieldViewSubject::SetColorBy(int v)
{ 
	if(mColorBy >= 0) this->ShowColorBars(false);
	mColorBy = v;
	if(v >= 0) this->ShowColorBars(true);
	if(this->IsColoredByScalars())
	{
		mActors[0]->SyncWithLimits(this->GetViewModule()->GetReader()->GetLimits(mScalarDataType),mColorBy);
		mActors[0]->ColorByArrayComponent(0,mColorByOffset+mColorBy); 
		mActors[0]->SetScalarVisibility(true);
	}
	else
	{
		mActors[0]->SetScalarVisibility(false);
		this->UpdateColorByExtra();
	}
	this->ClearCache();
}


void iFieldViewSubject::SetGlyphSampleRate(int v)
{ 
	if(v>0 && v<1600) 
	{
		mGlyphSampleRate = v;
		this->UpdateGlyphSampleRate();
		this->ClearCache();
	}
}


void iFieldViewSubject::ShowColorBarsBody(bool show)
{
	if(this->IsColoredByScalars()) this->GetViewModule()->GetColorBars()->ShowBar(ColorBarsPriority::Field,mColorBy,mScalarDataType,mPalettes[0],show);
}


//
//  Two functions used in saving/restoring the state and in creating new instances with
//
void iFieldViewSubject::SolidViewSubjectPackStateBody(iString &s) const
{
	this->PackValue(s,KeyColorBy(),mColorBy);
	this->PackValue(s,KeyGlyphSampleRate(),mGlyphSampleRate);
	this->PackValue(s,KeyConnectedToScalars(),this->IsConnectedToScalars());
	this->PackValue(s,KeyDataOffset(),this->GetDataOffset());

	this->PackValueDistance(s,KeyGlyphSize(),mGlyphSize);

	this->FieldViewSubjectPackStateBody(s);
}


void iFieldViewSubject::SolidViewSubjectUnPackStateBody(const iString &s)
{
	int i;

	if(this->UnPackValue(s,KeyColorBy(),i)) this->SetColorBy(i);
	if(this->UnPackValue(s,KeyGlyphSampleRate(),i)) this->SetGlyphSampleRate(i);
	
	if(this->UnPackValueDistance(s,KeyGlyphSize(),mGlyphSize)) this->SetGlyphSize(mGlyphSize);

	this->FieldViewSubjectUnPackStateBody(s);
}


bool iFieldViewSubject::IsConnectedToScalars() const
{
	return this->IsThereData() && mMergeDataFilter->HasData(0);
}


bool iFieldViewSubject::IsColoredByScalars() const
{
	return (this->IsConnectedToScalars() && this->GetViewModule()->GetReader()->GetLimits(mScalarDataType)!=0 && mColorBy>=0 && mColorBy<this->GetViewModule()->GetReader()->GetLimits(mScalarDataType)->GetNumVars());
}


void iFieldViewSubject::ViewSubjectSyncWithData(const iDataSyncRequest &r)
{
	if(this->IsThereData())
	{
		mMergeDataFilter->SetDataInput(0,this->GetViewModule()->GetReader()->GetData(mScalarDataType));
		mMergeDataFilter->SetInput(this->GetData());
		mMergeDataFilter->Update();
		this->ResetPipelineInput(mMergeDataFilter->GetOutput());
	}

	if((r.Var()==-1 || r.Var()==mColorBy) && this->IsColoredByScalars())
	{
		mActors[0]->SyncWithLimits(this->GetViewModule()->GetReader()->GetLimits(mScalarDataType),mColorBy);
	}

	if(!this->IsConnectedToScalars() && mColorBy>=0 && mColorBy<this->GetViewModule()->GetReader()->GetLimits(mScalarDataType)->GetNumVars())
	{
		this->SetColorBy(mColorByOffset);
	}

	this->FieldViewSubjectSyncWithData(r);
}

