/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2011 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "irendertool.h"


#include "iactor.h"
#include "icamera.h"
#include "icontrolmodule.h"
#include "ierror.h"
#include "ieventobserver.h"
#include "iextensionfactory.h"
#include "ilightkit.h"
#include "imagnifier.h"
#include "irendertoolbackground.h"
#include "ishell.h"
#include "ishellfactory.h"
#include "istereoimage.h"
#include "istereoimagearray.h"
#include "iviewmodule.h"

#include <vtkActor2D.h>
#include <vtkCamera.h>
#include <vtkCuller.h>
#include <vtkCullerCollection.h>
#include <vtkLight.h>
#include <vtkLightCollection.h>
#include <vtkLineSource.h>
#include <vtkOpenGLRenderWindow.h>
#include <vtkPolyDataMapper2D.h>
#include <vtkPropAssembly.h>
#include <vtkPropCollection.h>
#include <vtkProperty2D.h>
#include <vtkRenderer.h>
#include <vtkRenderWindow.h>
#include <vtkRenderWindowCollection.h>
#include <vtkRenderWindowInteractor.h>

//
//  Templates
//
#include "iarraytemplate.h"


using namespace iParameter;


//
//  Helper classes
//
class iDualWindowObserver : public iEventObserver
{

public:

	vtkTypeMacro(iDualWindowObserver,iEventObserver);
	static iDualWindowObserver* New(iRenderTool *rt = 0)
	{
		IERROR_ASSERT(rt);
		return new iDualWindowObserver(rt);
	}

protected:

	iDualWindowObserver(iRenderTool *rt) : iEventObserver(rt->GetViewModule()->GetControlModule()->GetShell())
	{
		mRenderTool = rt;
	}

	virtual void ExecuteBody(vtkObject *, unsigned long event, void *)
	{
		switch(event)
		{
		case EndEvent:
			{
				if(mRenderTool->mDualWin != 0) mRenderTool->mDualWin->Render();
				break;
			}
		case ModifiedEvent:
			{
				if(mRenderTool->mDualWin != 0) mRenderTool->mDualWin->SetSize(mRenderTool->mMainWin->GetSize());
				break;
			}
		}
	}

private:

	iRenderTool *mRenderTool;
};


class iRendererObserver : public iEventObserver
{

public:

	vtkTypeMacro(iRendererObserver,iEventObserver);
	static iRendererObserver* New(iRenderTool *rt = 0)
	{
		IERROR_ASSERT(rt);
		return new iRendererObserver(rt);
	}

protected:

	iRendererObserver(iRenderTool *rt) : iEventObserver(rt->GetViewModule()->GetControlModule()->GetShell())
	{
		mRenderTool = rt;
	}

	virtual void ExecuteBody(vtkObject *, unsigned long event, void *)
	{
		switch(event)
		{
		case ModifiedEvent:
			{
				mRenderTool->GetViewModule()->ClearCache(); // our size is cached by VM
				break;
			}
		case ResetCameraClippingRangeEvent:
			{
				double *cr = mRenderTool->mMainRen->GetActiveCamera()->GetClippingRange();
				if(mRenderTool->mAutoClippingRange)
				{
					mRenderTool->mClippingRange[0] = cr[0];
					mRenderTool->mClippingRange[1] = cr[1];
				}
				else
				{
					cr[0] = mRenderTool->mClippingRange[0];
					cr[1] = mRenderTool->mClippingRange[1];
				}
				break;
			}
		}
	}

private:

	iRenderTool *mRenderTool;
};


class iStereoRenderObserver : public iEventObserver
{

public:

	vtkTypeMacro(iStereoRenderObserver,iEventObserver);
	static iStereoRenderObserver* New(iRenderTool *rt = 0)
	{
		IERROR_ASSERT(rt);
		return new iStereoRenderObserver(rt);
	}

protected:

	iStereoRenderObserver(iRenderTool *rt) : iEventObserver(rt->GetViewModule()->GetControlModule()->GetShell())
	{
		mRenderTool = rt;
	}

	virtual void ExecuteBody(vtkObject *, unsigned long event, void *)
	{
		switch(event)
		{
		case ModifiedEvent:
			{
				if(mRenderTool->mMainWinStereoRender != (mRenderTool->mMainWin->GetStereoRender()!=0))
				{
					mRenderTool->mMainWin->SetStereoRender(mRenderTool->mMainWinStereoRender?1:0);
					iConsole::Display(iConsole::_Notification,"Pressing 3 to toggle the stereo mode is no longer supported.\nPlease use GUI/command-line controls to toggle the stereo mode.");
				}
				break;
			}
		}
	}

private:

	iRenderTool *mRenderTool;
};


//
//  Main class
//
iRenderTool* iRenderTool::New(iViewModule *vm)
{
	IERROR_ASSERT(vm);
	iRenderTool *tmp =  iExtensionFactory::CreateRenderTool(vm);
	IERROR_ASSERT(tmp);
	return tmp;
}


iRenderTool::iRenderTool(iViewModule *vm, vtkRenderer *ren, iMagnifier *mag) : mViewModule(vm), mStereoModeOffset(2-VTK_STEREO_CRYSTAL_EYES)
{
	//
	//  Keeping the rendering order (a workaround for a VTK bug)
	//
	mActorObjects = vtkPropCollection::New(); IERROR_ASSERT(mActorObjects);
	mVolumeObjects = vtkPropCollection::New(); IERROR_ASSERT(mVolumeObjects);

	//
	//  Main Window stuff
	//
	mDualWin = 0;
	mMainWin = iShellFactory::CreateRenderWindow(this); IERROR_ASSERT(mMainWin);
#ifdef I_NO_STEREO
	mMainWin->SetStereoCapableWindow(0);
#else
	mMainWin->SetStereoCapableWindow(1);
#endif
	if(ren == 0)
	{
		mMainRen = vtkRenderer::New(); IERROR_ASSERT(mMainRen);
		mMainInt = iShellFactory::CreateRenderWindowInteractor(this); IERROR_ASSERT(mMainInt);
		mMainCam = iCamera::New(this); IERROR_ASSERT(mMainCam);
		mMainRen->SetActiveCamera(mMainCam->GetDevice());
		//
		//  Configure Renderer. It is important to switch the backing store off for the
		//  stereo mode to work properly.
		//
		mMainRen->SetLayer(IRENDERTOOLBACKGROUND_FRONT_LAYER);
		mMainRen->BackingStoreOff();
		mMainRen->TwoSidedLightingOff();
		mMainRen->LightFollowCameraOn();
		mMainRen->GetActiveCamera()->SetEyeAngle(2.0);
		//
		//  Configure Interactor
		//
		mMainInt->SetRenderWindow(mMainWin);
		mMainInt->LightFollowCameraOff();
		//
		//  Create and initiate lights
		//
		mLights = iLightKit::New(); IERROR_ASSERT(mLights);	
		mLights->AddLightsToRenderer(mMainRen);
		//
		//  Image magnifier
		//
		mMagnifier = iMagnifier::New(); IERROR_ASSERT(mMagnifier);
	}
	else
	{
		mMainRen = ren;
		mMainInt = 0;
		mMainCam = 0;
		mLights = 0;
		mMagnifier = mag; IERROR_ASSERT(mMagnifier);
		mMagnifier->Register(ren);
	}

	//
	//  Create backgrounds
	//
	mMainBkg = iRenderToolBackground::New(); IERROR_ASSERT(mMainBkg);
	mDualBkg = iRenderToolBackground::New(); IERROR_ASSERT(mDualBkg);

	//
	//  Configure Window
	//
	mMainWin->SetSize(640,480);
	mMainWin->SetNumberOfLayers(2);
	mMainWin->AddRenderer(mMainBkg->GetRenderer());
	mMainWin->AddRenderer(mMainRen);
	mMainWin->SetWindowName("IFrIT - Visualization Window");

	//
	//  Dual window renderer and observer
	//
	mDualRen = this->CreateRenderer();

	mDualRen->SetActiveCamera(mMainRen->GetActiveCamera());
	mDualRen->LightFollowCameraOff();

	//
	//  Dual window observer
	//
	mDualWindowObserver = iDualWindowObserver::New(this); IERROR_ASSERT(mDualWindowObserver);

	//
	//  Dual window alignment markers (only needed for the primary window)
	//
	if(ren == 0)
	{
		mStereoAlignmentMarksActor = vtkPropAssembly::New(); IERROR_ASSERT(mStereoAlignmentMarksActor);
		mStereoAlignmentMarksActor->VisibilityOff();
		mStereoAlignmentMarksActor->PickableOff();
		vtkPolyDataMapper2D *mapper2D = vtkPolyDataMapper2D::New(); IERROR_ASSERT(mapper2D);
		vtkLineSource *ls;
		vtkActor2D *actor2D[8];

		float aml = 0.15;
		float amo = 0.1;
		int i;
		for(i=0; i<8; i++)
		{
			actor2D[i] = vtkActor2D::New(); IERROR_ASSERT(actor2D[i]);
			mapper2D = vtkPolyDataMapper2D::New(); IERROR_ASSERT(mapper2D);
			ls = vtkLineSource::New(); IERROR_ASSERT(ls);
			ls->SetResolution(1);
			if(i%2 == 0)
			{
				ls->SetPoint1(-aml,0.0,0.0);
				ls->SetPoint2(aml,0.0,0.0);
			}
			else
			{
				ls->SetPoint1(0.0,-aml,0.0);
				ls->SetPoint2(0.0,aml,0.0);
			}
			ls->Update();
			mapper2D->SetInput(ls->GetOutput());
			ls->Delete();
			vtkCoordinate *c = vtkCoordinate::New(); IERROR_ASSERT(c);
			c->SetCoordinateSystemToView();
			mapper2D->SetTransformCoordinate(c);
			c->Delete();
			actor2D[i]->SetMapper(mapper2D);
			mapper2D->Delete();
			actor2D[i]->GetProperty()->SetColor(0.0,0.0,0.0);
			actor2D[i]->GetPositionCoordinate()->SetCoordinateSystemToNormalizedViewport();
			actor2D[i]->PickableOff();	
			actor2D[i]->SetPosition((0.5-amo)*(2*((i/2)%2)-1),(0.5-amo)*(2*((i/4)%2)-1));
			actor2D[i]->SetLayerNumber(1);
			mStereoAlignmentMarksActor->AddPart(actor2D[i]);
			actor2D[i]->Delete();
		}

		this->AddObject(mStereoAlignmentMarksActor);
		mStereoAlignmentMarksActor->Delete();
	}
	else
	{
		mStereoAlignmentMarksActor = 0;
	}

	//
	//  Clipping range controls
	//
	mRendererObserver = iRendererObserver::New(this); IERROR_ASSERT(mRendererObserver);
	mAutoClippingRange = true;
	mMainRen->AddObserver(vtkCommand::ResetCameraClippingRangeEvent,mRendererObserver);
	mMainRen->AddObserver(vtkCommand::ModifiedEvent,mRendererObserver);

	//
	//  Disable pressing 3 in the render window
	//
	mStereoRenderObserver = iStereoRenderObserver::New(this); IERROR_ASSERT(mStereoRenderObserver);
	mMainWin->AddObserver(vtkCommand::ModifiedEvent,mStereoRenderObserver);

	//
	//  RenderWindow collection
	//
	mWindowCollection = vtkRenderWindowCollection::New(); IERROR_ASSERT(mWindowCollection);
	mWindowCollectionUpToDate = false;

	//
	//  Properties
	//
	mStereoMode = 0;
	this->SetMainWinStereoRender(false);
	mMainWin->SetStereoType(0);  // for pressing 3 in the render window
	mAntialiasing = mBackgroundImageFixedAspect = false;
	mStereoAlignmentMarksOn = true;

	//
	//  Set the properties
	//
	mFontScale = 0;
	mFontType = TextType::Arial;
	//this->SetAntialiasing(true);
}


iRenderTool::~iRenderTool()
{
	this->ShowDualWindow(false);

	mActorObjects->Delete();
	mVolumeObjects->Delete();

	mWindowCollection->RemoveAllItems();
	mWindowCollection->Delete();
	mDualWindowObserver->Delete();

	mMagnifier->Delete();

	mMainRen->RemoveObserver(mStereoRenderObserver);
	mStereoRenderObserver->Delete();

	mMainRen->RemoveObserver(mRendererObserver);
	mRendererObserver->Delete();

	if(mLights != 0) mLights->Delete();
	if(mMainInt != 0) mMainInt->Delete();
	if(mMainCam != 0) mMainCam->Delete();

	mMainRen->SetRenderWindow(0);
	mMainWin->RemoveRenderer(mMainRen);

	mDualRen->Delete();

	mMainWin->RemoveAllObservers();   

	mMainRen->Delete();
	mMainWin->Delete();
}


int iRenderTool::GetRenderingMagnification() const
{
	return mMagnifier->GetMagnification();
}


int iRenderTool::GetNumberOfActiveViews() const
{
	return (mDualWin == 0) ? 1 : 2;
}


void iRenderTool::Render()
{
	mMainWin->InvokeEvent(vtkCommand::StartEvent,NULL);
	this->ResetCameraClippingRange();
	mMainWin->Render();
	mMainWin->InvokeEvent(vtkCommand::EndEvent,NULL);
}


void iRenderTool::ResetCamera()
{
	const double scale = 1.6/sqrt(3.0);

	int i;
	double bounds[6], fp[3];
	mMainRen->ComputeVisiblePropBounds(bounds);
	mMainRen->GetActiveCamera()->GetFocalPoint(fp);

	for(i=0; i<3; i++)
	{
		if(fabs(bounds[2*i+0]-fp[i]) > fabs(bounds[2*i+1]-fp[i]))
		{
			bounds[2*i+1] = 2*fp[i] - bounds[2*i+0];
		}
		else
		{
			bounds[2*i+0] = 2*fp[i] - bounds[2*i+1];
		}
	}
	mMainRen->ResetCamera(bounds);

	//
	//  IFrIT-style reset
	//
	mMainCam->GetDevice()->SetParallelScale(scale*mMainCam->GetDevice()->GetParallelScale());
	mMainCam->GetDevice()->Dolly(1.0/scale);

	mMainRen->ResetCameraClippingRange();
}


void iRenderTool::AddObserver(unsigned long event, iEventObserver *command)
{
	Observer tmp;

	tmp.Event = event;
	tmp.Command = command;
	mObservers.Add(tmp);

	mMainWin->AddObserver(event,command);
	if(mDualWin != 0) mDualWin->AddObserver(event,command);
}


void iRenderTool::UpdateWindowName(const iString &base)
{
	if(mDualWin == 0)
	{
		mMainWin->SetWindowName(base.ToCharPointer());
	}
	else
	{
		mMainWin->SetWindowName((base+": Left Eye").ToCharPointer());
		mDualWin->SetWindowName((base+": Right Eye").ToCharPointer());
	}
}


vtkRenderWindowCollection* iRenderTool::GetRenderWindowCollection()
{
	if(!mWindowCollectionUpToDate)
	{
		this->UpdateWindowCollection();
		mWindowCollectionUpToDate = true;
	}
	return mWindowCollection;
}


void iRenderTool::UpdateWindowCollection()
{
	mWindowCollection->RemoveAllItems();
	mWindowCollection->AddItem(mMainWin);
	if(mDualWin != 0) mWindowCollection->AddItem(mDualWin);
}


void iRenderTool::WindowsModified()
{
	mWindowCollectionUpToDate = false;
}


void iRenderTool::ResetCameraClippingRange()
{
	if(mAutoClippingRange) mMainRen->ResetCameraClippingRange();
}


void iRenderTool::SetBackground(const iColor &color)
{
	mMainRen->SetBackground(color.ToVTK());
	mDualRen->SetBackground(color.ToVTK());

	mMainBkg->SetColor(color);
	mDualBkg->SetColor(color);
}


void iRenderTool::SetBackground(const iImage &image)
{
	mMainBkg->SetImage(image);
	mDualBkg->SetImage(image);
}


void iRenderTool::AddObject(vtkProp* p)
{
	if(p == 0) return;
	
	if(p->IsA("vtkVolume") != 0)
	{
		mVolumeObjects->AddItem(p);
		mMainRen->AddViewProp(p);
		mDualRen->AddViewProp(p);
	}
	else
	{
		//
		//  Make sure all actors are ahead of volumes (is it a VTK bug?)
		//
		mActorObjects->AddItem(p);

		mMainRen->RemoveAllViewProps();
		mDualRen->RemoveAllViewProps();

		vtkProp *p;
		mActorObjects->InitTraversal();
		while((p = mActorObjects->GetNextProp()) != 0)
		{
			mMainRen->AddViewProp(p);
			mDualRen->AddViewProp(p);
		}
		mVolumeObjects->InitTraversal();
		while((p = mVolumeObjects->GetNextProp()) != 0)
		{
			mMainRen->AddViewProp(p);
			mDualRen->AddViewProp(p);
		}
	}
}


void iRenderTool::RemoveObject(vtkProp* p)
{
	if(p == 0) return;

	if(p->IsA("vtkVolume") != 0)
	{
		mVolumeObjects->RemoveItem(p);
	}
	else
	{
		mActorObjects->RemoveItem(p);
	}

	mMainRen->RemoveViewProp(p);
	mDualRen->RemoveViewProp(p);
}


const int* iRenderTool::GetRenderWindowSize() const
{ 
	return mMainWin->GetSize(); 
}


void iRenderTool::SetRenderWindowSize(int w, int h)
{ 
	mMainWin->SetSize(w,h);
	if(mDualWin != 0) mDualWin->SetSize(w,h);
}


const int* iRenderTool::GetRenderWindowPosition() const
{ 
	return mMainWin->GetPosition(); 
}


void iRenderTool::SetRenderWindowPosition(int x, int y)
{ 
	mMainWin->SetPosition(x,y);
}


void iRenderTool::SetAntialiasing(bool s)
{
	mAntialiasing = s;
	if(s)
	{
		mMainWin->SetLineSmoothing(1);
		mMainWin->SetPointSmoothing(1);
		if(mDualWin != 0)
		{
			mDualWin->SetLineSmoothing(1);
			mDualWin->SetPointSmoothing(1);
		}
    } 
	else 
	{
		mMainWin->SetLineSmoothing(0);
		mMainWin->SetPointSmoothing(0);
		if(mDualWin != 0)
		{
			mDualWin->SetLineSmoothing(0);
			mDualWin->SetPointSmoothing(0);
		}
 	}

#ifndef I_OFFSCREEN
	vtkOpenGLRenderWindow *w = vtkOpenGLRenderWindow::SafeDownCast(mMainWin);
	if(w!=0 && w->GetNeverRendered()==0)
	{
		w->MakeCurrent();
		w->OpenGLInit();
	}
	if(mDualWin != 0)
	{
		w = vtkOpenGLRenderWindow::SafeDownCast(mDualWin);
		if(w!=0 && w->GetNeverRendered()==0)
		{
			w->MakeCurrent();
			w->OpenGLInit();
		}
	}
#endif
}


void iRenderTool::SetAdjustCameraClippingRangeAutomatically(bool s)
{
	mAutoClippingRange = s;
}


void iRenderTool::SetBackgroundImageFixedAspect(bool s)
{
	if(mBackgroundImageFixedAspect != s)
	{
		mBackgroundImageFixedAspect = s;
		mMainBkg->SetImageFixedAspect(s);
		mDualBkg->SetImageFixedAspect(s);
	}
}


void iRenderTool::SetBackgroundImageTile(float tx, float ty, float tw, float th)
{
	float tile[4];

	tile[0] = tx;
	tile[1] = ty;
	tile[2] = tw;
	tile[3] = th;

	this->SetBackgroundImageTile(tile);
}


void iRenderTool::SetBackgroundImageTile(const float tile[4])
{
	mMainBkg->SetImageTile(tile);
	mDualBkg->SetImageTile(tile);
}


void iRenderTool::SetCameraClippingRange(double cr[2])
{
	//
	//  This is a dirty trick to avoid the minimum bound on the nearest clipping range of 0.0001
	//
	if(!mAutoClippingRange)
	{
		double *d = mMainRen->GetActiveCamera()->GetClippingRange();
		mClippingRange[0] = d[0] = cr[0];
		mClippingRange[1] = d[1] = cr[1];
	}
}


void iRenderTool::RenderImages(int mag, iStereoImageArray &images)
{
	iStereoImage tmp;
	
	images.Clear();
	this->RenderStereoImage(mag,tmp);
	images.Add(tmp);
}


void iRenderTool::RenderStereoImage(int mag, iStereoImage &image)
{
	int v = 0;

	mDualWindowObserver->Block(true);
	if(mStereoAlignmentMarksActor != 0)
	{
		v = mStereoAlignmentMarksActor->GetVisibility();
		mStereoAlignmentMarksActor->VisibilityOff();
	}

	mMagnifier->SetMagnification(mag);

	//
	//  Render main window
	//
	int lfc = mMainRen->GetLightFollowCamera();
	mMainRen->SetLightFollowCamera(0);  //  Maintain correct lighting under magnification
	mMagnifier->SetInput(mMainRen);
	mMagnifier->Modified();
	mMagnifier->UpdateWholeExtent();
	image.ReplaceData(0,mMagnifier->GetOutput());
	mMainRen->SetLightFollowCamera(lfc);

	//
	//  Render dual window
	//
	if(mDualWin != 0)
	{
		mMagnifier->SetInput(mDualRen);
		mMagnifier->Modified();
		mMagnifier->UpdateWholeExtent();
		mMagnifier->Update();
		image.ReplaceData(1,mMagnifier->GetOutput());
	}

	mMagnifier->SetMagnification(1);

	if(mStereoAlignmentMarksActor != 0)
	{
		mStereoAlignmentMarksActor->SetVisibility(v);
	}
	mDualWindowObserver->Block(false);
}


float iRenderTool::GetLastRenderTimeInSeconds() const
{
	float s = mMainRen->GetLastRenderTimeInSeconds();
	if(mDualWin != 0) s += mDualRen->GetLastRenderTimeInSeconds();
	return s;
}


void iRenderTool::GetAspectRatio(double ar[2]) const
{
	mMainRen->GetAspect(ar);
}


//
//  Stereo operations
//
void iRenderTool::ShowStereoAlignmentMarks(bool s)
{
	mStereoAlignmentMarksOn = s;
	mStereoAlignmentMarksActor->SetVisibility((s && mDualWin!=0)?1:0);
}


void iRenderTool::SetStereoMode(int m)
{
	if(m==mStereoMode || m<0) return;

	if(mStereoMode == 1) this->ShowDualWindow(false);
	if(mStereoMode == 2)
	{
		//
		//  This seems extraneous, but that trick was needed to avoid flicker
		//  that Doug was complaining about. It was debugged on Robert's laptop.
		//
		this->SetMainWinStereoRender(false);
		mMainWin->Render();
	}
	mStereoMode = m;

	//
	//  Don't get stuck in a right-eye-only mode, it is not renderable in the non-stereo mode
	//
	mMainWin->SetStereoType(0);

	switch(mStereoMode)
	{
	case 0:
		{
			this->SetMainWinStereoRender(false);
			break;
		}
	case 1:
		{
			this->ShowDualWindow(true);
			break;
		}
	case 2:
		{
			if(mMainWin->GetStereoCapableWindow() == 0)
			{
				this->SetStereoMode(0);
				iConsole::Display(iConsole::_Notification,"Crystal Eyes mode is not supported on this machine.");
			}
			else
			{
				mMainWin->SetStereoType(VTK_STEREO_CRYSTAL_EYES);
				this->SetMainWinStereoRender(true);
			}
			break;
		}
	default:
		{
			mMainWin->SetStereoType(mStereoMode-mStereoModeOffset);
			mStereoMode = mMainWin->GetStereoType() + mStereoModeOffset;
			this->SetMainWinStereoRender(true);
			break;
		}
	}
}


void iRenderTool::ShowDualWindow(bool s)
{
	if(s)
	{
		if(mDualWin == 0) // DualWindow is not shown yet
		{
			mDualWin = iShellFactory::CreateRenderWindow(this); IERROR_ASSERT(mDualWin);
			mDualWin->SetStereoCapableWindow(0);
			mDualWin->SetStereoTypeToRight();
			mDualWin->StereoRenderOn();
			int *s = mMainWin->GetSize();
			mDualWin->SetSize(s[0],s[1]);
			this->WindowsModified();

			int i;
			for(i=0; i<mObservers.Size(); i++)
			{
				mDualWin->AddObserver(mObservers[i].Event,mObservers[i].Command);
			}
			//
			//  Copy RenderWindows
			//
			mDualWin->SetLineSmoothing(mMainWin->GetLineSmoothing());
			mDualWin->SetPointSmoothing(mMainWin->GetPointSmoothing());

			mDualWin->SetNumberOfLayers(2);
			mDualWin->AddRenderer(mDualBkg->GetRenderer());
			mDualWin->AddRenderer(mDualRen);

			mMainWin->SetStereoTypeToLeft();
			this->SetMainWinStereoRender(true);
			mMainWin->AddObserver(vtkCommand::StartEvent,mDualWindowObserver);
			mMainWin->AddObserver(vtkCommand::EndEvent,mDualWindowObserver);
			mMainWin->AddObserver(vtkCommand::ModifiedEvent,mDualWindowObserver);

			if(mStereoAlignmentMarksActor!=0 && mStereoAlignmentMarksOn) mStereoAlignmentMarksActor->VisibilityOn();
			this->GetViewModule()->UpdateWindowNumber();
		}
	}
	else
	{
		if(mDualWin != 0)
		{
			this->WindowsModified();
			mDualRen->SetRenderWindow(0);
			mDualBkg->GetRenderer()->SetRenderWindow(0);
			mDualWin->RemoveRenderer(mDualRen);
			mDualWin->RemoveRenderer(mDualBkg->GetRenderer());
			mDualWin->Delete();
			mDualWin = 0;

			this->SetMainWinStereoRender(false);
			mMainWin->RemoveObserver(mDualWindowObserver);

			if(mStereoAlignmentMarksActor != 0) mStereoAlignmentMarksActor->VisibilityOff();
			this->GetViewModule()->UpdateWindowNumber();
		}
	}
}


iRenderTool* iRenderTool::CreateInstance(vtkRenderer *ren) const
{
	return new iRenderTool(this->GetViewModule(),ren,mMagnifier);
}


vtkRenderer* iRenderTool::CreateRenderer() const
{
	vtkRenderer *ren = vtkRenderer::New();
	if(ren == 0) return 0;

	ren->SetLayer(IRENDERTOOLBACKGROUND_FRONT_LAYER);

	//
	//  Copy the state of the primary renderer
	//
	ren->SetBackingStore(mMainRen->GetBackingStore());
	ren->SetTwoSidedLighting(mMainRen->GetTwoSidedLighting());
	ren->SetBackground(mMainRen->GetBackground());
	ren->LightFollowCameraOff(); // only the primary renderer can control lights

	//
	//  Copy props. The trick is that we need to register the new renderer with every iActor
	//  so that iActors can create special mappers for this renderer. Otherwise, vtkPolyDataMappers 
	//  remember the last window they rendered into, and will reset at switching to a different window.
	//
	vtkPropCollection *pc = mMainRen->GetViewProps();
	vtkProp *p;
	pc->InitTraversal();
	while((p = pc->GetNextProp()) != 0)
	{
		ren->AddViewProp(p);
	}

	//
	//  Copy lights
	//
	vtkLight *l;
	vtkLightCollection *lc = mMainRen->GetLights();
	lc->InitTraversal();
	while((l = lc->GetNextItem()) != 0) ren->AddLight(l);

	//
	//  Copy cullers
	//
	vtkCuller *c;
	vtkCullerCollection *cc = mMainRen->GetCullers();
	cc->InitTraversal();
	ren->GetCullers()->RemoveAllItems();
	while((c = cc->GetNextItem()) != 0) ren->AddCuller(c);

	return ren;
}


int iRenderTool::GetFullScreenMode() const
{
	return 1;
}


void iRenderTool::CopyBackground(const iRenderTool *source)
{
	if(source != 0)
	{
		mMainBkg->Copy(source->mMainBkg);
		mDualBkg->Copy(source->mDualBkg);
	}
}


void iRenderTool::SetFontScale(int s)
{
	if(s>-10 && s<10)
	{
		mFontScale = s; 
	}
}


void iRenderTool::SetFontType(int s)
{
	if(TextType::IsValid(s))
	{
		mFontType = s; 
	}
}


void iRenderTool::SetMainWinStereoRender(bool s)
{
	mMainWinStereoRender = s;
	mMainWin->SetStereoRender(s?1:0);
}

