/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2011 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iconfigure.h"
#if ISHELL_INCLUDED(ISHELL_GG)


#include "ibgwidgettrackballsubject.h"


#include "icolor.h"
#include "icontrolmodule.h"
#include "ierror.h"
#include "imath.h"
#include "ishell.h"
#include "itransform.h"
#include "iviewmodule.h"
#include "ivtk.h"

#include "iggwidget.h"
#include "iggwidgetrendermodebutton.h"

#include <vtkCamera.h>
#include <vtkMath.h>
#include <vtkRenderer.h>


namespace ibgWidgetTrackBallSubject_Private
{
	const float PI = 3.1415927535f;
	const float TWOPI = 2*PI;

	unsigned char* GetData();
};


using namespace ibgWidgetTrackBallSubject_Private;

//
// Constructs a iTrackball.
//
ibgWidgetTrackBallSubject::ibgWidgetTrackBallSubject(iggWidget *owner, bool followCamera, iggWidgetRenderModeButton* rmb) : ibgWidgetSubject(owner), mFixedSize(64), mEarth(EarthData(256,128,GetData()))
{
	mOldX = mOldY = mOldRad = 0;
	mFollowCamera = followCamera;
	mDoubleBuffer = true;
	mResizable = false;

	mTransform = iTransform::New(); IERROR_ASSERT(mTransform);
	mBallTransform = iTransform::New(); IERROR_ASSERT(mBallTransform);
	mTransform->Identity();
	mBallTransform->Identity();

	mLightDirection[0] = 0.0;
	mLightDirection[1] = 0.0;
	mLightDirection[2] = sqrt(1.0-vtkMath::Dot2D(mLightDirection,mLightDirection));

	if(rmb != 0) rmb->ClearLaidOutFlag(this);
}


ibgWidgetTrackBallSubject::~ibgWidgetTrackBallSubject()
{
	mTransform->Delete();
	mBallTransform->Delete();
}


void ibgWidgetTrackBallSubject::SetDirection(float x, float y, float z)
{
	mTransform->Identity();
	mTransform->SetDirection(x,y,z);
}


void ibgWidgetTrackBallSubject::GetDirection(float *d)
{
	static const float base[3] = { 0.0, 0.0, 1.0 };
	mTransform->TransformVector(base,d);
}


void ibgWidgetTrackBallSubject::UpdateTransform(int newX, int newY)
{
	vtkTransform *ct = 0;
	int w0, h0, rad, cor;
	this->GetSizes(w0,h0,rad,cor);

	int rr = rad*rad;
	int dx0 = mOldX - w0;
	int dy0 = h0 - mOldY;
	int dx1 = newX - w0;
	int dy1 = h0 - newY;
	mOldX = newX;
	mOldY = newY;

	int rr0 = dx0*dx0 + dy0*dy0;
	int rr1 = dx1*dx1 + dy1*dy1;
	if(rr0>rr || rr1>rr) return;
	
	float lon0 = atan2(float(dy0),float(dx0));
	float lat0 = asin(sqrt(float(rr0)/float(rr)));
	
	float lon1 = atan2(float(dy1),float(dx1));
	float lat1 = asin(sqrt(float(rr1)/float(rr)));

	float dir[3], dir0[3], dir1[3], axis[3], d;

	//
	//  Old and new directions in camera frame
	//
	dir0[0] = sin(lat0)*cos(lon0);
	dir0[1] = sin(lat0)*sin(lon0);
	dir0[2] = cos(lat0);

	dir1[0] = sin(lat1)*cos(lon1);
	dir1[1] = sin(lat1)*sin(lon1);
	dir1[2] = cos(lat1);

	this->GetDirection(dir);
	if(!mFollowCamera)
	{
		ct = mOwner->GetShell()->GetControlModule()->GetViewModule()->GetRenderer()->GetActiveCamera()->GetViewTransformObject();
		ct->TransformVector(dir,dir); // transform into camera frame
	}

	vtkMath::Cross(dir1,dir0,axis);
#ifdef IVTK_PRE54
	d = vtkMath::RadiansToDegrees()*acos(vtkMath::Dot(dir0,dir1));
#else
	d = vtkMath::DegreesFromRadians(acos(vtkMath::Dot(dir0,dir1)));
#endif
	mTransform->Identity();
	mTransform->RotateWXYZ(-d,axis);
	mTransform->TransformVector(dir,dir);

	mBallTransform->RotateWXYZ(d,axis);

	if(!mFollowCamera)
	{
		ct->GetLinearInverse()->TransformVector(dir,dir);  // from camera into world frame
	}

	this->SetDirection(dir[0],dir[1],dir[2]);
	this->Repaint();
}


const iColor& ibgWidgetTrackBallSubject::ColorAtLonLat(float lon, float lat)
{
	static const iColor water(100,100,255);
	static const iColor grass(100,255,50);
	static const iColor ice(250,250,240);
	static const iColor sand(200,200,0);
	int i,j;

	i = round(mEarth.NumLon*(0.5+lon/TWOPI));
	if(i < 0) i += mEarth.NumLon;
	if(i >= mEarth.NumLon) i -= mEarth.NumLon;

	j = round(mEarth.NumLat*(0.5+lat/PI));
	if(j < 0) j += mEarth.NumLat;
	if(j >= mEarth.NumLat) j -= mEarth.NumLat;
	vtkIdType l = i + mEarth.NumLon*j;
	if(mEarth.Data[l] < 85) 
	{
		return water;
	} 
	else if(mEarth.Data[l] < 230)
	{
		return grass;
	} 
	else 
	{
		if(fabs(lat) > 0.7) 
		{
			return ice;
		} 
		else 
		{
			return sand;
		}
	}
}


namespace ibgWidgetTrackBallSubject_Private
{
	unsigned char* GetData()
	{
		static unsigned char data[] = 
		{
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,188,184,130,198,213,171,
			166, 84,113, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,147,193,
			231,250,220,205,233,248,236,239,205,226,236,224,226,239, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84,209,218,236,248,233,243,241,243,241,242,238,241,239,226,213,205,
			218,215,209,188,130, 84,226,242,218,218,212,130,236,231,226,222,205,
			180,243,243,246,248,248,248,248,248,188,233,246,252,252,226,198,250,
			254,130, 84, 84,231,236,198, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84,147,192, 84, 84,193, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,226,
			231,243,212,194, 84, 84,154,154, 84,175,209,236,241,243,241,246,238,
			202, 84, 84,249,236,233,202,202,243,246,246,246,246,246,248,248,246,
			248,246,246,246,248,248,248,248,248,248,248,246,248,248,248,248,241,
			243,248,243,241,246, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,162,156, 84,171, 84,
			130, 84, 84, 84, 84,171, 84, 84, 84, 84, 84, 84, 84, 84, 84,171,162,
			202,130, 84,222,177,147,239,184,177, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,213,248,248,
			243, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84,130,113, 84, 84, 84, 84,212,217,154, 84,171,
			84, 84,223,246,243,222,147,187,187,171,209,246,248,241,241,166, 84,
			84,147,209,205,246,246,246,246,246,246,246,248,248,246,248,248,248,
			246,246,248,248,248,248,248,248,248,248,248,248,248,248,248,246,248,
			248,249,248, 84,130, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84,113,246,238,190,234,246,254,218, 84,
			84, 84,175,113,113, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84,198,188,224,243,249,162, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84,130,171,113, 84,187,187, 84, 84, 84,207, 84, 84, 84, 84, 84,
			84,175,171, 84, 84,209, 84,190,197,162,193,171,239,239, 84, 84, 84,
			84,177, 84,188,202,208,246,246,248,246,246,246,246,248,248,248,248,
			248,248,246,248,246,248,248,248,248,248,248,248,248,248,248,248,248,
			248,248,249,243,162, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,234,246,162, 84,175,
			84,166, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84,130,211,202,113, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			223,190,224,113, 84,201,205,184, 84,143,201, 84, 84,162,194,209,184,
			201, 84, 84,113, 84,209,175, 84, 84, 84, 84, 84, 84,154,177, 84, 84,
			84, 84, 84, 84, 84,231,130,188, 84,130,236,250,246,246,246,246,246,
			246,246,246,246,246,246,246,248,248,248,248,248,248,248,248,248,248,
			246,249,248,246,231,113,154, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,184,241,234,254,252,188,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84,154,179,113,179,184,199,201,201,201,206,199,206,201,201, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,198,
			84,175, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			113,159,162, 84, 84, 84, 84,175,198,143, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84,130,130, 84,113,154, 84,202,194,187, 84,209,156, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,213,243,241,
			243,246,246,246,246,246,248,248,248,248,248,248,248,248,248,248,248,
			248,248,249,246,239,241,231,113, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84,196,250,207, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,113,187,199,201,
			194,201,196,201,196,201,191,179,178,183,191,173,178,183,154,170, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84,196,196,196,196,201,175,179,209, 84, 84, 84, 84,194,194, 84, 84,
			84,193,206,154,143,212,209,194, 84,209,212, 84,187,209,219,137,248,
			254, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,113,228,
			246,246,246,246,246,246,246,246,246,248,246,248,248,248,248,248,248,
			249,248,224,228,228,156,113, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84,154,226,205, 84, 84, 84, 84, 84, 84,
			84, 84, 84,193,154, 84,137, 84, 84, 84, 84,180,201,199,201,191,188,
			194,194,193,190,188,190,190,191,188,193,190,190,154,143,194,193,190,
			188,190,197,197,113, 84, 84,171,191,179,188, 84, 84, 84, 84, 84, 84,
			84, 84, 84,113,165, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84,113,199,196,199, 84,206,199,199,218,212,199,207,209,201,205, 84,
			84, 84,162,212,205, 84,154, 84, 84, 84,212,212,113,196,206,209,199,
			197,207,237,236, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			190,218,239,243,246,246,248,248,248,248,246,246,246,248,248,248,248,
			248,248,226,239,246,249,177, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84,194,231,194, 84, 84, 84, 84, 84, 84,
			84, 84, 84,196,194,190, 84,187, 84,160,178,186,130,159,186,178,193,
			194,194,193,194,193,193,193,190,187,188,179,187,193,201,196,196,193,
			193,193,190,190,190,190,190,190,190,190,196,193, 84, 84, 84, 84, 84,
			84, 84, 84,191,194,193,194,143,190,190, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,130,194,191,188,191,187,
			162, 84,113, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			165, 84, 84, 84, 84, 84, 84, 84,113,113, 84,165,199,199,193,205,199,
			199,171, 84, 84, 84, 84,196,201,215, 84, 84,177,191,205,196,193,196,
			191,147,206,212,215,224,226,231,230, 84, 84, 84, 84, 84, 84, 84, 84,
			84,143,218,228,180,246,239,246,248,246,248,246,246,248,248,248,248,
			248,249,249,188,224,147,113,205, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84,154,175,179,113,162,193,113, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,113, 84,130, 84, 84,
			84, 84, 84,194,190,190,194, 84,190,190,183,193,193,188,175,190,194,
			193,194,193,193,193,193,190,190,190,186,193,194,187,190,193,193,190,
			193,190,194,190,190,187,190,194,190,190,190,188,190,201,201,193,194,
			197,194,193,190,193,190,190,190,185,183,190,190,190,190,190,193,190,
			193, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,175, 84,
			84, 84, 84, 84, 84, 84, 84, 84,130,162,194,191,193,190,193,190,190,
			193,194,188,190,193,205,209,199,193,186,186,181,147,191,178,178,181,
			178,178,173,188,193,193,199,199,190,165,183, 84,196,196,196,162, 84,
			113,165,179, 84,130,199,198,162,178,191,193,113,199, 84, 84,193,209,
			179, 84,187, 84,185,170,196,215,196,202, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84,183,243,248,243,243,248,246,246,246,248,246,249,
			250,239,243,243,246,239,113, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84,143, 84,193,193,186,190,193,194,183,175,173,173,173,173,198,180,
			84, 84, 84, 84, 84, 84, 84, 84,190, 84, 84,143, 84, 84, 84,171, 84,
			190,190,190,190, 84,162,188,193, 84, 84,162,154,194,190,178,190,193,
			190,190,201,190,188,204,178,199,186,193,190,190,187,187,179,187,183,
			185,190,190,190,190,179,183,183,179,187,179,183,188,190,193,193,193,
			186,183,187,190,186,197,202,194,183,179,178,178,178,178,183,183,179,
			178,175,188,178,190,190,193,193,194, 84,190,183,193,196,190,193,193,
			183,188,171, 84, 84, 84, 84, 84, 84, 84, 84,190,190,190,190,190,193,
			194,190,187,190,190,186,178,173,173,176,176,180,176,186,180,178,178,
			178,173,170,175,170,170,178,196,191,191,188,178,188,185,191,196,143,
			196,196,196,196,188,196,191,186, 84,191,194,194,194,201, 84,196,201,
			196,190, 84, 84, 84,193, 84, 84,191,201,209,231,239,239, 84, 84, 84,
			84, 84, 84, 84,179,183,186,248,243,243,246,246,246,248,248,246,248,
			249,250, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84,165,191,194,186,178,178,178,178,178,173,173,175,175,193,170,
			170,188,190,190, 84, 84,190, 84, 84,183,187,190,190,190,188,190,190,
			188,190,186,190,196,187,194,190,191, 84,143,190,170,154,178,183,190,
			190,194,193,186,186,178,178,186,178,183,178,178,186,184,187,183,179,
			187,186,183,183,187,183,183,187,187,183,180,187,179,190,190,193,186,
			187,186,183,183,183,201,190,193,183,188,183,170,183,183,183,180,178,
			183,183,190,190,183,183,190,190,193,196,206,194,194,186,199,201,201,
			201,165,193,196,190,187,194,180, 84, 84,113,178, 84, 84,180,173,181,
			186,178,173,170,186,178,175,170,170,170,173,183,188,173,176,176,176,
			173,176,176,178,178,170,179,178,178,179,178,175,178,183,196,183,196,
			188,193,193,193,191,188,191,188,191,191,194,194,194,201,191,196, 84,
			170,137, 84, 84, 84, 84, 84, 84,196,207,154,191,179,183,207,250,171,
			84, 84, 84, 84, 84,175,237,222,243,241,246,246,248,248,248,248,231,
			239,188, 84, 84, 84, 84, 84, 84, 84, 84,184, 84, 84, 84,130,171, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84,113,179,194,187,173,178,170,183,183,183,178,183,173,175,178,162,
			84,130,170,170, 84,183,154,183,178,183,183,186,186,190,190,187,187,
			187,185,190,194,179,183,179,190,165,160,190,194,190,190,190,183,190,
			190,187,183,180,190,190,193,190,178,170,175,175,179,183,183,179,179,
			183,183,187,183,183,179,179,183,180,179,175,194,175,175,187,190,194,
			183,183,183,183,187,190,190,193,190,175,184,201,190,179,179,183,183,
			190,183,183,186,183,190,187,190,190,190,193,190,190,186,190,188,190,
			196,113, 84, 84, 84,196,198, 84, 84, 84, 84,196,193,188,186,181,190,
			173,170,178,170,170,173,170,170,178,183,186,178,175,193,191,191,196,
			193,196,196,193,188,178,170,170,178,175,173,173,178,178,179,178,185,
			188,188,193,190,196,188,191,191,196,188,188,191,188,191,193,191,171,
			201,194,193, 84, 84, 84,194,191,179,199,191,194,194,209,113, 84,177,
			84, 84, 84, 84, 84, 84, 84,113,175,243,242,243,248,249,249,166, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84,130,180,197,215,246,248,193,
			186, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84,130,194,193,183,173,173,173,170, 84, 84,148,178,183,183,175,183,
			173,113,130,148, 84,170,170,178,178,183,178,178,187,183,183,179,183,
			187,186,190,178,178,179,170,170,183,183,178,178,178,183,175,170,178,
			183,175,178,178,173,180,170,183,183,170,170,175,178,178,175,178,179,
			183,178,178,180,179,179,175,179,183,178,175,179,179,178,175,179,194,
			197,193,193,194,190,178,190,187,193,190,183,190,198,178,186,179,183,
			178,178,181,178,190,178,178,186,186,190,190,191,191,191,188,188,190,
			175,193, 84, 84, 84, 84, 84, 84,148,170,143, 84, 84, 84, 84,183,176,
			183,173,170,173,170,178,201,193,196,224,230,186,178,173,178,173,175,
			178,183,178,186,193,196,191,187,170,178,175,178,179,178,187,183,175,
			175,170,185,180,191,186,191,178,191,193,190,193,188,188,191, 84, 84,
			84, 84,148, 84, 84, 84, 84,170, 84, 84, 84,165,196,207, 84,212,171,
			84, 84, 84, 84, 84, 84, 84, 84, 84,194,226,246,246,246,249,246, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			130,186,179,183,175,178,183,178, 84, 84,170,175,175,178,179,175,178,
			183,179,175,178,179,178,178,178,175,180,179,178,183,178,179,179,178,
			178,175,186,178,178,179,175,178,178,178,179,179,178,178,178,178,170,
			178,187,183,179,180,178,178,170,183,187,178,187,178,183,178,178,180,
			178,178,179,178,178,180,178,179,178,178,178,178,178,181,178,178,178,
			179,179,178,179,179,178,183,187,194,193,186,186,193,193,178,186,187,
			183,187,170,183,193,196,186,183,188,190,190,193,186,193,190,186,190,
			190,190,190, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,178,180,186,
			183,186,173,173,183,183,181,170,207,196,175,175,236,207,183,183,183,
			178,183,193,170,175,190,209,193,191,173,173,170,173,170,170,178,178,
			179,183,187,183,178,186,191,188,186,188,190,188,188,188,130, 84, 84,
			84, 84, 84, 84, 84, 84,165, 84,191,199,196,201, 84, 84, 84, 84,212,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,175,243,243,243,248,231,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84,130, 84, 84, 84, 84, 84, 84, 84, 84,168,
			183,196,193,183,178,173,178,175, 84, 84, 84,170,173,178,178,179,185,
			175,183,183,179,175,183,179,183,178,183,178,175,175,175,178,179,179,
			170,178,178,175,178,175,183,178,178,175,183,178,175,175,178,183,183,
			183,178,178,178,178,183,178,179,178,178,178,173,183,181,183,176,178,
			178,179,179,178,181,181,176,178,179,178,178,183,178,178,181,181,173,
			183,178,173,173,178,173,183,178,198,196,193,201,208,201,194,183,183,
			186,186,186,183,188,196,197,175,193,186,193,190,190,190,194,196,190,
			190, 84, 84, 84, 84, 84, 84, 84, 84,113, 84, 84, 84, 84, 84,183,193,
			187,193,188,194,186,178,209,137,173,199, 84,130,236,236,231,236,246,
			188,183,186,175,178,186,173,173,173,178,173,170,175,173,173,178,175,
			170,173,179,175,175,170,175,174,178,178,186,193,190,178, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84,179,191,188,190,191,193, 84, 84,
			84,113, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,159, 84,187,231,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			165,190,187,173,175,170,179,178,175, 84, 84, 84,181,181,180,178,178,
			183,175,175,176,176,178,178,178,183,173,178,179,175,173,173,175,175,
			175,179,170,178,180,178,179,183,178,186,178,183,178,180,178,175,178,
			178,178,178,180,183,178,178,176,178,178,170,179,178,173,170,170,178,
			183,178,178,183,173,173,178,178,173,170,178,170,170,178,183,183,183,
			183,183,183,178,173,178,173,180,181,176,201,176,193,181,193,190,186,
			181,196,193,183,186, 84, 84, 84,148, 84,179,194,193,199,175,190,162,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84,154,175,193,193,193,193,113, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84,222,228,170,196,178,188,178,173,173,173,170,170,175,170,170,170,
			178,173,170,170,175,185,170,175,170,183,179,187,185,190,175, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84,193,193,193,190,188,193,113,
			84, 84,191,183, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,137, 84, 84, 84, 84,
			84, 84,194,178,159,160,170,178,175,175,154, 84, 84,113,181,181,173,
			176,178,183,176,173,178,181,178,176,176,173,173,170,178,170,183,178,
			175,178,183,183,180,183,175,173,173,178,178,183,175,178,178,178,173,
			178,178,178,178,178,183,173,178,180,178,178,176,176,176,176,176,173,
			176,173,176,181,181,181,173,173,170,170,183,187,178,176,173,178,173,
			178,181,173,176,173,176,173,183,178,178,181,181,181,181, 84, 84,113,
			84, 84, 84,178,165,113, 84, 84, 84,191,199,168,179, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84,170,188, 84,175, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84,170,148,199,170,173,173,183,173,173,170,173,170,170,
			170,173,170,175,175,170,183,179,179,180,175,175,183,187,183,190,190,
			178, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,190,190,193,194,186,
			179,190,190,194,193,190, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84,130,113,165,148, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84,175,170,170, 84, 84, 84, 84,154, 84,181,
			176,176,176,173,178,181,181,181,178,173,178,178,178,165,178,178,178,
			186,183,170,183,173,183,170,181,178,178,178,181,181,181,178,176,173,
			173,178,173,176,173,178,173,173,173,173,170,170,176,173,176,170,170,
			176,176,176,173,181,176,176,173,173,176,178,176,176,178,178,188,190,
			178,176,187,186,183,181,178,190,181,176,178,178,188, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84,170,194,190,196,201, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84,183, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84,143,154,209,180,170,170,170,170,173,173,173,
			170,170,170,173,170,173,173,175,175,170,175,175,175,170,179,178,183,
			178,183,187,183,178, 84, 84, 84, 84, 84, 84,148, 84,183,188,188,190,
			183,179,179,178,190,190,170, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,154,181,170, 84, 84,
			84, 84, 84, 84, 84,181,187, 84, 84,178,175, 84, 84, 84,170,173,179,
			173,176,175,170,173,176,181,181,181,176,176,181,176,178,173,183,173,
			178,183,178,181,170,181,175,178,173,181,183,181,181,181,187,179,181,
			187,187,181,185,181,181,181,173,170,170,181,173,181,170,181,170,187,
			176,170,170,170,173,173,170,170,173,198,193,178,193,173,178,186,186,
			181,183,183,176,173,181,188,178,178,181,181,181,185, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84,196,193,198,186,193,179, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84,154,113,113, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84,178,159,183,173,170,173,170,170,173,
			173,173,173,170,170,173,173,175,175,173,175,175,175,175,179,175,178,
			175,175,183,175,179,179,183,179,180,179, 84, 84, 84,178,178,179,188,
			190,190,179,190,190,183,186,194,187,175, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,168, 84,187,181,181,
			84, 84, 84, 84, 84, 84, 84,154,130, 84, 84, 84, 84, 84, 84, 84,181,
			181,178,185,181,181,181,178,181,176,181,176,181,181,170,176,181,181,
			181,178,183,181,181,181,178,170,176,178,181,185,181,179,181,181,181,
			181,187,187,181,180,180,187,181,181,187,181,173,170,181,181,170,176,
			170,170,170,170,181,170,173,170,178,173,179,188,181,181,176,176,183,
			180,178,181,173,173,175,173,173,178,181,178,178, 84,176, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84,183,193,173,193,198, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84,154, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84,178, 84,170,170,170,175,170,170,
			173,173,170,173,173,176,181,179,181,179,179,176,176,170,173,175,178,
			170,173,170,170,173,178,178,175,179,179,179, 84, 84,175,183,175,178,
			178,183,190,190,175,183,179,178,186,178,186,170,170, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,176,187,183, 84,130,170,
			181,173, 84, 84, 84, 84,137,160,176,181,181,187,181,181,181,181,181,
			181,185,181,188,187,181,187,183,187,183,187,187,187,181,181,178,188,
			176,181,178,181,181,181,176,181,181,185,176,181,181,185,181,176,181,
			187,185,200,187,187,188,181,181,181,181,181,187,173,170,170,187,179,
			173,181,193,181,185,173,181,176,176,170,178,170,181,181,173,178,175,
			173,173,170,173,173,173,173,173,175,173,173,178,193,173,165,181,173,
			148, 84, 84, 84, 84, 84, 84, 84, 84, 84,154,193,198, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,147,187,113, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,113, 84,159,170,175,175,
			170,170,175,170,181,170,176,185,181,181,183,183,181,181,173,176,179,
			173,175,173,173,170,173,173,178,175,175,178,183,178, 84, 84,178,179,
			179,178,170,178,178,178,178,180,178,178,170,178,173,173, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,165,187,137, 84,181,
			181,187,187,181, 84, 84,181,181,181,181,176,176,176,176,181,188,181,
			181,181,187,187,187,181,185,183,187,187,188,188,188,187,181,181,181,
			181,181,181,181,181,181,181,187,187,170,179,181,187,187,187,185,181,
			187,181,187,187,187,185,187,183,181,181,181,183,181,170,173,173,188,
			173,181,188,173,194,193,193,181,173,170,181,176,176,176,181,176,173,
			181,173,173,173,173,173,170,173,173,176,173,173,188,181,176,181,181,
			173,188,130, 84, 84, 84, 84, 84, 84, 84, 84, 84,178, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,165,147,178,
			198,178,178,170,178,180,183,179,181,187,183,188,185,187,187,176,176,
			170,179,181,173,173,170,170,175,173,173,170,178,170,178,178,179,178,
			173,178,183,178,175,173,175,173,178,173,178,170,160, 84,173, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			187,165,165,160, 84,181,187,181,181,181,181,181,181,181,183,181,176,
			181,181,176,183,188,187,188,187,187,188,187,183,188,181,181,176,181,
			187,181,187,187,187,187,187,185,187,187,181,188,181,187,193,187,193,
			193,181,187,187,181,187,179,187,181,170,187,170,181,176,173,181,193,
			181,196,188,173,176,193,193,187,187,185,187,176,170,173,178,173,178,
			187,183,173,176,173,173,173,173,173,173,176,173,173,173,181,165,178,
			180,130, 84,181, 84, 84, 84, 84, 84, 84, 84, 84,143, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,113,
			170, 84,170,173,181,170,175,170,173,181,187,183,187,187,183,181,181,
			181,176,176,181,178,175,175,175,170,175,170,170,170,178,170,170,170,
			173,170,175,178,170,175,183,170, 84, 84,165,130, 84, 84,170,170,113,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84,187,113,181,187,187,181,181,181,181,176,181,181,181,176,
			176,181,181,176,187,181,187,187,181,181,178,187,188,181,187,181,188,
			183,183,187,188,187,187,188,187,185,188,185,193,193,193,195,193,193,
			193,188,193,193,188,185,185,188,188,193,191,187,187,181,181,183,193,
			193,193,207,188,196,196,196,193,187,181,176,181,179,173,176,176,181,
			187,183,183,187,181,181,173,173,176,173,173,173,173,173,170,185,181,
			170,181,130, 84,178, 84, 84, 84, 84, 84, 84, 84,154, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84,175,143,178,187,183,178,170,179,185,204,188,181,206,187,187,
			185,181,181,181,176,176,173,176,175,173,175,172,179,173,173,175,170,
			170,170,170,170,170,170,173,173,170,165,130, 84, 84, 84,173,183,178,
			170, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84,165,187,187,187,187,181,181,187,178,181,181,181,176,176,176,
			185,181,176,181,181,176,176,187,183,183,181,187,183,185,185,181,181,
			181,185,187,187,185,191,208,206,187,194,193,193,198,193,196,196,205,
			193,193,193,193,193,193,193,193,193,193,188,193,188,187,178,203,196,
			181,198,203,208,221,194,192,200,187,187,193,187,193,193,192,187,192,
			187,187,193,192,180,176,176,173,176,176,181,173,170,173,181,185,179,
			170,170,170, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84,143,175,178,200,193,187,170,183,180,183,187,187,193,187,
			187,187,187,181,183,183,181,176,176,170,173,170,175,170,178,170,173,
			170,173,173,170,170,170,173,170,173,181,178,165,154,130,170, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84,181,187,181,181,181,176,173,209,188,176,181,181,
			176,176,181,187,181,181,181,181,183,187,187,137,173,187,187, 84,160,
			181,181,185,187,187,187,196,183,175,178,200,216,199,193,192,178,197,
			188,187,191,192,192,196,193,203,187,191,196,178,192,183,187,176,207,
			203,213,200,196,203,200,211,200,202,200,193,193,197,192,192,199,192,
			192,192,187,196,193,188,187,176,183,185,188,181,173,173,181,185,173,
			173,170,170, 84, 84,165,137, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84,137,170,193,193,183,178,175,170,176,170,183,183,187,
			193,193,188,183,187,181,187,187,179,176,181,176,176,179,175,173,175,
			175,175,173,170,173,176,183,176,173,173,170,130,148,170,178, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84,181,181,176,181,181,188,181,181,176, 84,
			188,181,181,181,178,176,176,176,176,183,185,185, 84, 84,178,183,137,
			187,181,176,187,187,187,192,175,178,179,220,217,217,209,209,179,175,
			209,217,191,226,209,200,209,208,211,207,221,203,181,188,191,198,203,
			217,220,227,212,211,209,209,208,193,199,209,224,217,217,213,211,217,
			199,199,192,193,193,193,193,187,179,187,193,183,181,181,173,179,180,
			181,178,173, 84, 84, 84,191, 84, 84, 84, 84,113, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84,178,178,178,188,193,185,176,181,200,176,178,193,
			187,193,193,193,192,183,181,187,187,187,187,176,181,188,175,173,176,
			181,173,176,178,178,173,170,181,173,173, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84,148,160, 84, 84,113,181,183,181,160,179,170, 84, 84,176,
			176, 84, 84,176,181,176,181,181,187,181,176,187, 84, 84, 84, 84, 84,
			84, 84, 84,181,176,187,181,181,175,172,172,200,211,209,213,224,196,
			223,223,215,213,215,203,185,213,196,196,179,193,187,200,204,192,181,
			186,195,187,188,181,198,198,209,212,212,209,211,209,193,196,211,216,
			213,209,192,195,195,196,185,187,192,183,187,187,181,176,173,173,176,
			181,173,176, 84, 84, 84, 84,187,170,188,113, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84,178,175,178,198,193,204,202,187,187,187,216,
			213,193,181,188,187,192,187,181,187,181,187,187,183,173,181,175,173,
			176,178,178,178,176,176,180,173,173,173, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84,176,176,187,193,181,196,173,176,176, 84, 84, 84, 84,130,
			84,170,176, 84, 84, 84,178,181,176,181,181,181, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84,181,170,173,188,181,171,172,172,223,223,220,217,
			186,212,221,217,217,215,212,187,176,193,193,188,188,179,201,209,208,
			207,206,205,199,212,209,211,216,216,217,217,220,209,211,211,209,216,
			217,211,220,208,208,192,191,192,183,187,192,187,187,181,173,173,173,
			113, 84, 84, 84, 84, 84, 84, 84, 84,183, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84,170,170,178,198,211,198,198,213,193,176,
			187,216,170,178,188,187,187,181,179,181,181,176,176,181,181,181,183,
			181,187,187,176,176,176,181,176,173,148, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84,176,188,187,187,187,187,181, 84, 84, 84, 84, 84,130,
			137, 84, 84, 84,181,181, 84,148,173,181,181,130,179,113,176,181,176,
			183,187,181,170,170,170,181,181,181,195,176,187,172,172,211,229,216,
			213,217,220,212,220,217,192,196,183,181,186,181,193,201,206,205,213,
			223,209,223,217,208,217,209,223,217,212,209,215,217,221,223,217,213,
			219,203,188,186,188,187,192,181,176,185,187,187,187,181,173,173,181,
			188, 84, 84, 84, 84, 84, 84, 84,179, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84,188,170,204,209,198,209,204,213,
			178,193,181,183,181,192,187,183,187,187,187,181,181,181,176,181,181,
			181,187,181,183,181,181,176,176,170,181, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84,113, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84,183,181,176,181,199,185, 84, 84,176, 84, 84, 84,
			84,160, 84, 84, 84, 84,148, 84, 84,176,183, 84, 84,154,181,176,192,
			187,192,187,193,181,193,181,176,176,192,187,187,200,173,172,199,213,
			216,217,219,220,220,223,213,193,187,188,201,209,193,203,211,208,208,
			227,229,223,227,223,226,226,216,217,217,217,212,217,211,211,223,216,
			224,227,211,194,200,199,187,188,179,192,183, 84, 84,183, 84,154,181,
			84, 84, 84, 84, 84, 84, 84, 84, 84,181,176, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84,154,188,180,217,217,213,180,
			178,204,198,183,178,178,193,193,187,183,187,181,187,181,181,181,176,
			176,181,181,176,176,173,173,176,173, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,113, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84,188,183,178,187,170,187, 84, 84, 84, 84, 84,
			84, 84, 84, 84,154,154,137, 84, 84, 84,113,176,188, 84, 84,181,188,
			180,196,208,187,187,188,187,181,181,185,192,196,181,178,167,167,172,
			215,187,193,203,197,200,211,211,193,203,186,181,193,215,221,205,192,
			227,227,228,223,223,223,212,217,221,220,223,217,211,203,197,193,193,
			192,228,214,208,200,203,187,187,187,192,195,197, 84, 84, 84, 84,154,
			185,181, 84, 84, 84, 84, 84, 84, 84,165,181, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,188,188,211,206,209,
			197,194,216,211,207,178,178,188,188,187,185,187,181,183,176,173,176,
			187,176,181,176,173,176,170,173,173,173,160, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84,181,137,154, 84, 84, 84,154,183,
			173,188,183,192, 84, 84, 84, 84, 84, 84, 84, 84,154, 84, 84, 84,130,
			176, 84,187,195,130,188,196,199,192,192,192,188,192,192,196,185,175,
			187,181,209,207,202,208,204,187,187,193,209,192,193,204,218,228,226,
			215,208,217,226,217,212,217,220,209,200,213,218,223,212,217,200,193,
			193,193,199,197,187,187,183,187,179,187,186,192,191,191,183, 84, 84,
			84,165,176,173, 84, 84, 84, 84, 84,185,178,178, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,207,209,211,
			207,198,183,207,207,178,207,183,198,187,187,181,187,176,181,181,176,
			176,181,173,181,176,173,165,176,178,176,176,113, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84,186, 84,137,113,188,191,217,
			213,206,188,191,204,113, 84, 84, 84, 84, 84, 84, 84, 84,170,165, 84,
			84, 84, 84, 84,170, 84,179,213,223,223,231,207,192,192,192,192,213,
			217,230,215,234,217,205,217,187,187,196,193,197,203,205,187,202,201,
			193,231,211,206,220,220,219,216,208,203,213,205,205,203,197,196,193,
			193,188,188,187,192,185,186,199,192,185,195,191,195,191,165, 84, 84,
			84, 84,165,176,137, 84, 84,176,176,178,173,170, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,130,197,
			178,213,209,186,170,170,170,200,187,198,187,192,185,187,178,176,178,
			170,181,181,181,173,178,170,178,170,181,181, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,187,176,188,216,220,213,
			217,211,237,239,227,224, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84,223,231,226,231,239,201,188,192,200,192,
			213,209,223,220,217,220,220,224,217,192,192,193,193,203,195,196,188,
			191,212,209,206,206,205,213,205,199,209,200,202,205,205,195,195,195,
			193,193,188,193,188,179,176,179,176,183,179,197,195,195,195,197, 84,
			84, 84, 84, 84, 84, 84,130,165,137, 84,137, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84,183,212,211,198,198,193,193,200,207,200,193,187,185,185,185,176,
			173,175,176,176,178,178,176,170,181,176, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84,205,216,187,211,211,209,216,
			217,239,230,239,246,243,236,237,227,226, 84, 84, 84, 84,197,227, 84,
			84, 84, 84, 84, 84, 84, 84,165,203,221,230,230,233,236,215,213,219,
			216,199,213,217,217,217,223,212,217,216,220,213,209,211,215,212,191,
			188,195,188,212,209,208,195,195,199,203,193,208,205,201,195,205,193,
			193,193,188,188,188,203,188,185,179,176,179,188,186,193,193,188,188,
			199, 84, 84, 84, 84, 84, 84,165, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84,194, 84,204,170,188,200,229,200,193,193,187,181,187,187,
			176,170,170,178,170,170,178,178,181,176,173, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84,154,195,213,209,221,217,235,
			239,236,243,220,239,241,241,243,217,232,232,235,223, 84, 84,234,230,
			228,232,235,209, 84,183,194,211,226,219,223,223,234,232,237,234,213,
			215,221,211,199,227,223,220,217,209,220,211,217,217,223,209,208,203,
			209,197,193,187,176,215,196,203,203,188,191,199,188,195,195,193,197,
			193,200,188,193,185,188,195,195,188,185,181,181,188,183,179,181,187,
			185,187, 84, 84, 84, 84, 84, 84,130, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84,194,154,183,181,180,170,186,198,202,204,193,176,176,
			181,181,187,137,176,180, 84, 84, 84, 84,178,178, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,197,213,221,223,228,223,
			232,248,236,217,223,239,239,236,237,213,211,213,228,232,237,239,243,
			239,235,236,236,234,234,231,230,234,228,232,227,236,236,232,236,234,
			235,234, 84,204,205,205,218,206,215,216,217,217,221,217,220,220,215,
			217,195,197,191,195,193,173,183,193,217,195,195,196,193,200,196,207,
			200,195,204,178,193,181,193,188,193,193,188,178,181,193,179,179,181,
			173,178,178,113, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84,171, 84,180,180,173,188,193,200,200,187,187,
			187,160, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,181, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84,130,148,159, 84,231,228,213,235,239,217,
			232,239,217,202,204,211,232,237,239,232,234,237,218,237,223,223,239,
			232,239,243,241,241,237,241,232,223,154,207,215,224,236,238,230,231,
			231,235,228, 84, 84,220,216,217,217,215,204,209,217,226,223,212,217,
			188,205,220,223,206,193,195,195,188,188,181,176,204,228,196,211,193,
			226,170,176,181,191,178,178,186,188,185,193,193,181,181,183,170,188,
			188,181,173,181, 84, 84, 84, 84, 84,137, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84,186, 84,154,180,170,193,211,204,193,
			188,187, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,176,179, 84,113,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,223,212,221,230,230,230,
			230,228,241,235,235,228,227,221,223,223,226,220,232,234,241,217,223,
			239,236,241,236,239,246,235,239,237,209,230, 84,217,221,234,220,223,
			221,234,239,239,234, 84, 84, 84,229,194,205,212,209,209,215,212,209,
			217,218,217,213,200,188,188,193,185,183,195,195,188,179,179,179,173,
			176,183,183,176,173,176,173,188,183,185,187,185,181,176,183,188,178,
			181,178,179,178, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84,198, 84,188,170,178,198,216,
			198,181,183, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,137, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,230,232,228,236,232,236,
			228,223,228,241,235,228,223,218,228,237,215,226,230,223,248,235,239,
			239,241,245,243,232,224,243,239,227,241,233,218, 84, 84,211,212,212,
			223,230,232,229,235,237,187, 84, 84, 84,217, 84,180,216,226,223,223,
			198,207,187,206,205,185,193,185,181,188,183,195,199,195,187,195,181,
			185,173,176,181,170,173,181,178,187,190,178,195,185,181,173,178,178,
			178,178,181,178,137, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,143,143, 84,180,175,188,
			193,202,188,173, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84,154, 84,113, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,113,228,232,236,236,234,
			227,223,232,236,241,237,228,230,232,230,217,234,241,236,248,243,227,
			238,237,234,235,237,243,218,236,241,237,237,233,223,217, 84,171,217,
			209,223,216,211,232,231,229,234, 84,187,234,209, 84, 84, 84, 84, 84,
			84, 84,197,209,205,193,181,181,185,185,181,185,185,183,191,188,191,
			188,187,188,173,173,170,173,173,176,178,183,185,176,176,181,183,176,
			176,181,181,181,170, 84,185, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,181,
			178,188,187,193,181, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,176,170,
			181,113, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,235,234,224,239,239,
			232,234,213,211,230,239,239,236,235,232,223,232,239,237,239,228,228,
			239,239,232,228,221,241,224,236,241,245,241,237,209,221,228, 84, 84,
			211,215,227,229,223,234,239,239,237,234,236,229,223,217,211, 84, 84,
			84, 84, 84, 84, 84,170,193,188,179,187,193,176,188,176,176,185,185,
			188,188,183,137,173,176,187,173,176,176,170,173,173,173,181,185,176,
			176,176,148, 84, 84, 84, 84,186, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84,178,178,178,181,176, 84, 84, 84, 84, 84,130,183,181, 84, 84, 84,
			84, 84, 84,181,130, 84,162, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,154,235,239,246,221,
			243,239,229,234,239,239,239,239,234,232,239,241,237,239,246,239,223,
			237,232,223,212,218,234,237,241,230,243,241,239,217,237,217,215,209,
			84,202,213,209,229,226,237,236,234,239,234,231,234,234,226,207, 84,
			84, 84, 84, 84, 84, 84,176,193,188,199,188,188,185,181,185,187,181,
			185,130, 84, 84, 84,173,181,193,185,173,173,170,173,181,191,181, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84,173,170,178,173,193,176, 84, 84, 84, 84,178,173, 84, 84, 84,
			84, 84, 84, 84,148,113, 84,148,130, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,237,224,228,
			243,243,243,241,245,245,245,236,237,221,223,239,239,232,223,248,243,
			235,237,246,239,221,228,241,239,230,241,239,239,234,236,234,230,224,
			212, 84, 84,165,217,217,230,237,234,234,234,239,231,241,237,217, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84,181,188,195,185,183,176,176,185,
			187,173, 84, 84, 84, 84, 84,165,197,173,173,181,173,170,178,187, 84,
			84,176,170, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84,170,181,178,173,176,185, 84,113,130,173,173, 84, 84,
			84, 84, 84, 84, 84,137, 84, 84,148,178,178,176, 84,130, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,248,241,
			241,232,245,243,249,246,246,243,243,237,223,236,237,232,236,217,215,
			248,246,250,248,239,232,234,235,217,232,243,239,236,239,230,232,223,
			227,221,188, 84, 84,211,221,236,237,239,232,236,237,239,237,227, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84,183,193,195,185,187,176,176,
			195,165, 84, 84, 84, 84, 84, 84, 84,193,178,173,178,170,183,165,185,
			84, 84,181, 84, 84, 84, 84, 84, 84, 84,165,113, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84,154,181,170,173,170,178,176,173,176, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84,130, 84, 84, 84, 84, 84, 84,219,
			230,223,224,227,235,231,231,228,228,228,236,223,220,227,223,220,224,
			243,245,249,246,239,235,238,237,234,220,239,243,237,237,237,217,228,
			217,220,220,221,113, 84, 84,211,217,239,237,220,230,227,223,154, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,185,198,188,197,195,
			187, 84, 84, 84, 84, 84, 84, 84, 84, 84,187,180,173,188,185,195,178,
			176,179, 84, 84, 84, 84, 84, 84, 84, 84, 84,188,181, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,176,176,178,181,
			185,185,187, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			209,203,216,209,193,193,199,216,209,196,229,227,220,228,214,211,223,
			228,231,236,237,239,239,241,239,236,221,221,231,227,216,239,230,217,
			223,209,193,211,211,217, 84, 84,200,220,234,227,220,220,207, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,181,183,191,176,
			200, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,173,195,195,197,
			195,185,176,188, 84, 84, 84, 84, 84, 84, 84, 84,195, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,187,178,
			176,183,181,185,143, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84,188,183,183,188,188,183,188,203,195,193,211,209,208,211,211,198,
			214,221,213,219,216,217,223,223,214,209,196,196,209,220,230,216,211,
			214,205,193,187,181,183,181,221, 84,206,208,211,234, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,137,187,199,
			188,187, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,181,197,188,
			180,176,173,170,170, 84, 84, 84, 84, 84, 84, 84, 84,181, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84,176,176,181, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84,183,176,187,183,183,183,183,187,183,193,186,194,188,193,193,
			193,183,193,203,193,193,205,193,193,188,193,187,193,187,211,192,196,
			199,188,188,179,176,176,188,181,205,212,113,113, 84, 84, 84, 84, 84,
			130, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,179,
			195,193,199, 84, 84, 84, 84, 84, 84, 84, 84,173, 84, 84, 84,113,197,
			160,181,185,173,173,193, 84, 84, 84, 84, 84, 84, 84, 84,137,137,113,
			130, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84,170,181, 84, 84, 84, 84, 84, 84, 84,137, 84,165,176,
			84, 84, 84, 84, 84,130, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84,113,181,176,183,183,183,176,183,187,183,183,183,183,187,
			183,176,183,188,188,183,183,188,183,183,183,187,187,183,193,193,193,
			193,183,187,187,183,183,178,176,183,215,212,175, 84, 84, 84,159,209,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			154,187,193, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84,176,193,183,185, 84, 84, 84, 84, 84, 84, 84,137, 84,165,
			113,173, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84,187,179, 84, 84, 84, 84, 84,181,181, 84,181,
			181,178,176,176,176,148, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84,113,187,176,176,183,183,187,183,183,183,187,188,
			188,181,181,186,181,183,187,183,183,187,181,183,183,176,183,183,183,
			183,183,176,176,176,183,183,180,183,183,188,206,193,204,211,217,205,
			211, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84,197,195, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			113, 84, 84, 84, 84,187,160, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84,130,113, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84,165,188,181, 84,170,181,181,173,176,
			176,176,181,183,176,183,181,130, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84,185,176,176,183,183,183,181,181,188,178,
			183,183,183,183,183,183,181,181,176,181,183,183,187,183,180,180,183,
			183,183,187,178,178,187,176,187,176,179,187,187,170,209,207,202,228,
			221,159, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84,187, 84,170, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84,173,191, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,113, 84,
			84, 84,113,191,113, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,176,183,176,183,
			181,176,181,187,183,170,183,183,176,170, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84,173,181,176,181,185,181,181,179,
			186,183,181,183,181,181,181,178,179,183,181,170,170,176,180,176,180,
			176,180,168,183,183,175,176,183,176,183,179,192,176,204,194,194,204,
			207,228, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84,181,188, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84,160, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84,173, 84,170,165, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,181,181,
			178,176,183,183,194,183,181,176,170,176,176,181,181,185,113, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,154,181,176,185,179,179,
			181,170, 84, 84, 84,188,183,176,176,175,170,170,180,170,176,176,176,
			178,181,183,183,176,183,183,181,181,181,170,179,176,197,183,213,209,
			207,228, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84,130, 84, 84, 84,176,185, 84, 84, 84, 84, 84, 84, 84, 84, 84,185,
			176, 84,113, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,183,
			181,179,183,188,181,173,165,170,176,176,197,176,178,176,176,178,178,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,113, 84, 84,
			84, 84, 84, 84, 84, 84, 84,154, 84,179,178,178,170,178,170,178,170,
			178,181,176,176,176,178,180,176,176,181,205,204,194,198,194,218,197,
			202,204,223, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84,176, 84, 84,185,181, 84, 84, 84, 84, 84, 84, 84,113,181,
			178,173, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,154,
			185,188,181,176,176,176,176,176,178,176,178,198,183,178,176,176,181,
			180, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,176,170,170,170,173,176,
			173,176,173,176,173,173,176,176,183,183,181,188,205,204,183,197,204,
			193,202,211, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84,178,173, 84,181, 84, 84, 84, 84, 84, 84,113,181,
			173,178, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,137,
			183,181,173,176,176,176,181,173,176,170,176,176,176,178,176,178,178,
			178,178,186, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,165,170,170,170,170,170,
			178,170,178,173,173,176,173,173,176,179,176,181,181,187,193,193,193,
			204,181, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84,183,179,154,170, 84, 84, 84,181,165,173,
			173,178,173,181, 84, 84, 84, 84,181, 84,188, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84,130, 84, 84, 84, 84, 84, 84, 84, 84,
			183,185,176,176,176,178,173,178,178,176,170,178,176,176,173,178,181,
			176,176,181,130, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,160,170,170,170,170,
			170,178,170,173,170,173,170,170,170,173,176,176,179,176,187,178,198,
			197,181, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84,113,173,183,181, 84, 84, 84, 84,181,176,
			173,176,173,178, 84, 84, 84, 84, 84, 84, 84,165, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			176,180,176,178,176,176,176,176,176,170,170,176,176,178,176,176,176,
			173,176,181,176,176,181,181,181, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,173,176,170,170,
			170,178,176,178,178,176,176,178,173,176,181,181,178,178,176,181,187,
			197,194, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84,179,173,181, 84, 84, 84,154,
			173,176,173,181,185, 84,170, 84,185, 84, 84, 84, 84, 84,113,176,180,
			113,137, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84,154,188,176,173,176,173,176,176,176,178,176,173,176,181,180,173,
			173,173,173,176,176,173,179,181,176,181,179,178,143, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,170,176,
			170,176,176,178,181,173,181,176,176,176,178,181,181,176,181,181,193,
			181,194, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,181,188,193,165,154,
			84,185,181,173,176, 84, 84,178, 84,154, 84, 84, 84, 84,137, 84,137,
			173, 84,178,181,181,181,113, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84,176,181,170,176,181,178,176,178,176,176,176,176,173,173,173,
			173,176,176,173,173,173,176,181,178,176,188,170,188,193,193,168, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			176,183,178,178,176,176,178,178,181,170,176,176,175,176,176,178,176,
			188,176,181, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,137,183,193, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84,185, 84, 84, 84, 84, 84, 84,
			84,113,170,185,188,188,181,181,180,160, 84, 84, 84, 84,130,148, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84,188,185,176,176,176,178,176,176,176,181,176,176,176,176,
			173,176,176,176,176,181,176,176,183,181,176,178,176,194,193,192,197,
			194,130, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84,183,178,180,178,176,176,173,173,176,176,176,178,178,180,180,176,
			178,176,176, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,130,113,170,
			84, 84, 84, 84, 84, 84, 84, 84, 84,178, 84,113,113, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84,178,173,185,181,181,137, 84,148,148,165, 84,
			137, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84,170,188,170,176,176,176,176,176,178,176,176,176,
			176,176,173,176,176,176,176,178,176,181,194,188,193,181,183,181,188,
			187,205,185, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84,181,176,183,176,176,183,176,183,181,178,176,181,183,183,
			176,183,183,178,176, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			137,188,193,154,185,143,137,137, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84,180,180,176,176,178,185,113, 84, 84, 84,
			84, 84,113, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84,165,196,176,176,176,176,176,176,181,181,181,
			176,173,176,176,173,173,173,176,181,181,181,181,183,188,176,181,187,
			188,193,186,188, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84,180,170,170,176,180,183,180,178,183,187,181,176,178,
			183,176,176,180,180,170, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84,137,176,188,183,148,113,170,113,137,130,137,
			84, 84, 84, 84, 84, 84, 84,130, 84,183,183,137, 84, 84,193, 84, 84,
			84, 84, 84, 84,137, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84,181,183,170,178,178,178,178,176,176,
			176,181,176,176,178,173,173,173,176,176,176,183,176,194,181,176,181,
			188,188,188,187, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84,178,183,176,180,176,183,176,180,178,176,187,178,
			176,178,183,187,170,180,170, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,165, 84, 84,173, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,154,170,
			84, 84, 84, 84, 84, 84, 84,176,113, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84,178,188,176,181,178,176,176,178,
			176,181,181,173,176,176,176,176,176,176,178,188,176,183,198,176,173,
			181,181,188,187, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84,154,178,183,176,183,176,176,176,176,178,180,
			183,183,176,178,176,180,178,178,148, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84,113, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84,113, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,186,188,188,188,176,176,
			176,178,176,176,176,176,176,176,176,178,176,173,183,183,178,173,176,
			176,170,181,185,185, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84,188,183,176,176,176,181,176,180,170,176,
			176,183,170,183,178,176,176,178,178,143, 84, 84, 84, 84, 84,168, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84,130,178,186,180,159, 84, 84, 84,193,160, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,154, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,226,193,187,176,181,
			181,176,183,176,178,176,176,181,181,176,176,183,183,176,183,183,173,
			194,176,181,181,181, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84,198,176,183,178,176,176,176,181,170,180,
			176,180,178,170,188,181,175,170,180,183,168, 84, 84, 84, 84, 84,187,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84,130, 84, 84,185,188,181,193, 84, 84, 84, 84,188,178, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,212,200,191,
			197,193,181,176,176,176,181,181,176,176,181,181,183,176,181,176,176,
			188,194,176,178,176,176, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84,207,194,186,176,183,176,178,178,176,
			176,176,180,178,178,181,176,180,183,180,183,165, 84, 84, 84, 84,180,
			183,165, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84,170,188,200,193,202,193,185,193, 84, 84, 84, 84,193,188,
			180, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,154,
			84, 84, 84, 84, 84, 84, 84, 84,130, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			212,195,197,183,176,181,178,176,181,176,183,176,178,183,176,181,181,
			183,181,176,176,181,176,181, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84,209,193,193,197,178,187,181,181,
			176,180,183,178,176,183,178,180,178,180,180,183, 84, 84, 84,180,181,
			178,183, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84,188,188,197,198,198,204,193,193,193,198,186,154, 84,194,
			193,188, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84,205,193,188,198,183,176,181,176,170,170,181,176,176,176,181,
			181,181,176,183,188,176,181, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84,205,188,203,193,193,188,176,
			181,181,183,178,183,183,183,183,176,178,181, 84, 84, 84, 84, 84,181,
			180,180,185, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84,159,202,200,200,196,193,200,199,200,196,187,198,193,193,
			188,193,178,148, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84,181,196,205,193,178,176,181,170,183,183,176,181,181,
			181,181,183,183,183,176,178,176, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,204,193,193,198,193,
			188,185,187,183,176,180,178,183,188,170,181, 84, 84, 84, 84, 84, 84,
			180,180,178, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84,130,199,202,202,196,199,200,198,199,200,202,196,187,194,
			198,193,186,188,188,165, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84,194,199,201,193,181,178,176,181,176,176,176,181,
			181,178,181,176,181,176,181,176,160, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,212,196,193,193,
			199,199,193,187,213,181,179,188,183,178,178,178, 84, 84, 84, 84, 84,
			84,176,183,181, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84,204,196,193,196,199,202,202,200,199,200,198,202,204,202,203,204,
			202,197,183,187,193,193,188,178, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84,173, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84,207,199,193,179,181,176,176,187,176,178,181,
			176,181,181,181,176,176,176,181,181, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,220,193,
			193,193,199,196,188,192,187,181,185,180,181,178,176, 84, 84, 84, 84,
			84,176,179,181,179, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			183,202,188,193,193,193,204,205,196,196,202,196,200,202,197,200,198,
			198,202,199,198,188,204,197,193,186, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84,196,208,205,202,181,181,176,181,187,176,170,
			176,176,181,176,181,181,193, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,209,
			209,200,199,200,199,192,196,183,170,188,176,183,180,181, 84, 84, 84,
			84, 84, 84,193,180, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84,205,204,198,193,193,202,200,193,198,196,200,199,196,196,196,204,
			199,204,200,203,193,196,193,193,185,178,181, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84,170,209,209,188,181,178,183,181,176,187,
			170,176,170,176,176,181,154, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			216,207,209,203,199,204,193,183,187,187,181,176,178,148, 84, 84, 84,
			84, 84, 84, 84,181,148, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84,183,203,200,198,204,198,197,198,197,197,198,198,198,198,199,
			199,202,209,203,196,198,196,181,193,178,181,178,180,148, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84,212,215,220,204,181,170,178,176,176,
			181,181,170,176,178,170,179, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84,228,207,211,200,200,200,195,192,187,183,180,178,178, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84,175,194,198,196,197,193,198,198,197,202,197,197,197,197,
			202,199,205,205,199,193,193,200,198,193,188,188,181,181,154, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84,205,207,208,188,176,178,176,183,
			176,176,176,181,181,181,181,181, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84,198,220,203,199,196,188,193,187,187,179,178,181, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84,197,198,198,197,193,198,197,197,198,202,193,202,
			198,198,196,200,212,205,208,198,196,198,198,193,181,181,181,183, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84,205,200,194,197,183,183,183,
			178,176,181,178,181,183,176,176, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84,200,209,213,193,193,193,188,181,185,185, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84,193,193,194,198,193,198,193,198,193,197,197,
			198,197,198,198,200,205,200,202,199,205,193,198,198,188,185,173,180,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84,148,201,192,194,188,176,178,
			181,183,181,178,178,178,181,176,178, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84,193,205,205,193,193,188,188,181,181, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84,154,187,187,197,188,198,193,193,197,202,
			200,198,193,198,204,202,209,196,202,200,199,200,188,193,188,188,173,
			130, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,193,198,187,183,188,
			181,181,181,181,181,183,181,181,170, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84,200,198,188,193,188,193,193, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84,178,187,183,188,178,194,194,194,
			159, 84, 84, 84,165,193,205,202,202,200,200,199,197,188,183,173,180,
			173, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84,113, 84, 84, 84, 84,148,194,194,202,193,
			181,188,176,178,176,179,181,181,181, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84,193,193,193,188,183,137, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84,173,181,187,185,181,181,160, 84,
			84, 84, 84, 84, 84, 84,137,181,113,193,193,193,193,190,181,181,181,
			173, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,181,201,187,194,
			194,183,181,181,181,181,187, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,130,148, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84,165,187,193,185,188,185,181,181,
			181,180, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,183,202,195,
			197,193,176,181,185,179,181,181, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,181,181,181,181,181,
			173,173, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84,130, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,178,176,194,
			188,197,194,188,187,181,188,185,187, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,179,181,179,181,
			178,173,165, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84,181, 84,165, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,176,
			198,188,191,197,188,176,154, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84,188,188, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,187,
			178,193,181,187,193,183,176,113, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84,143, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84,130, 84,181, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84,178,188,183,181,193, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84,178,176, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84,165,188,113, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84,178,181,187,192,187, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84,173, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84,170,186, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84,154,178,188,206,192,176, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84,137,183,188, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84,175,170,186,188,195,188, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84,137,187, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84,143,209,188,188,188,196,192, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84,188,252,183,196,188,178, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84,137,154, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84,143,239,193,181,160, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84,165,206,187,194,202, 84, 84, 84, 84, 84,162,
			196, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84,148,130,180,187, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84,113,170,226,183,187, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84,147, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84,209, 84,246,213,188, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84,171,249, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84,202,226,184, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84,209, 84, 84,177,228,228,166, 84, 84, 84,130,239,250, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84,156,162,234, 84,113, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84,156,184,246, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84,147,177,222,246,241,252,248,250,234,130, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,237,254,249,246,243,243,
			246,243,246,246,248,239,246,241,243,246,243,239,241,241,243,243,237,
			246,250,249,252,239,246,248,250,252,249,250,249,246,249,246,241,246,
			246,248,250, 84,130, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84,243, 84,147, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84,130, 84, 84, 84, 84, 84,237,
			248,250,243,249,248,246,243,250,252,246,248,246,243,243,252,252,249,
			246,246,216, 84, 84, 84, 84, 84,174,243,246,248,248,248,246,246,243,
			246,243,243,246,248,248,246,243,246,248,243,246,241,241,239,239,239,
			243,243,248,248,249,248,246,248,250,248,246,250,246,241,241,242,243,
			248,246,250,249,252,252,248,246,254,243,209,238, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84,224, 84, 84,228,236, 84,246,248,248,248,130, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,205,
			84,171, 84, 84, 84, 84, 84, 84, 84,147,202,231,249,250,246,213,249,
			246,246,246,248,250,249,250,249,248,248,250,249,249,249,249,246,243,
			239,239,236, 84, 84,113,246,249,248,243,242,243,246,246,246,246,246,
			246,246,243,243,243,246,246,246,246,246,246,246,246,243,241,243,243,
			243,243,243,243,248,248,249,252,252,252,252,246,246,243,243,243,246,
			246,248,250,252,250,250,246,246,246,248,250,246,246,246,248,248,248,
			84, 84,143, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84,213,218,254,246,248,231,248,246,248,248, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,198,254,162,
			249,231,234,222,188, 84,246,248,248,246,246,243,243,246,237,234,241,
			243,246,241,241,239,231,243,243,246,243,246,243,243,242,228,243,246,
			246,246,246,248,248,248,246,246,248,249,249,249,248,248,246,246,246,
			243,241,234,213, 84,166,239,243,243,241,248,242,246,249,249,249,248,
			248,246,246,243,243,243,241,243,246,246,248,248,248,243,243,246,243,
			246,246,246,248,249,249,249,250,250,250,249,249,246,246,246,243,243,
			243,248,246,246,248,246,243,248,246,243,243,243,246,248,248,246,252,
			241,234,228,246,239,238,236,188, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84,198, 84, 84,147, 84, 84,236,156,193, 84, 84, 84, 84, 84,
			84,254, 84,147, 84, 84,171,113, 84,246,248,249,243,237, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,188,249,246,
			248,248,249,249,242,249,249,248,249,248,246,246,246,246,243,243,246,
			243,246,246,243,246,241,237,236,242,243,246,246,243,243,243,237,246,
			246,243,246,246,243,246,243,246,246,246,248,246,246,243,242,241,234,
			239,228,224,202,220,236,241,243,239,231,248,246,249,248,249,248,248,
			248,248,246,246,243,243,243,241,241,241,243,243,243,243,243,243,243,
			246,248,250,249,252,252,249,250,252,250,249,248,248,246,243,241,243,
			241,243,243,241,248,246,243,246,243,246,243,243,243,243,246,246,248,
			248,248,236,246,222,239,226,188, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,177, 84,
			84, 84, 84,220, 84,249,218, 84,162, 84,233,147,130,147, 84, 84, 84,
			84, 84, 84,254,249,250,252,252,252,238,248,250,248,246,249,243,248,
			243,239,243,241,238,246,243,241,239,239,243,242,246,246,228, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,147,237,243,246,
			246,246,246,243,246,246,246,243,246,246,242,243,246,246,246,246,248,
			248,246,248,248,248,248,246,246,246,248,246,246,249,243,243,243,243,
			243,246,246,246,246,246,246,243,246,246,246,241,243,243,243,241,243,
			234,236,234,236,228,241,243,241,241,237,238,241,243,246,243,248,246,
			246,246,246,246,243,243,243,241,241,241,241,241,239,241,241,242,246,
			246,248,250,250,252,252,252,252,252,249,246,246,246,246,241,241,241,
			243,243,246,246,248,246,246,246,246,246,246,246,248,246,246,246,246,
			246,246,250,228,241,216, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84,113,209,241,243,248,243,243,248,236,249,236,248,
			243,243,248,246,243,246,241,241,239,246,231,243,246,241,248,241,243,
			243,243,243,243,241,241,241,241,239,241,246,243,246,246,248,246,246,
			243,243,243,246,243,243,241,238,246,239,243,250,250,216, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84,130,162,177,239,239,246,243,246,246,
			243,243,241,243,243,246,246,246,246,246,243,242,246,246,246,246,246,
			241,243,246,246,248,246,248,248,248,248,246,248,246,246,246,243,243,
			243,241,243,243,248,243,243,243,246,246,246,242,239,241,236,236,234,
			239,238,239,241,242,241,241,241,241,241,241,241,243,243,246,246,246,
			241,243,243,243,241,241,241,241,239,239,239,239,242,239,241,243,243,
			246,246,248,249,249,249,249,248,246,249,249,246,246,243,243,243,246,
			243,246,246,246,243,243,243,243,246,243,243,243,246,243,243,241,241,
			234,241,243,239,234, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,113,236,
			236,216,147,192,130,220,185,246,243,246,246,246,243,243,246,248,248,
			248,248,246,229,243,239,241,241,241,243,238,243,246,243,243,243,242,
			242,243,243,243,241,243,246,246,243,243,243,243,243,246,243,243,246,
			242,246,239,246,246,228, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84,199,236,236,238,236,237,234,239,241,241,239,243,243,
			241,243,246,243,243,246,243,243,243,246,248,243,243,243,246,246,246,
			246,246,246,248,248,248,246,248,246,248,246,246,246,246,248,248,246,
			246,246,246,246,243,243,246,246,243,243,243,246,246,246,243,243,248,
			241,246,246,243,243,243,243,242,239,241,239,239,241,241,241,239,241,
			241,243,241,239,241,241,242,241,237,239,239,241,243,241,243,243,239,
			241,242,243,246,246,246,246,246,246,243,243,243,243,243,243,243,243,
			241,241,241,243,241,243,241,241,239,241,241,243,243,243,246,242,241,
			241,241,241,243,222,209,250, 84, 84,202,192, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84,113,241,248,243,241,241,243,243,243,241,241,243,243,246,243,243,
			243,239,241,241,241,243,243,246,241,241,243,243,242,243,243,241,241,
			241,243,239,239,241,239,241,243,246,246,243,243,243,241,241,238,238,
			226,241,226,239,243,224, 84, 84, 84, 84, 84, 84, 84,113, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84,229,224,228,250, 84, 84, 84, 84,
			84, 84,216,236,243,237,234,231,234,239,238,239,241,239,238,243,239,
			239,241,243,246,246,248,248,246,248,246,246,248,246,246,243,246,243,
			243,243,242,242,246,246,246,246,246,246,246,248,248,248,248,248,246,
			246,246,246,243,243,243,241,243,241,241,239,239,243,243,246,249,248,
			246,246,246,243,243,241,243,243,243,241,241,241,241,241,241,241,241,
			239,241,239,239,239,241,241,239,241,241,241,241,241,242,241,242,243,
			243,243,243,246,241,246,246,246,246,243,243,241,243,246,246,243,241,
			241,242,241,239,241,241,241,241,241,241,241,241,243,243,243,243,241,
			243,241,239,243,246,226,246,236,154,147, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84,234,239,236,237,234,236,234,234,236,237,239,
			241,239,241,241,241,241,243,241,239,243,243,241,243,243,243,243,243,
			243,243,243,241,243,241,243,241,241,241,241,239,241,239,241,246,246,
			243,243,234,236,236,222,222,202, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84,229,224,228,234,243,222, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84,228,231,229,233,239,239,231,236,237,
			234,234,239,241,236,237,238,236,248,246,246,246,246,243,248,246,243,
			242,243,246,243,243,241,241,241,243,241,243,241,241,243,243,243,243,
			241,243,243,241,241,241,239,239,241,241,241,241,241,241,241,241,239,
			242,239,241,241,241,239,241,241,239,241,241,241,242,239,241,241,242,
			241,242,242,239,239,241,241,239,241,241,241,243,243,243,243,243,243,
			246,243,243,243,243,246,243,246,243,243,243,241,243,241,243,246,239,
			237,241,241,239,239,239,241,241,241,241,241,243,241,243,243,243,243,
			241,241,243,239,231,197,147, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84,143,222,239,239,234,231,231,231,234,234,233,231,231,236,
			236,236,238,238,236,239,241,239,239,241,243,243,241,239,239,243,243,
			241,239,241,241,239,239,239,239,241,241,236,241,236,239,243,241,241,
			241,243,241,239,239,239,241,237,237,236,234,239,236,239,236,191, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			224,233,241,233,224,222,239,236,238,237,238,234,233,239,236,237,236,
			218,243,246,249,246,243,250,248,246,248,243,242,238,241,241,236,243,
			243,241,239,239,241,239,239,239,239,239,239,239,239,239,239,239,239,
			239,239,239,239,241,239,241,241,241,241,239,239,241,241,239,239,241,
			239,241,239,239,239,239,239,239,239,241,241,241,241,239,239,241,239,
			239,239,241,239,241,241,243,243,243,243,243,243,243,243,243,241,243,
			243,246,243,241,243,241,243,241,239,241,241,239,241,237,239,241,241,
			241,241,241,241,241,241,243,243,243,241,243,243,243,241,243,241,241,
			241,238,241,237,237,241,228,231,184, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84,
			84, 84,147,162,216,237,233,234,236,236,236,233,234,231,233,231,231,
			236,239,238,237,241,239,239,239,241,241,241,241,241,241,239,241,241,
			241,239,239,241,238,239,237,243,246,243,243,241,241,243,241,241,241,
			246,241,241,239,241,236,237,239,239,237,238,236,237,239,239,241,237,
			243,243, 84, 84, 84, 84, 84,220,239,228,236,239,241,243,209,239,224,
			212,224,236,229,237,241,237,237,239,236,237,238,236,239,241,241,241,
			239,241,237,239,243,241,241,241,243,241,246,243,248,236,239,237,237,
			239,239,239,239,236,239,239,239,239,239,237,238,236,239,239,236,238,
			239,239,239,239,239,241,241,241,239,241,239,241,239,239,239,241,239,
			239,242,239,239,239,236,224,234,241,239,239,239,239,241,241,241,241,
			241,241,241,243,243,243,241,243,241,241,241,241,241,241,241,241,241,
			241,241,241,241,241,239,241,241,239,238,238,239,241,246,243,243,243,
			246,243,239,241,239,243,241,243,246,241,241,243,243,243,243,243,243,
			243,243,243,241,226,228,233,218,218,231,228,222,213,147, 84, 84, 84,
			84, 84, 84, 84, 84,185,147,113,130, 84, 84, 84, 84, 84, 84, 84, 84,
			84,184,243,239,241,239,239,241,239,239,239,237,237,236,237,239,239,
			239,239,239,239,239,237,239,239,241,243,243,243,243,243,243,239,233,
			237,239,241,237,241,239,237,241,239,239,241,241,241,241,241,241,241,
			241,241,241,238,238,236,234,231,231,233,231,231,228,228,231,236,242,
			239,243,228,239,233,237,243,220,237,233,239,243,243,241,241,239,241,
			239,241,239,233,234,231,229,231,231,236,241,241,246,229,239,236,239,
			241,239,239,239,239,236,236,234,239,243,237,236,237,236,236,236,236,
			234,236,236,236,237,236,236,237,238,236,237,237,237,238,237,239,239,
			239,239,241,239,241,239,242,243,241,239,239,242,239,239,241,241,241,
			241,239,242,239,239,239,239,241,239,239,241,239,241,241,241,241,241,
			241,241,241,241,241,241,241,241,241,241,239,239,241,241,241,239,239,
			239,239,239,239,239,241,239,239,241,236,233,234,236,243,241,241,241,
			241,238,243,241,241,241,241,241,241,243,243,241,243,243,243,243,243,
			243,243,241,241,241,239,239,239,239,241,228,233,228,239,239,209,231,
			218,224,213,231,228,228,237,236,239,239,243,246,248,246,241,226,218,
			234,228,206,213,226,228,224,222,231,237,236,224,231,239,241,241,228,
			228,237,234,229,228,243,224,206,228,237,243,248,241,234,236,236,239,
			241,241,239,241,239,239,237,239,239,241,243,239,243,241,243,246,246,
			241,234,238,238,236,239,241,241,241,243,241,241,243,243,239,231,239,
			239,236,241,239,241,239,237,239,241,241,241,243,241,243,243,243,243,
			243,246,243,246,246,246,246,243,241,241,241,241,243,241,243,246,246,
			246,246,243,243,243,243,241,238,241,239,238,239,239,236,237,236,237,
			238,239,239,239,241,239,241,239,241,242,239,241,239,242,241,241,241,
			239,239,239,239,239,239,239,243,242,239,239,239,242,239,242,239,239,
			239,239,239,241,239,239,239,239,241,239,241,239,241,241,241,241,241,
			241,241,241,241,241,241,241,241,241,241,241,241,241,241,241,241,241,
			241,243,241,241,241,241,241,241,241,241,241,241,241,239,239,241,239,
			241,241,241,241,243,243,243,243,243,241,241,243,243,243,243,243,243,
			243,243,243,243,243,243,246,243,246,246,243,243,243,239,239,239,237,
			243,241,226,238,236,236,248,243,243,243,243,241,243,243,243,246,243,
			246,246,246,241,239,241,246,243,243,243,243,246,239,241,243,246,243,
			241,241,241,243,243,241,241,243,243,239,243,239,239,239,241,241,243,
			241,243,243,243,243,246,248,248,246,249,246,246,248,246,243,246,248,
			246,243,243,246,243,243,243,243,243,243,237,237,237,241,243,239,239,
			239,241,241,241,241,241,241,239,237,241,237,237,237,239,241,241,243,
			243,241,241,246,246,239,243,243,246,246,248,246,243,248,241,243,246,
			243,246,243,246,243,243,243,246,243,241,243,243,246,243,243,243,243,
			243,246,243,246,243,243,241,241,241,241,241,241,241,243,241,241,241,
			241,241,241,241,241,241,241,241,241,241,241,241,241,241,241,241,239,
			241,241,241,241,241,241,241,241,241,241,241,243,241,241,241,243,241,
			243,243,243,243,243,241,243,243,243,243,243,243,243,243,243,243,246,
			243,243,243,243,243,243,243,243,246,243,246,243,243,243,243,241,241,
			243,241,243,241,241,241,241,239,241,241,241,241,243,243,243,246,243,
			243,243,246,243,243,246,246,243,246,246,243,243,246,243,243,243,246,
			241,243,243,243,246,246,246,246,246,248,248,246,246,248,248,248,248,
			248,248,248,246,248,246,248,248,248,248,248,248,248,248,248,248,246,
			243,243,246,246,248,248,250,248,246,246,243,243,246,246,243,243,237,
			237,239,241,243,246,248,243,243,243,246,243,246,246,246,248,248,248,
			248,246,246,243,243,243,246,246,246,246,248,246,246,246,248,246,248,
			246,246,246,246,248,248,248,248,248,248,248,248,248,248,246,246,248,
			246,246,246,248,246,246,246,246,246,246,248,246,243,248,246,246,246,
			246,246,248,246,248,248,248,248,248,248,246,248,248,246,246,246,246,
			246,246,248,246,246,248,246,246,246,246,246,248,246,246,246,243,243,
			243,243,246,246,243,246,246,246,243,243,243,243,243,246,246,246,246,
			246,246,246,243,246,246,246,246,246,246,248,246,243,246,246,248,250,
			246,246,246,243,243,246,246,246,246,246,246,246,246,246,246,246,246,
			246,246,246,246,246,246,246,246,246,246,246,246,246,248,246,246,246,
			246,246,246,246,248,246,246,246,246,246,246,246,246,246,246,246,246,
			246,246,246,246,246,246,246,246,248,246,246,248,248,246,246,246,246,
			246,248,246,248,246,248,246,248,248
		};
	
		return data;
	}
};

#endif
