#include <stdio.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/types.h>
#include <linux/ipmi_ioctls.h>
#include <sdr.h>
#include <sensor_events.h>
#include <math.h>

static SENSOR *sensor_create(void);
static void sensor_destroy(SENSOR *sensor);
static void process_sdr_record(unsigned char *buffer);
static double linearize_sensor_reading(SENSOR *slist, unsigned char reading);
static char *linearization_to_str(unsigned char value);
static char *reading_type_to_str(unsigned char value);
int return_device_type_entry(unsigned char type_code,
                             unsigned char type_modifier,
                             char          *buffer);


SENSOR	*sensor_head = NULL;


static SENSOR *sensor_create()
{
	SENSOR	*new, *scan;

	new = (SENSOR *) malloc(sizeof(SENSOR));
	memset(new, 0, sizeof(SENSOR));
	if (!sensor_head)
		sensor_head = new;
	else
		{
		scan = sensor_head;
		while (scan->next)
			scan = scan -> next;
		scan->next = new;
		}
	return(new);
}

static void sensor_destroy(SENSOR *sensor)
{
	SENSOR	*scan;

	if (sensor == sensor_head)
		sensor_head = sensor_head->next;
	else
		{
		scan = sensor_head;
		while(sensor != scan->next)
			scan = scan -> next;
		scan->next = sensor->next;
		}
	return;
}

int dl_sdr(int fd)
{
	RESERVE_SDR_REPOSITORY_RESPONSE		sdr_res_resp;
	SDR_REPOSITORY_INFO_RESPONSE			sdr_info_resp;
	GET_SDR_REQUEST										request;
	SDR_HEADER												*sdr_hdr_buffer;
	unsigned short										*next_record_buffer;
	unsigned char											hunk_buffer[8192];
	IPMI_XFER													xfer;
	IPMI_XFER													*xferp = &xfer;
	unsigned short										sdr_current_record, sdr_next_record;
	unsigned char											cc, sdr_buffer[8192];
	int																rc, len, hunk_size, cnt;
	unsigned int											sdr_buffer_pos, brtl;

	printf("Reserving SDR repository -=> ");
	fflush(stdout);

	INIT_XFER(xferp);
	SET_REQUEST_LUN(xferp, 0);
	SET_REQUEST_NETFN(xferp, STORAGE_REQUEST);
	SET_REQUEST_CMD(xferp, CMD_RESERVE_SDR_REPOSITORY);
	if ((rc = ioctl(fd, IOCTL_IPMI_XFER, (void *) &xfer))<0)
		{
		printf("dl_sdr(): failed (%m)\n");
		return(-1);
		}
	GET_RESPONSE_CC(xferp, cc);
	if (cc != 0x00)
		{
		printf("dl_sdr(): failed (cc = 0x%.2x)\n",cc);
		return(-1);
		}
	memset(&sdr_res_resp, 0, sizeof(sdr_res_resp));
	GET_RESPONSE_DATA(xferp, (unsigned char *) &sdr_res_resp);
	printf("0x%.4x\n",sdr_res_resp.res_id);

	printf("Retrieving SDR info -=> ");
	fflush(stdout);
	INIT_XFER(xferp);
	SET_REQUEST_LUN(xferp, 0);
	SET_REQUEST_NETFN(xferp, STORAGE_REQUEST);
	SET_REQUEST_CMD(xferp, CMD_GET_SDR_REPOSITORY_INFO);
	if ((rc = ioctl(fd, IOCTL_IPMI_XFER, (void *) &xfer))<0)
		{
		printf("dl_sdr(): failed (%m)\n");
		return(-1);
		}
	GET_RESPONSE_CC(xferp, cc);
	if (cc != 0x00)
		{
		printf("dl_sdr(): failed (cc = 0x%.2x)\n",cc);
		return(-1);
		}
	memset(&sdr_info_resp, 0, sizeof(sdr_info_resp));
	GET_RESPONSE_DATA(xferp, (unsigned char *) &sdr_info_resp);
	printf("%d records (%d bytes free)\n",sdr_info_resp.record_count, sdr_info_resp.free_space);
	cnt = sdr_info_resp.record_count;

	sdr_next_record = 0x0000;
	sdr_current_record = 0x0000;

	printf("Downloading records..");
	fflush(stdout);
	while(1)
		{
		printf("%d..",cnt);
		fflush(stdout);

		request.reservation_id = sdr_res_resp.res_id;
		request.record_id = sdr_current_record;
		request.offset_into_record = 0;
		request.bytes_to_read = sizeof(SDR_HEADER);
		INIT_XFER(xferp);
		SET_REQUEST_LUN(xferp, 0);
		SET_REQUEST_NETFN(xferp, STORAGE_REQUEST);
		SET_REQUEST_CMD(xferp, CMD_GET_SDR);
		SET_REQUEST_DATA(xferp, (unsigned char *) &request, sizeof(request));
		if ((rc = ioctl(fd, IOCTL_IPMI_XFER, (void *) &xfer))<0)
			{
			printf("dl_sdr(): failed (%m)\n");
			return(-1);
			}
		GET_RESPONSE_CC(xferp, cc);
		if (cc != 0x00)
			{
			printf("dl_sdr(): failed (cc = 0x%.2x)\n",cc);
			return(-1);
			}
		memset(&hunk_buffer[0], 0, sizeof(hunk_buffer));
		GET_RESPONSE_DATA(xferp, (unsigned char *) &hunk_buffer[0]);

		next_record_buffer = (unsigned short *) &hunk_buffer[0];
		sdr_hdr_buffer = (SDR_HEADER *) &hunk_buffer[(sizeof(unsigned short))];
		sdr_next_record = *next_record_buffer;
		sdr_current_record = sdr_hdr_buffer->record_id;
//		printf("record 0x%.4x, rec len = %d, next 0x%.4x\n",sdr_current_record,sdr_hdr_buffer->record_length,sdr_next_record);
	
		/* Download full record */
		sdr_buffer_pos = 0;
		brtl = sdr_hdr_buffer->record_length + sizeof(SDR_HEADER);
		while(brtl)
			{
			if (brtl <= 16)
				hunk_size = brtl;
			else
				hunk_size = 16;

			memset(&request, 0, sizeof(request));
			request.reservation_id = sdr_res_resp.res_id;
			request.record_id = sdr_current_record;
			request.offset_into_record = sdr_buffer_pos;
			request.bytes_to_read = hunk_size;

#if 0
      printf("SDR block request (record 0x%.4x, size %d, remain %d, pos %d, res 0x%.4x)\n",
             sdr_current_record,
             hunk_size,
             brtl, sdr_buffer_pos, sdr_res_resp.res_id);
#endif
			INIT_XFER(xferp);
			SET_REQUEST_LUN(xferp, 0);
			SET_REQUEST_NETFN(xferp, STORAGE_REQUEST);
			SET_REQUEST_CMD(xferp, CMD_GET_SDR);
			SET_REQUEST_DATA(xferp, (unsigned char *) &request, sizeof(request));
			if ((rc = ioctl(fd, IOCTL_IPMI_XFER, (void *) &xfer))<0)
				{
				printf("dl_sdr(): failed (%m)\n");
				return(-1);
				}
			GET_RESPONSE_CC(xferp, cc);
			if (cc != 0x00)
				{
				printf("dl_sdr(): failed (cc = 0x%.2x)\n",cc);
				return(-1);
				}
			len = GET_RESPONSE_DATA_LENGTH(xferp, len);
			len -=2;
			memset(&hunk_buffer[0], 0, sizeof(hunk_buffer));
			GET_RESPONSE_DATA(xferp, (unsigned char *) &hunk_buffer[0]);
			next_record_buffer = (unsigned short *) &hunk_buffer[0];
			sdr_hdr_buffer = (SDR_HEADER *) &hunk_buffer[(sizeof(unsigned short))];
			memcpy(&sdr_buffer[sdr_buffer_pos], &hunk_buffer[2], (len));
			sdr_buffer_pos += len;
			brtl -= len;
			}
		process_sdr_record(&sdr_buffer[0]);
		if (sdr_next_record == (unsigned short) 0xffff)
			{
			printf("\n");
			return(0);
			}
		sdr_current_record = sdr_next_record;
		cnt--;
		}
}

/*
 * SDR decode tables and tools
 */
 
/*
 * Tables and whatnot for looking up english..
 */
 
struct  ipmb_i2c_device_type_codes
  {
  unsigned  char  code;
  unsigned  char  modifier_type;  /* 00 = no modifiers, 0x1 = bit, 0x2 = byte */
  unsigned  char  modifier_code;  
  unsigned  char  *name;
  };
  
struct  ipmb_i2c_device_type_codes  ipmb_i2c_device_type_table[] = {
  { 0x00, 0x01, 0x00, "IPMB Device (Firmware Transfer Support)" }, 
  { 0x00, 0x01, 0x01, "IPMB Device (ICMB Bridge)"               }, 
  { 0x00, 0x01, 0x02, "IPMB Device (Event Generator)"           }, 
  { 0x00, 0x01, 0x03, "IPMB Device (Primary Event Receiver)"    }, 
  { 0x00, 0x01, 0x04, "IPMB Device (SDR Repository Device)"     }, 
  { 0x00, 0x01, 0x05, "IPMB Device (SEL Device)"                }, 
  { 0x00, 0x01, 0x06, "IPMB Device (Chassis Device)"            }, 
  { 0x00, 0x01, 0x07, "IPMB Device (FRU Inventory Device)"      }, 
  
  { 0x01, 0x01, 0x00, "SMM Card IPMB Device (Firmware Transfer Support)"  },
  { 0x01, 0x01, 0x01, "SMM Card IPMB Device (ICMB Bridge)"                },
  { 0x01, 0x01, 0x02, "SMM Card IPMB Device (Event Receiver)"             },
  { 0x01, 0x01, 0x03, "SMM Card IPMB Device (Reserved)"                   },
  { 0x01, 0x01, 0x04, "SMM Card IPMB Device (Reserved)"                   },
  { 0x01, 0x01, 0x05, "SMM Card IPMB Device (Reserved)"                   },
  { 0x01, 0x01, 0x06, "SMM Card IPMB Device (Reserved)"                   },
  { 0x01, 0x01, 0x07, "SMM Card IPMB Device (Reserved)"                   },
  
  { 0x02, 0x00, 0x00, "DS1624 Temperature Sensor / EEPROM or Equivalent"  },
  { 0x03, 0x00, 0x00, "DS1621 Temperature Sensor or Equivalent"           },
  { 0x04, 0x00, 0x00, "LM75 Temperature Sensor or Equivalent"             },
  
  { 0x05, 0x02, 0x00, "Heceta ASIC or equivalent (HECETA 1 [LM 78])"  },
  { 0x05, 0x02, 0x01, "Heceta ASIC or equivalent (HECETA 2 [LM 79])"  },
  { 0x05, 0x02, 0x02, "Heceta ASIC or equivalent (HECETA 3)"          },
  
  { 0x08, 0x02, 0x00, "EEPROM, 24C01 or equivalent (Use: unspecified)"                },
  { 0x08, 0x02, 0x01, "EEPROM, 24C01 or equivalent (Use: DIMM Memory ID)"             },
  { 0x08, 0x02, 0x02, "EEPROM, 24C01 or equivalent (Use: FRU Inventory)"              },
  { 0x08, 0x02, 0x03, "EEPROM, 24C01 or equivalent (Use: System Processor SECC FRU)"  },
  
  { 0x09, 0x02, 0x00, "EEPROM, 24C02 or equivalent (Use: unspecified)"                },
  { 0x09, 0x02, 0x01, "EEPROM, 24C02 or equivalent (Use: DIMM Memory ID)"             },
  { 0x09, 0x02, 0x02, "EEPROM, 24C02 or equivalent (Use: FRU Inventory)"              },
  { 0x09, 0x02, 0x03, "EEPROM, 24C02 or equivalent (Use: System Processor SECC FRU)"  },
  
  { 0x0A, 0x02, 0x00, "EEPROM, 24C04 or equivalent (Use: unspecified)"                },
  { 0x0A, 0x02, 0x01, "EEPROM, 24C04 or equivalent (Use: DIMM Memory ID)"             },
  { 0x0A, 0x02, 0x02, "EEPROM, 24C04 or equivalent (Use: FRU Inventory)"              },
  { 0x0A, 0x02, 0x03, "EEPROM, 24C04 or equivalent (Use: System Processor SECC FRU)"  },
  
  { 0x0B, 0x02, 0x00, "EEPROM, 24C08 or equivalent (Use: unspecified)"                },
  { 0x0B, 0x02, 0x01, "EEPROM, 24C08 or equivalent (Use: DIMM Memory ID)"             },
  { 0x0B, 0x02, 0x02, "EEPROM, 24C08 or equivalent (Use: FRU Inventory)"              },
  { 0x0B, 0x02, 0x03, "EEPROM, 24C08 or equivalent (Use: System Processor SECC FRU)"  },
  
  { 0x0C, 0x02, 0x00, "EEPROM, 24C16 or equivalent (Use: unspecified)"                },
  { 0x0C, 0x02, 0x01, "EEPROM, 24C16 or equivalent (Use: DIMM Memory ID)"             },
  { 0x0C, 0x02, 0x02, "EEPROM, 24C16 or equivalent (Use: FRU Inventory)"              },
  { 0x0C, 0x02, 0x03, "EEPROM, 24C16 or equivalent (Use: System Processor SECC FRU)"  },
  
  { 0x0D, 0x02, 0x00, "EEPROM, 24C17 or equivalent (Use: unspecified)"                },
  { 0x0D, 0x02, 0x01, "EEPROM, 24C17 or equivalent (Use: DIMM Memory ID)"             },
  { 0x0D, 0x02, 0x02, "EEPROM, 24C17 or equivalent (Use: FRU Inventory)"              },
  { 0x0D, 0x02, 0x03, "EEPROM, 24C17 or equivalent (Use: System Processor SECC FRU)"  },
  
  { 0x0E, 0x02, 0x00, "EEPROM, 24C32 or equivalent (Use: unspecified)"                },
  { 0x0E, 0x02, 0x01, "EEPROM, 24C32 or equivalent (Use: DIMM Memory ID)"             },
  { 0x0E, 0x02, 0x02, "EEPROM, 24C32 or equivalent (Use: FRU Inventory)"              },
  { 0x0E, 0x02, 0x03, "EEPROM, 24C32 or equivalent (Use: System Processor SECC FRU)"  },
  
  { 0x0F, 0x02, 0x00, "EEPROM, 24C64 or equivalent (Use: unspecified)"                },
  { 0x0F, 0x02, 0x01, "EEPROM, 24C64 or equivalent (Use: DIMM Memory ID)"             },
  { 0x0F, 0x02, 0x02, "EEPROM, 24C64 or equivalent (Use: FRU Inventory)"              },
  { 0x0F, 0x02, 0x03, "EEPROM, 24C64 or equivalent (Use: System Processor SECC FRU)"  },
  
  { 0x10, 0x00, 0x00, "Other FRU Inventory Device Behind Management Controller" },
  
  { 0x14, 0x00, 0x00, "PCF 8570 256 byte RAM or equivalent"                           },
  { 0x15, 0x00, 0x00, "PCF 8573 clock/calendar or equivalent"                         },
  { 0x16, 0x00, 0x00, "PCF 8574A I/O port or equivalent"                              },
  { 0x17, 0x00, 0x00, "PCF 8583 clock/calendar or equivalent"                         },
  { 0x18, 0x00, 0x00, "PCF 8593 clock/calendar or equivalent"                         },
  { 0x19, 0x00, 0x00, "Clock/calendar, type not specified"                            },
  { 0x1A, 0x00, 0x00, "PCF 8591 A/D, D/A Converter or equivalent"                     },
  { 0x1B, 0x00, 0x00, "I/O port, specific device not specified"                       },
  { 0x1C, 0x00, 0x00, "A/D Converter, specific device not specified"                  },
  { 0x1D, 0x00, 0x00, "D/A Converter, specific device not specified"                  },
  { 0x1E, 0x00, 0x00, "A/D,D/A Converter, specific device not specified"              },
  { 0x1F, 0x00, 0x00, "LCD Controller/Driver, specific device not specified"          },
  { 0x20, 0x00, 0x00, "Core Logic Device, specific device not specified"              },
  { 0x21, 0x00, 0x00, "LMC6874 Intelligent Battery Controller or equivalent"          },
  { 0x22, 0x00, 0x00, "Intelligent Battery Controller, specific device not specified" },
  { 0x23, 0x00, 0x00, "Combo Management ASIC, specific device not specified"          },
  { 0xFF, 0xFF, 0xFF, NULL}};
  
  
struct  device_location_codes
  {
  unsigned  char  code;
  char            *name;
  };
  
static struct  device_location_codes device_location_table[] = {
  { 0x00, "Reserved"                  },
  { 0x01, "Other"                     },
  { 0x02, "Unknown (Unspecified)"     },
  { 0x03, "Processor"                 },
  { 0x04, "Disk or Disk Bay"          },
  { 0x05, "Peripheral Bay"            },
  { 0x06, "System Management Module"  },
  { 0x07, "System Board"              },
  { 0x08, "Memory Module"             },
  { 0x09, "Processor Module"          },
  { 0x0A, "Power Unit"                },
  { 0x0B, "Add-in Card"               },
  { 0x0C, "Front Panel Board"         },
  { 0x0D, "Back Panel Board"          },
  { 0x0E, "Drive Backplane"           },
  { 0xFF, NULL}};
  
struct  entity_id_codes
  {
  unsigned  char  entity_id;
  char            *name;
  };
  
static struct entity_id_codes  entity_id_table[] = {
  { 0x00, "ID Unspecified"              },
  { 0x01, "Main System Board"           },
  { 0x02, "System Board 2"              },
  { 0x03, "System Board 3"              },
  { 0x04, "System Board 4"              },
  { 0x08, "System Expansion Board 1"    },
  { 0x09, "System Expansion Board 2"    },
  { 0x0A, "System Expansion Board 3"    },
  { 0x0B, "System Expansion Board 4"    },
  { 0x10, "Processor Board 1"           },
  { 0x11, "Processor Board 2"           },
  { 0x12, "Processor Board 3"           },
  { 0x13, "Processor Board 4"           },
  { 0x18, "Processor 1"                 },
  { 0x19, "Processor 2"                 },
  { 0x1A, "Processor 3"                 },
  { 0x1B, "Processor 4"                 },
  { 0x1C, "Processor 5"                 },
  { 0x1D, "Processor 6"                 },
  { 0x1E, "Processor 7"                 },
  { 0x1F, "Processor 8"                 },
  { 0x20, "Memory Module 1"             },
  { 0x21, "Memory Module 2"             },
  { 0x22, "Memory Module 3"             },
  { 0x23, "Memory Module 4"             },
  { 0x28, "Disk Backplane 1"            },
  { 0x29, "Disk Backplane 2"            },
  { 0x2A, "Disk Backplane 3"            },
  { 0x2B, "Disk Backplane 4"            },
  { 0x2C, "Power Supply 1"              },
  { 0x2D, "Power Supply 2"              },
  { 0x2E, "Power Supply 3"              },
  { 0x2F, "Power Supply 4"              },
  { 0x30, "Power Supply 5"              },
  { 0x31, "Power Supply 6"              },
  { 0x32, "Power Supply 7"              },
  { 0x33, "Power Supply 8"              },
  { 0x34, "Power Management Board 1"    },
  { 0x35, "Power Management Board 2"    },
  { 0x36, "Power Management Board 3"    },
  { 0x37, "Power Management Board 4"    },
  { 0x3C, "Power Module 1"              },
  { 0x3D, "Power Module 2"              },
  { 0x3E, "Power Module 3"              },
  { 0x3F, "Power Module 4"              },
  { 0x44, "System Management Card 1"    },
  { 0x45, "System Management Card 2"    },
  { 0x46, "System Management Card 3"    },
  { 0x47, "System Management Card 4"    },
  { 0x4C, "Chassis Board 1"             },
  { 0x4D, "Chassis Board 2"             },
  { 0x4E, "Chassis Board 3"             },
  { 0x4F, "Chassis Board 4"             },
  { 0x54, "Chassis Front Panel Board 1" },
  { 0x55, "Chassis Front Panel Board 2" },
  { 0x56, "Chassis Front Panel Board 3" },
  { 0x57, "Chassis Front Panel Board 4" },
  { 0x5C, "Chassis Back Panel Board 1"  },
  { 0x5D, "Chassis Back Panel Board 2"  },
  { 0x5E, "Chassis Back Panel Board 3"  },
  { 0x5F, "Chassis Back Panel Board 4"  },
  { 0x64, "Chassis Management Board 1"  },
  { 0x65, "Chassis Management Board 2"  },
  { 0x66, "Chassis Management Board 3"  },
  { 0x67, "Chassis Management Board 4"  },
  { 0x6C, "Disk Drive Bay 1"            },
  { 0x6D, "Disk Drive Bay 2"            },
  { 0x6E, "Disk Drive Bay 3"            },
  { 0x6F, "Disk Drive Bay 4"            },
  { 0x70, "Disk Drive Bay 5"            },
  { 0x71, "Disk Drive Bay 6"            },
  { 0x72, "Disk Drive Bay 7"            },
  { 0x73, "Disk Drive Bay 8"            },
  { 0x74, "Disk Drive Bay 9"            },
  { 0x75, "Disk Drive Bay 10"           },
  { 0x76, "Disk Drive Bay 11"           },
  { 0x77, "Disk Drive Bay 12"           },
  { 0x78, "Disk Drive Bay 13"           },
  { 0x79, "Disk Drive Bay 14"           },
  { 0x7A, "Disk Drive Bay 15"           },
  { 0x7B, "Disk Drive Bay 16"           },
  { 0x7C, "Peripheral Bay 1"            },
  { 0x7D, "Peripheral Bay 2"            },
  { 0x7E, "Peripheral Bay 3"            },
  { 0x7F, "Peripheral Bay 4"            },
  { 0x80, "Peripheral Bay 5"            },
  { 0x81, "Peripheral Bay 6"            },
  { 0x82, "Peripheral Bay 7"            },
  { 0x83, "Peripheral Bay 8"            },
  { 0xFF, NULL                          }};
  
#define STRING_DATA_TYPE_BINARY         0x00
#define STRING_DATA_TYPE_BCD_PLUS       0x01
#define STRING_DATA_TYPE_SIX_BIT_ASCII  0x02
#define STRING_DATA_TYPE_LANG_DEPENDANT 0x03

void decode_string(unsigned char type,
                          unsigned char language_code,
                          unsigned char *source,
                          char          *target,
                          int           size)
{                         
  unsigned char *s = &source[0];
  unsigned char *d = &target[0];
  
  memset(target, 0, size);
  if (type == STRING_DATA_TYPE_BCD_PLUS)
    {
    while(size)
      {
//   if ((*s >= (unsigned char) 0x00) && (*s <= (unsigned char) 0x09))
      if (*s <= (unsigned char) 0x09) 
        *d++ = (unsigned char) 0x30 + (unsigned char) *s++;
      else
        {
        if (*s == (unsigned char) 0x0A)
          *d++ = (unsigned char) ' ';
        else if (*s == (unsigned char) 0x0B)
          *d++ = (unsigned char) '-';
        else if (*s == (unsigned char) 0x0C)
          *d++ = (unsigned char) '.';
        else if ((*s <= (unsigned char) 0x0D) && (*s <= (unsigned char) 0x0F))
          {
          *d++ = (unsigned char) '*';
          }
        s++;
        }
      size --;
      }
    } 
  else if (type == STRING_DATA_TYPE_SIX_BIT_ASCII)
    {
    printf("Six bit ASCII decode not supported\n");
    }
  else if (type == STRING_DATA_TYPE_LANG_DEPENDANT)
    {
    if ((language_code == 0x00) || (language_code == 0x25))
      {
      strncpy(target, source, size);
      target[size] = 0x0;
      }
    else
      {
      printf("Language 0x%x dependant decode not supported\n",language_code);
      return;
      }
    } 
  else
    {
    sprintf(target,"Unknown Language Type 0x%.2x",type);
    return;
    }
}   

static void process_sdr_record(unsigned char *buffer)
{
  SDR_HEADER    *rec_header;
  SENSOR        *sensor_new;
  char          tmp[255];
  
  rec_header = (SDR_HEADER *) &buffer[0];
  
  if ((rec_header->record_type != 0x01) &&
      (rec_header->record_type != 0x02) &&
      (rec_header->record_type != 0x10) &&
      (rec_header->record_type != 0xC0))
    {
    printf(
            "Bad SDR header (record 0x%.4x, type 0x%x)\n",
            rec_header->record_id,
            rec_header->record_type);
    return;
    }

 #if 0
  printf("process_sdr_record(): Processing record 0x%.4x, type 0x%x\n",
          rec_header->record_id,
          rec_header->record_type);
#endif
  sensor_new = sensor_create();
  
  memcpy(&sensor_new->rec_header,
         rec_header,
         sizeof(SDR_HEADER));
  
  if (sensor_new->rec_header.record_type == 0x01)
    {
    memcpy(&sensor_new->sensor_type.type_01_sensor,
           &buffer[0],
           sensor_new->rec_header.record_length+sizeof(SDR_HEADER));
 #if 0
    decode_string(sensor_new->sensor_type.type_01_sensor.id_string_type_code,
                  0,
                  &sensor_new->sensor_type.type_01_sensor.id_string[0],
                  &tmp[0],
                  sensor_new->sensor_type.type_01_sensor.id_string_num_bytes);
    printf("Internal Sensor Found -> '%s'\n",tmp);
#endif
    }
  else if (sensor_new->rec_header.record_type == 0x02)
    {
    memcpy(&sensor_new->sensor_type.type_02_sensor,
           &buffer[0],
           sensor_new->rec_header.record_length + sizeof(SDR_HEADER));
 #if 0
    decode_string(sensor_new->sensor_type.type_02_sensor.id_string_type_code,
                  0,
                  &sensor_new->sensor_type.type_02_sensor.id_string[0],
                  &tmp[0],
                  sensor_new->sensor_type.type_02_sensor.id_string_num_bytes);
    printf("Discrete/Digital Sensor Found -> '%s'\n",tmp);
#endif
    }
  else if (sensor_new->rec_header.record_type == 0x10)
    {
    memcpy(&sensor_new->sensor_type.type_10_sensor,
           &buffer[0],
           sensor_new->rec_header.record_length+sizeof(SDR_HEADER));
    if (!return_device_type_entry(
            sensor_new->sensor_type.type_10_sensor.device_type,
            sensor_new->sensor_type.type_10_sensor.device_type_modifier,
            &tmp[0]))
      {
      printf(
              "Unable to lookup device with code 0x%x, modifier 0x%x\n",
              sensor_new->sensor_type.type_10_sensor.device_type,
              sensor_new->sensor_type.type_10_sensor.device_type_modifier);
      }
    else
      {
 #if 0 
      printf("process_sdr_record(): IPMB '%s' found \n",tmp);
#endif
      }
    } 
  else if (sensor_new->rec_header.record_type == 0xC0)
    {
#if 0
    memcpy(&sensor_new->sensor_type.type_c0_sensor,
           &buffer[0],
           sensor_new->rec_header.record_length +5);
    printf("process_sdr_record(): OEM Record Found\n");
#endif
    }
}

/*
 * Looks up the proper type_code and type_modifier in the ipmb i2c device
 * type table and returns the type. If multiple types are available, they
 * are returned '*' separated 
 */
int return_device_type_entry(unsigned char type_code,
                             unsigned char type_modifier,
                             char          *buffer)
{                            
  int i;                     
  
  for (i=0; ipmb_i2c_device_type_table[i].code != 0xFF; i++)
    { 
    if (ipmb_i2c_device_type_table[i].code == type_code)
      {
      if (ipmb_i2c_device_type_table[i].modifier_type == 0)
        {
        strcpy(buffer, ipmb_i2c_device_type_table[i].name);
        return(1);
        }
      else if (ipmb_i2c_device_type_table[i].modifier_type == 1)
        {
        unsigned char bitmask;
        
        strcpy(buffer, "");
        for (; ipmb_i2c_device_type_table[i].code == type_code; i++)
          { 
          bitmask = 0x01 << ipmb_i2c_device_type_table[i].modifier_code;
          if (bitmask & type_modifier)
            {
            strcat(buffer, ipmb_i2c_device_type_table[i].name);
            strcat(buffer, "*");
            }
          } 
        if (buffer[0] == (char) 0x00)
          {
          printf("return_device_type_entry(): T 0x%x, B mod mask 0x%x not found\n",
                 type_code,
                 type_modifier);
          return(0);
          }
        else
          {
          buffer[(strlen(buffer)-1)] = 0x00;
          }
        return(1);
        }
      else if (ipmb_i2c_device_type_table[i].modifier_type == 2)
        {
        for (; ipmb_i2c_device_type_table[i].code == type_code; i++)
          { 
          if (ipmb_i2c_device_type_table[i].modifier_code == type_modifier)
            {
            strcpy(buffer, ipmb_i2c_device_type_table[i].name);
            return(1);
            }
          printf("return_device_type_entry(): T 0x%x, B mod 0x%x not found\n",
                 type_code,
                 type_modifier);
          return(0);
          }
        } 
      else
        {
        printf("return_device_type_entry(): T 0x%x has bad M type 0x%x\n",
               ipmb_i2c_device_type_table[i].code, 
               ipmb_i2c_device_type_table[i].modifier_type);
        return(0);
        }
      } 
    } 
  return(0);
} 

static char *reading_type_to_str(unsigned char value)
{
  switch (value)
    {
    case NO_PRESENT_READING_SUPPORT:
      return("NO_PRESENT_READING_SUPPORT");
    case PRESENT_READING_TYPE_NOT_SPECIFIED:
      return("PRESENT_READING_TYPE_NOT_SPECIFIED");
    case DIGITAL_DISCRETE:
      return("DIGITAL_DISCRETE");
    case ANALOG:
      return("ANALOG");
    default:
      return("UNKNOWN");
    } 
}   

static char *linearization_to_str(unsigned char value)
{
  switch (value)
    {
    case LINEARIZATION_LINEAR:
      return("LINEAR");
    case LINEARIZATION_LN:
      return("LN");
    case LINEARIZATION_LOG10:
      return("LOG10");
    case LINEARIZATION_LOG2:
      return("LOG2");
    case LINEARIZATION_E:
      return("E");
    case LINEARIZATION_EXP10:
      return("EXP10");
    case LINEARIZATION_EXP2:
      return("EXP2");
    case LINEARIZATION_1_DIV_X:
      return("1 DIV X");
    case LINEARIZATION_SQR_X:
      return("SQR X");
    case LINEARIZATION_CUBE_X:
      return("CUBE X");
    case LINEARIZATION_SQRT_X:
      return("SQRT X");
    case LINEARIZATION_CUBE_MINUS_1_X:
      return("CUBE MINUS 1 X");
    default:
      return("UNKNOWN");
    } 
}   

static double linearize_sensor_reading(SENSOR *slist, unsigned char reading)
{
  struct bit10 {
  int bits : 10;
  };
  struct bit10 M, B;
  char K1 = 0, K2 = 0;
#if 0
  char            M = 0, B = 0, K1 = 0, K2 = 0;
#endif            
  double          pre_lin;
  unsigned char   x;
  unsigned char   function;
  
  x = reading;
  if (slist->rec_header.record_type == 0x01)
    {
    M.bits =  slist->sensor_type.type_01_sensor.M_ms2bits;
    M.bits =  M.bits << 8;
    M.bits |= (slist->sensor_type.type_01_sensor.M_ls8bits & 0xff);
    
    B.bits =  slist->sensor_type.type_01_sensor.B_ms2bits;
    B.bits =  B.bits << 8;
    B.bits |= (slist->sensor_type.type_01_sensor.B_ls8bits & 0xff);
    
    K1 = slist->sensor_type.type_01_sensor.B_exp;
    K2 = slist->sensor_type.type_01_sensor.R_exp;
    
    function = slist->sensor_type.type_01_sensor.linearization;
    }
  else
    {
    printf("linearize_sensor_reading(): record type 0x%x not supported\n",
            slist->rec_header.record_type);
    return(0);
    }
//  printf("x=%d, m=%d, b=%d, k1=%d, k2=%d, linear=%d\n",x,M.bits,B.bits,K1,K2,function);

  pre_lin = ((M.bits * x) + (B.bits * pow(10, K1))) * pow(10, K2);
  switch(function)
    {
    case LINEARIZATION_LINEAR:
      break;
    default:
      printf("linearize_sensor_reading(): linearization type %x not supported\n",function);
      return(0);
    } 
  return(pre_lin);
}

/* Sensor event stuff */
struct sensor_event_descriptions
  {
  unsigned char code;
  unsigned char sensor_class;
  unsigned char offset;
  char          *desc;
  };
  
static struct sensor_event_descriptions event_desc_table[] = {
  { 0x01, SENSOR_CLASS_THRESHOLD, 0x00, "Lower Non-critical - going low"      },
  { 0x01, SENSOR_CLASS_THRESHOLD, 0x01, "Lower Non-critical - going high"   },
  { 0x01, SENSOR_CLASS_THRESHOLD, 0x02, "Lower Critical - going low"          },
  { 0x01, SENSOR_CLASS_THRESHOLD, 0x03, "Lower Critical - going high"       },
  { 0x01, SENSOR_CLASS_THRESHOLD, 0x04, "Lower Non-recoverable - going low" },
  { 0x01, SENSOR_CLASS_THRESHOLD, 0x05, "Lower Non-recoverable - going high"  },
  { 0x01, SENSOR_CLASS_THRESHOLD, 0x06, "Upper Non-critical - going low"      },
  { 0x01, SENSOR_CLASS_THRESHOLD, 0x07, "Upper Non-critical - going high"   },
  { 0x01, SENSOR_CLASS_THRESHOLD, 0x08, "Upper Critical - going low"          },
  { 0x01, SENSOR_CLASS_THRESHOLD, 0x09, "Upper Critical - going high"       },
  { 0x01, SENSOR_CLASS_THRESHOLD, 0x0A, "Upper Non-recoverable - going low" },
  { 0x01, SENSOR_CLASS_THRESHOLD, 0x0B, "Upper Non-recoverable - going low" },
  
  { 0x02, SENSOR_CLASS_DISCRETE,  0x00, "Transition to Idle"    },
  { 0x02, SENSOR_CLASS_DISCRETE,  0x01, "Transition to Active"  },
  { 0x02, SENSOR_CLASS_DISCRETE,  0x02, "Transition to Busy"    },
  
  { 0x03, SENSOR_CLASS_DIGITAL, 0x00, "State Deasserted"  },
  { 0x03, SENSOR_CLASS_DIGITAL, 0x01, "State Asserted"    },
  
  { 0x04, SENSOR_CLASS_DIGITAL, 0x00, "Predictive Failure Deasserted" },
  { 0x04, SENSOR_CLASS_DIGITAL, 0x01, "Predictive Failure Asserted"   },
  
  { 0x05, SENSOR_CLASS_DIGITAL, 0x00, "Limit Not Exceeded"  },
  { 0x05, SENSOR_CLASS_DIGITAL, 0x01, "Limit Exceeded"      },
  
  { 0x06, SENSOR_CLASS_DIGITAL, 0x00, "Performance Met" },
  { 0x06, SENSOR_CLASS_DIGITAL, 0x01, "Performance Lags"  },
  
  { 0x07, SENSOR_CLASS_DISCRETE, 0x00, "Transition to OK"                                 },
  { 0x07, SENSOR_CLASS_DISCRETE, 0x01, "Transition to Non-Critical from OK"             },
  { 0x07, SENSOR_CLASS_DISCRETE, 0x02, "Transition to Critical from less severe"          },
  { 0x07, SENSOR_CLASS_DISCRETE, 0x03, "Transition to Non-recoverable from less severe" },
  { 0x07, SENSOR_CLASS_DISCRETE, 0x04, "Transition to Non-Critical from more severe"      },
  { 0x07, SENSOR_CLASS_DISCRETE, 0x05, "Transition to Critical from Non-recoverable"      },
  { 0x07, SENSOR_CLASS_DISCRETE, 0x06, "Transition to Non-recoverable"                    },
  { 0x07, SENSOR_CLASS_DISCRETE, 0x07, "Monitor"                                        },
  { 0x07, SENSOR_CLASS_DISCRETE, 0x08, "Informational"                                  },
  
  { 0x08, SENSOR_CLASS_DIGITAL, 0x00, "Device Removed"    },
  { 0x08, SENSOR_CLASS_DIGITAL, 0x01, "Device Inserted" },
  
  { 0x09, SENSOR_CLASS_DIGITAL, 0x00, "Device Disabled" },
  { 0x09, SENSOR_CLASS_DIGITAL, 0x01, "Device Enabled"    },
  
  { 0x0A, SENSOR_CLASS_DISCRETE, 0x00, "Transition to Running"        },
  { 0x0A, SENSOR_CLASS_DISCRETE, 0x01, "Transition to In Test"        },
  { 0x0A, SENSOR_CLASS_DISCRETE, 0x02, "Transition to Power Off"      },
  { 0x0A, SENSOR_CLASS_DISCRETE, 0x03, "Transition to On Line"        },
  { 0x0A, SENSOR_CLASS_DISCRETE, 0x04, "Transition to Off Line"     },
  { 0x0A, SENSOR_CLASS_DISCRETE, 0x05, "Transition to Off Duty"     },
  { 0x0A, SENSOR_CLASS_DISCRETE, 0x06, "Transition to Degraded"     },
  { 0x0A, SENSOR_CLASS_DISCRETE, 0x07, "Transition to Power Save"   },
  { 0x0A, SENSOR_CLASS_DISCRETE, 0x08, "Transition to Install Error"  },
  
  { 0x0B, SENSOR_CLASS_DISCRETE, 0x00, "Redundancy Regained"  },
  { 0x0B, SENSOR_CLASS_DISCRETE, 0x01, "Redundancy Lost"      },
  { 0x0B, SENSOR_CLASS_DISCRETE, 0x02, "Redundancy Degraded"  },
  { 0x00, SENSOR_CLASS_UNKNOWN, 0x00, NULL}};
int sensor_get_class(unsigned char sensor_reading_type_code)
{
  int i = 0;
  
  while (event_desc_table[i].desc != NULL)
    {
    if (event_desc_table[i].code == sensor_reading_type_code)
      return(event_desc_table[i].sensor_class);
    i++;
    }
  return(SENSOR_CLASS_UNKNOWN);
} 

char *sensor_get_trigger_description(unsigned char sensor_reading_type_code,
                                   unsigned char offset)
{                                  
  int i = 0;                       
  
  while(event_desc_table[i].desc != NULL)
    {
    if ((event_desc_table[i].code == sensor_reading_type_code) &&
        (event_desc_table[i].offset == offset))
      return(event_desc_table[i].desc);
    i++;
    }
  return(NULL);
} 

struct sensor_unit_type
  {
  unsigned char code;
  unsigned char *unit;
  };
  
static struct sensor_unit_type sensorunittypelist[] = {
  { 0, "unspecified"      },
  { 1, "degrees C"      },
  { 2, "degrees F"      },
  { 3, "degrees K"      },
  { 4, "Volts"      },  
  { 5, "Amps"     },
  { 6, "Watts"      },
  { 7, "Joules"     },
  { 8, "Coulombs"     },
  { 9, "VA"     },    
  { 10, "Nits"      },
  { 11, "lumen"     },
  { 12, "lux"     },
  { 13, "Candela"     },
  { 14, "kPa"     },  
  { 15, "PSI"     },  
  { 16, "Newton"      },
  { 17, "CFM"     },  
  { 18, "RPM"     },  
  { 19, "Hz"      },  
  { 20, "microsecond"     },
  { 21, "millisecond"     },
  { 22, "second"      },  
  { 23, "minute"      },  
  { 24, "hour"      },
  { 25, "day"     },
  { 26, "week"      },
  { 27, "mil"     },
  { 28, "inches"      },
  { 29, "feet"      },
  { 30, "cu in"     },
  { 31, "cu feet"     },
  { 32, "mm"      },  
  { 33, "cm"      },  
  { 34, "m"     },
  { 35, "cu cm"     },
  { 36, "cu m"      },
  { 37, "liters"      },
  { 38, "fluid ounce"     },
  { 39, "radians"     },  
  { 40, "steradians"      },
  { 41, "revolutions"     },
  { 42, "cycles"      },  
  { 43, "gravities"     },
  { 44, "ounce"     },  
  { 45, "pound"     },  
  { 46, "ft-lb"     },  
  { 47, "oz-in"     },  
  { 48, "gauss"     },  
  { 49, "gilberts"      },
  { 50, "henry"     },  
  { 51, "millihenry"      },
  { 52, "farad"     },    
  { 53, "microfarad"      },
  { 54, "ohms"      },    
  { 55, "siemens"     },  
  { 56, "mole"      },
  { 57, "becquerel"     },
  { 58, "PPM"     },    
  { 59, "reserved"      },
  { 60, "Decibels"      },
  { 61, "DbA"     },    
  { 62, "DbC"     },    
  { 63, "gray"      },  
  { 64, "sievert"     },
  { 65, "color temp deg K"      },
  { 66, "bit"     },
  { 67, "kilobit"     },
  { 68, "megabit"     },
  { 69, "gigabit"     },
  { 70, "byte"      },
  { 71, "kilobyte"      },
  { 72, "megabyte"      },
  { 73, "gigabyte"      },
  { 74, "word"      },  
  { 75, "dword"     },  
  { 76, "qword"     },  
  { 77, "line"      },  
  { 78, "hit"     },
  { 79, "miss"      },
  { 80, "retry"     },
  { 81, "reset"     },
  { 82, "overrun / overflow"      },
  { 83, "underrun"      },
  { 84, "collision"     },
  { 85, "packets"     },
  { 86, "messages"      },
  { 87, "characters"      },
  { 88, "error"     },    
  { 89, "correctable error"     },
  { 90, "uncorrectable error"     },
  { 0, NULL}};
  
struct sensortype
  {
  unsigned char code;
  char          *name;
  };
char *sensor_get_unit_type(unsigned char sensor_unit_type_code)
{
  int i = 0;
  
  while(sensorunittypelist[i].unit != NULL)
    {
    if (sensor_unit_type_code == sensorunittypelist[i].code)
      return(sensorunittypelist[i].unit);
    i++;
    }
  return(NULL);
} 
static struct sensortype sensortypelist[] = {
  { 0x01, "Temperature"              },
  { 0x02, "Voltage"                  },
  { 0x03, "Current"                  },
  { 0x04, "Fan"                      },
  { 0x05, "Physical Security"        },
  { 0x06, "Platform Security"        },
  { 0x07, "Processor"                },
  { 0x08, "Power Supply"             },
  { 0x09, "Power Unit"               },
  { 0x0a, "Cooling Device"           },
  { 0x0b, "Other Sensor"             },
  { 0x0c, "Memory"                   },
  { 0x0d, "Drive Slot (Bay)"         },
  { 0x0e, "POST Memory Resize"       },
  { 0x0f, "POST Error"               },
  { 0x10, "Event Logging Disabled"   },
  { 0x11, "Watchdog 1"               },
  { 0x12, "System Event"             },
  { 0x13, "Critical Interrupt"       },
  { 0x14, "Button"                   },
  { 0x15, "Module / Board"           },
  { 0x16, "Microcontroller / Coprocessor" },
  { 0x17, "Add-in Card"              },
  { 0x18, "Chasis"                   },
  { 0x19, "Chipset"                  },
  { 0x1a, "Other FRU"                },
  { 0x1b, "Cable / Interconnect"     },
  { 0x1c, "Terminator"               },
  { 0x1d, "System Boot"              },
  { 0x1e, "Boot Error"               },
  { 0x1f, "O/S Boot"                 },
  { 0x20, "O/S Critical Stop"        },
  { 0x21, "Slot / Connector"         },
  { 0x22, "System ACPI Power State"  },
  { 0x00, 0                         }};
  
char *sensor_get_type(unsigned char sensor_type_code)
{
  int i = 0;
  
  while(sensortypelist[i].name != NULL)
    {
    if (sensor_type_code == sensortypelist[i].code)
      return(sensortypelist[i].name);
    i++;
    }
  return(NULL);
} 

struct eventdesc
  {
  unsigned char sensor_type_code;
  unsigned char sensor_specific_offset;
  char          *desc;
  };
  
static struct eventdesc sensor_event_list[] = {
  { 0x05, 0x00, "General Chasis Intrusion" }, 
  { 0x05, 0x01, "Drive Bay Intrusion"      }, 
  { 0x05, 0x02, "I/O Card Area Intrusion"  }, 
  { 0x05, 0x03, "Processor Area Intrusion" }, 
  { 0x05, 0x04, "LAN Leash Lost"           }, 
  { 0x05, 0x05, "Unauthorized Dock/Undock" }, 
  
  { 0x06, 0x00, "Secure Mode Violation attempt"                       },
  { 0x06, 0x01, "Pre-boot Password Violation - user password"         },
  { 0x06, 0x02, "Pre-boot Password Violation - sys password"          },
  { 0x06, 0x03, "Pre-boot Password Violation - network boot password" },
  { 0x06, 0x04, "Other Pre-boot Password Violation"                   },
  { 0x06, 0x05, "Out-of-band Access Password Violation"               },
  
  { 0x07, 0x00, "IERR"                                        },
  { 0x07, 0x01, "Thermal Trip"                                },
  { 0x07, 0x02, "FBR1/BIST failure"                           },
  { 0x07, 0x03, "FBR2/Hang in POST failure"                   },
  { 0x07, 0x04, "FBR3/Processor Startup / Initialize failure" },
  { 0x07, 0x05, "Configuration Error"                         },
  { 0x07, 0x06, "SM BIOS 'Uncorrectable CPU-complex' error"   },
  { 0x07, 0x07, "Processor Presence Detected"                 },
  { 0x07, 0x08, "Processor Disabled"                          },
  { 0x07, 0x09, "Terminator Presence Detected"                },
  
  { 0x08, 0x00, "Presence Detected"             },
  { 0x08, 0x01, "Power Supply Failure Detected" },
  { 0x08, 0x02, "Predictive Failure Asserted"   },
  
  { 0x09, 0x00, "Power Off / Power Down"      },
  { 0x09, 0x01, "Power Cycle"                 },
  { 0x09, 0x02, "240 UA Power Down"           },
  { 0x09, 0x03, "Interlock Power Down"        },
  { 0x09, 0x04, "A/C Lost"                    },
  { 0x09, 0x05, "Soft Power Control Failure"  },
  { 0x09, 0x06, "Power Unit Failure Detected" },
  
  { 0x0c, 0x00, "Correctable ECC"     },
  { 0x0c, 0x01, "Uncorrectable ECC"   },
  { 0x0c, 0x02, "Parity"              },
  { 0x0c, 0x03, "Memory Scrub Failed" },
  
  { 0x10, 0x00, "Correctable Memory Error Logging Disabled" },
  { 0x10, 0x01, "Event 'Type' Logging Disabled"             },
  { 0x10, 0x02, "Log Area Reset / Cleared"                  },
  { 0x10, 0x03, "All Event Logging Disabled"                },
  
  { 0x11, 0x00, "BIOS Watchdog Reset"               },
  { 0x11, 0x01, "OS Watchdog Reset"                 },
  { 0x11, 0x02, "OS Watchdog Shutdown"              },
  { 0x11, 0x03, "OS Watchdog Power Down"            },
  { 0x11, 0x04, "OS Watchdog Power Cycle"           },
  { 0x11, 0x05, "OS Watchdog NMI"                   },
  { 0x11, 0x06, "OS Watchdog Expired"               },
  { 0x11, 0x07, "OS Watchdog Pre-timeout Interrupt" },
  
  { 0x12, 0x00, "System Reconfigured"                  },
  { 0x12, 0x01, "OEM System Boot Event"                },
  { 0x12, 0x02, "Undetermined System Hardware Failure" },
  
  { 0x13, 0x00, "Front Panel NMI"             },
  { 0x13, 0x01, "Bus Timeout"                 },
  { 0x13, 0x02, "I/O Channel Check NMI"       },
  { 0x13, 0x03, "Software NMI"                },
  { 0x13, 0x04, "PCI PERR"                    },
  { 0x13, 0x05, "PCI SERR"                    },
  { 0x13, 0x06, "EISA Fail Safe Timeout"      },
  { 0x13, 0x07, "Fatal NMI (port 61h, bit 7)" },
  
  { 0x1d, 0x00, "Initiated By Power up"        },
  { 0x1d, 0x01, "Initiated By Hard Reset"      },
  { 0x1d, 0x02, "Initiated By Warm Reset"      },
  { 0x1d, 0x03, "User Requested PXE Boot"      },
  { 0x1d, 0x04, "Autumatic Boot To Diagnostic" },
  
  { 0x1e, 0x00, "No Bootable Media"                                 },
  { 0x1e, 0x01, "Non-Bootable Diskette Left In Drive"               },
  { 0x1e, 0x02, "PXE Server Not Found"                              },
  { 0x1e, 0x03, "Invalid Boot Sector"                               },
  { 0x1e, 0x04, "Timeout Waiting For User Selection Of Boot Source" },
  
  { 0x1f, 0x00, "A: Boot Completed"                          },
  { 0x1f, 0x01, "C: Boot Completed"                          },
  { 0x1f, 0x02, "PXE Boot Completed"                         },
  { 0x1f, 0x03, "Diagnostic Boot Completed"                  },
  { 0x1f, 0x04, "CD-ROM Boot Completed"                      },
  { 0x1f, 0x05, "ROM Boot Completed"                         },
  { 0x1f, 0x06, "Boot Completed - Boot Device Not Specified" },
  
  { 0x20, 0x00, "Stop During O/S Load / Initialization" },
  { 0x20, 0x01, "Run-time Stop"                         },
  
  { 0x21, 0x00, "Fault Status Asserted"                          },
  { 0x21, 0x01, "Identify Status Asserted"                       },
  { 0x21, 0x02, "Slot / Connector Device Installed"              },
  { 0x21, 0x03, "Slot / Connector Ready For Device Installation" },
  { 0x21, 0x04, "Slot / Connector Ready For Device Removal"      },
  { 0x21, 0x05, "Slot Power Is Off"                              },
  { 0x21, 0x06, "Slot / Connector Device Removal Request"        },
  { 0x21, 0x07, "Interlock Asserted"                             },
  
  { 0x22, 0x00, "S0 / G0 'working'"                                          },
  { 0x22, 0x01, "S1 Sleeping With System H/W & Processor Context Maintained" },
  { 0x22, 0x02, "S2 Sleeping, Processor Context Lost"                        },
  { 0x22, 0x03, "S3 Sleeping, Processor & H/W Context Lost, Mem Retained"    },
  { 0x22, 0x04, "S4 Non-volatile Sleep / Suspend To Disk"                    },
  { 0x22, 0x05, "S5 / G2 Soft-Off"                                           },
  { 0x22, 0x06, "S4 / S5 Soft-Off, Exact S4 / S5 State Cannot Be Determined" },
  { 0x22, 0x07, "G3 / Mechanical Off"                                        },
  { 0x22, 0x08, "Sleeping In An S1, S2, Or S3 State"                         },
  { 0x22, 0x09, "G1 Sleeping"                                                },
  { 0x00, 0x00, NULL}};
  
char *sensor_get_canned_event_desc(unsigned char sensor_type_code,
                            unsigned char sensor_specific_offset)
{                           
  int i = 0;                
  
  while(sensor_event_list[i].desc != NULL)
    {
    if ((sensor_event_list[i].sensor_type_code == sensor_type_code) &&
        (sensor_event_list[i].sensor_specific_offset == sensor_specific_offset))
      return(sensor_event_list[i].desc);
      
    i++;
    }
  return(NULL);
} 
void sensor_str_convert(
                        unsigned char sensor_number,
                        unsigned char *data,
                        char          *string_msg)
{                       
 /* DISCRETE_SENSOR_READING_RESPONSE  *discrete; */
  DIGITAL_SENSOR_READING_RESPONSE   *digital;
  ANALOG_SENSOR_READING_RESPONSE    *analog;
  SENSOR                            *sensor;
  
  /* Make heads or tails out of the data */
	sensor = sensor_head;
  while(sensor)
    {
    if (lookup_sensor_number(sensor) == sensor_number)
      break;
    sensor = sensor->next;
    }
  if (!sensor)
    {
		printf("BAM!\n");
    return;
    }
  if (sensor->rec_header.record_type == 0x01)
    {
    if (sensor->sensor_type.type_01_sensor.sens_capable_reading_type == ANALOG)
      {
      double  final_value;
      
      analog = (ANALOG_SENSOR_READING_RESPONSE *) &data[0];
      final_value = linearize_sensor_reading(sensor, analog->reading);
      sprintf(string_msg,
              "Sensor -> %d, Reading -> %.2f %s",
              sensor_number,
              final_value,
              sensor_get_unit_type(sensor->sensor_type.type_01_sensor.sensor_units_2_base_unit));
      return; 
      }
    else
      {
      printf(
              "sensor reading type 0x%x not supported\n",
              sensor->sensor_type.type_01_sensor.sens_capable_reading_type);
      }
    } 
  else if (sensor->rec_header.record_type == 0x02)
    {
    int sensor_class;
    
    /* This is a digital / discrete reading without thresholds */
    sensor_class = sensor_get_class(sensor->sensor_type.type_02_sensor.event_reading_type_code);
    if (sensor_class == SENSOR_CLASS_DIGITAL)
      {
      digital = (DIGITAL_SENSOR_READING_RESPONSE *) &data[0];
      sprintf(string_msg,
              "Sensor -> %d, Reading -> %s",
              sensor_number,
              sensor_get_trigger_description(
                sensor->sensor_type.type_02_sensor.event_reading_type_code,
                digital->digital_status_offset));
      return;   
      }
    else if (sensor_class == SENSOR_CLASS_DISCRETE)
      {
      printf("discrete sensors not currently supported\n");
      }
    else if (sensor_class == SENSOR_CLASS_UNKNOWN)
      {
      sprintf(string_msg, "JOB_ERROR:SENSOR_NOT_SUPPORTED");
      return;
      }
    else
      {
      printf(
              "unknown type 02 sensor class 0x%x\n",
              sensor_class);
      sprintf(string_msg, "JOB_ERROR:SENSOR_NOT_SUPPORTED");
      return;
      }
    } 
  else
    {
    printf(
            "unsupported sensor record type 0x%x\n",
           sensor->rec_header.record_type);
    sprintf(string_msg, "JOB_ERROR:SENSOR_NOT_SUPPORTED");
    return;
    }
}

int lookup_sensor_number(SENSOR *sensor)
{
  if (sensor->rec_header.record_type == 0x01)
    return(sensor->sensor_type.type_01_sensor.sensor_number);
  else if (sensor->rec_header.record_type == 0x02)
    return(sensor->sensor_type.type_02_sensor.sensor_number);
  else
    return(-1);
}
 

int read_sensor(int fd, unsigned char sensor_number)
{
	IPMI_XFER			xfer;
	IPMI_XFER			*xferp = &xfer;
	unsigned char	cc;
	int						rc;
	unsigned char	buffer[32];
	char					string[255];

	INIT_XFER(xferp);
	SET_REQUEST_LUN(xferp, 0);
	SET_REQUEST_NETFN(xferp, SENSOR_REQUEST);
	SET_REQUEST_CMD(xferp, 0x2d);
	SET_REQUEST_DATA(xferp, (unsigned char *) &sensor_number, sizeof(sensor_number));

	if ((rc = ioctl(fd, IOCTL_IPMI_XFER, (void *) &xfer))<0)
		{
		printf("read_sensor(): failed (%m)\n");
		return(-1);
		}
	GET_RESPONSE_CC(xferp, cc);
	if (cc != 0x00)
		{
		printf("read_sensor(): failed (0x%.2x)\n",cc);
		return(-1);
		}
	GET_RESPONSE_DATA(xferp, &buffer[0]);
	sensor_str_convert(sensor_number, &buffer[0], string);
	printf("%s\n",string);
}

