# typofix.pl - when someone uses s/foo/bar typofixing, this script really
# goes and modifies the original text on screen.

# version 0.91
# For irssi 0.7.99 by Timo Sirainen

#  /SET typofix_modify_string  [fixed]    - append string after replaced text
#                                           (%s expands to the replacement made)
#  /SET typofix_hide_replace ON           - hide the s/foo/bar/ line
#  /SET typofix_quiet OFF                 - don't print any errors with
#                                           /TYPOUNDO and /TYPOREDO

#  /TYPOUNDO - Undo the previous replace
#  /TYPOREDO - Redo the previous undo

use strict;
use Irssi::TextUI;
use vars qw($VERSION %IRSSI);
$VERSION = "0.91";
%IRSSI = (
    authors     => "Timo Sirainen",
    contact	=> "tss\@iki.fi",
    name	=> "Typofix (cras)",
    description	=> "When someone uses s/foo/bar/, this really modifies the text",
    bugs	=> "Removes color, sorry",
    license	=> "Public Domain",
    url		=> "http://irssi.org/",
    changed	=> "2002-03-04T22:47+0100"
);

my $UNDO_BUFFER_SIZE = 10;
my $color = "(?:\cd(?:[a-h]|[/-?][/-7]))*";

# window specific undo buffer
my %undo_buffer;
my %undo_buffer_pos;

sub replace {
  my ($window, $nick, $from, $to, $opt) = @_;

  my $view = $window->view();
  my $buffer = $view->{buffer};
  my $line = $view->{bottom_startline};

  my $last_line;
  while ($line) {
    my $text = $line->get_text(1);
    if (($line->{info}->{level} & (MSGLEVEL_PUBLIC|MSGLEVEL_MSGS)) != 0 &&
	$text !~ m,s/.*/, &&
        $text =~ /(?:<(?:${color}[ +@])?$color| \* )\Q$nick\E(?:$color>)?$color .*$from/) {
      $last_line = $line
    }
    $line = $line->next();
  }
  return 0 if (!$last_line);
  my $text = $last_line->get_text(1);

  # save undo info
  my @buf = ();
  @buf = @{$undo_buffer{$window->{refnum}}} if ($undo_buffer{$window->{refnum}});
  splice @buf, 0, 1 if (scalar @buf >= $UNDO_BUFFER_SIZE);
  push @buf, {
    "window" => $window->{refnum},
    "id" => $last_line->{_irssi},
    "text" => $text
  };
  $undo_buffer{$window->{refnum}} = \@buf;
  $undo_buffer_pos{$window->{refnum}} = scalar @buf;

  # text replacing
  $text =~ s/(.*(?:<(?:${color}[ +@])?$color| \* )\Q$nick\E(?:$color>)?$color )//;
  my $pre = $1;

  if ($opt =~ /g/) {
    $text =~ s/$from/$to/g;
  } else {
    $text =~ s/$from/$to/;
  }

  my $fixed = Irssi::settings_get_str('typofix_modify_string');
  $fixed =~ s,%s,s/$from/$to/$opt,;

  $text = $pre.$text.$fixed;

  my $prev_line = $last_line->prev();
  my $info = $last_line->{info};
  $view->remove_line($last_line);

  $window->gui_printtext_after($prev_line, $info->{level}, "$text\n");
  $view->redraw();

  return 1;
}

sub event_privmsg {
  my ($server, $data, $nick, $address) = @_;
  my ($target, $text) = $data =~ /^(\S*)\s:(.*)/;

  return if ($text !~ m,^s/([^/]*)/([^/]*)/*([^ ]*)(.*),);
  my $from = $1;
  my $to = $2;
  my $opt = $3;
  my $extra = $4;

  my $hide = Irssi::settings_get_bool('typofix_hide_replace') && !$extra;

  my $private = !$server->ischannel($target);
  my $level = $private ? MSGLEVEL_MSGS : MSGLEVEL_PUBLIC;
  $target = $nick if $private;
  my $window = $server->window_find_closest($target, $level);

  Irssi::signal_stop() if replace($window, $nick, $from, $to, $opt) && $hide;
}

sub event_own_public {
  my ($server, $text, $target) = @_;

  return if ($text !~ m,^s/([^/]*)/([^/]*)/*([^ ]*)(.*),);
  my $from = $1;
  my $to = $2;
  my $opt = $3;
  my $extra = $4;

  my $hide = Irssi::settings_get_bool('typofix_hide_replace') && !$extra;

  my $private = !$server->ischannel($target);
  my $level = $private ? MSGLEVEL_MSGS : MSGLEVEL_PUBLIC;
  my $nick = $server->{nick};
  my $window = $server->window_find_closest($target, $level);

  Irssi::signal_stop() if replace($window, $nick, $from, $to, $opt) && $hide;
}

sub replace_line {
  my ($window, $line, $text) = @_;

  my $view = $window->view();
  my $buffer = $view->{buffer};
  my $prev_line = $line->prev();
  my $info = $line->{info};
  $view->remove_line($line);

  $window->gui_printtext_after($prev_line, $info->{level}, "$text\n");
  $line = $window->last_line_insert();
  $view->redraw();
  return $line->{_irssi};
}

sub restore_line {
  my ($window, $pos) = @_;

  my $view = $window->view();
  my $item = @{$undo_buffer{$window->{refnum}}}[$pos];

  my $line = $view->{buffer}->{first_line};
  while ($line) {
    if ($line->{_irssi} == $item->{id}) {
      my $old_text = $line->get_text(1);
      $item->{id} = replace_line($window, $line, $item->{text});
      $item->{text} = $old_text;
      return;
    }
    $line = $line->next();
  }

  # undo buffer contains only old entries
  $window->print('Undo contains only old data') if (!Irssi::settings_get_bool('typofix_quiet'));
}

sub cmd_undo {
  my $window = Irssi::active_win();
  my $refnum = $window->{refnum};
  if (!$undo_buffer{$refnum}) {
    $window->print('Undo buffer is empty') if (!Irssi::settings_get_bool('typofix_quiet'));
    return;
  }

  my $pos = $undo_buffer_pos{$refnum}-1;
  if ($pos < 0) {
    $window->print('Nothing to undo anymore') if (!Irssi::settings_get_bool('typofix_quiet'));
    return;
  }
  $undo_buffer_pos{$refnum} = $pos;
  restore_line($window, $pos);
}

sub cmd_redo {
  my $window = Irssi::active_win();
  my $refnum = $window->{refnum};
  if (!$undo_buffer{$refnum}) {
    $window->print('Undo buffer is empty') if (!Irssi::settings_get_bool('typofix_quiet'));
    return;
  }

  my $pos = $undo_buffer_pos{$refnum};
  if ($pos >= scalar @{$undo_buffer{$refnum}}) {
    $window->print('Nothing to redo') if (!Irssi::settings_get_bool('typofix_quiet'));
    return;
  }
  $undo_buffer_pos{$refnum} = $pos+1;
  restore_line($window, $pos);
}

Irssi::settings_add_str('typofix', 'typofix_modify_string', ' [fixed %s]');
Irssi::settings_add_bool('typofix', 'typofix_hide_replace', 1);
Irssi::settings_add_bool('typofix', 'typofix_quiet', 0);

Irssi::signal_add('event privmsg', 'event_privmsg');
Irssi::signal_add('message own_public', 'event_own_public');

Irssi::command_bind('typoundo', 'cmd_undo');
Irssi::command_bind('typoredo', 'cmd_redo');
