#! /usr/bin/perl -w
#-----------------------------------------------------------------------------
# $Id: jpilot_archive.pl,v 1.1.1.1 2003/02/04 18:01:30 rousseau Exp $
#
# Peter Williams <ptr_wllms@yahoo.com>
#
# Simple utility to save files backed up by Jason Day's jpilot-backup
# jpilot extension to an archive directory so that they are never
# lost.
#
# Works by collecting the names of all "archive directories" made by
# jpilot-backup, then comparing the files from oldest to newest, and
# finally comparing them with the contents of the Archive, and keeping
# the newest one.
#
#-----------------------------------------------------------------------------

use strict;
use Digest::MD5;
use File::Copy;
use Time::Local;
use File::Path;

##########
# Configurable variables
##########
# location of the jpilot directory
my $Jpilot_Dir = "$ENV{'HOME'}/.jpilot";
# where archive dirs are kept by jpilot-backup
my $Backup_TopDir = "${Jpilot_Dir}/Backup";
# where the persistent archive should live
my $Arch_Dir = "${Backup_TopDir}/MainArchive";

##########
# Internal variables
##########
my %Time_to_Dir; # mapping of epoch time -> directory
my $Ref; # 

#-----------------------------------------------------------------------------
# main routine
#-----------------------------------------------------------------------------

MAIN: {
	unless (-d $Arch_Dir) {
		mkpath($Arch_Dir,1,0711) or die "Can't mkpath $Arch_Dir: $!";
	}
	scan_dirs();
	compare_backups();
}

#-----------------------------------------------------------------------------
# Scan directories and organize them by time backed up
#-----------------------------------------------------------------------------

sub scan_dirs {
	print "Scanning directories now...\n";
	opendir(JPILOT,$Backup_TopDir) or die "Can't opendir $Backup_TopDir: $!";
	foreach my $dir (grep((
		-d "${Backup_TopDir}/$_" and
		/^Archive_\S+/),
		readdir(JPILOT))) {
		my ($year,$month,$day,$hour,$min,$sec) =
			(
			$dir =~ m#Archive_(\d{4})-(\d{2})-(\d{2})\@(\d{2}):(\d{2}):(\d{2})#
			);
		my $time = timelocal($sec,$min,$hour,$day,--$month,$year);	
		print "\t$dir, time: $time\n";
		$Time_to_Dir{$time} = "${Backup_TopDir}/$dir";
	}
	closedir JPILOT;
}

#-----------------------------------------------------------------------------
# Sort by directory, then compare the files with subsequent archived files
#-----------------------------------------------------------------------------

sub compare_backups {
	my $lastdir = "";
	print "\n";

	foreach my $mtime (sort {$a <=> $b} keys(%Time_to_Dir)) {
		my $dir = $Time_to_Dir{$mtime};
		my $print_last = $lastdir ? $lastdir : "<none>";
		print "\nComparing the following directories now:\n";
		print "\tCurrent: $dir\n";
		print "\tLast:    $print_last\n\n";
		opendir(TMP,$dir) or die "Can't open $dir: $!";
		foreach my $file (grep(!/^\.\.?$/,readdir(TMP))) {
			print "\n\tchecking file $file:\n";
			my $md5 = Digest::MD5->new();
			open(FILE,"${dir}/$file") or die "Can't open ${dir}/file: $!";
			while (<FILE>) { $md5->add($_) }
			close FILE;
			$Ref->{$dir}{$file} = $md5->b64digest();
			if (defined $Ref->{$lastdir}{$file}) {
				print "\t\tin last backup, skipping\n";
				next;
			} else {
				print "\t\tnot in last backup\n";
				compare_against_archive($dir,$file);
			}
		}
		$lastdir = $dir;
		closedir TMP;
	}
}

#-----------------------------------------------------------------------------
# Compare last found version against archived version of file
#-----------------------------------------------------------------------------

sub compare_against_archive {
	my ($dir,$file) = @_;
	my $afile = "${Arch_Dir}/$file";
	if (-f $afile) {
		print "\t\tpresent in archive (checking for diffs)...\n";
		my $chk_md5 = Digest::MD5->new();
		open(AFILE,$afile) or die "Can't open $afile: $!";
		while (<AFILE>) { $chk_md5->add($_) }
		close AFILE;
		if ($Ref->{$dir}{$file} eq $chk_md5->b64digest()) {
			print "\t\tno diffs.\n";
			next;
		} else {
			print "\t\tfiles differ!\n";
			print "\t\tCopying $file to archive\n";
			copy("${dir}/$file","${Arch_Dir}/$file")
		}
	} else {
		print "\t\tcopying to archive\n";
		copy("${dir}/$file","${Arch_Dir}/$file");
		print "\t\tcopied to archive\n";
	}
}

#-----------------------------------------------------------------------------

__END__

