#ifndef K3DSDK_VIEWPORT_H
#define K3DSDK_VIEWPORT_H

// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Declares k3d::viewport::drawable, a boilerplate implementation of k3d::viewport::idrawable
		\author Tim Shead (tshead@k-3d.com)
*/

#include "glutility.h"
#include "iviewport.h"
#include "persistence.h"
#include "property_collection.h"
#include "transform.h"

#include <sdpgl/sdpgl.h>

namespace k3d
{

// Forward declarations
class bounding_box;
class idocument;

namespace viewport
{

/// Convenience function for refreshing all viewports
void redraw_all(idocument& Document, const iviewport::redraw_type_t RedrawType);
/// Convenience function for setting-up materials
void setup_material(iunknown* const Material);
/// Draws a standard bounding box to make it easier to visualize an object
void draw_bounding_box(const bounding_box& Box);

/** \brief Provides a boilerplate implementation of k3d::iviewport_drawable
		\param base_t Must derive from k3d::transformable
*/
template<typename base_t>
class drawable :
	public base_t,
	public idrawable
{
public:
	drawable(idocument& Document) :
		base_t(Document),
		m_visible(init_name("viewport_visible") + init_description("Visible [boolean]") + init_value(true) + init_document(Document)),
		m_nurbs_renderer(0)
	{
		base_t::enable_serialization(persistence::proxy(m_visible));
		base_t::register_property(m_visible);
		m_visible.changed_signal().connect(SigC::slot(*this, &drawable::async_redraw_all));
	}

	~drawable()
	{
		if(m_nurbs_renderer)
			gluDeleteNurbsRenderer(m_nurbs_renderer);
	}

	void viewport_draw(const render_state& State)
	{
		if(!m_visible.property_value())
			return;

		sdpgl::store_attributes attributes();

		push_draw();
		on_viewport_draw(State);
		pop_draw();
	}

	void viewport_select(const render_state& State)
	{
		if(!m_visible.property_value())
			return;

		sdpgl::store_attributes attributes();

		push_draw();
		on_viewport_select(State);
		pop_draw();
	}

protected:
	void async_redraw_all()
	{
		redraw_all(base_t::document(), iviewport::ASYNCHRONOUS);
	}

	typedef GLUnurbsObj* nurbs_renderer_t;
	nurbs_renderer_t nurbs_renderer(const render_state& State)
	{
		if(!m_nurbs_renderer)
			{
				m_nurbs_renderer = gluNewNurbsRenderer();

				// Important!  We load our own matrices for efficiency (saves round-trips to the server) and to prevent problems with selection
				gluNurbsProperty(m_nurbs_renderer, GLU_AUTO_LOAD_MATRIX, GL_FALSE);
				gluNurbsProperty(m_nurbs_renderer, GLU_CULLING, GL_TRUE);
			}

		GLfloat gl_modelview_matrix[16];
		glGetFloatv(GL_MODELVIEW_MATRIX, gl_modelview_matrix);
		gluLoadSamplingMatrices(m_nurbs_renderer, gl_modelview_matrix, State.gl_projection_matrix, State.gl_viewport);

		return m_nurbs_renderer;
	}

private:
	void push_draw()
	{
		glMatrixMode(GL_MODELVIEW);
		glPushMatrix();
		push_opengl_matrix(base_t::matrix());
	}

	void pop_draw()
	{
		glMatrixMode(GL_MODELVIEW);
		glPopMatrix();
	}

	/// Set to true iff the object should be visible in the viewport
	k3d_data_property(bool, immutable_name, change_signal, with_undo, local_storage, no_constraint) m_visible;

	virtual void on_viewport_draw(const render_state& State) = 0;
	virtual void on_viewport_select(const render_state& State) = 0;

	/// OpenGL NURBS renderer
	nurbs_renderer_t m_nurbs_renderer;
};

} // namespace viewport

} // namespace k3d

#endif // K3DSDK_VIEWPORT_H


