/***************************************************************************
 *   Copyright (C) 2004 by Fred Schaettgen <kdebluetooth@schaettgen.de>*
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/
#include "deviceinputwidget.h"
#include <qtimer.h>
#include "inquiry.h"
#include <kglobal.h>
#include <kpixmap.h>
#include <kiconloader.h>
#include "devicemimeconverter.h"
#include "namecache.h"
#include <qlistbox.h>
#include <qlineedit.h>
#include <kdialogbase.h>
#include <klocale.h>
#include <ksqueezedtextlabel.h>
#include <kdebug.h>

namespace KBluetooth {


DeviceInputWidget::DeviceInputWidget(QWidget *parent) :
    DeviceInputWidgetBase(parent, "DeviceInputWidget")
{
    checkAddress();
    this->inquiry = NULL;
    connect(this->deviceList, SIGNAL(selectionChanged()),
        this, SLOT(selectionChanged()));
    connect(this->addressEdit, SIGNAL(textChanged(const QString&)),
        this, SLOT(checkAddress()));
}

DeviceInputWidget::~DeviceInputWidget()
{
}

void DeviceInputWidget::init(bool useDiscoveryService)
{
    deviceList->clear();
}

void DeviceInputWidget::startSearch()
{
    inquiry = new KBluetooth::Inquiry(NULL, this);
    connect(inquiry, SIGNAL(neighbourFound(const KBluetooth::DeviceAddress&,int)),
        this, SLOT(neighbourFound(const KBluetooth::DeviceAddress&,int)));
    connect(inquiry, SIGNAL(finnished()), this, SLOT(searchFinished()));
    connect(inquiry, SIGNAL(error(int,QString)), this, SLOT(searchFinished()));
    inquiryStart();
    checkAddress();
}

void DeviceInputWidget::stopSearch()
{
    delete inquiry;
    inquiry = NULL;
}

void DeviceInputWidget::neighbourFound(const KBluetooth::DeviceAddress& bdaddr, 
    int deviceClass)
{
    QString name = QString(bdaddr);
    KBluetooth::NameCache::getCachedName(bdaddr, name);
    if (addrToNameMap.find(bdaddr) == addrToNameMap.end()) {
        addrToNameMap[bdaddr] = name;
        nameToAddrMap[name] = bdaddr;
        QString iconName = DeviceClassMimeConverter::classToIconName(deviceClass);
        KPixmap icon = KGlobal::iconLoader()->loadIcon(
            iconName, KIcon::Small, 16);
        deviceList->insertItem(icon, name);
        kdDebug() << "insert item " << name << " " << iconName << endl;
    }
}

void DeviceInputWidget::searchFinished()
{
    statusLabel->setText("");
    if (inquiry) {
        QTimer::singleShot(500, this, SLOT(inquiryStart()));
    }
}

void DeviceInputWidget::inquiryStart()
{
    if (inquiry) {
        if (inquiry->inquiry() == false) {
            QTimer::singleShot(3000, this, SLOT(inquiryStart()));    
        }
        statusLabel->setText(i18n("Searching..."));
    }
}

void DeviceInputWidget::selectionChanged()
{
    if (deviceList->currentItem() >= 0) {
        addressEdit->setText(QString(nameToAddrMap[deviceList->text(
            deviceList->currentItem())]));
        checkAddress();
    }
}

void DeviceInputWidget::checkAddress()
{
    emit addressValid(currentAddress() != DeviceAddress::invalid);
}

bool DeviceInputWidget::showSelectionDialog(QWidget* parent,
        KBluetooth::DeviceAddress &address, bool bInitFromDiscoveryService)
{
    KDialogBase dialog(parent, "deviceselectiondlg", true,
        i18n("Bluetooth Device Selection"), KDialogBase::Ok|KDialogBase::Cancel);
    DeviceInputWidget *widget = new DeviceInputWidget(&dialog);
    dialog.setMainWidget(widget);
    connect(widget->deviceList, SIGNAL(returnPressed(QListBoxItem*)),
        &dialog, SLOT(accept()));
    connect(widget, SIGNAL(addressValid(bool)), &dialog, SLOT(enableButtonOK(bool)));
    widget->startSearch();
    bool ret;
    if (dialog.exec() == QDialog::Accepted) {
        address = widget->currentAddress();
        ret = true;
    }
    else {
        ret = false;
    }
    delete widget;
    return ret;
}

KBluetooth::DeviceAddress DeviceInputWidget::currentAddress()
{
    return KBluetooth::DeviceAddress(addressEdit->text());
}

}
#include "deviceinputwidget.moc"
