\ Solution of cubic equation with real coefficients -- ANS compatible version of 10/6/1994

\ Forth Scientific Library Algorithm #6


\ Environmental dependencies:
\
\       Integrated data/floating point stack *** This is the integrated stack version ***
\       FLOAT, FLOAT EXT wordsets
\       double precision floats (64 bits for output tests)
\
\ Source of algorithms:
\ Complex roots: Abramowitz & Stegun, p. 17 3.8.2
\ Real roots:    Press, et al., "Numerical Recipes" (1st ed.) p. 146
\
\       0 = x^3 + ax^2 + bx + c
\       q = b / 3 - a^2 / 9
\       r = ( b a/3 - c) / 2 - ( a / 3 )^3
\
\       D = q^3 + r^2
\
\       If D > 0, 1 real and a pair of cc roots;
\          D = 0, all roots real, at least 2 equal;
\          D < 0, all roots real
\
\ Case I: D > 0
\
\       s1 = [ r + sqrt(D) ]^1/3
\       s2 = [ r - sqrt(D) ]^1/3
\
\       x1 = s1 + s2 - a / 3
\       Re(x2) = - (s1 + s2) / 2  - a / 3
\       Im(x2) = sqrt(3) * (s1 - s2) / 2
\
\       x3 = conjg(x2)
\
\ Case II: D < 0
\
\       K = -2 * sqrt(-q)
\       phi = acos( -r / sqrt(-q^3) )
\
\       x1 = K * cos(phi / 3) - a / 3
\       x2 = K * cos( (phi + 2*pi) / 3) - a / 3
\       x3 = K * cos( (phi + 4*pi) / 3) - a / 3
\
\     (c) Copyright 1994  Julian V. Noble.     Permission is granted
\     by the author to use this software for any application provided
\     the copyright notice is preserved.

\ =====================================================================
\ The is the kForth version, which requires the following files:
\
\  ans-words.4th
\  fsl-util.4th
\  complex.4th
\
\ Additional definitions for kForth:

: f0>     0e FSWAP F< ;
: f**2    FDUP  F* ;
: f**3    FDUP  FDUP  F* F*  ;
\
\ Revisions:
\
\   2005-08-26  ported to kForth  K. Myneni
\   2005-09-02  fixed problem with word "cubic-roots"  KM
\ ======================================================================

CR .( CUBIC-SOLVER      V1.1c         02 September 2005  JVN, KM )

\ Cube root by Newton's method

: X'    ( N x -- x')
        ftuck   f**2  F/  FSWAP  F2*  F+  3e  F/  ;           \ X' = (N/X^2 + 2*X)/3

: fcbrt  ( N -- N^1/3)   
        FDUP  F0<  >R  FABS
        FDUP  FSQRT		         (  -- N x)
        BEGIN   ZDUP     X'              (  -- N x x')
          ftuck    F-  FOVER   F/  FABS
          1.E-8  F<  
	UNTIL
        X'  R> IF  FNEGATE  THEN  ;

\ Solve cubic

FVARIABLE A     FVARIABLE B     FVARIABLE C     \ coeffs
FVARIABLE Q     FVARIABLE R                     \ derived stuff
FVARIABLE S1    FVARIABLE S2

ZVARIABLE Z1
ZVARIABLE Z2
ZVARIABLE Z3

: discriminant    ( a b c -- D )
        C F!    B F!   3e F/  A F!
        B F@  3e F/  A F@ f**2 F-  Q F!                       \  Q = B/3 - A^2
        A F@  B  F@  F*   C F@ F-  F2/  A F@ f**3  F-  R F!   \  R = (A*B - C)/2 - A^3
        Q F@  f**3   R F@ f**2  F+    ;                       \  d = Q^3 + R^2


: 1real     ( D -- | roots stored in z1, z2, z3 )  
        FSQRT   R F@  FOVER  F+  fcbrt S1 F!    ( sqrt[D] )  \  S1 = (R + sqrt(D))^1/3
        FNEGATE  R F@  F+        fcbrt S2 F!    (  -- )      \  S2 = (R - sqrt(D))^1/3
        S1 F@ S2 F@  F+  FDUP   A F@  F-        ( s1+s2 x1)  \  x1 = S1 + S2 - A/3 + i*0
        0e z1 z!                                \ real root
        FNEGATE  F2/  A F@ F-   FDUP            ( Re[x2]  Re[x2] )  
	\  x2 = -(S1 + S2)/2 - A/3 + i*SQRT(3)*(S1 - S2)/2
        S1 F@  S2 F@  F-  F2/  3e  FSQRT  F*   ftuck   ( Re[x2] Im[x2] Re[x2] Im[x2] )
        z2 z!                                    ( Re[x2] Im[x2] )
        FNEGATE z3 z! ;                          \  x3 = conjg(x2)


: root     ( K angle -- K x )
        3e F/  FCOS   FOVER  F*  A F@  F-  ;

: 3real   ( -- | roots stored in z1, z2, z3 ) 
        Q F@  FABS  FSQRT                   ( K )            \  K=SQRT(ABS(Q))
        R F@  FNEGATE                       ( K  -r )        
        FOVER   f**3  F/   FACOS            ( K phi )        \  phi = ACOS(-R/K^3)
        FSWAP   F2*   FNEGATE               ( -- phi K )     \  K = -2*K
        FOVER                      root 0e z1 z!             \  x1
        FOVER   PI  F2*       F+   root 0e z2 z!             \  x2
        FSWAP   PI  F2* F2*   F+   root 0e z3 z!             \  x3
        FDROP   ;


: +?    FDUP F0> IF ." + "  ELSE  ." - "  FABS  THEN  ;

: .equ'n   CR  ." x^3 "
        A F@  3e F*   +?  ( E.) F.  ."  x^2 "
        B F@  +?  ( E.) F.  ." x "  C F@  +?  ( E.) F.  ."  = 0"  ;

: >roots     ( a b c -- | roots are computed and displayed; roots also stored in z1,z2,z3 )
        discriminant                          ( -- D )
        CR  .equ'n
        FDUP  f0>                           \ test discriminant
        IF      CR  ." 1 real, 2 complex conjugate roots" 
	        1real
		CR z1 z@ FDROP F.
		CR z2 z@       z.
		CR z3 z@       z.
        ELSE    FDROP   
                CR  ." 3 real roots"  
		3real
		CR z1 z@ FDROP F.
		CR z2 z@ FDROP F.
		CR z3 z@ FDROP F. 
        THEN  CR  ;

: cubic-roots    ( a b c -- flag | flag is TRUE if all 3 roots are real; roots stored in z1,z2,z3 )
       discriminant FDUP F0> IF           1real  FALSE
                             ELSE   FDROP 3real  TRUE 
                             THEN ;

: RESIDUAL   ( z -- | z = x + iy )   
        S2 F!  S1 F!
        S1 F@ f**2   S2 F@ f**2  3e F*   F-  B F@  F+     \  x*(x^2 - 3*y^2) + A*(x^2 - y^2) + x*B + C
        S1 F@ F*
        A F@ 3e F*
        S1 F@ f**2   S2 F@ f**2 F-   F*  F+
        C F@  F+ 
        3e  S1 F@  f**2 F*   S2 F@  f**2   F-             \  y*(3*x^2 - y^2 + 2*x*A + B)
        S1 F@ F2*   A F@  F*  3e F*   F+
        B  F@ F+    S2 F@ F*    CR z.  ;
\ Use to test quality of solution


\ Examples

\ -2e 1e -2e >roots

\ x^3 - 2 x^2 + 1 x - 2 = 0
\ 1 real, 2 complex conjugate roots
\ 2
\ 6.66134e-16  + i 1
\ 6.66134e-16  - i 1 
\  ok
                ( The exact solutions of x^3 - 2x^2 + x - 2 = 0 are  2, i, -i )

\ -2e -1e 2e >roots

\ x^3 - 2 x^2 - 1 x + 2 = 0
\ 3 real roots
\ -1
\ 2
\ 1 
\  ok
                ( The exact solutions of x^3 - 2x^2 - x - 2 = 0 are  2, 1, -1 )

\ -4e -1e 22e >roots

\ x^3 - 4 x^2 - 1 x + 22 = 0
\ 1 real, 2 complex conjugate roots
\ -2
\ 3  + i 1.41421
\ 3  - i 1.41421 
\  ok
               ( The exact solutions of x^3 - 4x^2 - x + 22 = 0 are  -2, 3+i*sqrt[2], 3-i*sqrt[2] )

\ -2e 0e residual
\ 0 + i 0  ok
\ 3e 2e FSQRT residual
\ -7.10543e-15 + i 0  ok
\ 3e 2e FSQRT FNEGATE residual
\ -7.10543e-15 + i -0  ok
