/* -*- c++ -*-
 *
 * giftmessage.h
 *
 * Copyright (C) 2003 Sebastian Sauer <mail@dipe.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

/*!
 * The base parsing classes to handle the giFT-protocol are defined in
 * this file. Take a look at http://gift.sourceforge.net/docs.php?document=interface.html
 * to note how the giFT-protocol works.
 */

#ifndef __libkmldonkey_giftmessage_h__
#define __libkmldonkey_giftmessage_h__

#include <qstring.h>
#include <qmap.h>
#include <qregexp.h>
#include <kdebug.h>

/*!
 * A GiftMessage could contain numbers of GiftMessageItem's. They are structed as
 * tree what means each GiftMessageItem could have sub/child GiftMessageItem's.
 */
class GiftMessageItem
{
public:
    GiftMessageItem(GiftMessageItem* ParentItem = 0);
    ~GiftMessageItem();

    /*!
     * Return the MainKey.
     */
    QString getMainKey() { return MainKey; }
    /*!
     * Return the MainValue.
     */
    QString getMainValue() { return MainValue; }
    /*!
     * Return the Argumentvalue of the defined Argumentkey.
     */
    QString getArgumentValue(const QString& key) { return Arguments[key]; }
    /*!
     * Return a list of child-/subcommand's.
     */
    QValueList<GiftMessageItem*> getSubCommands() { return SubCommands; }

    /*!
     * Remove any previous parsed messages.
     */
    void clearMessage();

    /*!
     * Returns the message as giFT-protocolmessage.
     */
    QString getMessage();

    /*!
     * Enumeration of setMessage() returnvalues.
     */
    enum enumSetMessage {
        smOk = 0,
        smFinalizerMissing,
        smParseError
    };
    /*!
     * The setMessage() function parses some string like the following and
     * converts it into a GiftMessageItem-tree.
     *
     *    key(value)                                  Rootcommand + value
     *      argument_key(argument_value)              Arguments
     *      argument_key(argument_value)
     *      subcommand_key(subcommand_value) {        Subcommand + value
     *        argument_key(argument_value)            Arguments of the subcommand
     *        argument_key(argument_value)
     *      }
     *    ;                                           Close Rootcommand, message is done and could be parsed now
     */
    enumSetMessage setMessage(const QString& message);

private:
    GiftMessageItem* ParentItem; // if =0 then this GiftMessageItem is the RootItem of GiftMessage
    QString MainKey, MainValue; // each GiftMessageItem does have a (possible empty) MainKey and MainValue
    QMap<QString, QString> Arguments; // key(argument) pairs of this GiftMessageItem
    QValueList<GiftMessageItem*> SubCommands; // Each GiftMessageItem could have sub/child GiftMessageItem's.

    QString msg;

    // following stuff is used to parse the message
    static bool getPrevArg(const QString& msg, int& pos, QString& key, QString& value);
    static bool getNextArg(const QString& msg, int& pos, QString& key, QString& value);
    static bool getNextSubcommand(QString& msg, int& pos, QString& subkey, QString& subvalue, QString& submsg);
    static int getToken(const QString& message, const QString& token, const int startpos, bool backwards = false);
    static QString simplifyString(const QString& s);
    static bool isEscaped(const QString& s, const int pos);
    static QString escapeString(const QString& msg, bool EscapeWhiteSpaces = false);
    static QString unescapeString(const QString& msg);
};

/*!
 * The main giFT-Interfaceprotocol class. This class manages the GiftMessageItem's.
 */
class GiftMessage
{
public:
    /*!
     * Constructor.
     */
    GiftMessage();

    /*!
     * Destructor.
     */
    ~GiftMessage();

    /*!
     * Return the MainKey of the RootItem.
     */
    QString getMainKey();

    /*!
     * Return the MainValue of the RootItem.
     */
    QString getMainValue();

    /*!
     * Return the the Argumentvalue of the defined Argumentkey.
     */
    QString getArgumentValue(const QString& key);

    /*!
     * Returns the message as giFT-protocolmessage.
     */
    QString getMessage();

    /*!
     * Set the giFT-protocolmessage. On success (smrOk) the RootItem is
     * filled with the parsed message.
     */
    GiftMessageItem::enumSetMessage setMessage(const QString&);

private:
    GiftMessageItem* RootItem;
    GiftMessageItem::enumSetMessage smresult;
};

#endif
