/*
	msystemtray.cpp - A system tray icon
	Copyright (C) 2003  Konrad Twardowski <kdtonline@poczta.onet.pl>

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "configuration.h"
#include "mmainwindow.h"
#include "msystemtray.h"

#include <qtimer.h>

#include <kdebug.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kpixmap.h>
#include <kpixmapeffect.h>
#include <kpopupmenu.h>
#include <kwin.h>

const double
	KS_TRAY_FADE_CHANGE = 0.05f,
	KS_TRAY_FADE_MAX = 0.7f;
const int KS_TRAY_FADE_TIMEOUT = 50;
const QColor KS_TRAY_FADE_COLOR = Qt::yellow;

MSystemTray::Mode MSystemTray::_mode = Always;
MSystemTray *MSystemTray::_instance = 0;

// public

MSystemTray::~MSystemTray()
{
	if (_flashIcon)
		delete _flashIcon;
	if (_saveIcon)
		delete _saveIcon;
}

void MSystemTray::flashIcon()
{
	if (_saveIcon)
		delete _saveIcon;
	_saveIcon = new QPixmap();
	*_saveIcon = *pixmap();
	_flashCount = 0;
	setPixmap(*_flashIcon);
	_flashTimer->start(200);
}

void MSystemTray::setActive(const bool yes)
{
	// kdDebug() << "MSystemTray::setActive: " << yes << endl;

	if (_active == yes)
		return;

	_active = yes;
	_incFade = _active;
	_fadeTimer->start(KS_TRAY_FADE_TIMEOUT);
}

void MSystemTray::setMode(const Mode mode)
{
	// kdDebug() << "MSystemTray::setMode: " << mode << endl;

	_mode = mode;
	switch (_mode)
	{
		case Always:
			ks_tray; // init instance
			if (ks_actions->active())
				_instance->setActive(true);
// FIXME: 0.9: XSMP error (?) - bad state
			kapp->enableSessionManagement();
			break;
		case IfActive:
			if (ks_actions->active())
			{
				ks_tray; // init instance
				_instance->setActive(true);
				kapp->enableSessionManagement();
			}
			else
			{
				delete _instance;
				_instance = 0;
				kapp->disableSessionManagement();
			}
			break;
		case Never:
			delete _instance;
			_instance = 0;
			kapp->disableSessionManagement();
			break;
	}
}

// protected

void MSystemTray::contextMenuAboutToShow(KPopupMenu *menu)
{
	if (ks_actions->active())
	{
		menu->changeItem(_cancelItem, i18n("&Cancel") + ": " + ks_actions->getCurrentName());
		menu->setItemEnabled(_cancelItem, true);
	}
	else
	{
		menu->changeItem(_cancelItem, i18n("&Cancel"));
		menu->setItemEnabled(_cancelItem, false);
	}
}

void MSystemTray::mousePressEvent(QMouseEvent *e)
{
	// middle mouse button - display actions menu
// TODO: 0.9: configure middle button action
	if (e->button() == MidButton)
		_actionsMenu->popup(e->globalPos());
	else
		KSystemTray::mousePressEvent(e);
}

// private

MSystemTray::MSystemTray()
	: KSystemTray(ks_main, "MSystemTray"),
	_active(false),
	_incFade(true),
	_fadeValue(0.0f),
	_flashCount(0),
	_saveIcon(0)
{
	setPixmap(kapp->miniIcon());

// TODO: 0.9: mouse wheel: decrease/increase end time by 5min. (?)

	// init fade timer
	_fadeTimer = new QTimer(this);
	connect(_fadeTimer, SIGNAL(timeout()), SLOT(slotFade()));

	// init flash timer
	_flashTimer = new QTimer(this);
	connect(_flashTimer, SIGNAL(timeout()), SLOT(slotFlashTimeout()));

	// init icons
	KPixmap pm(kapp->miniIcon());
	_flashIcon = new KPixmap(KPixmapEffect::fade(pm, 0.7f, green));

	show();

	// add some items to the context menu
	KPopupMenu *cm = contextMenu();
	cm->setCaption(i18n("KShutDown Menu"));
	_actionsMenu = new KPopupMenu(cm, "KPopupMenu::_actionsMenu");
	_actionsMenu->setCaption(i18n("KShutDown Actions (no delay!)"));
	_actionsMenu->insertTitle(kapp->miniIcon(), i18n("Actions (no delay!)"));
	connect(_actionsMenu, SIGNAL(aboutToShow()), SLOT(slotActionsMenuAboutToShow()));

	// shut down
	_shutDownItem = _actionsMenu->insertItem(
		ks_actions->getIcon(Action::ShutDown), i18n("&Turn Off Computer"),
		this, SLOT(slotShutDown())
	);
	// reboot
	_rebootItem = _actionsMenu->insertItem(
		ks_actions->getIcon(Action::Reboot), i18n("&Restart Computer"),
		this, SLOT(slotReboot())
	);
	// lock screen
	_lockScreenItem = _actionsMenu->insertItem(
		ks_actions->getIcon(Action::LockScreen), i18n("&Lock Session"),
		this, SLOT(slotLockScreen())
	);
	// logout
	_logoutItem = _actionsMenu->insertItem(
		ks_actions->getIcon(Action::Logout), i18n("&End Current Session"),
		this, SLOT(slotLogout())
	);
	_actionsMenu->insertTearOffHandle();
	// id 0 = title
	cm->insertItem(i18n("&Immediate Action"), _actionsMenu, -1, 1);
	// cancel
	_cancelItem = cm->insertItem(
		SmallIcon("stop"), i18n("&Stop"),
		ks_main, SLOT(slotCancel()), 0, -1, 2
	);
	// -
	cm->insertSeparator(3);
	// configure
	cm->insertItem(
		SmallIcon("configure"), i18n("Configure &KShutDown..."),
		ks_main, SLOT(slotConfigure()), 0, -1, 4
	);
	// help
	cm->insertItem(
		SmallIcon("help"), i18n("&Help"),
		ks_main->helpMenu(), -1, 5
	);
	// -
	cm->insertTearOffHandle();

	// quit signal
	connect(
		this, SIGNAL(quitSelected()),
		ks_main, SLOT(slotQuit()));
}

// private slots

void MSystemTray::slotActionsMenuAboutToShow()
{
	_actionsMenu->setItemEnabled(_shutDownItem, ks_actions->isEnabled(Action::ShutDown));
	_actionsMenu->setItemEnabled(_rebootItem, ks_actions->isEnabled(Action::Reboot));
	_actionsMenu->setItemEnabled(_lockScreenItem, ks_actions->isEnabled(Action::LockScreen));
	_actionsMenu->setItemEnabled(_logoutItem, ks_actions->isEnabled(Action::Logout));
}

void MSystemTray::slotFade()
{
	if (_incFade)
	{
		_fadeValue += KS_TRAY_FADE_CHANGE;
		if (_fadeValue > KS_TRAY_FADE_MAX)
		{
			_fadeTimer->stop();
			_fadeValue = KS_TRAY_FADE_MAX;
		}
	}
	else
	{
		_fadeValue -= KS_TRAY_FADE_CHANGE;
		if (_fadeValue < 0.0f)
		{
			_fadeTimer->stop();
			_fadeValue = 0.0f;
		}
	}
	KPixmap p(kapp->miniIcon());
	setPixmap(KPixmapEffect::fade(p, _fadeValue, KS_TRAY_FADE_COLOR));
}

void MSystemTray::slotFlashTimeout()
{
	_flashCount++;

	if ((_flashCount % 2) == 0)
		setPixmap(*_saveIcon);
	else
		setPixmap(*_flashIcon);

	if (_flashCount == 6)
		_flashTimer->stop();
}

void MSystemTray::slotLockScreen()
{
	ks_actions->execConfirm(Action::LockScreen);
}

void MSystemTray::slotLogout()
{
	ks_actions->execConfirm(Action::Logout);
}

void MSystemTray::slotReboot()
{
	ks_actions->execConfirm(Action::Reboot);
}

void MSystemTray::slotShutDown()
{
	ks_actions->execConfirm(Action::ShutDown);
}
