// =============================================================================
//
//      --- kvi_identifiers.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviIdentifiers"

#include <time.h>

#include "kvi_app.h"
#include "kvi_command.h"
#include "kvi_console.h"
#include "kvi_defines.h"
#include "kvi_frame.h"
#include "kvi_mdi_manager.h"
#include "kvi_mirccntrl.h"
#include "kvi_script_object.h"
#include "kvi_script_objectcontroller.h"
#include "kvi_systray_onlinetimer.h"
#include "kvi_userlistbox.h"
#include "kvi_userparser.h"

KviIdentifierEntry KviUserParser::idnTable_A_G[] =
{
	{ "DAY"              , &KviUserParser::parseIdn_DAY               },
	{ "DIALOGRESULT"     , &KviUserParser::parseIdn_DIALOGRESULT      },
	{ "DIALOGMAGIC"      , &KviUserParser::parseIdn_DIALOGMAGIC       },
	{ "B"                , &KviUserParser::parseIdn_B                 },
	{ "CR"               , &KviUserParser::parseIdn_CR                },
	{ "DATE"             , &KviUserParser::parseIdn_DATE              },
	{ "CONSOLE"          , &KviUserParser::parseIdn_CONSOLE           },
	{ "CHAN"             , &KviUserParser::parseIdn_CHAN              },
	{ "CHAT"             , &KviUserParser::parseIdn_CHAT              },
	{ "ALLWINDOWS"       , &KviUserParser::parseIdn_ALLWINDOWS        },
	{ "ALLCHANNELS"      , &KviUserParser::parseIdn_ALLCHANNELS       },
	{ "ALLQUERIES"       , &KviUserParser::parseIdn_ALLQUERIES        },
	{ "ALLCHATS"         , &KviUserParser::parseIdn_ALLCHATS          },
	{ "CONNTIME"         , &KviUserParser::parseIdn_CONNTIME          },
	{ "CONNTIMES"        , &KviUserParser::parseIdn_CONNTIMES         },
	{ "ACTIVEWINDOW"     , &KviUserParser::parseIdn_ACTIVEWINDOW      },
	{ "CONNSTARTTIME"    , &KviUserParser::parseIdn_CONNSTARTTIME     },
	{ "EXECFILENAME"     , &KviUserParser::parseIdn_EXECFILENAME      },
	{ "GETRETURN"        , &KviUserParser::parseIdn_GETRETURN         },
	{ 0                  , 0                                          }
};

KviIdentifierEntry KviUserParser::idnTable_H_N[] =
{
	{ "ME"               , &KviUserParser::parseIdn_ME                },
	{ "MDISIZE"          , &KviUserParser::parseIdn_MDISIZE           },
	{ "MONTH"            , &KviUserParser::parseIdn_MONTH             },
	{ "MYNICK"           , &KviUserParser::parseIdn_ME                },
	{ "MYUSERNAME"       , &KviUserParser::parseIdn_MYUSERNAME        },
	{ "MYREALNAME"       , &KviUserParser::parseIdn_MYREALNAME        },
	{ "MYHOSTNAME"       , &KviUserParser::parseIdn_MYHOSTNAME        },
	{ "MYIP"             , &KviUserParser::parseIdn_MYIP              },
	{ "MYMASK"           , &KviUserParser::parseIdn_MYMASK            },
	{ "MYUMODE"          , &KviUserParser::parseIdn_MYUMODE           },
	{ "LF"               , &KviUserParser::parseIdn_LF                },
	{ "LOCALHOST"        , &KviUserParser::parseIdn_MYHOSTNAME        },
	{ "KVIRCDIR"         , &KviUserParser::parseIdn_KVIRCDIR          },
	{ "KVIRCLOGDIR"      , &KviUserParser::parseIdn_KVIRCLOGDIR       },
	{ "KVIRCSCRIPTDIR"   , &KviUserParser::parseIdn_KVIRCSCRIPTDIR    },
	{ "ISCONNECTED"      , &KviUserParser::parseIdn_ISCONNECTED       },
	{ "ISREMOTEEXEC"     , &KviUserParser::parseIdn_ISREMOTEEXEC      },
	{ "K"                , &KviUserParser::parseIdn_K                 },
	{ 0                  , 0                                          }
};

KviIdentifierEntry KviUserParser::idnTable_O_S[] =
{
	{ "SELECTED"         , &KviUserParser::parseIdn_SELECTED          },
	{ "R"                , &KviUserParser::parseIdn_R                 },
	{ "O"                , &KviUserParser::parseIdn_O                 },
	{ "QUERY"            , &KviUserParser::parseIdn_QUERY             },
	{ "SERVERHOSTNAME"   , &KviUserParser::parseIdn_SERVERHOSTNAME    },
	{ "SERVERIP"         , &KviUserParser::parseIdn_SERVERIP          },
	{ "SERVERPORT"       , &KviUserParser::parseIdn_SERVERPORT        },
	{ "SERVERNETWORK"    , &KviUserParser::parseIdn_SERVERNETWORK     },
	{ "STATUS"           , &KviUserParser::parseIdn_CONSOLE           },
	{ "SELECTEDMASKS"    , &KviUserParser::parseIdn_SELECTEDMASKS     },
	{ "SELECTEDHOSTS"    , &KviUserParser::parseIdn_SELECTEDHOSTS     },
	{ "SELECTEDUSERNAMES", &KviUserParser::parseIdn_SELECTEDUSERNAMES },
	{ "PROCCMDLINE"      , &KviUserParser::parseIdn_PROCCMDLINE       },
	{ "PROCPID"          , &KviUserParser::parseIdn_PROCPID           },
	{ "PROCSTDOUT"       , &KviUserParser::parseIdn_PROCSTDOUT        },
	{ "PROCSTDERR"       , &KviUserParser::parseIdn_PROCSTDERR        },
	{ "PROCMAGIC"        , &KviUserParser::parseIdn_PROCMAGIC         },
	{ "PROCWINDOW"       , &KviUserParser::parseIdn_PROCWINDOW        },
	{ "PROCEXITCODE"     , &KviUserParser::parseIdn_PROCEXITCODE      },
	{ "ROOT"             , &KviUserParser::parseIdn_ROOT              },
	{ "SIGNALSENDER"     , &KviUserParser::parseIdn_SIGNALSENDER      },
	{ 0                  , 0                                          }
};

KviIdentifierEntry KviUserParser::idnTable_T_Z[] =
{
	{ "WINDOW"           , &KviUserParser::parseIdn_WINDOW            },
	{ "U"                , &KviUserParser::parseIdn_U                 },
	{ "TIME"             , &KviUserParser::parseIdn_TIME              },
	{ "TIMES"            , &KviUserParser::parseIdn_TIMES             },
	{ "THIS"             , &KviUserParser::parseIdn_THIS              },
	{ "TIMERMAGIC"       , &KviUserParser::parseIdn_TIMERMAGIC        },
	{ "TIMERNAME"        , &KviUserParser::parseIdn_TIMERNAME         },
	{ "VERSION"          , &KviUserParser::parseIdn_VERSION           },
	{ "WHOISNICK"        , &KviUserParser::parseIdn_WHOISNICK		  },
	{ "WHOISUSER"        , &KviUserParser::parseIdn_WHOISUSER		  },
	{ "WHOISHOST"        , &KviUserParser::parseIdn_WHOISHOST		  },
	{ "WHOISREAL"        , &KviUserParser::parseIdn_WHOISREAL		  },
	{ "WHOISSERVER"      , &KviUserParser::parseIdn_WHOISSERVER	      },
	{ "WHOISIRCOP"       , &KviUserParser::parseIdn_WHOISIRCOP		  },
	{ "WHOISIDLE"        , &KviUserParser::parseIdn_WHOISIDLE		  },
	{ "WHOISSTATUS"      , &KviUserParser::parseIdn_WHOISSTATUS	      },
	{ "WHOISCHANNELS"    , &KviUserParser::parseIdn_WHOISCHANNELS	  },
	{ "WHOISMAGIC"       , &KviUserParser::parseIdn_WHOISMAGIC		  },
	{ "WHOISSUCCESS"     , &KviUserParser::parseIdn_WHOISSUCCESS      },
	{ "TRYRESULT"        , &KviUserParser::parseIdn_TRYRESULT         },
	{ "UPTIME"           , &KviUserParser::parseIdn_UPTIME            },
	{ "YEAR"             , &KviUserParser::parseIdn_YEAR              },
	{ 0                  , 0                                          }
};

// TODO: TARGET, QUERYTARGET, ISTIMER, CURRENTDIR, PWD
// TODO: $RemoteEnd and $RemoteEnd(<window>)

/*
	@identifier: MyUMode
	@short:
		Returns the current user mode
	@description:
		Returns the current user mode flags as a string.<br>
		Please note that this string may also be empty.<br>
		This identifier is guaranteed to be "in sync" after
		the login operations have been completed.
	@examples:
		<example>
		<a href="echo.kvihelp">echo</a> My user mode is $myUMode
		</example>
*/
bool KviUserParser::parseIdn_MYUMODE(KviCommand *cmd, KviStr &buffer)
{
	buffer.append(m_pFrm->m_global.szCurrentUserMode);
	return true;
}

/*
	@identifier: Console
	@short:
		Returns the name of the console window
	@description:
		Returns the name of the console window.<br>
		Equivalent to <a href="s_status.kvihelp">$Status</a>
	@examples:
		<example>
		<a href="echo.kvihelp">echo</a> -w $console This text goes to the console window.
		</example>
*/

/*
	@identifier: Status
	@short:
		Equivalent to $Console
	@description:
		Equivalent to <a href="s_console.kvihelp">$Console</a>
*/
bool KviUserParser::parseIdn_CONSOLE(KviCommand *cmd, KviStr &buffer)
{
	buffer.append(KVI_CONSOLE_WINDOW_NAME);
	return true;
}

/*
	@identifier: Version
	@short:
		Returns the KVIrc version string
	@description:
		Returns the KVIrc version string.
	@examples:
		<example>
		<a href="say.kvihelp">say</a> <a href="s_window.kvihelp">$window</a> Running $version
		</example>
*/
bool KviUserParser::parseIdn_VERSION(KviCommand *cmd, KviStr &buffer)
{
	buffer.append(KVI_VERSION);
	return true;
}

/*
	@identifier: Selected
	@short:
		Returns the list of selected nicknames
	@syntax:
		$selected<br>
		$selected(&lt;channel&gt;)
	@description:
		This identifier has also a related function.<br>
		The function returns a comma separated list of selected nicknames
		int the channel &lt;channel&gt;<br>
		The identifier (without the parameter) acts as $selected(<a href="s_window.kvihelp">$window</a>).<br>
		Please note that the list of nicknames may also be empty.!<br>
		This function/identifier returns an empty string if the window that
		the list is requested is not a channel (or console).
	@examples:
		<example>
		<a href="if.kvihelp">if</a>("$selected"=="")<a href="echo.kvihelp">echo</a> No selected nicknames.
		<a href="if.kvihelp">else</a> <a href="foreach.kvihelp">foreach</a>(%tmp, $selected(<a href="s_chan.kvihelp">$chan</a>))<a href="echo.kvihelp">echo</a> %tmp was selected.
		</example>
	@seealso:
		<a href="s_selectedmasks.kvihelp">$SelectedMasks</a>
*/
bool KviUserParser::parseIdn_SELECTED(KviCommand *cmd, KviStr &buffer)
{
	if( cmd->m_wnd->m_pListBox )
		cmd->m_wnd->m_pListBox->appendSelectedNicknames(buffer);
	return true;
}

/*
	@identifier: SelectedMasks
	@short:
		Returns the list of selected users masks
	@syntax:
		$selectedMasks<br>
		$selectedMasks(&lt;channel&gt;)
	@description:
		This identifier has also a related function.<br>
		The function returns a comma separated list of selected users masks
		int the channel &lt;channel&gt;<br>
		The identifier (without the parameter) acts as $selectedMasks(<a href="s_window.kvihelp">$window</a>).<br>
		Please note that the list of masks may also be empty.<br>
		A mask may also be incomplete (such as nick!*@*) if the channel
		userlist has not been updated completely yet.<br>
		This function/identifier returns an empty string if the window that
		the list is requested is not a channel (or console).
	@examples:
		<example>
		<a href="if.kvihelp">if</a>("$selectedMasks"=="")<a href="echo.kvihelp">echo</a> No selected masks.
		<a href="if.kvihelp">else</a> <a href="foreach.kvihelp">foreach</a>(%tmp, $selectedMasks(<a href="s_chan.kvihelp">$chan</a>))<a href="echo.kvihelp">echo</a> %tmp was selected.
		</example>
	@seealso:
		<a href="s_selected.kvihelp">$Selected</a>
*/
bool KviUserParser::parseIdn_SELECTEDMASKS(KviCommand *cmd, KviStr &buffer)
{
	if( cmd->m_wnd->m_pListBox )
		cmd->m_wnd->m_pListBox->appendSelectedMasks(buffer);
	return true;
}

/*
	@identifier: MdiSize
	@short:
		Returns the size of the MDI surface.
	@syntax:
		$MdiSize<br>
	@description:
		This identifier returns the size of the MDI surface (the surface containing all windows).<br>
		$mdisize will return the width and the height, separated by a comma.
	@examples:
		<example>
		<a href="echo.kvihelp">echo</a> $mdisize
		</example>
		This will print something like 1023, 722.
	@seealso:
		<a href="window.kvihelp">WINDOW</a>
*/
bool KviUserParser::parseIdn_MDISIZE(KviCommand *cmd, KviStr &buffer)
{
	KviStr tmp;
	tmp.setNum(m_pFrm->m_pMdi->width());
	buffer += tmp;
	buffer += ", ";
	tmp.setNum(m_pFrm->m_pMdi->height());
	buffer += tmp;
	return true;
}

/*
	@identifier: SelectedHosts
	@short:
		Returns the list of selected users hosts
	@syntax:
		$selectedHosts<br>
		$selectedHosts(&lt;channel&gt;)
	@description:
		This identifier has also a related function.<br>
		The function returns a comma separated list of selected users hosts
		int the channel &lt;channel&gt;<br>
		The identifier (without the parameter) acts as $selectedHosts(<a href="s_window.kvihelp">$window</a>).<br>
		Please note that the list of hosts may also be empty.<br>
		A host may also be incomplete (*) if the channel
		userlist has not been updated completely yet.<br>
		This function/identifier returns an empty string if the window that
		the list is requested is not a channel (or console).
	@examples:
		<example>
		<a href="if.kvihelp">if</a>("$selectedHosts"=="")<a href="echo.kvihelp">echo</a> No selected nicknames.
		<a href="if.kvihelp">else</a> {
		@tab@<a href="foreach.kvihelp">foreach</a>(%tmp, $selectedHosts(<a href="s_chan.kvihelp">$chan</a>)){
		@tab@@tab@<a href="if.kvihelp">if</a>("%tmp" <a href="operators.kvihelp">!=</a> "")<a href="host.kvihelp">host</a> %tmp
		@tab@}
		}
		</example>
	@seealso:
		<a href="s_selected.kvihelp">$Selected</a>,
		<a href="s_selectedmasks.kvihelp">$SelectedMasks</a>,
		<a href="s_selectedusernames.kvihelp">$SelectedUsernames</a>
*/
bool KviUserParser::parseIdn_SELECTEDHOSTS(KviCommand *cmd, KviStr &buffer)
{
	if( cmd->m_wnd->m_pListBox )
		cmd->m_wnd->m_pListBox->appendSelectedHosts(buffer);
	return true;
}

/*
	@identifier: SelectedUsernames
	@short:
		Returns the list of selected usernames
	@syntax:
		$selectedUsernames<br>
		$selectedUsernames(&lt;channel&gt;)
	@description:
		This identifier has also a related function.<br>
		The function returns a comma separated list of selected usernames
		int the channel &lt;channel&gt;<br>
		The identifier (without the parameter) acts as $selectedUsernames(<a href="s_window.kvihelp">$window</a>).<br>
		Please note that the list of usernames may also be empty.<br>
		A username may also be incomplete (*) if the channel
		userlist has not been updated completely yet.<br>
		This function/identifier returns an empty string if the window that
		the list is requested is not a channel (or console).
	@seealso:
		<a href="s_selected.kvihelp">$Selected</a>,
		<a href="s_selectedmasks.kvihelp">$SelectedMasks</a>,
		<a href="s_selectedhosts.kvihelp">$SelectedHosts</a>
*/
bool KviUserParser::parseIdn_SELECTEDUSERNAMES(KviCommand *cmd, KviStr &buffer)
{
	if( cmd->m_wnd->m_pListBox )
		cmd->m_wnd->m_pListBox->appendSelectedUsernames(buffer);
	return true;
}

/*
	@identifier: ServerNetwork
	@short:
		Returns the current IRC network name.
	@description:
		Returns the current IRC network name (if known).
	@examples:
		<example>
		<a href="say.kvihelp">say</a> <a href="s_window.kvihelp">$window</a> We are on the $ServerNetwork network.
		</example>
	@seealso:
		<a href="s_serverport.kvihelp">$ServerPort</a>,
		<a href="s_serverhostname.kvihelp">$ServerHostName</a>,
		<a href="s_serverip.kvihelp">$ServerIp</a>
*/
bool KviUserParser::parseIdn_SERVERNETWORK(KviCommand *cmd, KviStr &buffer)
{
	buffer.append(m_pFrm->m_global.szCurrentNetwork);
	return true;
}

/*
	@identifier: ServerPort
	@short:
		Returns the current IRC server port
	@description:
		Returns the IRC server port that we are connected to.
	@examples:
		<example>
		<a href="say.kvihelp">say</a> <a href="s_window.kvihelp">$window</a> I am connected to $ServerHostName on port $ServerPort
		</example>
	@seealso:
		<a href="s_servernetwork.kvihelp">$ServerNetwork</a>,
		<a href="s_serverhostname.kvihelp">$ServerHostName</a>,
		<a href="s_serverip.kvihelp">$ServerIp</a>
*/
bool KviUserParser::parseIdn_SERVERPORT(KviCommand *cmd, KviStr &buffer)
{
	buffer.append(m_pFrm->m_global.szCurrentServerPort);
	return true;
}

/*
	@identifier: ServerHostName
	@short:
		Returns the current IRC server hostname
	@description:
		Returns the current IRC server hostname as you have passed to the
		<a href="server.kvihelp">/SERVER</a> command.<br>
		Please note that this may be also an IP.
	@examples:
		<example>
		<a href="say.kvihelp">say</a> <a href="s_window.kvihelp">$window</a> I am connected to $ServerHostName on port $ServerPort
		</example>
	@seealso:
		<a href="s_serverport.kvihelp">$ServerPort</a>,
		<a href="s_serverhostname.kvihelp">$ServerHostName</a>,
		<a href="s_serverip.kvihelp">$ServerIp</a>
*/
bool KviUserParser::parseIdn_SERVERHOSTNAME(KviCommand *cmd, KviStr &buffer)
{
	buffer.append(m_pFrm->m_global.szCurrentServerHost);
	return true;
}

/*
	@identifier: ServerIp
	@short:
		Returns the current IRC server IP address
	@description:
		Returns the current IRC server IP address.
	@examples:
		<example>
		<a href="say.kvihelp">say</a> <a href="s_window.kvihelp">$window</a> I am connected to $ServerHostName ($ServerIp) on port $ServerPort
		</example>
	@seealso:
		<a href="s_serverport.kvihelp">$ServerPort</a>,
		<a href="s_serverhostname.kvihelp">$ServerHostName</a>,
		<a href="s_servernetwork.kvihelp">$ServerNetwork</a>
*/

bool KviUserParser::parseIdn_SERVERIP(KviCommand *cmd, KviStr &buffer)
{
	QHostAddress addr = m_pFrm->m_global.szCurrentServerAddresses.first();
	if( !addr.isNull() )
		buffer.append(addr.toString().ascii());
	return true;
}

/*
	@identifier: MyMask
	@short:
		Returns your current mask
	@description:
		Returns your current mask as known by the IRC network.<br>
		The result is undefined if you are NOT connected to a server.
	@examples:
		<example>
		<a href="mode.kvihelp">mode</a> #myChan +l $myMask
		</example>
	@seealso:
		<a href="s_myhostname.kvihelp">$MyHostName</a>,
		<a href="s_myip.kvihelp">$MyIp</a>,
		<a href="s_myusername.kvihelp">$MyUserName</a>,
		<a href="s_myrealname.kvihelp">$MyRealName</a>,
		<a href="s_me.kvihelp">$Me</a>
*/
bool KviUserParser::parseIdn_MYMASK(KviCommand *cmd, KviStr &buffer)
{
	buffer.append(m_pFrm->m_global.szCurrentMaskFromServer);
	return true;
}

/*
	@identifier: MyHostName
	@short:
		Returns your current host name
	@description:
		Returns your current host name as known on the IRC network that you are connected to.<br>
		The result is undefined if you are NOT connected to a server.
	@examples:
		<example>
		<a href="msg.kvihelp">msg</a> <a href="s_selected.kvihelp">$selected</a> My hostname is $MyHostName
		</example>
	@seealso:
		<a href="s_mymask.kvihelp">$MyMask</a>,
		<a href="s_myip.kvihelp">$MyIp</a>,
		<a href="s_myusername.kvihelp">$MyUserName</a>,
		<a href="s_myrealname.kvihelp">$MyRealName</a>,
		<a href="s_me.kvihelp">$Me</a>
*/
bool KviUserParser::parseIdn_MYHOSTNAME(KviCommand *cmd, KviStr &buffer)
{
	buffer.append(m_pFrm->m_global.szLocalHostName);
	return true;
}

/*
	@identifier: MyIp
	@short:
		Returns your current IP address
	@description:
		Returns your current IP address.<br>
		The result is undefined if you are not connected to a server.<br>
	@examples:
		<example>
		<a href="msg.kvihelp">msg</a> MyFriend Please telnet me : $MyIp
		</example>
	@seealso:
		<a href="s_mymask.kvihelp">$MyMask</a>,
		<a href="s_myhostname.kvihelp">$MyHostName</a>,
		<a href="s_myusername.kvihelp">$MyUserName</a>,
		<a href="s_myrealname.kvihelp">$MyRealName</a>,
		<a href="s_me.kvihelp">$Me</a>
*/
bool KviUserParser::parseIdn_MYIP(KviCommand *cmd, KviStr &buffer)
{
	buffer.append(m_pFrm->m_global.szLocalHostIp);
	return true;
}

/*
	@identifier: MyUserName
	@short:
		Returns your current user name
	@description:
		Returns your current user name as specified in the user options dialog.<br>
		The result is undefined if you are NOT connected to a server.<br>
		Please note that this user name may not correspond to the user name
		accepted by the server (that may use the IDENT service).
	@examples:
		<example>
		<a href="echo.kvihelp">echo</a> My user name is $MyUserName
		</example>
	@seealso:
		<a href="s_mymask.kvihelp">$MyMask</a>,
		<a href="s_myhostname.kvihelp">$MyHostName</a>,
		<a href="s_myip.kvihelp">$MyIp</a>,
		<a href="s_myrealname.kvihelp">$MyRealName</a>,
		<a href="s_me.kvihelp">$Me</a>
*/
bool KviUserParser::parseIdn_MYUSERNAME(KviCommand *cmd, KviStr &buffer)
{
	buffer.append(m_pFrm->m_global.szCurrentUserName);
	return true;
}

/*
	@identifier: MyRealName
	@short:
		Returns your current "real name"
	@description:
		Returns your current real name as specified in the user options dialog.<br>
		The result is undefined if you are NOT connected to a server.
	@examples:
		<example>
		<a href="echo.kvihelp">echo</a> My real name string is $MyRealName
		</example>
	@seealso:
		<a href="s_mymask.kvihelp">$MyMask</a>,
		<a href="s_myhostname.kvihelp">$MyHostName</a>,
		<a href="s_myip.kvihelp">$MyIp</a>,
		<a href="s_myusername.kvihelp">$MyUserName</a>,
		<a href="s_me.kvihelp">$Me</a>
*/
bool KviUserParser::parseIdn_MYREALNAME(KviCommand *cmd, KviStr &buffer)
{
	buffer.append(m_pFrm->m_global.szCurrentRealName);
	return true;
}

/*
	@identifier: MyNick
	@short:
		Returns your current nickname
	@description:
		Returns your current nickname as accepted by the current IRC server.<br>
		The result is undefined if you are NOT connected to a server.<br>
		Equivalent to <a href="s_me.kvihelp">$Me</a>
	@seealso:
		<a href="s_mymask.kvihelp">$MyMask</a>,
		<a href="s_myhostname.kvihelp">$MyHostName</a>,
		<a href="s_myip.kvihelp">$MyIp</a>,
		<a href="s_myusername.kvihelp">$MyUserName</a>
*/

/*
	@identifier: Me
	@short:
		Equivalent to $MyNick
	@description:
		Equivalent to <a href="s_mynick.kvihelp">$MyNick</a>
*/
bool KviUserParser::parseIdn_ME(KviCommand *cmd, KviStr &buffer)
{
	buffer.append(m_pFrm->m_global.szCurrentNick);
	return true;
}

/*
	@identifier: Window
	@short:
		Returns the current window name
	@description:
		Returns the window name that this command is bound to.
	@examples:
		<example>
		<a href="echo.kvihelp">echo</a> -w <a href="s_console.kvihelp">$console</a> This command was executed in window $window
		</example>
	@seealso:
		<a href="s_console.kvihelp">$Console</a>
*/
bool KviUserParser::parseIdn_WINDOW(KviCommand *cmd, KviStr &buffer)
{
	buffer.append(cmd->m_wnd->caption());
	return true;
}

/*
	@identifier: Chan
	@short:
		Returns the current channel name.
	@description:
		If the window that this command is executed in is a channel window,
		this identifier returns its name (and is equivalent to <a href="s_window.kvihelp">$window</a>),
		otherwise returns an empty string.
	@examples:
		<example>
		<a href="if.kvihelp">if</a>("$chan" == "")<a href="echo.kvihelp">echo</a> This is not a channel;
		else <a href="echo.kvihelp">echo</a> This is a channel window ($chan)
		</example>
	@seealso:
		<a href="s_window.kvihelp">$Window</a>
		<a href="s_console.kvihelp">$Console</a>
*/
bool KviUserParser::parseIdn_CHAN(KviCommand *cmd, KviStr &buffer)
{
	if( cmd->m_wnd->type() == KVI_WND_TYPE_CHANNEL )
		buffer.append(cmd->m_wnd->caption());
	return true;
}

/*
	@identifier: Query
	@short:
		Returns the current query name.
	@description:
		If the window that this command is executed in is a query window,
		this identifier returns its name (and is equivalent to <a href="s_window.kvihelp">$window</a>),
		otherwise returns an empty string.
	@examples:
		<example>
		<a href="if.kvihelp">if</a>("$query" == "")<a href="echo.kvihelp">echo</a> This is not a query
		else <a href="echo.kvihelp">echo</a> This is a query window ($query)
		</example>
	@seealso:
		<a href="s_chat.kvihelp">$Chat</a>
		<a href="s_chan.kvihelp">$Chan</a>
		<a href="s_window.kvihelp">$Window</a>
		<a href="s_console.kvihelp">$Console</a>
*/
bool KviUserParser::parseIdn_QUERY(KviCommand *cmd, KviStr &buffer)
{
	if( cmd->m_wnd->type() == KVI_WND_TYPE_QUERY )
		buffer.append(cmd->m_wnd->caption());
	return true;
}

/*
	@identifier: Chat
	@short:
		Returns the current DCC chat name.
	@description:
		If the window that this command is executed in is a DCC chat window,
		this identifier returns its name (and is equivalent to <a href="s_window.kvihelp">$window</a>),
		otherwise returns an empty string.
	@examples:
		<example>
		<a href="if.kvihelp">if</a>("$chat" == "")<a href="echo.kvihelp">echo</a> This is not a DCC chat window
		else <a href="echo.kvihelp">echo</a> This is a DCC chat window ($chat)
		</example>
	@seealso:
		<a href="s_query.kvihelp">$Query</a>
		<a href="s_chan.kvihelp">$Chan</a>
		<a href="s_window.kvihelp">$Window</a>
		<a href="s_console.kvihelp">$Console</a>
*/
bool KviUserParser::parseIdn_CHAT(KviCommand *cmd, KviStr &buffer)
{
	if( cmd->m_wnd->type() == KVI_WND_TYPE_CHAT )
		buffer.append(cmd->m_wnd->caption());
	return true;
}

/*
	@identifier: Cr
	@short:
		Evaluates to the Carriage return character
	@description:
		Evaluates to the Carriage return character
	@examples:
		Send two irc messages using one raw command
		<example>
		<a href="raw.kvihelp">raw</a> privmsg #channel :Hey$Cr$Lf\privmsg #channel :Whassup?
		</example>
	@seealso:
		<a href="s_lf.kvihelp">$Lf</a>
*/
bool KviUserParser::parseIdn_CR(KviCommand *cmd, KviStr &buffer)
{
	buffer.append("\r");
	return true;
}

/*
	@identifier: Lf
	@short:
		Evaluates to the Line feed character
	@description:
		Evaluates to the Line feed character
	@examples:
		Send two irc messages using one raw command.<br>
		The escape character is used to separate the $LF identifier from the following string.
		<example>
		<a href="raw.kvihelp">raw</a> privmsg #channel :Hey$Cr$Lf\privmsg #channel :Whassup?
		</example>
		Echo two lines using one echo command
		<example>
		<a href="echo.kvihelp">echo</a> Line1$LF\Line2
		</example>
	@seealso:
		<a href="s_cr.kvihelp">$Cr</a>
*/
bool KviUserParser::parseIdn_LF(KviCommand *cmd, KviStr &buffer)
{
	buffer.append("\n");
	return true;
}

/*
	@identifier: Time
	@short:
		Returns the current time as a string
	@description:
		Returns the current time as a string.<br>
		Equivalent to <a href="s_numtimetostring.kvihelp">$numtimetostring</a>(<a href="s_time.kvihelp">$time</a>).<br>
	@examples:
		<example>
		<a href="echo.kvihelp">echo</a> Current time : $time
		</example>
	@seealso:
		<a href="s_date.kvihelp">$Date</a>,
		<a href="s_times.kvihelp">$TimeS</a>
*/
bool KviUserParser::parseIdn_TIME(KviCommand *cmd, KviStr &buffer)
{
	buffer.append(QTime::currentTime().toString());
	return true;
}

/*
	@identifier: TimeS
	@short:
		Returns the current time as a number
	@description:
		Returns the current time since
		the Epoch (00:00:00 UTC, January 1, 1970), measured in seconds.<br>
	@examples:
		<example>
		<a href="echo.kvihelp">echo</a> Current time : $times (<a href="s_numtimetostring.kvihelp">$numtimetostring</a>($times))
		</example>
	@seealso:
		<a href="s_time.kvihelp">$Time</a>,
		<a href="s_date.kvihelp">$Date</a>,
		<a href="s_numtimetostring.kvihelp">$NumTimeToString()</a>

*/
bool KviUserParser::parseIdn_TIMES(KviCommand *cmd, KviStr &buffer)
{
	KviStr tmp;
	tmp.setNum(time(0));
	buffer.append(tmp.ptr());
	return true;
}

/*
	@identifier: ConnStartTime
	@short:
		Returns the time of the beginning of the current connection
	@description:
		Returns the time at which the current connection was started expressed
		as UNIX time.<br>
		It is useful to identify the current connection.<br>
	@seealso:
		<a href="s_time.kvihelp">$Time</a>,
		<a href="s_date.kvihelp">$Date</a>,
		<a href="s_numtimetostring.kvihelp">$NumTimeToString()</a>

*/
bool KviUserParser::parseIdn_CONNSTARTTIME(KviCommand *cmd, KviStr &buffer)
{
	KviStr tmp;
	tmp.setNum(m_pFrm->m_pSysTrayOnLineTimer->getConnectionStartTime());
	buffer.append(tmp.ptr());
	return true;
}

/*
	@identifier: Uptime
	@short:
		Returns the time since KVIrc was started
	@description:
		Returns the time since KVIrc was started measured in seconds,
	@seealso:
		<a href="s_time.kvihelp">$Time</a>,
		<a href="s_date.kvihelp">$Date</a>,
		<a href="s_times.kvihelp">$Times</a>,
		<a href="s_durationtostring.kvihelp">$DurationToString()</a>
*/
bool KviUserParser::parseIdn_UPTIME(KviCommand *cmd, KviStr &buffer)
{
	KviStr tmp;
	tmp.setNum(g_pApp->m_startupTime.secsTo(QDateTime::currentDateTime()));
	buffer.append(tmp.ptr());
	return true;
}

/*
	@identifier: Date
	@short:
		Returns the current date as a string
	@description:
		Returns the current date as a string
	@examples:
		<example>
		<a href="echo.kvihelp">echo</a> Current date : $date
		</example>
	@seealso:
		<a href="s_time.kvihelp">$Time</a>
		<a href="s_times.kvihelp">$TimeS</a>
*/
bool KviUserParser::parseIdn_DATE(KviCommand *cmd, KviStr &buffer)
{
	buffer.append(QDate::currentDate().toString());
	return true;
}

/*
	@identifier: Day
	@short:
		Returns the current day
	@description:
		Returns the current day of month (with leading 0, if appropriate)
	@examples:
		<example>
		<a href="echo.kvihelp">echo</a> day of month : $day
		</example>
	@seealso:
		<a href="s_date.kvihelp">$Date</a>,
		<a href="s_time.kvihelp">$Time</a>,
		<a href="s_times.kvihelp">$TimeS</a>
*/
bool KviUserParser::parseIdn_DAY(KviCommand *cmd, KviStr &buffer)
{
	QString dateStr;
	dateStr.sprintf("%02d", QDate::currentDate().day());
	buffer.append(dateStr);
	return true;
}

/*
	@identifier: Month
	@short:
		Returns the current month
	@description:
		Returns the current month (with leading 0, if appropriate)
	@examples:
		<example>
		<a href="echo.kvihelp">echo</a> the current month is : $month
		</example>
	@seealso:
		<a href="s_date.kvihelp">$Date</a>,
		<a href="s_time.kvihelp">$Time</a>,
		<a href="s_times.kvihelp">$TimeS</a>
*/
bool KviUserParser::parseIdn_MONTH(KviCommand *cmd, KviStr &buffer)
{
	QString dateStr;
	dateStr.sprintf("%02d", QDate::currentDate().month());
	buffer.append(dateStr);
	return true;
}

/*
	@identifier: Year
	@short:
		Returns the current year
	@description:
		Returns the current year (4 digits)
	@examples:
		<example>
		<a href="echo.kvihelp">echo</a> we ar in the year : $year
		</example>
	@seealso:
		<a href="s_date.kvihelp">$Date</a>,
		<a href="s_time.kvihelp">$Time</a>,
		<a href="s_times.kvihelp">$TimeS</a>
*/
bool KviUserParser::parseIdn_YEAR(KviCommand *cmd, KviStr &buffer)
{
	KviStr dateStr;
	dateStr.sprintf("%d", QDate::currentDate().year());
	buffer.append(dateStr);
	return true;
}

/*
	@identifier: AllWindows
	@short:
		Returns a list of currently open windows
	@description:
		Returns a comma separated list of currently open windows.
	@examples:
		<example>
		<a href="foreach.kvihelp">foreach</a>(%t, $allWindows)<a href="echo.kvihelp">echo</a> This is the window %t
		</example>
	@seealso:
		<a href="s_allchannels.kvihelp">$AllChannels</a>,
		<a href="s_allchats.kvihelp">$AllChats</a>,
		<a href="s_allqueries.kvihelp">$AllQueries</a>
*/
bool KviUserParser::parseIdn_ALLWINDOWS(KviCommand *cmd, KviStr &buffer)
{
	bool bAddComma = false;
	for( KviWindow *w = m_pFrm->m_pWinList->first(); w; w = m_pFrm->m_pWinList->next() ) {
		if( bAddComma )
			buffer.append(',');
		else
			bAddComma = true;
		buffer.append(w->caption());
	}
	return true;
}

/*
	@identifier: AllChannels
	@short:
		Returns a list of currently open channel windows.
	@description:
		Returns a comma separated list of currently open channel windows.
	@examples:
		<example>
		<a href="foreach.kvihelp">foreach</a>(%t, $allChannels)<a href="say.kvihelp">say</a> %t Hello all!
		</example>
	@seealso:
		<a href="s_allwindows.kvihelp">$AllWindows</a>,
		<a href="s_allchats.kvihelp">$AllChats</a>,
		<a href="s_allqueries.kvihelp">$AllQueries</a>
*/
bool KviUserParser::parseIdn_ALLCHANNELS(KviCommand *cmd, KviStr &buffer)
{
	bool bAddComma = false;
	for( KviWindow *w = m_pFrm->m_pWinList->first(); w; w = m_pFrm->m_pWinList->next() ) {
		if( w->type() ==  KVI_WND_TYPE_CHANNEL ) {
			if( bAddComma )
				buffer.append(',');
			else
				bAddComma = true;
			buffer.append(w->caption());
		}
	}
	return true;
}

/*
	@identifier: AllQueries
	@short:
		Returns a list of currently open query windows.
	@description:
		Returns a comma separated list of currently open query windows.
	@examples:
		<example>
		<a href="foreach.kvihelp">foreach</a>(%t, $allQueries)<a href="say.kvihelp">say</a> %t Cya... I will go sleep...
		</example>
	@seealso:
		<a href="s_allwindows.kvihelp">$AllWindows</a>,
		<a href="s_allchats.kvihelp">$AllChats</a>,
		<a href="s_allchannels.kvihelp">$AllChannels</a>
*/
bool KviUserParser::parseIdn_ALLQUERIES(KviCommand *cmd, KviStr &buffer)
{
	bool bAddComma = false;
	for( KviWindow *w = m_pFrm->m_pWinList->first(); w; w = m_pFrm->m_pWinList->next() ) {
		if( w->type() ==  KVI_WND_TYPE_QUERY ) {
			if( bAddComma )
				buffer.append(',');
			else
				bAddComma = true;
			buffer.append(w->caption());
		}
	}
	return true;
}

/*
	@identifier: AllChats
	@short:
		Returns a list of currently open DCC chat windows.
	@description:
		Returns a comma separated list of currently open DCC chat windows.
	@examples:
		<example>
		<a href="foreach.kvihelp">foreach</a>(%t, $allChats)<a href="say.kvihelp">say</a> %t Cya... I will go sleep...
		</example>
	@seealso:
		<a href="s_allwindows.kvihelp">$AllWindows</a>,
		<a href="s_allqueries.kvihelp">$AllQueries</a>,
		<a href="s_allchannels.kvihelp">$AllChannels</a>
*/
bool KviUserParser::parseIdn_ALLCHATS(KviCommand *cmd, KviStr &buffer)
{
	bool bAddComma = false;
	for( KviWindow *w = m_pFrm->m_pWinList->first(); w; w = m_pFrm->m_pWinList->next() ) {
		if( w->type() ==  KVI_WND_TYPE_CHAT ) {
			if( bAddComma )
				buffer.append(',');
			else
				bAddComma = true;
			buffer.append(w->caption());
		}
	}
	return true;
}

/*
	@identifier: KVIrcDir
	@short:
		Returns the path of the KVIrc directory
	@description:
		Returns the path of the local KVIrc directory (ie. the one that the user select when running KVIrc for the first time).
	@seealso:
		<a href="s_kvirclogdir.kvihelp">$KVIrcLogDir</a>
*/
bool KviUserParser::parseIdn_KVIRCDIR(KviCommand *cmd, KviStr &buffer)
{
	KviStr tmp;
	g_pApp->getLocalKVIrcDirectory(tmp);
	buffer.append(tmp);
	return true;
}

/*
	@identifier: KVIrcLogDir
	@short:
		Returns the path of the KVIrc logs directory
	@description:
		Returns the path of the local KVIrc logs directory.
	@seealso:
		<a href="s_kvircdir.kvihelp">$KVIrcDir</a>
*/
bool KviUserParser::parseIdn_KVIRCLOGDIR(KviCommand *cmd, KviStr &buffer)
{
	KviStr tmp;
	g_pApp->getLocalKVIrcDirectory(tmp, KviApp::Log);
	buffer.append(tmp);
	return true;
}

/*
	@identifier: KVIrcScriptDir
	@short:
		Returns the path of the KVIrc scripts directory
	@description:
		Returns the path of the local KVIrc scripts directory.
	@seealso:
		<a href="s_kvircdir.kvihelp">$KVIrcDir</a>
*/
bool KviUserParser::parseIdn_KVIRCSCRIPTDIR(KviCommand *cmd, KviStr &buffer)
{
	KviStr tmp;
	g_pApp->getLocalKVIrcDirectory(tmp, KviApp::Scripts);
	buffer.append(tmp);
	return true;
}

/*
	@identifier: K
	@short:
		Evaluates to the CTRL+K color control code
	@description:
		Evaluates to the CTRL+K color control code.<br>
		Since this control code requires two numeric
		values to be specified after and does not allow a space
		you should use this identifier as $K\color1, color2.<br>
		The escape character will make the KVIrc's parser happy :)
	@seealso:
		<a href="s_b.kvihelp">$B</a>, <a href="s_u.kvihelp">$U</a>, <a href="s_r.kvihelp">$R</a>, <a href="s_o.kvihelp">$O</a>,
*/
bool KviUserParser::parseIdn_K(KviCommand *cmd, KviStr &buffer)
{
	buffer.append(KVI_TEXT_COLOR);
	return true;
}

/*
	@identifier: B
	@short:
		Evaluates to the CTRL+B bold control code
	@description:
		Evaluates to the CTRL+B bold control code
	@seealso:
		<a href="s_k.kvihelp">$K</a>, <a href="s_u.kvihelp">$U</a>, <a href="s_r.kvihelp">$R</a>, <a href="s_o.kvihelp">$O</a>,
*/
bool KviUserParser::parseIdn_B(KviCommand *cmd, KviStr &buffer)
{
	buffer.append(KVI_TEXT_BOLD);
	return true;
}

/*
	@identifier: U
	@short:
		Evaluates to the CTRL+U underline control code
	@description:
		Evaluates to the CTRL+U underline control code
	@seealso:
		<a href="s_k.kvihelp">$K</a>, <a href="s_b.kvihelp">$B</a>, <a href="s_r.kvihelp">$R</a>, <a href="s_o.kvihelp">$O</a>,
*/
bool KviUserParser::parseIdn_U(KviCommand *cmd, KviStr &buffer)
{
	buffer.append(KVI_TEXT_UNDERLINE);
	return true;
}

/*
	@identifier: R
	@short:
		Evaluates to the CTRL+R reverse control code
	@description:
		Evaluates to the CTRL+R reverse control code
	@seealso:
		<a href="s_k.kvihelp">$K</a>, <a href="s_b.kvihelp">$B</a>, <a href="s_u.kvihelp">$U</a>, <a href="s_o.kvihelp">$O</a>,
*/
bool KviUserParser::parseIdn_R(KviCommand *cmd, KviStr &buffer)
{
	buffer.append(KVI_TEXT_REVERSE);
	return true;
}

/*
	@identifier: O
	@short:
		Evaluates to the CTRL+O reset control code
	@description:
		Evaluates to the CTRL+O reset control code
	@seealso:
		<a href="s_k.kvihelp">$K</a>, <a href="s_b.kvihelp">$B</a>, <a href="s_u.kvihelp">$U</a>, <a href="s_r.kvihelp">$R</a>,
*/
bool KviUserParser::parseIdn_O(KviCommand *cmd, KviStr &buffer)
{
	buffer.append(KVI_TEXT_RESET);
	return true;
}

/*
	@identifier: ProcMagic
	@short:
		Evaluates to the terminating slave process magic data
	@description:
		Evaluates to the terminating slave process magic data.<br>
	@seealso:
		<a href="execv.kvihelp">EXECV</a>,
		<a href="s_procpid.kvihelp">$procpid</a>,
		<a href="s_procstderr.kvihelp">$procstderr</a>,
		<a href="s_procstdout.kvihelp">$procstdout</a>,
		<a href="s_proccmdline.kvihelp">$proccmdline</a>
		<a href="s_procwindow.kvihelp">$procwindow</a>
		<a href="s_procexitcode.kvihelp">$procexitcode</a>
*/
bool KviUserParser::parseIdn_PROCMAGIC(KviCommand *cmd, KviStr &buffer)
{
	buffer.append(m_szLastProcMagic);
	return true;
}

/*
	@identifier: ProcCmdLine
	@short:
		Evaluates to the terminating slave process commandline
	@description:
		Evaluates to the terminating slave process commandline.<br>
	@seealso:
		<a href="execv.kvihelp">EXECV</a>,
		<a href="s_procpid.kvihelp">$procpid</a>,
		<a href="s_procstderr.kvihelp">$procstderr</a>,
		<a href="s_procstdout.kvihelp">$procstdout</a>,
		<a href="s_procmagic.kvihelp">$procmagic</a>
		<a href="s_procwindow.kvihelp">$procwindow</a>
		<a href="s_procexitcode.kvihelp">$procexitcode</a>
*/
bool KviUserParser::parseIdn_PROCCMDLINE(KviCommand *cmd, KviStr &buffer)
{
	buffer.append(m_szLastProcCommandline);
	return true;
}

/*
	@identifier: ProcStderr
	@short:
		Evaluates to the terminating slave process stderr output.
	@description:
		Evaluates to the terminating slave process stderr output.<br>
	@seealso:
		<a href="execv.kvihelp">EXECV</a>,
		<a href="s_procpid.kvihelp">$procpid</a>,
		<a href="s_proccmdline.kvihelp">$proccmdline</a>,
		<a href="s_procstdout.kvihelp">$procstdout</a>,
		<a href="s_procmagic.kvihelp">$procmagic</a>
		<a href="s_procwindow.kvihelp">$procwindow</a>
		<a href="s_procexitcode.kvihelp">$procexitcode</a>
*/
bool KviUserParser::parseIdn_PROCSTDERR(KviCommand *cmd, KviStr &buffer)
{
	buffer.append(m_szLastProcStderr);
	return true;
}

/*
	@identifier: ProcStdout
	@short:
		Evaluates to the terminating slave process stdout output.
	@description:
		Evaluates to the terminating slave process stdout output.<br>
	@seealso:
		<a href="execv.kvihelp">EXECV</a>,
		<a href="s_procpid.kvihelp">$procpid</a>,
		<a href="s_proccmdline.kvihelp">$proccmdline</a>,
		<a href="s_procstderr.kvihelp">$procstderr</a>,
		<a href="s_procmagic.kvihelp">$procmagic</a>
		<a href="s_procwindow.kvihelp">$procwindow</a>
		<a href="s_procexitcode.kvihelp">$procexitcode</a>
*/
bool KviUserParser::parseIdn_PROCSTDOUT(KviCommand *cmd, KviStr &buffer)
{
	buffer.append(m_szLastProcStdout);
	return true;
}

/*
	@identifier: WhoisNick
	@short:
		Returns the nickname of the current AWHOIS call target user
	@description:
		This identifier contains valid information only in the <a href="awhois.kvihelp">AWHOIS</a>
		callback command.<br> It returns the nickname of the user that the WHOIS information
		was requested for.<br>
	@seealso:
		<a href="awhois.kvihelp">AWHOIS</a>,
		<a href="s_whoissuccess.kvihelp">$whoisSuccess</a>,
		<a href="s_whoisuser.kvihelp">$whoisUser</a>,
		<a href="s_whoishost.kvihelp">$whoisHost</a>,
		<a href="s_whoisreal.kvihelp">$whoisReal</a>
*/
bool KviUserParser::parseIdn_WHOISNICK(KviCommand *cmd, KviStr &buffer)
{
	buffer.append(m_szLastAsyncWhoisNick);
	return true;
}

/*
	@identifier: WhoisSuccess
	@short:
		Returns the status of the current AWHOIS call
	@description:
		This identifier contains valid information only in the <a href="awhois.kvihelp">AWHOIS</a>
		callback command.<br> It returns 0 if the requested user is not on IRC (KVIrc
		has received a "No such nickname" error), 1 otherwise.
	@seealso:
		<a href="awhois.kvihelp">AWHOIS</a>,
		<a href="s_whoisnick.kvihelp">$whoisNick</a>,
		<a href="s_whoisuser.kvihelp">$whoisUser</a>,
		<a href="s_whoishost.kvihelp">$whoisHost</a>,
		<a href="s_whoisreal.kvihelp">$whoisReal</a>
*/
bool KviUserParser::parseIdn_WHOISSUCCESS(KviCommand *cmd, KviStr &buffer)
{
	buffer.append(m_szLastAsyncWhoisSuccess);
	return true;
}

/*
	@identifier: WhoisUser
	@short:
		Returns the username of the current AWHOIS call target user
	@description:
		This identifier contains valid information only in the <a href="awhois.kvihelp">AWHOIS</a>
		callback command.<br>
		It returns the username of the user that the WHOIS information
		was requested for.<br>
		Please note that this identifier will be empty if <a href="s_whoisSuccess.kvihelp">$whoisSuccess</a>
		returns 0.<br>
	@seealso:
		<a href="awhois.kvihelp">AWHOIS</a>,
		<a href="s_whoisnick.kvihelp">$whoisNick</a>,
		<a href="s_whoissuccess.kvihelp">$whoisSuccess</a>,
		<a href="s_whoishost.kvihelp">$whoisHost</a>,
		<a href="s_whoisreal.kvihelp">$whoisReal</a>
*/
bool KviUserParser::parseIdn_WHOISUSER(KviCommand *cmd, KviStr &buffer)
{
	buffer.append(m_szLastAsyncWhoisUsername);
	return true;
}

/*
	@identifier: WhoisHost
	@short:
		Returns the hostname of the current AWHOIS call target user
	@description:
		This identifier contains valid information only in the <a href="awhois.kvihelp">AWHOIS</a>
		callback command.<br>
		It returns the hostname of the user that the WHOIS information
		was requested for.<br>
		Please note that this identifier will be empty if <a href="s_whoisSuccess.kvihelp">$whoisSuccess</a>
		returns 0.<br>
	@seealso:
		<a href="awhois.kvihelp">AWHOIS</a>,
		<a href="s_whoisnick.kvihelp">$whoisNick</a>,
		<a href="s_whoissuccess.kvihelp">$whoisSuccess</a>,
		<a href="s_whoisuser.kvihelp">$whoisUser</a>,
		<a href="s_whoisreal.kvihelp">$whoisReal</a>
*/
bool KviUserParser::parseIdn_WHOISHOST(KviCommand *cmd, KviStr &buffer)
{
	buffer.append(m_szLastAsyncWhoisHost);
	return true;
}

/*
	@identifier: WhoisReal
	@short:
		Returns the real name field of the current AWHOIS call target user
	@description:
		This identifier contains valid information only in the <a href="awhois.kvihelp">AWHOIS</a>
		callback command.<br>
		It returns the real name field of the user that the WHOIS information
		was requested for.<br>
		Please note that this identifier will be empty if <a href="s_whoisSuccess.kvihelp">$whoisSuccess</a>
		returns 0.<br>
	@seealso:
		<a href="awhois.kvihelp">AWHOIS</a>,
		<a href="s_whoisnick.kvihelp">$whoisNick</a>,
		<a href="s_whoissuccess.kvihelp">$whoisSuccess</a>,
		<a href="s_whoishost.kvihelp">$whoisHost</a>,
		<a href="s_whoisuser.kvihelp">$whoisUser</a>
*/
bool KviUserParser::parseIdn_WHOISREAL(KviCommand *cmd, KviStr &buffer)
{
	buffer.append(m_szLastAsyncWhoisRealName);
	return true;
}

/*
	@identifier: WhoisServer
	@short:
		Returns the server of the current AWHOIS call target user
	@description:
		This identifier contains valid information only in the <a href="awhois.kvihelp">AWHOIS</a>
		callback command.<br>
		It returns the name of the server that the user that the WHOIS information
		was requested for is using.<br>
		Please note that this identifier will be empty if <a href="s_whoisSuccess.kvihelp">$whoisSuccess</a>
		returns 0.<br>
	@seealso:
		<a href="awhois.kvihelp">AWHOIS</a>,
		<a href="s_whoisnick.kvihelp">$whoisNick</a>,
		<a href="s_whoissuccess.kvihelp">$whoisSuccess</a>,
		<a href="s_whoishost.kvihelp">$whoisHost</a>,
		<a href="s_whoisreal.kvihelp">$whoisReal</a>
*/
bool KviUserParser::parseIdn_WHOISSERVER(KviCommand *cmd, KviStr &buffer)
{
	buffer.append(m_szLastAsyncWhoisServer);
	return true;
}

/*
	@identifier: WhoisIRCop
	@short:
		Returns the status of the current AWHOIS call target user
	@description:
		This identifier contains valid information only in the <a href="awhois.kvihelp">AWHOIS</a>
		callback command.<br>
		Returns 1 if the user that the WHOIS information was requested for is an IRCOperator.<br>
		Please note that this identifier will be empty if <a href="s_whoisSuccess.kvihelp">$whoisSuccess</a>
		returns 0.<br>
	@seealso:
		<a href="awhois.kvihelp">AWHOIS</a>,
		<a href="s_whoisnick.kvihelp">$whoisNick</a>,
		<a href="s_whoissuccess.kvihelp">$whoisSuccess</a>,
		<a href="s_whoishost.kvihelp">$whoisHost</a>,
		<a href="s_whoisreal.kvihelp">$whoisReal</a>
*/
bool KviUserParser::parseIdn_WHOISIRCOP(KviCommand *cmd, KviStr &buffer)
{
	buffer.append(m_szLastAsyncWhoisIrcOp);
	return true;
}

/*
	@identifier: WhoisIdle
	@short:
		Returns the idle time of the current AWHOIS call target user
	@description:
		This identifier contains valid information only in the <a href="awhois.kvihelp">AWHOIS</a>
		callback command.<br>
		Returns the idle time of the user that the WHOIS information
		was requested for.<br>
		Please note that this identifier will be empty if <a href="s_whoisSuccess.kvihelp">$whoisSuccess</a>
		returns 0.<br>
		This information will be sent only by the user's local server.<br>
	@seealso:
		<a href="awhois.kvihelp">AWHOIS</a>,
		<a href="s_whoisnick.kvihelp">$whoisNick</a>,
		<a href="s_whoissuccess.kvihelp">$whoisSuccess</a>,
		<a href="s_whoishost.kvihelp">$whoisHost</a>,
		<a href="s_whoisreal.kvihelp">$whoisReal</a>
*/
bool KviUserParser::parseIdn_WHOISIDLE(KviCommand *cmd, KviStr &buffer)
{
	buffer.append(m_szLastAsyncWhoisIdleTime);
	return true;
}

/*
	@identifier: WhoisStatus
	@short:
		Returns the away message of the current AWHOIS call target user
	@description:
		This identifier contains valid information only in the <a href="awhois.kvihelp">AWHOIS</a>
		callback command.<br>
		Returns the away message of the user that the WHOIS information was requested for (obviously if the user is away).<br>
		Please note that this identifier will be empty if <a href="s_whoisSuccess.kvihelp">$whoisSuccess</a>
		returns 0.<br>
	@seealso:
		<a href="awhois.kvihelp">AWHOIS</a>,
		<a href="s_whoisnick.kvihelp">$whoisNick</a>,
		<a href="s_whoissuccess.kvihelp">$whoisSuccess</a>,
		<a href="s_whoishost.kvihelp">$whoisHost</a>,
		<a href="s_whoisreal.kvihelp">$whoisReal</a>
*/
bool KviUserParser::parseIdn_WHOISSTATUS(KviCommand *cmd, KviStr &buffer)
{
	buffer.append(m_szLastAsyncWhoisStatus);
	return true;
}

/*
	@identifier: WhoisChannels
	@short:
		Returns the channel list of the current AWHOIS call target user
	@description:
		This identifier contains valid information only in the <a href="awhois.kvihelp">AWHOIS</a>
		callback command.<br>
		Returns a comma separated list of channels that the target user is currently on.<br>
		Please note that this identifier will be empty if <a href="s_whoisSuccess.kvihelp">$whoisSuccess</a>
		returns 0.<br>
	@seealso:
		<a href="awhois.kvihelp">AWHOIS</a>,
		<a href="s_whoisnick.kvihelp">$whoisNick</a>,
		<a href="s_whoissuccess.kvihelp">$whoisSuccess</a>,
		<a href="s_whoishost.kvihelp">$whoisHost</a>,
		<a href="s_whoisreal.kvihelp">$whoisReal</a>
*/
bool KviUserParser::parseIdn_WHOISCHANNELS(KviCommand *cmd, KviStr &buffer)
{
	buffer.append(m_szLastAsyncWhoisChannels);
	return true;
}

/*
	@identifier: WhoisMagic
	@short:
		Returns the magic parameter of the current AWHOIS call target user
	@description:
		This identifier contains valid information only in the <a href="awhois.kvihelp">AWHOIS</a>
		callback command.<br>
		Returns the magic parameter[s] passed to the AWHOIS call.<br>
	@seealso:
		<a href="awhois.kvihelp">AWHOIS</a>,
		<a href="s_whoisnick.kvihelp">$whoisNick</a>,
		<a href="s_whoissuccess.kvihelp">$whoisSuccess</a>,
		<a href="s_whoishost.kvihelp">$whoisHost</a>,
		<a href="s_whoisreal.kvihelp">$whoisReal</a>
*/
bool KviUserParser::parseIdn_WHOISMAGIC(KviCommand *cmd, KviStr &buffer)
{
	buffer.append(m_szLastAsyncWhoisMagic);
	return true;
}

/*
	@identifier: ProcPid
	@short:
		Evaluates to the terminating slave process pid.
	@description:
		Evaluates to the terminating slave process pid.<br>
	@seealso:
		<a href="execv.kvihelp">EXECV</a>,
		<a href="s_procstdout.kvihelp">$procstdout</a>,
		<a href="s_proccmdline.kvihelp">$proccmdline</a>,
		<a href="s_procstderr.kvihelp">$procstderr</a>,
		<a href="s_procmagic.kvihelp">$procmagic</a>
		<a href="s_procwindow.kvihelp">$procwindow</a>
		<a href="s_procexitcode.kvihelp">$procexitcode</a>
*/
bool KviUserParser::parseIdn_PROCPID(KviCommand *cmd, KviStr &buffer)
{
	buffer.append(m_szLastProcPid);
	return true;
}

/*
	@identifier: ProcWindow
	@short:
		Evaluates to the terminating slave process window.
	@description:
		Evaluates to the name of the window that the terminating
		process was started in.<br>
		Please note that this window may no longer exist.<br>
	@seealso:
		<a href="execv.kvihelp">EXECV</a>,
		<a href="s_procstdout.kvihelp">$procstdout</a>,
		<a href="s_proccmdline.kvihelp">$proccmdline</a>,
		<a href="s_procstderr.kvihelp">$procstderr</a>,
		<a href="s_procmagic.kvihelp">$procmagic</a>,
		<a href="s_procpid.kvihelp">$procpid</a>,
		<a href="s_procexitcode.kvihelp">$procexitcode</a>
*/
bool KviUserParser::parseIdn_PROCWINDOW(KviCommand *cmd, KviStr &buffer)
{
	buffer.append(m_szLastProcWindow);
	return true;
}

/*
	@identifier: ProcExitCode
	@short:
		Evaluates to the terminating slave process exit code.
	@description:
		Evaluates to the terminating slave process exit code.
	@seealso:
		<a href="execv.kvihelp">EXECV</a>,
		<a href="s_procstdout.kvihelp">$procstdout</a>,
		<a href="s_proccmdline.kvihelp">$proccmdline</a>,
		<a href="s_procstderr.kvihelp">$procstderr</a>,
		<a href="s_procmagic.kvihelp">$procmagic</a>,
		<a href="s_procpid.kvihelp">$procpid</a>,
		<a href="s_procwindow.kvihelp">$procwindow</a>
*/
bool KviUserParser::parseIdn_PROCEXITCODE(KviCommand *cmd, KviStr &buffer)
{
	buffer.append(m_szLastProcExitCode);
	return true;
}

/*
	@identifier: IsConnected
	@short:
		Check if we are connected to a server
	@description:
		Returns 1 if we are connected to a server, 0 otherwise.
	@seealso:
		<a href="s_conntimes.kvihelp">$conntimes</a>, <a href="s_conntime.kvihelp">$conntime</a>
*/
bool KviUserParser::parseIdn_ISCONNECTED(KviCommand *cmd, KviStr &buffer)
{
	KviStr tmp;
	buffer.append(m_pFrm->isConnected() ? '1' : '0');
	return true;
}

/*
	@identifier: ConnTime
	@short:
		Returns the duration of the current irc connection
	@description:
		Returns a string containing the duraition of the current IRC connection.
	@seealso:
		<a href="s_conntimes.kvihelp">$conntimes</a>, <a href="s_uptime.kvihelp">$uptime</a>,
		<a href="s_isconnected.kvihelp">$IsConnected</a>
*/
bool KviUserParser::parseIdn_CONNTIME(KviCommand *cmd, KviStr &buffer)
{
	KviStr tmp;
	m_pFrm->m_pSysTrayOnLineTimer->getConnectionTime(tmp);
	buffer.append(tmp);
	return true;
}

/*
	@identifier: ConnTimeS
	@short:
		Returns the duration of the current irc connection
	@description:
		Returns the duration (in seconds) of the current IRC connection.
	@seealso:
		<a href="s_conntime.kvihelp">$conntime</a>, <a href="s_uptime.kvihelp">$uptime</a>,
		<a href="s_isconnected.kvihelp">$IsConnected</a>
*/
bool KviUserParser::parseIdn_CONNTIMES(KviCommand *cmd, KviStr &buffer)
{
	KviStr tmp;
	tmp.setNum(m_pFrm->m_pSysTrayOnLineTimer->getIntConnectionTime());
	buffer.append(tmp);
	return true;
}

/*
	@identifier: ActiveWindow
	@short:
		Returns the active window
	@description:
		Returns a string containing the name of the window that is currently activated.
	@seealso:
		<a href="s_window.kvihelp">$window</a>
*/
bool KviUserParser::parseIdn_ACTIVEWINDOW(KviCommand *cmd, KviStr &buffer)
{
	buffer.append((m_pFrm->activeWindow())->caption());
	return true;
}

/*
	@identifier: DialogResult
	@short:
		Returns the result of the currently terminated dialog
	@description:
		This identifier returns valid data only in commands called
		by an async dialog routine.<br>
		It evaluates to the result of the <a href="dialog.kvihelp">DIALOG</a> call.<br>
	@seealso:
		<a href="dialog.kvihelp">DIALOG</a>
		<a href="s_dialogmagic.kvihelp">$dialogMagic</a>
*/
bool KviUserParser::parseIdn_DIALOGRESULT(KviCommand *cmd, KviStr &buffer)
{
	buffer.append(m_szLastDialogResult);
	return true;
}

/*
	@identifier: DialogMagic
	@short:
		Returns the magic data of the currently terminated dialog
	@description:
		This identifier returns valid data only in commands called
		by an async dialog routine.<br>
		It evaluates to the magic data passed to the <a href="dialog.kvihelp">DIALOG</a> call.<br>
	@seealso:
		<a href="dialog.kvihelp">DIALOG</a>
		<a href="s_dialogresult.kvihelp">$dialogResult</a>
*/
bool KviUserParser::parseIdn_DIALOGMAGIC(KviCommand *cmd, KviStr &buffer)
{
	buffer.append(m_szLastDialogMagic);
	return true;
}

/*
	@identifier: TimerMagic
	@short:
		Returns the magic data of the currently "fired" timer.
	@description:
		This identifier returns valid data only in commands called
		by a timer.<br>
		It evaluates to the magic data passed to the <a href="timer.kvihelp">TIMER</a> call.<br>
	@seealso:
		<a href="timer.kvihelp">TIMER</a>
		<a href="s_timername.kvihelp">$timername</a>
*/
bool KviUserParser::parseIdn_TIMERMAGIC(KviCommand *cmd, KviStr &buffer)
{
	buffer.append(m_szLastTimerMagic);
	return true;
}

/*
	@identifier: TimerName
	@short:
		Returns the name of the currently "fired" timer.
	@description:
		This identifier returns valid data only in commands called
		by a timer.<br>
		It evaluates to the name passed to the <a href="timer.kvihelp">TIMER</a> call.<br>
	@seealso:
		<a href="timer.kvihelp">TIMER</a>
		<a href="s_timermagic.kvihelp">$timerMagic</a>
*/
bool KviUserParser::parseIdn_TIMERNAME(KviCommand *cmd, KviStr &buffer)
{
	buffer.append(m_szLastTimerName);
	return true;
}

/*
	@identifier: TryResult
	@short:
		Returns the result of the last TRY command
	@description:
		Evaluates to 1 if the last execution of a <a href="try.kvihelp">TRY</a>
		command has been successful, 0 otherwise.<br>
	@seealso:
		<a href="try.kvihelp">TRY</a>
*/
bool KviUserParser::parseIdn_TRYRESULT(KviCommand *cmd, KviStr &buffer)
{
	buffer.append(m_szLastTryResult);
	return true;
}

/*
	@identifier: IsRemoteExec
	@short:
		Mimetypes scripts: returns 1 if it is a remote execution.
	@description:
		This identifier is meaningful only in Mimetype execution scripts.<br>
		If the execution of a "file" has been triggered by a remote user
		via <a href="doc_ctcp_multimedia.kvihelp">CTCP MULTIMEDIA</a> this identifier
		will contain 1, otherwise 0.<br>
		(Please note that if the "remote exec safe" flag is set to false the
		command will be never executed!).<br>
	@seealso:
		<a href="doc_mimetypes.kvihelp">mime types documentation</a>, <br>
		<a href="s_execfilename.kvihelp">$ExecFileName</a>
*/
bool KviUserParser::parseIdn_ISREMOTEEXEC(KviCommand *cmd, KviStr &buffer)
{
	buffer.append(m_bLastIsRemoteExec ? '1' : '0');
	return true;
}

/*
	@identifier: ExecFileName
	@short:
		Mimetypes scripts: returns the filename to "run"
	@description:
		This identifier is meaningful only in Mimetype execution scripts.<br>
		Returns the filename (usually with the complete path) of the file
		being "executed".<br>
	@seealso:
		<a href="doc_mimetypes.kvihelp">mime types documentation</a>, <br>
		<a href="s_isremoteexec.kvihelp">$IsRemoteExec</a>
*/
bool KviUserParser::parseIdn_EXECFILENAME(KviCommand *cmd, KviStr &buffer)
{
	buffer.append(m_szLastExecFilename);
	return true;
}

/*
	@identifier: This
	@short:
		Returns the id of the current object
	@description:
		This identifier contains valid information only in the event triggered by an object.<br>
		It returns the id of the object that has triggered the event.<br>
	@seealso:
		<a href="syntax_objects.kvihelp">Objects documentation</a>
*/
bool KviUserParser::parseIdn_THIS(KviCommand *cmd, KviStr &buffer)
{
	buffer.append(cmd->thisId());
	return true;
}

/*
	@identifier: GetReturn
	@short:
		Returns the return value for the current command sequence
	@description:
		Returns the return value for the current command sequence as set
		by setreturn.<br>
	@seealso:
		<a href="setreturn.kvihelp">SETRETURN</a>
*/
bool KviUserParser::parseIdn_GETRETURN(KviCommand *cmd, KviStr &buffer)
{
	buffer.append(cmd->m_retBuffer);
	return true;
}

/*
	@identifier: Root
	@short:
		Returns the id of the current root object
	@description:
		This identifier returns the id of the toplevel object bound to the
		server window that this command is executed in.<br>
		All the objects in this server window are children of this object
	@seealso:
		<a href="syntax_objects.kvihelp">Objects documentation</a>
*/
bool KviUserParser::parseIdn_ROOT(KviCommand *cmd, KviStr &buffer)
{
	buffer.append(m_pScriptObjectController->topLevelObject()->id());
	return true;
}

/*
	@identifier: SignalSender
	@short:
		Returns the id of the current signal sender object
	@description:
		This identifier returns the id of the last signal sender object.<br>
		Warning : this identifier is like a global variable:
		it is set when a signal is triggered by an object
		and remain set until another signal has to be triggered: after that
		it gets overwritten.<br>
		If you interested in the signal sender object identity
		you should copy its value somewhere at the beginning of the connected slot function.<br>
	@seealso:
		<a href="syntax_objects.kvihelp">Objects documentation</a>
*/
bool KviUserParser::parseIdn_SIGNALSENDER(KviCommand *cmd, KviStr &buffer)
{
	buffer.append(m_szLastSignalSender);
	return true;
}
