/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Arun Rodrigues, and Brian Barrett with
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted subject to the conditions specified in the
 * LICENSE file.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1996 The Ohio State University
 *	RBD/JRV
 *
 *	$Id: scatterv.c,v 6.4 1999/12/31 21:27:37 jsquyres Exp $
 *
 *	Function:	- scatter varying len. buffers from root in rank order
 *	Accepts:	- send buffer
 *			- send counts
 *			- displacements
 *			- send datatype
 *			- recv buffer
 *			- recv count
 *			- recv datatype
 *			- root
 *			- communicator
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <lam_config.h>
#include <app_mgmt.h>
#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>
#include <rpisys.h>

/*@

MPI_Scatterv - Scatters a buffer in parts to all tasks in a group

Input Parameters:
+ sbuf - address of send buffer (choice, significant only at 'root') 
. scounts - integer array (of length group size) 
specifying the number of elements to send to each processor  
. disps - integer array (of length group size). Entry 
 'i'  specifies the displacement (relative to sendbuf  from
which to take the outgoing data to process  'i' 
. sdtype - data type of send buffer elements (handle) 
. rcount - number of elements in receive buffer (integer) 
. rtype - data type of receive buffer elements (handle) 
. root - rank of sending process (integer) 
- comm - communicator (handle) 

Output Parameter:
. rbuf - address of receive buffer (choice) 

.N fortran

.N Errors
.N MPI_SUCCESS
.N MPI_ERR_COMM
.N MPI_ERR_INTERCOMM
.N MPI_ERR_IMPI
.N MPI_ERR_COUNT
.N MPI_ERR_TYPE
.N MPI_ERR_BUFFER
.N MPI_ERR_ROOT
.N MPI_ERR_ARG

.N ACK
@*/
int MPI_Scatterv(void *sbuf, int *scounts, int *disps, 
		 MPI_Datatype sdtype, void* rbuf, int rcount, 
		 MPI_Datatype rdtype, int root, MPI_Comm comm)
{
	int		i;			/* favourite index */
	int		rank;			/* my rank */
	int		size;			/* group size */
	int		err;			/* error code */
	char		*ptmp;			/* temp. buffer */
	MPI_Aint	extent;			/* datatype extent */
	MPI_Status	status;			/* recv status */
	struct _gps	*p;			/* favourite pointer */

	lam_initerr();
	lam_setfunc(BLKMPISCATTERV);
/*
 * Check for invalid arguments.
 */
	if ((comm == MPI_COMM_NULL) || LAM_IS_INTER(comm)) {
		return(lam_errfunc(comm, BLKMPISCATTERV,
				lam_mkerr(MPI_ERR_COMM, 0)));
	}

	MPI_Comm_rank(comm, &rank);
	MPI_Comm_size(comm, &size);

	if ((root >= size) || (root < 0)) {
		return(lam_errfunc(comm, BLKMPISCATTERV,
				lam_mkerr(MPI_ERR_ROOT, 0)));
	}

	if ((rank == root) && (disps == 0)) {
		return(lam_errfunc(comm, BLKMPISCATTERV,
				lam_mkerr(MPI_ERR_ARG, 0)));
	}

	if ((root == rank && sdtype == MPI_DATATYPE_NULL)
			|| (rdtype == MPI_DATATYPE_NULL)) {
		return(lam_errfunc(comm, BLKMPISCATTERV,
				lam_mkerr(MPI_ERR_TYPE, 0)));
	}

	if ((rcount < 0) || (rank == root && scounts == 0)) {
		return(lam_errfunc(comm, BLKMPISCATTERV,
				lam_mkerr(MPI_ERR_COUNT, 0)));
	}

#if LAM_WANT_IMPI
	/* Remove this when IMPI collectives are implemented */

        if (LAM_IS_IMPI(comm)) {
	  return lam_err_comm(comm, MPI_ERR_COMM, 0, 
			      "Collectives not yet implemented on IMPI communicators");
	}
#endif

	LAM_TRACE(lam_tr_cffstart(BLKMPISCATTERV));
/*
 * Remember required parameters.
 */
	p = &(comm->c_group->g_procs[root]->p_gps);

	lam_setparam(BLKMPISCATTERV, root | (p->gps_grank << 16),
				(p->gps_node << 16) | p->gps_idx);
/*
 * Switch to collective communicator.
 */
	lam_mkcoll(comm);
/*
 * If not root, receive data.
 */
	if (rank != root) {
		err = MPI_Recv(rbuf, rcount, rdtype,
					root, BLKMPISCATTERV, comm, &status);
		lam_mkpt(comm);

		if (err != MPI_SUCCESS) {
			return(lam_errfunc(comm, BLKMPISCATTERV, err));
		}

		LAM_TRACE(lam_tr_cffend(BLKMPISCATTERV,
						root, comm, rdtype, rcount));
		
		lam_resetfunc(BLKMPISCATTERV);
		return(MPI_SUCCESS);
	}
/*
 * I am the root, loop sending data.
 */
	MPI_Type_extent(sdtype, &extent);

	for (i = 0; i < size; ++i) {

		ptmp = ((char *) sbuf) + (extent * disps[i]);
/*
 * simple optimization
 */
		if (i == rank) {
			err = lam_dtsndrcv(ptmp, scounts[i], sdtype, rbuf,
					rcount, rdtype, BLKMPISCATTERV,	comm);
		} else {
			err = MPI_Send(ptmp, scounts[i], sdtype,
					i, BLKMPISCATTERV, comm);
		}

		if (err != MPI_SUCCESS) {
			lam_mkpt(comm);
			return(lam_errfunc(comm, BLKMPISCATTERV, err));
		}
	}

	lam_mkpt(comm);

	LAM_TRACE(lam_tr_cffend(BLKMPISCATTERV, root, comm, rdtype, rcount));
	
	lam_resetfunc(BLKMPISCATTERV);
	return(MPI_SUCCESS);
}
