/*
    ldapdiff
    Copyright (C) 2000-2002 Thomas.Reith@rhoen.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <lber.h>
#include <ldap.h>

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif

#include "ldapdiff.h"

/* global variables */
char *profile  = NULL;
char *logfile  = NULL;
int   facility = -1;
int   loglevel = 0;

int main(int argc,char* argv[])
{
 extern char     *optarg;
 extern int       optind;
 int              option;
 int              ldapoption;
 int              rc;
 int              errflag     = 0;
 char            *cfile       = NULL;
 char            *ifile       = NULL;
 LDAP            *ld;
 struct s_ldif   *sldif       = NULL;
 struct s_ldif   *sldifstart  = NULL;
 struct s_mod    *smod        = NULL;
 struct s_schema *sschema     = NULL;

 while ((option = getopt(argc, argv, "c:d:f:l:p:s:v")) != EOF){
  switch(option){
   case 'c':
        cfile    = optarg;
        break;
   case 'd':
        loglevel = atoi(optarg);
        break;
   case 'f':
        ifile    = optarg;
        break;
   case 'l':
        logfile  = optarg;
        break;
   case 'p':
        profile  = optarg;
        break;
   case 's':
        if((facility = ldifgetfacility(optarg)) == -1){
         errflag++;
        }
        break;
   case 'v':
        printf("%s\n",VERSION);
        printf("%s\n",COPYING);
        exit(0);
        break;
   default:
        errflag++;
  }
 }

 if(errflag != 0 || profile == NULL){
  printf("usage: ldapdiff [options] -p profile\n");
  printf("                options: -c configfile\n");
  printf("                options: -d loglevel 0|1|2\n");
  printf("                options: -f ldiffile\n");
  printf("                options: -l logfile\n");
  printf("                options: -s local[0-7] (syslog facility)\n");
  printf("       ldapdiff -v (print version and exit)\n");
  exit(-1);
 }

 /* log version information, always */
 ldiflog(LOG1,"%s",VERSION);
 ldiflog(LOG1,"%s",COPYING);
 ldiflog(LOG1,"processing profile: [%s]",profile);

 if(cfile == NULL){
  cfile = LDAPCONF;
 }

 ldifinitconf(cfile); 

 if((ld = ldap_open(ldifgetgconf(CONFLDAPHOST),atoi(ldifgetgconf(CONFLDAPPORT)))) == NULL){
  fprintf(stderr,"ldap_open() failed: file: %s, line: %d\n",__FILE__,__LINE__);
  perror("perror(): ");
  exit(-1);
 }

 ldapoption = LDAP_NO_LIMIT;
 if(ldap_set_option(ld,LDAP_OPT_SIZELIMIT,&ldapoption) == -1){
  fprintf(stderr,"ldap_set_option() failed: file: %s, line: %d\n",__FILE__,__LINE__);
  exit(-1);
 }

 ldapoption = LDAP_VERSION3;
 if(ldap_set_option(ld,LDAP_OPT_PROTOCOL_VERSION,&ldapoption) == -1){
  fprintf(stderr,"ldap_set_option() failed: file: %s, line: %d\n",__FILE__,__LINE__);
  exit(-1);
 }

 if((rc = ldap_bind_s(ld,ldifgetgconf(CONFROOTDN),ldifgetgconf(CONFROOTPW),LDAP_AUTH_SIMPLE)) != LDAP_SUCCESS){
  fprintf(stderr,"ldap_bind_s() failed: file: %s, line: %d\n",__FILE__,__LINE__);
  ldap_get_option(ld,LDAP_OPT_ERROR_NUMBER,&rc); 
  fprintf(stderr,"ldap_err2string(): %s",ldap_err2string(rc));
  exit(-1);
 }

 /* read ldif input */
 ldifread(&sldif,ifile);
 if(sldif == NULL){
  ldiflog(LOG1,"no ldif entries found, normal exit...");
  exit(0);
 }
 sldifstart = sldif; 

 /* build the schema cache */
 ldifschemacreate(ld,&sschema);

 /* replace all aliases with the real attribute names */
 ldifschemareplacealiases(sldif,sschema);

 /* compare ldif file against ldap server */
 ldifiterldif(ld,sldifstart,&smod,sschema);

 /* compare ldap server against ldif file */
 ldifiterldap(ld,sldifstart,&smod,sschema);

 ldiffree(&sldifstart);

 ldifstats(smod);

 if(strcmp(ldifgetgconf(CONFONLINEUPDATE),"yes") == 0){
  ldifmodify(ld,smod,sschema);
 }

 if(strcmp(ldifgetgconf(CONFOFFLINEUPDATE),"yes") == 0){
  ldifwrite(smod,sschema);
 }

 if(strcmp(ldifgetgconf(CONFPLUGIN),"yes") == 0){
  ldifdl(smod);
 }

 /* free the schema cache */
 ldifschemafree(&sschema);

 if(ldap_unbind(ld) == -1){
  fprintf(stderr,"ldap_unbind() failed: file: %s, line: %d\n",__FILE__,__LINE__);
  ldap_get_option(ld,LDAP_OPT_ERROR_NUMBER,&rc); 
  fprintf(stderr,"ldap_err2string(): %s",ldap_err2string(rc));
  exit(-1);
 }

 return 0;
}
