/* lightlab, Copyright (c) 2002 Jamie Zawinski <jwz@jwz.org>
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.  No representations are made about the suitability of this
 * software for any purpose.  It is provided "as is" without express or 
 * implied warranty.
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <stdlib.h>
#include <stdio.h>

#include <gtk/gtk.h>
#include <gdk/gdkx.h>

#include <gdk-pixbuf/gdk-pixbuf.h>
#include <gdk-pixbuf/gdk-pixbuf-xlib.h>

#include "lightlab.h"
#include "version.h"
#include "scene.h"
#include "interface.h"
#include "callbacks.h"
#include "support.h"
#include "leopard.xpm"

#undef countof
#define countof(x) (sizeof((x))/sizeof(*(x)))


#define ABOUT_FONT_1 "-*-helvetica-bold-r-*-*-*-180-*-*-*-*-iso8859-1"
#define ABOUT_FONT_2 "-*-helvetica-bold-r-*-*-*-120-*-*-*-*-iso8859-1"


static GtkWidget *toplevel;
const char *progname;

void
get_light_info (int which, light_info *info_return)
{
  GtkWidget *check, *x, *y, *z, *w, *color;
  char buf[255];
  gdouble rgba[4];

  if (which < 0 || which > 2) abort();

  which++;

# define LOOKUP(W,N) \
    sprintf (buf, N "%d", which); \
    W = GTK_WIDGET (lookup_widget (toplevel, buf))

  LOOKUP(check, "light_check");
  LOOKUP(x,     "x_hscale");
  LOOKUP(y,     "y_hscale");
  LOOKUP(z,     "z_hscale");
  LOOKUP(w,     "w_hscale");

  info_return->enabled_p =
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (check));

  info_return->x = gtk_range_get_adjustment (GTK_RANGE (x))->value;
  info_return->y = gtk_range_get_adjustment (GTK_RANGE (y))->value;
  info_return->z = gtk_range_get_adjustment (GTK_RANGE (z))->value;
  info_return->w = gtk_range_get_adjustment (GTK_RANGE (w))->value;
  
  LOOKUP(color, "colorselection_ambient");
  gtk_color_selection_get_color (GTK_COLOR_SELECTION (color), rgba);
  info_return->ambient.r = rgba[0];
  info_return->ambient.g = rgba[1];
  info_return->ambient.b = rgba[2];

  LOOKUP(color, "colorselection_diffuse");
  gtk_color_selection_get_color (GTK_COLOR_SELECTION (color), rgba);
  info_return->diffuse.r = rgba[0];
  info_return->diffuse.g = rgba[1];
  info_return->diffuse.b = rgba[2];

  LOOKUP(color, "colorselection_specular");
  gtk_color_selection_get_color (GTK_COLOR_SELECTION (color), rgba);
  info_return->specular.r = rgba[0];
  info_return->specular.g = rgba[1];
  info_return->specular.b = rgba[2];

# undef LOOKUP
}


void
get_object_info (int which, object_info *info_return)
{
  GtkWidget *render, *texture, *type, *shininess, *color;
  char buf[255];
  gdouble rgba[4];

  if (which != 0) abort();

  which += 4;

# define LOOKUP(W,N) \
    sprintf (buf, N "%d", which); \
    W = GTK_WIDGET (lookup_widget (toplevel, buf))

  LOOKUP(render,    "object_render_menu");
  LOOKUP(texture,   "object_texture_menu");
  LOOKUP(type,      "object_type_menu");
  LOOKUP(shininess, "shininess_hscale");

# define MENU_NUM(OPT,STORE) do { \
    GtkMenu *m = GTK_MENU(gtk_option_menu_get_menu(GTK_OPTION_MENU(OPT))); \
    GtkWidget *selected = gtk_menu_get_active (m); \
    GList *kids = gtk_container_children (GTK_CONTAINER (m)); \
    int menu_elt = g_list_index (kids, (gpointer) selected); \
    (STORE) = menu_elt; \
  } while(0)
  
  MENU_NUM (render,  info_return->render);
  MENU_NUM (texture, info_return->solid_p);
  MENU_NUM (type,    info_return->type);
# undef MENU_NUM

  info_return->shininess =
    gtk_range_get_adjustment (GTK_RANGE (shininess))->value;

  LOOKUP(color, "colorselection_ambient");
  gtk_color_selection_get_color (GTK_COLOR_SELECTION (color), rgba);
  info_return->ambient.r = rgba[0];
  info_return->ambient.g = rgba[1];
  info_return->ambient.b = rgba[2];

  LOOKUP(color, "colorselection_diffuse");
  gtk_color_selection_get_color (GTK_COLOR_SELECTION (color), rgba);
  info_return->diffuse.r = rgba[0];
  info_return->diffuse.g = rgba[1];
  info_return->diffuse.b = rgba[2];

  LOOKUP(color, "colorselection_specular");
  gtk_color_selection_get_color (GTK_COLOR_SELECTION (color), rgba);
  info_return->specular.r = rgba[0];
  info_return->specular.g = rgba[1];
  info_return->specular.b = rgba[2];

# undef LOOKUP
}



static void
set_widget_defaults (int which)
{
  /* The OpenGL defaults are:

     Objects:
        GL_AMBIENT:         [0.2, 0.2, 0.2, 1.0]
        GL_DIFFUSE:         [0.8, 0.8, 0.8, 1.0]
        GL_SPECULAR:        [0.0, 0.0, 0.0, 1.0]
        GL_EMISSION:        [0.0, 0.0, 0.0, 1.0]
        GL_SHININESS:       0.0

     Lights:
        GL_AMBIENT:         [0.0, 0.0, 0.0, 1.0]
        GL_DIFFUSE 0:       [1.0, 1.0, 1.0, 1.0]
        GL_DIFFUSE n:       [0.0, 0.0, 0.0, 1.0]
        GL_SPECULAR 0:      [1.0, 1.0, 1.0, 1.0]
        GL_SPECULAR n:      [0.0, 0.0, 0.0, 1.0]
        GL_POSITION:        [0.0, 0.0  1.0, 0.0]

        GL_SPOT_DIRECTION:  [0.0, 0.0, -1.0]
        GL_SPOT_EXPONENT:   0.0
        GL_SPOT_CUTOFF:     180 (from set [0-90,180])

        GL_CONSTANT_ATTENUATION:   [1.0, 0.0, 0.0]
        GL_LINEAR_ATTENUATION:     [1.0, 0.0, 0.0]
        GL_QUADRATIC_ATTENUATION:  [1.0, 0.0, 0.0]
   */

  static struct { const char *name; gdouble rgba[4]; } csettings[] = {

    { "colorselection_ambient1",  { 0.0, 0.0, 0.0, 1.0 }},
    { "colorselection_ambient2",  { 0.0, 0.0, 0.0, 1.0 }},
    { "colorselection_ambient3",  { 0.0, 0.0, 0.0, 1.0 }},

    { "colorselection_diffuse1",  { 1.0, 1.0, 1.0, 1.0 }},
    { "colorselection_diffuse2",  { 0.0, 0.0, 0.0, 1.0 }},
    { "colorselection_diffuse3",  { 0.0, 0.0, 0.0, 1.0 }},

    { "colorselection_specular1", { 1.0, 1.0, 1.0, 1.0 }},
    { "colorselection_specular2", { 0.0, 0.0, 0.0, 1.0 }},
    { "colorselection_specular3", { 0.0, 0.0, 0.0, 1.0 }},

    { "colorselection_ambient4",  { 0.2, 0.2, 0.2, 1.0 }},
    { "colorselection_diffuse4",  { 0.8, 0.8, 0.8, 1.0 }},
    { "colorselection_specular4", { 0.0, 0.0, 0.0, 1.0 }},
 /* { "colorselection_emission4", { 0.0, 0.0, 0.0, 1.0 }},  */
  };

  static struct { const char *name; double v; } fsettings[] = {

    { "x_hscale1", 0.0 },
    { "y_hscale1", 0.0 },
    { "z_hscale1", 1.0 },
    { "w_hscale1", 0.0 },

    { "x_hscale2", 0.0 },
    { "y_hscale2", 0.0 },
    { "z_hscale2", 1.0 },
    { "w_hscale2", 0.0 },

    { "x_hscale3", 0.0 },
    { "y_hscale3", 0.0 },
    { "z_hscale3", 1.0 },
    { "w_hscale3", 0.0 },

    { "shininess_hscale4", 0.0 },

 /* { "spot_x_hscale1",    0.0 },
    { "spot_y_hscale1",    0.0 },
    { "spot_z_hscale1",   -1.0 },
    { "spot_e_hscale1",    0.0 },
    { "spot_cut_ hscale1", 180 },

    { "spot_x_hscale2",    0.0 },
    { "spot_y_hscale2",    0.0 },
    { "spot_z_hscale2",   -1.0 },
    { "spot_e_hscale2",    0.0 },
    { "spot_cut_ hscale3", 180 },

    { "spot_x_hscale3",    0.0 },
    { "spot_y_hscale3",    0.0 },
    { "spot_z_hscale3",   -1.0 },
    { "spot_e_hscale3",    0.0 },
    { "spot_cut_hscale3",  180 }, */
  };

  int i;

  for (i = 0; i < countof(csettings); i++)
    {
      const char *name = csettings[i].name;
      GtkWidget *w;

      if (which != -1 &&
          name[strlen(name)-1] != (which + 1 + '0'))
        continue;

      w = GTK_WIDGET (lookup_widget (toplevel, name));
      gtk_color_selection_set_color (GTK_COLOR_SELECTION (w),
                                     csettings[i].rgba);
    }

  for (i = 0; i < countof(fsettings); i++)
    {
      const char *name = fsettings[i].name;
      GtkWidget *w;

      if (which != -1 &&
          name[strlen(name)-1] != (which + 1 + '0'))
        continue;

      w = GTK_WIDGET (lookup_widget (toplevel, name));
      gtk_adjustment_set_value (gtk_range_get_adjustment GTK_RANGE (w),
                                fsettings[i].v);
    }
}


static void
enable_widgets (int which, gboolean sensitive_p)
{
  const char *names[] = { 
    "x_hscale1", "x_hscale2", "x_hscale3",
    "y_hscale1", "y_hscale2", "y_hscale3",
    "z_hscale1", "z_hscale2", "z_hscale3",
    "w_hscale1", "w_hscale2", "w_hscale3",
    "type_notebook1", "type_notebook2", "type_notebook3",
    "reset1", "reset2", "reset3",
 /*  "shininess_hscale4", */
 /* "spot_x_hscale1", "spot_x_hscale2", "spot_x_hscale3",
    "spot_y_hscale1", "spot_y_hscale2", "spot_y_hscale3",
    "spot_z_hscale1", "spot_z_hscale2", "spot_z_hscale3",
    "spot_e_hscale1", "spot_e_hscale2", "spot_e_hscale3",
    "spot_cut_ hscale1", "spot_cut_ hscale2", "spot_cut_ hscale3", */
  };

  int i;
  for (i = 0; i < countof(names); i++)
    {
      GtkWidget *w;
      const char *name = names[i];
      if (which != -1 &&
          name[strlen(name)-1] != (which + 1 + '0'))
        continue;

      w = GTK_WIDGET (lookup_widget (toplevel, name));
      gtk_widget_set_sensitive (w, sensitive_p);
    }
}



static GdkVisual *
x_visual_to_gdk_visual (Visual *xv)
{
  GList *gvs = gdk_list_visuals();
  if (!xv) return gdk_visual_get_system();
  for (; gvs; gvs = gvs->next)
    {
      GdkVisual *gv = (GdkVisual *) gvs->data;
      if (xv == GDK_VISUAL_XVISUAL (gv))
        return gv;
    }
  fprintf (stderr, "%s: couldn't convert X Visual 0x%lx to a GdkVisual\n",
           progname, (unsigned long) xv->visualid);
  abort();
}


static scene_data *
init_scene_widget (void)
{
  Display *dpy = GDK_DISPLAY();
  int screen_num = DefaultScreen (dpy);
  GtkWidget *widget = GTK_WIDGET (lookup_widget (toplevel, "gl_area"));
  Visual *xvisual = get_gl_visual (dpy, screen_num);
  GdkVisual *visual = x_visual_to_gdk_visual (xvisual);
  GdkVisual *dvisual = gdk_visual_get_system();
  GdkColormap *cmap = (visual == dvisual
                       ? gdk_colormap_get_system ()
                       : gdk_colormap_new (visual, False));

  if (!GTK_WIDGET_REALIZED (widget) ||
      gtk_widget_get_visual (widget) != visual)
    {
      gtk_widget_unrealize (widget);
      gtk_widget_set_visual (widget, visual);
      gtk_widget_set_colormap (widget, cmap);
      gtk_widget_realize (widget);
    }

  gtk_widget_show (widget);
  gtk_widget_realize (widget);

  return init_scene (dpy, GDK_WINDOW_XWINDOW (widget->window));
}


static void
load_texture (scene_data *sd)
{
  Display *dpy = GDK_DISPLAY();
  GdkPixbuf *pb;

  gdk_pixbuf_xlib_init (dpy, DefaultScreen (dpy));
  xlib_rgb_init (dpy, DefaultScreenOfDisplay (dpy));

  pb = gdk_pixbuf_new_from_xpm_data ((const char **) leopard_xpm);
  if (pb)
    {
      int w = gdk_pixbuf_get_width (pb);
      int h = gdk_pixbuf_get_height (pb);
      guchar *data = gdk_pixbuf_get_pixels (pb);
      init_texture (sd, w, h, data);
    }
}


static int fps = 0;
static gboolean update_allowed_p = TRUE;

static int
display_tick_timer (gpointer data)
{
  update_allowed_p = TRUE;
  return TRUE;  /* re-execute timer */
}


static int
display_timer (gpointer data)
{
  static int last_w = 0;
  static int last_h = 0;
  scene_data *sd = (scene_data *) data;
  GtkWidget *widget;

  if (!update_allowed_p)
    goto END;
  update_allowed_p = FALSE;

  widget = GTK_WIDGET (lookup_widget (toplevel, "gl_area"));

  if (widget->allocation.width != last_w ||
      widget->allocation.height != last_h)
    {
      reshape_scene (sd);
      last_w = widget->allocation.width;
      last_h = widget->allocation.height;
    }

  update_scene (sd);
  fps++;
  XSync (GDK_DISPLAY(), False);

 END:
  return TRUE;  /* re-execute timer */
}


#if 0
static int
fps_timer (gpointer data)
{
  fprintf (stderr, "%d\n", fps);
  fps = 0;
  return TRUE;  /* re-execute timer */
}
#endif


void
about_dialog (int up)
{
  static GtkWidget *about = 0;
  if (up)
    {
      if (!about)
        {
          GtkWidget *L;
          char *s1 = 0;
          char *s2 = 0;
          about = create_about_dialog ();
          gtk_signal_connect (GTK_OBJECT (about), "delete_event",
                              GTK_SIGNAL_FUNC (on_about_ok_clicked),
                              0);

          L = lookup_widget (about, "about1");

          gtk_label_get (GTK_LABEL (L), &s1);
          s2 = (char *) malloc (strlen (s1) + 20);
          sprintf (s2, "%s %s", s1, version);
          gtk_label_set_text (GTK_LABEL (L), s2);
          free (s2);

          L->style = gtk_style_copy (L->style);
          L->style->font = gdk_font_load (ABOUT_FONT_1);
          gtk_widget_set_style (L, L->style);

          L = lookup_widget (about, "about2");
          L->style = gtk_style_copy (L->style);
          L->style->font = gdk_font_load (ABOUT_FONT_2);
          gtk_widget_set_style (L, L->style);

          gtk_widget_realize (about);
          gtk_window_set_transient_for (GTK_WINDOW (about),
                                        GTK_WINDOW (toplevel));
        }
      gtk_widget_show (about);
    }
  else if (about)
    {
      gtk_widget_hide (about);
    }
}

void
enable_light (int which, int enabled_p)
{
  enable_widgets (which, enabled_p);
}

void reset_light (int which)
{
  if (which < 0 || which > 4) abort();
  set_widget_defaults (which);
}


static void
wm_toplevel_close_cb (GtkWidget *widget, GdkEvent *event, gpointer data)
{
  gtk_main_quit ();
}

int
main (int argc, char **argv)
{
  scene_data *sd;
  char *s;

  progname = argv[0];
  s = strrchr (progname, '/');
  if (s) progname = s+1;

  gtk_set_locale ();
  gtk_init (&argc, &argv);

  toplevel = create_lightlab ();
  set_widget_defaults (-1);
  enable_widgets (0, TRUE);
  enable_widgets (1, FALSE);
  enable_widgets (2, FALSE);

  {
    GtkNotebook *n;
    n = GTK_NOTEBOOK (lookup_widget (toplevel, "notebook1"));
    gtk_notebook_set_page (n, 3);
    n = GTK_NOTEBOOK (lookup_widget (toplevel, "type_notebook1"));
    gtk_notebook_set_page (n, 1);
    n = GTK_NOTEBOOK (lookup_widget (toplevel, "type_notebook2"));
    gtk_notebook_set_page (n, 1);
    n = GTK_NOTEBOOK (lookup_widget (toplevel, "type_notebook3"));
    gtk_notebook_set_page (n, 1);
    n = GTK_NOTEBOOK (lookup_widget (toplevel, "type_notebook4"));
    gtk_notebook_set_page (n, 1);
  }

  gtk_signal_connect (GTK_OBJECT (toplevel), "delete_event",
                      GTK_SIGNAL_FUNC (wm_toplevel_close_cb),
                      0);

  sd = init_scene_widget ();
  load_texture (sd);
  gtk_widget_show (toplevel);

  gtk_idle_add (display_timer, (gpointer) sd);
  gtk_timeout_add (33, display_tick_timer, (gpointer) sd);    /* 30x/sec */

/*  gtk_timeout_add (1000, fps_timer, (gpointer) sd); */

  gtk_main ();
  return 0;
}
