#include <arpa/inet.h>
#include <syslog.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/types.h>
#include <string.h>
#include "times.h"
#include "global_vars.h"
#include "cmd.h"
#include "cmd_lcp3.h"
#include "srv_def.h"
#include "execute.h"
#include "con.h"

/*-------------*/
int clean_clients(struct client_t*);
int kick_client(struct client_t *who, char mode);
/*-------------*/

void client_log_times(struct client_t *who)
{
    time_t now = time(NULL);
    struct in_addr ia;
    if ( !server->logfile ) return;
    memset(&ia, 0, sizeof(struct in_addr));
    ia.s_addr = who->ip;
    fprintf(server->logfile, "user: ");
    switch ( server->use_user_accounting )
    {
	case 0:
	    fprintf(server->logfile, "%s ", inet_ntoa(ia));
	    break;
	case 1:
	    fprintf(server->logfile, "%s ", who->name);
	    break;
    }
    fprintf(server->logfile, "'%s' ", cor_ctime(&(who->started)));
    fprintf(server->logfile, "'%s' %d\n", cor_ctime(&now), (int)(now-who->started));
    fflush(server->logfile);
}

int client_get_sock_type(struct client_t *who)
{
    switch ( who->type )
    {
	case LCS_CLT_LCP3:
	case LCS_CLT_UNKNOWN_TCP:
	case LCS_CLT_LC: return(SOCK_TYPE_TCP); break;
	case LCS_CLT_DC: return(SOCK_TYPE_UDP); break;
     }
     syslog(LOG_ERR, "client_get_sock_type(): invalid client type!");
     return(-1);
}

int clean_clients(struct client_t *only)
{
 void *dum;
 fd_set fds;
 struct timeval tv;
 time_t now = time(NULL);
 struct client_t *who;
 int single = 0;
 if ( only )
 { // check only one client
     single++;
     who = only;
 }
 else who = (struct client_t*)cltlist.first; // check all clients
 while ( who )
 {
     dum = who->next;
     // check whether tcp connection is still active
     if ( client_get_sock_type(who) == SOCK_TYPE_TCP )
     {
         FD_ZERO(&fds);
         FD_SET(who->tcp_sock, &fds);
         tv.tv_sec = 0;
         tv.tv_usec = 100; // (buf could be full) give some time?
         if ( !select(who->tcp_sock+1, NULL, &fds, NULL, &tv) )
         { // unable to write to tcp_sock --> assume con is down.
             kick_client(who, 2);
	     who = dum;
	     if ( !single ) continue; else break;
         }
     }
     // check for timeout
     if ( who->timeout < now )
     {
         kick_client(who, 3);
     }
     who = dum;
     if ( single ) break;
 }
 return(0);
}

int kick_cmd_direct_wrap(int cmd, struct client_t* who, struct sockaddr_in *trg, char *msg, int mlen, char type)
{
    switch ( who->type )
    {
	case LCS_CLT_LC:
	case LCS_CLT_DC:
	    return(cmd_direct(cmd, who->tcp_sock, trg, msg, mlen, type));
	    break;
	case LCS_CLT_LCP3:
	    return(lcp3_cmd_direct(CMD3_UNREG, who->tcp_sock, NULL, 0));
	    break;
    }
    return(0);
}

int kick_client(struct client_t *who, char mode)
// mode: see switch() statement below
{
 struct ordq_item_t * oq;
 struct ackq_item_t * aq;
 struct t_lcp3_ordq_item* oq3;
 struct sockaddr_in trg;
 struct line_t *line;
 void *dum;
 int type = 0;
 if ( who == NULL ) return(-10);
 trg.sin_family = AF_INET;
 trg.sin_port = who->port;
 trg.sin_addr.s_addr = who->ip;
 type = client_get_sock_type(who);
#ifdef DEBUG
 syslog(LOG_DEBUG, "%s:%d: kick_client()", inet_ntoa(trg.sin_addr), ntohs(trg.sin_port));
#endif
 switch ( mode )
 {
     case 0: kick_cmd_direct_wrap(CMD_KICK, who, &trg, NULL, 0, type);
         syslog(LOG_INFO, "%s:%d kicked: unknown reason", inet_ntoa(trg.sin_addr), ntohs(trg.sin_port));
	 break;
     case 1: kick_cmd_direct_wrap(CMD_UNREG, who, &trg, NULL, 0, type);
         syslog(LOG_INFO, "%s:%d unregistered", inet_ntoa(trg.sin_addr), ntohs(trg.sin_port));
	 break;
     case 2:
         syslog(LOG_INFO, "%s:%d closed tcp connection", inet_ntoa(trg.sin_addr), ntohs(trg.sin_port));
	 break;
     case 3: kick_cmd_direct_wrap(CMD_KICK, who, &trg, NULL, 0, type);
         syslog(LOG_INFO, "%s:%d timed out", inet_ntoa(trg.sin_addr), ntohs(trg.sin_port));
	 break;
     case 4: kick_cmd_direct_wrap(CMD_KICK, who, &trg, NULL, 0, type);
         syslog(LOG_INFO, "%s:%d kicked: no ack from clt", inet_ntoa(trg.sin_addr), ntohs(trg.sin_port));
	 break;
     case 5: kick_cmd_direct_wrap(CMD_KICK, who, &trg, NULL, 0, type);
         syslog(LOG_INFO, "%s:%d kicked: srv unable to communicate", inet_ntoa(trg.sin_addr), ntohs(trg.sin_port));
	 break;
     case 6:
         kick_cmd_direct_wrap(CMD_KICK, who, &trg, NULL, 0, type); // 20001002
         syslog(LOG_INFO, "%s:%d bad user. Access denied.", inet_ntoa(trg.sin_addr), ntohs(trg.sin_port));
	 break;
     case 7: // silent kick
         kick_cmd_direct_wrap(CMD_KICK, who, &trg, NULL, 0, type);
#ifdef DEBUG
	 syslog(LOG_DEBUG, "%s:%d: silent kick", inet_ntoa(trg.sin_addr), ntohs(trg.sin_port));
#endif
         break;
     case 8: // bad proto
         kick_cmd_direct_wrap(CMD_KICK, who, &trg, NULL, 0, type);
	 syslog(LOG_INFO, "%s:%d kicked: uses bad protocol", inet_ntoa(trg.sin_addr), ntohs(trg.sin_port));
	 break;
 }
 line = (struct line_t*)lstlines.first;
 while ( line )
 {
     if ( ((struct client_t*)line->con_action.who) == who ) line->con_action.who = NULL;
     line = (struct line_t*)line->next;
 }
 oq3 = (struct t_lcp3_ordq_item*)lcp3_ordq.first;
 while ( oq3 )
 {
     dum = oq3->next;
     if ( oq3->who == who ) list_del(&lcp3_ordq, (struct list_hdr_t*)oq3);
     oq3 = dum;
 }
 oq = (struct ordq_item_t *)ordq.first;
 while ( oq != NULL )
 {
  dum = oq->next;
  if ( oq->who == who ) list_del(&ordq, (struct list_hdr_t *)oq);
  oq = dum;
 }
 aq = (struct ackq_item_t *)ackq.first;
 while ( aq != NULL )
 {
  dum = aq->next;
  if ( aq->who == who ) list_del(&ackq, (struct list_hdr_t *)aq);
  aq = dum;
 }
 if ( who->status == CLT_ONLINE )
 {
     client_log_times(who);
     who->line->online--;
     if ( who->line->client_offline[0] )
         exec_dont_care_param(who->line->client_offline, inet_ntoa(trg.sin_addr));
 }
 line = who->line;
 if ( type == SOCK_TYPE_TCP )
 {
     close(who->tcp_sock);
 }
 list_del(&cltlist, (struct list_hdr_t *)who);
 // check, whether to close the line or not...
 // kind of additional security check...
 // thanx jpcl...
/*
 if ( (!line->online) || (!cltlist.count) )
 {
     set_offline(NULL); // set all clients of any line offline!
                        // open lines get closed by set_offline().
 }
*/
 // replacement:
 if ( (!line->online) && (line->con_stat == CST_UP_SERVER) )
    set_offline(line);
 return(0);
}

struct client_t *get_client(int ip, short port)
{
    struct client_t *who = (struct client_t*)cltlist.first;
    while ( who )
    {
	if ( (ip == who->ip) && (port == who->port) )  // got the right one!
	    return(who);
	who = (struct client_t*)who->next;
    }
    return(NULL);
}

int client_exists(struct client_t *who)
{
    struct client_t *it = (struct client_t*)cltlist.first;
    if ( !who ) return(0);
    while ( it )
    {
	if ( it == who ) return(1);
	it = (struct client_t*)it->next;
    }
    return(0);
}
