/*****************************************************************************/
/* linleech - a program to selectively download usenet articles              */
/* Copyright (C) 1988  Curtiss Howard                                        */ 
/*                                                                           */
/* This program is free software; you can redistribute it and/or modify      */
/* it under the terms of the GNU General Public License as published by      */
/* the Free Software Foundation; either version 2 of the License, or         */
/* (at your option) any later version.                                       */
/*                                                                           */
/* This program is distributed in the hope that it will be useful,           */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of            */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             */
/* GNU General Public License for more details.                              */
/*                                                                           */
/* You should have received a copy of the GNU General Public License         */
/* along with this program; if not, write to the Free Software               */
/* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */
/*****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "linleech_slang.h"
#include "linleech_socket.h"

void SLang_Init (void)
{
	printf ("linleech> Initializing S-Lang... ");
	
	fflush (stdout);

	if ((SLang_init_slang() == -1) || (SLang_init_slfile() == -1) ||
		(SLang_init_slunix() == -1) ||
		(SLadd_intrin_fun_table (Functions, NULL) == -1) ||
		(SLadd_intrinsic_variable ("connected", &Misc.Connected, INT_TYPE, 
			0) == -1))
	{
		printf ("failure.\n");

		exit (0);
	}

	printf ("okay.\n");
}

void Connect (char *Host, int *Port)
{
	int Code;
	
	if (Misc.Connected) 
	{
		printf ("linleech> Function connect unavailable; already "
			"connected.\n");

		return;
	}

	Socket = HostConnect (Host, *Port);

	if (Socket == -1)
	{
		return;
	}

	printf ("linleech> Connected to %s.\n", Host);

	Sock = CreateSock (Socket);

	if (Sock != NULL)
	{
		printf ("linleech> Attempting to use server... ");
	
		fflush (stdout);
		
		if (fgets (RecvBuffer, sizeof (RecvBuffer), Sock) != NULL)
		{
			Misc.BytesReceived += strlen (RecvBuffer);

			(int *) Token = strtok (RecvBuffer, " ");

			Code = atoi (Token);

			if (Code == 200 || Code == 201)
			{
				printf ("okay.\n");

				Misc.Connected = TRUE;
			}

			if (Code != 200 && Code != 201)
			{
				printf ("failure.  Try using auth_connect.\n");
			}
		}
	}
}

void DeleteMatchList (void)
{
	printf ("linleech> Attempting to delete match list... ");
	
	fflush (stdout);
	
	if (ListVars.ListCreated == FALSE)
	{
		printf ("failure.  List not created.\n");
	}

	if (ListVars.ListCreated == TRUE)
	{
		free (Match);
		
		ListVars.ListSize = 0;
		ListVars.ListCount = 0;
		ListVars.ListCreated = FALSE;

		printf ("okay.\n");
	}
}

void DisableMatchAll (void)
{
	printf ("linleech> Only saving matched articles.\n");

	Misc.SaveAll = FALSE;
}

void DisableResume (void)
{
	printf ("linleech> Disabling article resume.\n");

	Misc.Resume = FALSE;
}

void Disconnect (void)
{
	if (!Misc.Connected)
	{
		printf ("linleech> Unable to disconnect; not connected.\n");

		return;
	}
	
	fclose (Sock);

	close (Socket);

	Misc.Connected = FALSE;

	printf ("linleech> Disconnected.\n");
}

void EnableMatchAll (void)
{
	printf ("linleech> Saving all articles.\n");
	
	Misc.SaveAll = TRUE;
}

void EnableResume (void)
{
	printf ("linleech> Enabling article resume.\n");

	Misc.Resume = TRUE;
}

void Filter (char *Program, int *Output)
{
	Misc.FilterSet = TRUE;

	strcpy (Misc.Filter, Program);

	if (*Output < 0 || *Output > 1)
	{
		printf ("linleech> Invalid output value for function filter; "
			"defaulting to 0 (don't show article filter output).\n");
		
		*Output = 0;
	}
	
	printf ("linleech> Set article filter to \"%s\" ", Program);

	if (!*Output)
	{
		Misc.FilterOutput = FALSE;

		printf ("(not showing output).\n");
	}

	if (*Output)
	{
		Misc.FilterOutput = TRUE;

		printf ("(showing output).\n");
	}
}

void HideSubjects (void)
{
	printf ("linleech> Hiding subjects.\n");
	
	Misc.ShowHead = FALSE;
}

void JoinGroup (char *Group)
{
	int Code, NumArticles, Start, Stop;

	if(!Misc.Connected)
	{
		printf ("linleech> Function group unavailable; not connected.\n");

		return;
	}

	printf ("linleech> Joining group %s... ", Group);
	
	fflush (stdout);
	
	sprintf (SendBuffer, "GROUP %s\r\n", Group);

	send (Socket, SendBuffer, strlen (SendBuffer), 0);

	Misc.BytesSent += strlen (SendBuffer);
	
	if (fgets (RecvBuffer, sizeof (RecvBuffer), Sock) != NULL)
	{
		Misc.BytesReceived += strlen (RecvBuffer);

		(int *) Token = strtok (RecvBuffer, " ");

		Code = atoi (Token);

		if (Code == 411)
		{
			printf ("failure.\n");
		}

		if (Code == 211)
		{
			(int *) Token = strtok (NULL, " ");

			NumArticles = atoi (Token);

			(int *) Token = strtok (NULL, " ");

			Start = atoi (Token);

			(int *) Token = strtok (NULL, " ");

			Stop = atoi (Token);

			printf ("okay.\n");

			GetArticles (Group, NumArticles, Start, Stop);
		}
	}
}
		
void MatchFunc (char *Str, int *Case)
{
	if (ListVars.ListCreated == FALSE)
	{
		printf ("linleech> Function match unavailable; no match list.\n");

		return;
	}

	if (ListVars.ListCount == ListVars.ListSize)
	{
		printf ("linleech> Function match unavailable; match list size "
			"reached.\n");

		return;
	}
	
	strcpy (Match[ListVars.ListCount].Item, Str);
	
	if (*Case < 0 || *Case > 1)
	{
		printf ("linleech> Invalid case value for function match; "
			"defaulting to 0 (case insensitive).\n");

		*Case = 0;
	}

	Match[ListVars.ListCount].Case = *Case;
	Match[ListVars.ListCount].HasTrigger = FALSE;
	Match[ListVars.ListCount].RegexpMatch = FALSE;
	
	ListVars.ListCount++;
	
	printf ("linleech> Added \"%s\" to match list ", 
		Match[ListVars.ListCount - 1].Item);

	if (!*Case)
	{
		printf ("(case insensitive).\n");
	}

	if (*Case)
	{
		printf ("(case sensitive).\n");
	}
}

void MatchRegexp (char *Str)
{
	if (ListVars.ListCreated == FALSE)
	{
		printf ("linleech> Function match_regexp unavailable; no match "
			"list.\n");

		return;
	}

	if (ListVars.ListCount == ListVars.ListSize)
	{
		printf ("linleech> Function match_regexp unavailable; match list "
			"size reached.\n");

		return;
	}
	
	strcpy (Match[ListVars.ListCount].Item, Str);

	Match[ListVars.ListCount].HasTrigger = FALSE;
	Match[ListVars.ListCount].RegexpMatch = TRUE;
	
	Match[ListVars.ListCount].PatBuf.translate = 0;
	Match[ListVars.ListCount].PatBuf.fastmap = 0;
	Match[ListVars.ListCount].PatBuf.buffer = 0;
	Match[ListVars.ListCount].PatBuf.allocated = 0;
	
	if (re_compile_pattern (Match[ListVars.ListCount].Item, 
		strlen (Match[ListVars.ListCount].Item),
		&Match[ListVars.ListCount].PatBuf) != NULL)
	{
		printf ("linleech> Error compiling pattern buffer.\n");

		exit (0);
	}

	ListVars.ListCount++;

	printf ("linleech> Added \"%s\" to match list (using regexp).\n",
		Match[ListVars.ListCount - 1].Item);
}

void NewMatchList (int *Size)
{
	printf ("linleech> Attempting to create match list... ");

	fflush (stdout);
	
	if (ListVars.ListCreated == TRUE)
	{
		printf ("failure.  List already created.\n");
	}

	if (ListVars.ListCreated == FALSE)
	{
		Match = (MATCH *) calloc (*Size, sizeof (MATCH));

		if (Match == (MATCH *) NULL)
		{
			printf ("failure.  Couldn't allocate list.\n");
		}

		if (Match != (MATCH *) NULL)
		{
			ListVars.ListSize = *Size;
			ListVars.ListCount = 0;
			ListVars.ListCreated = TRUE;

			printf ("okay.\n");
		}
	}
}

void RawConnect (char *Host, int *Port)
{
	if (Misc.Connected)
	{
		printf ("linleech> Cannot connect to %s; already connected.\n",
			Host);

		return;
	}

	if (!Misc.Connected)
	{
		Socket =  HostConnect (Host, *Port);

		if (Socket == -1)
		{
			return;
		}

		if (Socket != -1)
		{
			printf ("linleech> Connected to %s.\n", Host);
			
			Sock = CreateSock (Socket);

			if (Sock != NULL)
			{
				Misc.Connected = TRUE;
			}
		}
	}
}

char *RawReceive (void)
{
	char *Ret;

	if (!Misc.Connected)
	{
		printf ("linleech> Unable to receive data; not connected.\n");

		return (char *) NULL;
	}

	if (fgets (RecvBuffer, sizeof (RecvBuffer), Sock) != NULL)
	{
		Ret = RecvBuffer;

		Misc.BytesReceived += strlen (Ret);

		strcat (Ret, "\0");

		return Ret;
	}

	return (char *) NULL;
}

void RawSend (char *Text)
{
	if (!Misc.Connected)
	{
		printf ("linleech> Unable to send data; not connected.\n");

		return;
	}

	if (Misc.Connected)
	{
		send (Socket, Text, strlen (Text), 0);

		Misc.BytesSent += strlen (Text);
	}
}

void RemoveFilter (void)
{
	Misc.FilterOutput = FALSE;
	Misc.FilterSet = FALSE;

	strcpy (Misc.Filter, "");

	printf ("linleech> Removed article filter.\n");
}

void SaveTo (char *Dir)
{
	printf ("linleech> Saving articles to %s.\n", Dir);

	strcpy (Misc.SaveDir, Dir);
}

void SaveToDef (void)
{
	printf ("linleech> Saving articles to current directory.\n");

	strcpy (Misc.SaveDir, "");
}

void SetTrigger (char *Trigger, char *Find)
{
	if (ListVars.ListCreated == FALSE)
	{
		printf ("linleech> Function set_trigger unavailable; no match "
			"list.\n");

		return;
	}

	if (ListVars.ListCount == ListVars.ListSize)
	{
		printf ("linleech> Function set_trigger unavailable; match list "
			"size reached.\n");

		return;
	}
	
	strcpy (Match[ListVars.ListCount].Item, Find);
	strcpy (Match[ListVars.ListCount].Trigger, Trigger);
	
	Match[ListVars.ListCount].HasTrigger = TRUE;
	Match[ListVars.ListCount].RegexpMatch = TRUE;

	Match[ListVars.ListCount].PatBuf.translate = 0;
	Match[ListVars.ListCount].PatBuf.fastmap = 0;
	Match[ListVars.ListCount].PatBuf.buffer = 0;
	Match[ListVars.ListCount].PatBuf.allocated = 0;
	
	Misc.ReqTrigger = TRUE;
	
	if (re_compile_pattern (Match[ListVars.ListCount].Item,
		strlen (Match[ListVars.ListCount].Item),
		&Match[ListVars.ListCount].PatBuf) != NULL)
	{
		printf ("linleech> Error compiling pattern buffer.\n");

		exit (0);
	}

	ListVars.ListCount++;
	
	printf ("linleech> Added trigger \"%s\" to match list with item "
		"\"%s\".\n", Match[ListVars.ListCount - 1].Trigger, 
		Match[ListVars.ListCount - 1].Item);
}

void ShowSubjects (void)
{
	printf ("linleech> Showing subjects.\n");
	
	Misc.ShowHead = TRUE;
}

/* These functions are not S-Lang function hanlders. */

void ArticleMatchInsensitive (char *Subject, MATCH MatchVar, int Article,
	char *Group)
{
	int Count;
	char Filename[255];
	
	for (Count = 0; Count < strlen (Subject); Count++)
	{
		Subject[Count] = tolower (Subject[Count]);
	}

	for (Count = 0; Count < strlen (MatchVar.Item); Count++)
	{
		MatchVar.Item[Count] = tolower (MatchVar.Item[Count]);
	}

	if (((char *) strstr (Subject, MatchVar.Item)) != NULL)
	{
		sprintf (Filename, "%s%s.%d", Misc.SaveDir, Group, Article);
		
		printf ("linleech> Found \"%s\" in article %d; saving to %s... ", 
			MatchVar.Item, Article, Filename);

		fflush (stdout);
		
		Misc.Matched++;
		
		SaveArticle (Filename, Article);
	}
}

void ArticleMatchRegexp (char *Subject, MATCH MatchVar, int Article, 
	char *Group)
{
	char Filename[255];

	if (re_search (&MatchVar.PatBuf, Subject, strlen (Subject), 0,
		strlen (Subject), &MatchVar.Regs) > 0)
	{
		sprintf (Filename, "%s%s.%d", Misc.SaveDir, Group, Article);

		printf ("linleech> Found \"%s\" in article %d; saving to %s... ",
			MatchVar.Item, Article, Filename);
		
		fflush (stdout);

		Misc.Matched++;
		
		SaveArticle (Filename, Article);
	}
}

void ArticleMatchSensitive (char *Subject, MATCH MatchVar, int Article,
	char *Group)
{
	char Filename[255];
	
	if (((char *) strstr (Subject, MatchVar.Item)) != NULL)
	{
		sprintf (Filename, "%s%s.%d", Misc.SaveDir, Group, Article);
		
		printf ("linleech> Found \"%s\" in article %d; saving to %s... ", 
			MatchVar.Item, Article, Filename);

		fflush (stdout);

		Misc.Matched++;
		
		SaveArticle (Filename, Article);
	}
}

void GetArticles (char *Group, int NumArticles, int Start, int Stop)
{
	int Count, Count2, TriggerCount, TriggerFound;
	char Filename[255];
	FILE *Fptr;
	
	if (Misc.Resume)
	{
		sprintf (Filename, "%s.last", Group);

		Fptr = fopen (Filename, "r");

		if (!Fptr)
		{
			Count = Start;
		}

		if (Fptr)
		{
			fscanf (Fptr, "%d", &Count);

			if (Count < Start || Count > Stop)
			{
				Count = Start;
			}

			if (Count >= Start && Count <= Stop)
			{
				printf ("linleech> Resuming at article %d...\n", Count);
			}
		}

		printf ("linleech> Searching through %d articles in %s...\n",
			(Stop - Count), Group);
	}
	
	if (!Misc.Resume)
	{
		printf ("linleech> Searching through %d articles in %s...\n", 
			NumArticles, Group);

		Count = Start;
	}
	
	Misc.Matched = 0;
	
	TriggerCount = 0;
	
	TriggerFound = 0;
	
	sprintf (SendBuffer, "HEAD %d\r\n", Start);

	send (Socket, SendBuffer, strlen (SendBuffer), 0);
	
	Misc.BytesSent += strlen (SendBuffer);
	
	for (Count2 = 0; Count2 < ListVars.ListSize; Count2++)
	{
		if (Match[Count2].HasTrigger)
		{
			TriggerCount++;
		}
	}

	while ((Count <= Stop) && (Count >= Start))
	{
		if (Misc.Resume)
		{
			sprintf (Filename, "%s.last", Group);

			Fptr = fopen (Filename, "w");

			if (!Fptr)
			{
				printf ("linleech> Can't open file for saving article "
					"status.\n");
			}

			if (Fptr)
			{
				fprintf (Fptr, "%d\n", Count);

				fclose (Fptr);
			}
		}
		
		if (fgets (RecvBuffer, sizeof (RecvBuffer), Sock) != NULL)
		{
			Misc.BytesReceived += strlen (RecvBuffer);
			
			if (!strncmp (RecvBuffer, "423", 3))
			{
				printf ("linleech> Article %d: Not found.\n", Count);
			}
		
			for (Count2 = 0; Count2 < ListVars.ListSize; Count2++)
			{
				if ((strncmp (RecvBuffer, "423", 3)) &&
					Match[Count2].HasTrigger &&
					((char *) strstr (RecvBuffer, Match[Count2].Trigger) 
						!= NULL) &&
					(re_search (&Match[Count2].PatBuf, RecvBuffer,
					strlen (RecvBuffer), 0, strlen (RecvBuffer),
					&Match[Count2].Regs) > 0))
				{	
					if (TriggerFound < TriggerCount)
					{
						TriggerFound++;
					}
				}
			}
		
			if ((strncmp (RecvBuffer, "423", 3)) && 
				((char *) strstr (RecvBuffer, "Subject:") != NULL))
			{
				if (Misc.ShowHead == TRUE)
				{
					printf ("linleech> Article %d: %s", Count, 
						RecvBuffer);
				}

				if (!Misc.SaveAll)
				{
					for (Count2 = 0; Count2 < ListVars.ListSize; Count2++)
					{
						if ((!Match[Count2].Case) && 
							(!Match[Count2].RegexpMatch) &&
							(!Match[Count2].HasTrigger) &&
							strcmp (Match[Count2].Item, ""))
						{
							if ((Misc.ReqTrigger && (TriggerFound ==
								TriggerCount)) || !Misc.ReqTrigger)
							{
								ArticleMatchInsensitive (RecvBuffer, 
									Match[Count2], Count, Group);
							}
						}

						if ((Match[Count2].Case) &&
							(!Match[Count2].RegexpMatch) &&
							(!Match[Count2].HasTrigger) &&
							strcmp (Match[Count2].Item, ""))
						{
							if ((Misc.ReqTrigger && (TriggerFound ==
								TriggerCount)) || !Misc.ReqTrigger)
							{
								ArticleMatchSensitive (RecvBuffer,
									Match[Count2], Count, Group);
							}
						}

						if ((Match[Count2].RegexpMatch) &&
							(!Match[Count2].HasTrigger) &&
							strcmp (Match[Count2].Item, ""))
						{
							if ((Misc.ReqTrigger && (TriggerFound ==
								TriggerCount)) || !Misc.ReqTrigger)
							{
								ArticleMatchRegexp (RecvBuffer,
									Match[Count2], Count, Group);
							}
						}
					}
				}

				if (Misc.SaveAll)
				{
					if ((Misc.ReqTrigger && 
						(TriggerFound == TriggerCount)) ||
						!Misc.ReqTrigger)
					{
						sprintf (Filename, "%s%s.%d", Group, Count);

						printf ("linleech> Saving article %d to %s... ", 
							Count, Filename);

						fflush (stdout);

						SaveArticle (Filename, Count);
					}
				}
			}
		}

		if (!strncmp (RecvBuffer, ".\r\n", 3) || !strncmp (RecvBuffer,
			"423", 3))
		{
			Count++;

			sprintf (SendBuffer, "HEAD %d\r\n", Count);

			send (Socket, SendBuffer, strlen (SendBuffer), 0);

			Misc.BytesSent += strlen (SendBuffer);
		}
	}

	/* There always seems to be an extra header after leaving the loop, */
	/* so eat it up so it won't mess with JoinGroup().                  */
	
	fgets (RecvBuffer, sizeof (RecvBuffer), Sock);

	printf ("linleech> Matched %d article(s) in %s.\n", Misc.Matched,
		Group);
}

void SaveArticle (char *Filename, int Article)
{
	int Chars, End;
	FILE *Fptr;
	char Temp[255];
	
	Fptr = fopen (Filename, "w");

	if (!Fptr)
	{
		printf ("can't open %s.\n", Filename);

		return;
	}

	sprintf (SendBuffer, "ARTICLE %d\r\n", Article);

	send (Socket, SendBuffer, strlen (SendBuffer), 0);

	Misc.BytesSent += strlen (SendBuffer);

	End = 0;
	
	while (End != 2)
	{
		if (fgets (RecvBuffer, sizeof (RecvBuffer), Sock) != NULL)
		{
			Misc.BytesReceived += strlen (RecvBuffer);

			if (End)
			{
				if (fprintf (Fptr, "%s", RecvBuffer) != 
					strlen (RecvBuffer))
				{
					printf ("failure writing to disk.\n");

					exit (0);
				}
			}
			
			if (!strncmp (RecvBuffer, ".\r\n", 3))
			{
				End++;

				if (End == 2)
				{
					fclose (Fptr);

					printf ("done.\n");
					
					if (Misc.FilterSet)
					{
						sprintf (Temp, Misc.Filter, Filename);

						Misc.FilterFptr = popen (Temp, "r");

						printf ("linleech> Running %s... ", Temp);

						fflush (stdout);
						
						if (Misc.FilterOutput)
						{
							printf ("\n");
						}
						
						Chars = fread (Temp, sizeof (char), 
							sizeof (Temp), Misc.FilterFptr);

						while (Chars > 0)
						{
								if (Misc.FilterOutput)
								{
									printf ("%s\n", Temp);
								}
								
								Chars = fread (Temp, sizeof (char),
									sizeof (Temp), Misc.FilterFptr);
						}

						if (!Misc.FilterOutput)
						{
							printf ("done.\n");
						}

						pclose (Misc.FilterFptr);
					}
				}
			}
		}
	}
}
