/*
 * $Id: mft.c,v 1.11 2001/12/15 05:13:08 antona Exp $
 *
 * mft.c - Mft record handling code. Part of the Linux-NTFS project.
 *
 * Copyright (c) 2000,2001 Anton Altaparmakov.
 *
 * This program/include file is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program/include file is distributed in the hope that it will be 
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty 
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (in the main directory of the Linux-NTFS 
 * distribution in the file COPYING); if not, write to the Free Software
 * Foundation,Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdlib.h>
#include <stdio.h>
#include <errno.h>

#include "types.h"
#include "list.h"
#include "mft.h"
#include "endians.h"
#include "disk_io.h"
#include "support.h"
#include "attrib.h"

inline __u32 get_mft_record_data_size(const MFT_RECORD *b)
{
	if (!b || !is_mft_recordp(b))
		return 0;
	/* Get the number of used bytes and return it. */
	return le32_to_cpu(b->bytes_in_use);
}

/*

int __ntfs_unmap_mft_entry(mft_entry *me)
{
	int i;

	if (me->m_count)
		return -EBUSY;
	if (!test_and_clear_bit(ME_mapped, me->m_flags))
		return 0;
	if (MftEntryDirty(me) && (i = ntfs_flush_mft_entry(me)))
		goto err_map_ret;
	if (MftEntryDirty(me)) {
		i = -EBUSY;
		goto err_map_ret;
	}
	free(me->m_rec);
	return 0;
err_map_ret:
	SetMftEntryMapped(me);
	return i;
}

int insert_mft_entry(ntfs_volume *vol, mft_entry **me, const MFT_REF mref,
		     MFT_RECORD *mrec, const BOOL dirty, ntfs_file *f)
{
	struct list_head *tmp;
	mft_entry *ment, *m;
	int i;
	
	if (!*me && !(*me = __allocate_mft_entry()))
		return -errno;
	ment = *me;
	ment->m_ref = mref;
	ment->m_vol = vol;
	if (mrec) {
		ment->m_rec = mrec;
		SetMftEntryMapped(ment);
		if (dirty)
			SetMftEntryDirty(ment);
	} else
		__ntfs_map_mft_entry(ment);
	list_for_each(tmp, &vol->mft_entries) {
		m = list_entry(tmp, mft_entry, m_list);
		if (MREF(m->m_ref) < MREF(ment->m_ref))
			continue;
		if (MREF(m->m_ref) > MREF(ment->m_ref)) {
			__list_add(&ment->m_list, tmp->prev, tmp);
			break;
		}
		fprintf(stderr, "Linux-NTFS: BUG! insert_mft_entry(): mft "
				"entry already present in volume.\n");
		if (MSEQNO(m->m_ref) != MSEQNO(ment->m_ref) &&
		    MSEQNO(m->m_ref) != 0 && MSEQNO(ment->m_ref) != 0)
			fprintf(stderr, "On top of this, the sequence numbers "
					"are mismatched.\n");
		return -EEXIST;
	}
	vol->nr_mft_entries++;
	ment->m_file = f;
	if (f)
		ment->m_count++;
	if (!MftEntryDirty(ment))
		return 0;
        list_add_tail(&ment->m_dirty_list, &vol->dirty_mft_entries);
	vol->nr_dirty_mft_entries++;
	return 0;
try_to_clean_ret:
	return ntfs_flush_mft_entry(ment);
}

int remove_mft_entry(mft_entry *me)
{
	ntfs_volume *vol = me->m_vol;
	int err = 0;
	
	if (MftEntryDirty(me))
		err = ntfs_flush_mft_entry(me);
	if (MftEntryDirty(me))
		return err < 0 ? err : -EBUSY;
	if (me->m_file && (err = __remove_mft_entry_from_ntfs_file(me)))
		return err < 0 ? err : -EBUSY;
	if (me->m_count)
		return -EBUSY;
	free(me->m_rec);
	ClearMftEntryMapped(me);
	list_del(&me->m_list);
	vol->nr_mft_entries--;
	free(me);
	return 0;
}

int __add_mtf_entry_to_ntfs_file(ntfs_file *f, mft_entry *me)
{
	int i;
	void *t;
	
	if (!MftEntryMapped(me) && (i = __ntfs_map_mft_entry(me)))
		return i;
	if (me->m_file == f)
		return 0;
	me->m_count++;
	me->m_file = f;
	for (i = 0; i < f->nr_m_refs; i++)
		if (MREF(f->m_refs[i]) == MREF(me->m_ref))
			return 0;
	f->nr_m_refs++;
	t = realloc(f->m_refs, f->nr_m_refs * sizeof(MFT_REF));
	if (!t)
		goto undo_add_ret;
	f->m_refs = t;
	t = realloc(f->m_entries, f->nr_m_refs * sizeof(mft_entry*));
	if (!t) {
		t = realloc(f->m_refs, (f->nr_m_refs - 1) *
							sizeof(MFT_REF));
		if (t)
			f->m_refs = t;
		goto undo_add_ret;
	}
	f->m_entries = t;
	f->m_refs[f->nr_m_refs - 1] = me->m_ref;
	f->m_entries[f->nr_m_refs - 1] = me;
	return 0;
undo_add_ret:
	f->nr_m_refs--;
	me->m_count--;
	me->m_file = NULL;
	return -errno;
}

*/



/* OLD STUFF ABOVE THIS LINE. BEGIN OF NEW STUFF BELOW THIS LINE. */


















/**
 * ntfs_flush_mft_entry - flush a mft_entry to disk
 * @m:		mft_entry to flush
 * 
 * The mft entry is written to disk (if dirty), marked clean, and removed from
 * the dirty list of its volume.
 *
 * Return 0 on success and -ERRNO on error. Following failure return values
 * are defined:
 * 	-EBADF		- Mft entry is corrupt (error flag set).
 * 	-EIO		- Writing to disk failed.
 * 	-ENOTSUP	- Found a sparse mft record. BUG!
 * 	-EPERM		- Tried to write to cluster 0, which would overwrite the
 * 			  bootsector. BUG!
 * 	-EINVAL		- Found dirty mft entry but it is not mapped. BUG!
 * 	others		- Error codes returned by vcn_to_lcn() function; meaning
 * 			  we couldn't locate the position on disk for the entry.
 */
int ntfs_flush_mft_entry(mft_entry *m)
{
	ntfs_volume *v;
	__s64 ofs;
	VCN mref;
	LCN lcn;
	__u32 i;
	
	if (!MftEntryDirty(m))
		return 0;
	if (MftEntryError(m))
		return -EBADF;
#ifdef DEBUG
	if (!MftEntryMapped(m)) {
		fprintf(stderr, "DEBUG Linux-NTFS: ntfs_flush_mft_entry() "
				"found unmapped, dirty mft entry. Aborting "
				"flush!\n");
		return -EINVAL;
	}
#endif
	mref = (VCN)MREF(m->m_ref);
	v = m->m_vol;
	lcn = vcn_to_lcn(v->mft_runlist, mref << v->mft_record_size_bits >>
							v->cluster_size_bits);
	if (!lcn) {
		if (!errno) {
			fprintf(stderr, "ERROR Linux-NTFS: "
					"ntfs_flush_mft_entry() tried to write "
					"to cluster zero!\n");
			errno = EPERM;
		}
		return -errno;
	} else if (lcn == -1) {
		/* It is sparse. This should never happen as we would have
		 * instantiated it when reading the entry. */
		fprintf(stderr, "ERROR Linux-NTFS: BUG! ntfs_flush_mft_entry() "
				"called on sparse mft entry.\n");
		return -ENOTSUP;
	}
	ofs = mref << v->mft_record_size_bits & v->cluster_size - 1;
	lcn = (LCN)mst_pwrite(v->fd, m->m_rec, v->mft_record_size, 
					(lcn << v->cluster_size_bits) + ofs);
	if (lcn != v->mft_record_size) {
		perror("ERROR Linux-NTFS: ntfs_flush_mft_entry() mst_write() "
		       "failed");
		return -EIO;
	}
	v->nr_dirty_mft_entries--;
	list_del(&m->m_dirty_list);
	ClearMftEntryDirty(m);
	return 0;
}

/**
 * __ntfs_set_mft_entry_dirty - add mft_entry to its volume's dirty list
 * @m:         mft entry to add to dirty list
 *
 * The mft_entry @m better be marked dirty! You have been warned. Better use
 * ntfs_set_mft_entry_dirty() instead.
 */
void __ntfs_set_mft_entry_dirty(mft_entry *m)
{
	ntfs_volume *v = m->m_vol;
	list_add_tail(&m->m_dirty_list, &v->dirty_mft_entries);
	v->nr_dirty_mft_entries++;
	return;
}

/**
 * __ntfs_map_mft_entry - map an mft entry into memory
 * @m:		mft_entry to map
 *
 * Maps an mft_entry into memory. The mft_entry better be unmapped or massive
 * data loss/corruption/memory leaks, etc. will occur. You have been warned.
 * Be a good hobbit and use ntfs_map_mft_entry() instead.
 *
 * Return 0 on success or -ERRNO on error. The exact error return values are:
 *
 *	-ENOMEM		- Not enough memory to allocate necessary structure(s).
 *	-EPERM		- Access denied. (Tried to access cluster zero.)
 *	-ENOTSUP	- Found a sparse mft record! Can this really happen?
 *			  Not implemented until someone complains.
 *	Other codes returned by mst_pread().
 */
int __ntfs_map_mft_entry(mft_entry *m)
{
	MFT_RECORD *mrec;
	ntfs_volume *v;
	__s64 ofs;
	VCN mref;
	LCN lcn;
	
	mref = (VCN)MREF(m->m_ref);
	mrec = (MFT_RECORD*)malloc(v->mft_record_size);
	if (!mrec)
		goto err_ret_errno;
	v = m->m_vol;
	lcn = vcn_to_lcn(v->mft_runlist, mref << v->mft_record_size_bits >>
							v->cluster_size_bits);
	if (!lcn) {
		if (!errno) {
			fprintf(stderr, "ERROR Linux-NTFS: __map_mft_entry() "
					"tried to access cluster zero!\n");
			errno = EPERM;
		}
		free(mrec);
		return -errno;
	} else if (lcn == -1) {
		/* It is sparse. We would need to instantiate it. */
		fprintf(stderr, "Linux-NTFS: __map_mft_entry() encountered "
				"sparse mft entry. Not supported yet.\n"
				"Please email me, Anton <aia21@cam.ac.uk> and "
				"tell me you got this error,\n"
				"so I know I have to implement it.\n");
		free(mrec);
		return -ENOTSUP;
	}
	ofs = mref << v->mft_record_size_bits & v->cluster_size - 1;
	lcn = (LCN)mst_pread(v->fd, (__u8*)mrec, v->mft_record_size, 1, 
					(lcn << v->cluster_size_bits) + ofs);
	m->m_rec = mrec;
	SetMftEntryMapped(m);
	if (lcn != (LCN)1) {
		SetMftEntryError(m);
		if (lcn != (LCN)-1)
			return -EIO;
		goto err_ret_errno;
	}
	if (ntfs_get_bit(v->mft_bitmap, mref) !=
				le16_to_cpu(mrec->flags & MFT_RECORD_IN_USE)) {
		fprintf(stderr, "WARNING Linux-NTFS: __map_mft_entry() found "
				"mft record in use when bitmap says that it is "
				"not in use or vice versa.\n");
		if (MSEQNO(mref) && MSEQNO(mref) !=
					le16_to_cpu(mrec->sequence_number)) {
			fprintf(stderr, "Sequence numbers are also mismatched."
					"\nAssuming bitmap is correct and "
					"fixing mft record.\n");
			mrec->flags ^= MFT_RECORD_IN_USE;
			ntfs_set_mft_entry_dirty(m);
		} else {
			fprintf(stderr, "Assuming mft record is correct and "
					"fixing bitmap.\n");
			ntfs_change_bit(v->mft_bitmap, mref);
			//SetNtfsVolMftBitmapDirty(v);
		}
	} else if (MSEQNO(mref) && MSEQNO(mref) !=
					le16_to_cpu(mrec->sequence_number)) {
		fprintf(stderr, "ERROR Linux-NTFS: __map_mft_entry(): sequence "
				"numbers do not match.\n");
		return -EIO;
	}
	return 0;
err_ret_errno:
	return -errno;
}

/**
 * __lookup_mft_entry_in_vol_by_mref - find an mft_entry in the volume's cache
 * @v:		ntfs_volume whose cache to search
 * @mref:	mft record number to search for
 * @me:		address of variable in which to return the mft entry's address
 *
 * Search for an mft_entry in @v that has a mft reference of MREF(@mref). If
 * MSEQNO(@mref) is not zero, consistency check the sequence numbers. If they do
 * not match return error.
 *
 * Return 1 if the entry was found. *@me then contains the address of the found
 * mft_entry.
 *
 * Return 0 if the entry was not found. *@me then contains the address of the
 * first entry that collates _after_ the entry looked for. You must use
 * list_add_tail(&new_entry->m_list, &*me->m_list) to insert a new entry, if
 * desired, or you will break the sort order causing massive corruption. You
 * have been warned! Further, you must not use *me or any of *me->struct_member,
 * except for *me->m_list, or you will cause massive corruption and a possible
 * dereference of invalid memory, thus a potential crash! Again, you have been
 * warned!
 *
 * Return -1 on error. This means that MREF(@mref) does exist in @v's cache,
 * but it has a different sequence number from MSEQNO(@mref). Something is thus
 * corrupt somewhere! *me then points to the found mft_entry, so caller can do
 * something about the corruption if they want to (e.g. caller is ntfsck).
 */
static __inline__ int __lookup_mft_entry_in_vol_by_mref(ntfs_volume *v,
							const MFT_REF mref,
							mft_entry **me)
{
	struct list_head *tmp;
	mft_entry *m;

	list_for_each(tmp, &v->mft_entries) {
		m = list_entry(tmp, mft_entry, m_list);
		/* Not there yet. -> Keep going... */
		if (MREF(m->m_ref) < MREF(mref))
			continue;
		*me = m;
		/* Overshot! -> Not present. Done. */
		if (MREF(m->m_ref) > MREF(mref))
			return 0;
		/* Found. -> Check sequence number. */
		if (MSEQNO(mref)) {
			if (!MSEQNO(m->m_ref))
				fprintf(stderr, "WARNING Linux-NTFS: "
					"__lookup_mft_entry_in_vol_by_mref(): "
					"requested mref has sequence number "
					"not zero but found mref has zero "
					"sequence number!\n");
			else if (MSEQNO(m->m_ref) != MSEQNO(mref)) {
				fprintf(stderr, "ERROR Linux-NTFS: "
					"__lookup_mft_entry_in_vol_by_mref(): "
					"sequence numbers do not match.\n");
				return -1;
			}
		}
		/* Got it! Done. */
		return 1;
	}
	/*
	 * Not present and end of list reached so return a bogus pointer into
	 * @v, which, when dereferenced with ->m_list, will return the list
	 * itself! This is not beautiful code, but it works, as long as you
	 * adhere stringently to _only_ use *me->m_list and _nothing_ else. It
	 * is the only simple solution without the caller having to do do more
	 * return value checking. (AIA)
	 */
	*me = list_entry(&v->mft_entries, mft_entry, m_list);
	return 0;
}

/**
 * __insert_mft_entry_in_vol - insert a mft_entry in a ntfs_volume
 * @v:		ntfs_volume to insert @me into
 * @me:		mft_entry to insert into @v
 * @pos:	optinal position in @v at which to insert the loaded entry
 *
 * Insert @me into @v and increment the use count. This should only be called on
 * a mft_entry which has just been allocated and has not been inserted yet. You
 * have been warned!
 *
 * Return 0 on success and -1 if the entry already exists in the volume's cache.
 */
static __inline__ int __insert_mft_entry_in_vol(ntfs_volume *v, mft_entry *me,
						struct list_head *pos)
{
	me->m_count++;
	if (pos)
		/* Insert the entry in the correct position. */
		list_add_tail(&me->m_list, pos);
	else {
		mft_entry *m;

		if (__lookup_mft_entry_in_vol_by_mref(v, m->m_ref, &m)) {
			fprintf(stderr, "ERROR Linux-NTFS: "
					"__insert_mft_entry_in_vol(): mft "
					"entry already present in volume.\n");
			me->m_count--;
			return -1;
		}
		/* Insert the entry in the correct position. */
		list_add_tail(&me->m_list, &m->m_list);
	}
	v->nr_mft_entries++;
	me->m_vol = v;
	if (MftEntryDirty(me)) {
	        list_add_tail(&me->m_dirty_list, &v->dirty_mft_entries);
		v->nr_dirty_mft_entries++;
	}
	return 0;
}

/**
 * __remove_mft_entry_from_vol - remove a mft_entry from its ntfs_volume
 * @m:		mft_entry to remove from @v
 *
 * Remove @m from @m->m_vol and decrement the use count. Make sure the mft_entry
 * is not in use, is clean and no attributes are mapped.
 *
 * Return 0 on success and -1 if the mft_entry is busy.
 */
static __inline__ int __remove_mft_entry_from_vol(mft_entry *m)
{
	if (m->m_count > 1 || MftEntryDirty(m) || m->nr_mapped_attrs)
		return -1;
	if (m->m_count) {
		/* Remove the entry from the list and clean up. */
		m->m_vol->nr_mft_entries--;
		list_del_init(&m->m_list);
		m->m_vol = NULL;
		m->m_count = 0;
	}
	return 0;
}

/**
 * __allocate_mft_entry - allocate a new mft_entry and initialize it
 *
 * Allocate a new mft_entry and initialize it. Return the new entry or NULL on
 * error with errno set to ENOMEM.
 */
static __inline__ mft_entry *__allocate_mft_entry(void)
{
	mft_entry *me = (mft_entry *)malloc(sizeof(mft_entry));
	if (!me)
		return NULL;
	memset(me, 0, sizeof(mft_entry));
	INIT_LIST_HEAD(&me->m_list);
	INIT_LIST_HEAD(&me->m_dirty_list);
	return me;
}

/**
 * __free_mft_entry - free an unused, unmapped mft_entry
 * @m:		mft_entry to free
 *
 * Check that @m is unused and unmapped. Then free() it.
 *
 * Return 0 on success and -1 if mft_entry is busy.
 */
static __inline__ int __free_mft_entry(mft_entry *m)
{
	if (m->m_count || MftEntryMapped(m))
		return -1;
	free(m);
	return 0;
}

/**
 * __ntfs_load_mft_entry - load mft record number @mref from ntfs_volume @v
 * @v:		ntfs_volume to load the mft record from
 * @mref:	mft reference of the mft record to load
 * @pos:	optinal position in @v at which to insert the loaded entry
 *
 * Allocate a new mft_entry, load the mft_record number MREF(@mref) from the
 * storage medium described by @v and insert the mft_entry in @v.
 *
 * If MSEQNO(mref) is not zero consistency check the sequence numbers. If not
 * equal return an error.
 *
 * Return a pointer to the new mft_entry or NULL on error.
 *
 * On error, errno is set to the error code. Defined error codes are:
 *
 *	ENOMEM	- Not enough memory to allocate necessary structure(s).
 * 	EEXIST	- The MREF(@mref) is already present in the cache!
 * 	ESPIPE	- Mft record @mref is outside the mft.
 * 	EIO	- Mft record is corrupt or I/O error.
 * 	Other codes returned by __ntfs_map_mft_entry().
 */
static __inline__ mft_entry* __ntfs_load_mft_entry(ntfs_volume *v,
						   const MFT_REF mref,
						   struct list_head *pos)
{
	int err;
	mft_entry *m;

	if (MREF(mref) >= v->number_of_mft_records) {
		errno = ESPIPE;
		return NULL;
	}
	/* Get a new mft_entry and initialize it. */
	m = __allocate_mft_entry();
	if (!m)
		return NULL;
	m->m_ref = mref;
	/* Add the new entry to the volume. */
	if (__insert_mft_entry_in_vol(v, m, pos) == -1) {
		errno = EEXIST;
		return NULL;
	}
	/* Map the new entry. */
	if ((err = __ntfs_map_mft_entry(m))) {
		/* No need to check error codes. These will succeed. */
		__remove_mft_entry_from_vol(m);
		if (test_and_clear_bit(ME_mapped, m->m_flags))
			free(m->m_rec);
		__free_mft_entry(m);
		errno = -err;
		return NULL;
	}
	return m;
}

/**
 * __ntfs_setup_mft_entry - initialize m_flags of base mft entry
 * @m:		mft entry to initialize
 *
 * Make sure the mft_entry @m is mapped, that it is not marked as a base mft
 * record yet, but that it is a base mft record and then set up @m->m_flags
 * to show the entry as being a base mft record and determine whether or not
 * an attribute list is present and reflect this in @m->m_flags.
 *
 * If any of the checks fail we return without any error reporting.
 */
static __inline__ void __ntfs_setup_mft_entry(mft_entry *m)
{
	attr_search_context ctx;
	
	if (!MftEntryMapped(m) || MftEntryIsBaseRecord(m))
		return;
	if (!m->m_rec->base_mft_record) {
		SetMftEntryIsBaseRecord(m);
		/* Check if we have an attribute list attribute present. */
		memset(&ctx, 0, sizeof(attr_search_context));
		ctx.mrec = m->m_rec;
		if (find_first_attr($ATTRIBUTE_LIST, NULL, 0, 0, NULL, 0,
								NULL, 0, &ctx))
			SetMftEntryAttrListPresent(m);
		return;
	}
	/* Do any extension mft record setup here. */
}

/**
 * __ntfs_load_file_entry - load mft record number @mref from ntfs_volume @v
 * @v:		ntfs_volume to load the mft record from
 * @mref:	mft reference of the mft record to load
 * @pos:	optinal position in @v at which to insert the loaded entry
 *
 * Allocate a new mft_entry, load the mft_record number MREF(@mref) from the
 * storage medium described by @v and insert the mft_entry in @v.
 * 
 * Check that loaded mft record has the magic FILE, that it is in use, and that
 * it has a matching sequence number (if MSEQNO(@mref) != 0). If either of these
 * fails, return -EIO.
 *
 * Finally, initialize the mft_entry's m_flags to reflect whether this is a base
 * mft record and whether an attribute list is present. 
 *
 * Return a pointer to the new mft_entry or NULL on error.
 *
 * On error, errno is set to the error code. Defined error codes are:
 *
 *	ENOMEM	- Not enough memory to allocate necessary structure(s).
 * 	EEXIST	- The MREF(@mref) is already present in the cache!
 * 	ENOENT	- Mft record @mref does not exist.
 * 	EIO	- I/O error or file record corrupt.
 * 	Other codes returned by __ntfs_map_mft_entry().
 */
static __inline__ mft_entry* __ntfs_load_file_entry(ntfs_volume *v,
						    const MFT_REF mref,
						    struct list_head *pos)
{
	mft_entry *m;
	MFT_RECORD *mrec;

	if (!ntfs_get_bit(v->mft_bitmap, MREF(mref))) {
		errno = ENOENT;
		return NULL;
	}
	m = __ntfs_load_mft_entry(v, mref, pos);
	if (!m)
		return NULL;
	mrec = m->m_rec;
	/* Check the new entry. */
	if (!is_file_record(mrec->magic) || 
	    !(mrec->flags & MFT_RECORD_IN_USE)) {
		errno = EIO;
		return NULL;
	}
	/* Check if this is a base or extension mft record, and setup the mft
	 * entry accordingly. */
	__ntfs_setup_mft_entry(m);
	return m;
}

/**
 * ntfs_map_file_entry - map mft record number @mref of ntfs_volume @v
 * @v:		ntfs_volume to get the mft record from
 * @mref:	mft reference of the mft record to map
 *
 * Search the volume's mft_entries list for @mref. If already present, return
 * it.
 *
 * If not, use __ntfs_load_file_entry() to allocate a new mft_entry, load the
 * mft_record number MREF(@mref) from the storage medium described by @v,
 * insert the mft_entry in @v, check that loaded mft record is ok, and
 * initialize the mft_entry's m_flags.
 *
 * Return a pointer to the mft_entry or NULL on error, with errno set to the
 * error code. Defined error codes are:
 *
 * 	EEXIST	- The MREF(@mref) is already in the cache but sequence numbers
 * 		  do not match.
 * 	Other codes returned by __ntfs_load_file_entry().
 */
mft_entry *ntfs_map_file_entry(ntfs_volume *v, const MFT_REF mref)
{
	mft_entry *m;
	int err = __lookup_mft_entry_in_vol_by_mref(v, mref, &m);
	switch (err) {
	case -1: 	/* In cache, but sequence number mismatch. */
		errno = EEXIST;
		return NULL;
	case 0:		/* Not in @v's cache, so load it. */
		if ((m = __ntfs_load_file_entry(v, mref, &m->m_list)))
			m->m_count++;
		break;
	case 1:		/* In cache already. */
		if ((err = ntfs_map_mft_entry(m)) < 0) {
			errno = -err;
			return NULL;
		}
		__ntfs_setup_mft_entry(m);
		break;
	}
	return m;
}

/**
 * __insert_base_mft_entry_in_file - insert a base mft entry in a ntfs_file
 * @f:		ntfs_file to insert the base mft_entry into
 * @m:		mft_entry to connect to @f
 *
 * Increment the use count of @m, initialize @f with @m. Finally, make @m
 * point to @f.
 *
 * Return 0 on success or -ERRNO on error. Defined error return values are:
 *
 *	-ENOMEM - Not enough memory to allocate necessary structure(s).
 */
static __inline__ int __insert_base_mft_entry_in_file(ntfs_file *f,
						      mft_entry *m)
{
	/* Make sure m doesn't disappear under us. */
	m->m_count++;
	/* Add m to f. */
	f->f_m_refs = (MFT_REF*)malloc(sizeof(MFT_REF));
	if (!f->f_m_refs)
		goto ret_merr_mcnt;
	f->f_m_entries = (mft_entry**)malloc(sizeof(mft_entry*));
	if (!f->f_m_entries) {
		free(f->f_m_refs);
		f->f_m_refs = NULL;
		goto ret_merr_mcnt;
	}
	f->f_m_refs[0] = m->m_ref;
	f->f_m_entries[0] = m;
	f->nr_m_refs = 1;
	/* Add f to m. */
	m->m_file = f;
	return 0;
ret_merr_mcnt:
	m->m_count--;
	return -errno;
}

/**
 * __remove_mft_entry_from_file - remove an mft entry from its file
 * @m:		mft_entry to remove from its file
 *
 * Removes the mft_entry @m from its file by dissociating @m from @m->m_file
 * and removing @m from the arrays in @m->m_file.
 *
 * Return 0 on success or -1 if the file or mft_entry was busy.
 *
 * For the function to succeed the file has to be unused (except for being in
 * its ntfs_volume). Also if @m is the base mft entry of its file, it cannot be
 * removed from the file if any extension mft entries are present in the file.
 */
static __inline__ int __remove_mft_entry_from_file(mft_entry *m)
{
	MFT_REF mref;
	void *t;
	int i;
	ntfs_file *f = m->m_file;
	if (!f)
		return 0;
	if (f->f_count > 1)
		return -1;
	if (MftEntryIsBaseRecord(m) && f->nr_m_refs > 1)
		return -1;
	/* Find the position of the mft entry in the file's arrays. */
	for (i = f->nr_m_refs - 1; i >= 0; i--) {
		if (MREF(m->m_ref) != MREF(f->f_m_refs[i]))
			continue;
		mref = f->f_m_refs[i];
		break;
	}
	/* Dissociate mft entry from file and unmap it. */
	m->m_file = NULL;
	m->m_count--;
	/* Remove mft entry from file. */
	f->nr_m_refs--;
	if (!i) {
		/* The entry is the base mft entry, hence no more entries. */
		free(f->f_m_refs);
		free(f->f_m_entries);
		return 0;
	}
	if (i < f->nr_m_refs) {
		/* The entry being removed is not the last entry, so have to
		 * move all following entries forward by one. */
		memmove(f->f_m_refs + i, f->f_m_refs + i + 1,
				(f->nr_m_refs - i) * sizeof(MFT_REF));
		memmove(f->f_m_entries + i, f->f_m_entries + i + 1,
				(f->nr_m_refs - i) * sizeof(mft_entry*));
	}
	/* Resize the arrays, ignoring errors, as no memory would be leaked. */
	t = realloc(f->f_m_refs, f->nr_m_refs * sizeof(MFT_REF));
	if (t)
		f->f_m_refs = t;
	t = realloc(f->f_m_entries, f->nr_m_refs * sizeof(mft_entry*));
	if (t)
		f->f_m_entries = t;
	return 0;
}

/**
 * __remove_all_mft_entries_from_file - remove all mft entries from a ntfs_file
 * @f:		ntfs_file to remove the all mft entries from
 *
 * For all mft entries associated with the file, dissociate each from the file,
 * starting with the last extension mft entry and going down to the base mft
 * entry.
 *
 * Return 0 on success or -1 if busy.
 */
static __inline__ int __remove_all_mft_entries_from_file(ntfs_file *f)
{
	int i;
	mft_entry *m;
	
	if (f->f_count > 1)
		return -1;
	/* Fast remove all mft entries from the file starting with last one. */
	for (i = f->nr_m_refs - 1; i >= 0; i--) {
		m = f->f_m_entries[i];
		if (m->m_count > 2) {
			f->nr_m_refs = i + 1;
			return -1;
		}
		/* Remove entry and unmap. */
		m->m_file = NULL;
		m->m_count--;
	}
	/* We removed all entries. Clean up. */
	f->nr_m_refs = 0;
	free(f->f_m_entries);
	free(f->f_m_refs);
	return 0;
}

/**
 * __insert_ntfs_file_in_volume - insert a ntfs_file in a ntfs_volume
 * @v:		ntfs_volume to insert @f into
 * @f:		file to insert into @v
 *
 * Insert @f into @v and increment the use count. This should only be called on
 * a file which has just been allocated. You have been warned!
 */
static __inline__ void __insert_ntfs_file_in_volume(ntfs_volume *v,
						    ntfs_file *f)
{
	f->f_count++;
	f->f_vol = v;
	v->nr_open_files++;
	list_add_tail(&f->f_list, &v->open_files);
}

/**
 * __remove_ntfs_file_from_volume - remove a ntfs_file from it's ntfs_volume
 * @f:		file to remove from it's volume
 *
 * Remove @f from @f->f_vol and decrement the use count. This should never be
 * called on a file which has references to mft_entries or is still open. You
 * have been warned!
 */
static __inline__ void __remove_ntfs_file_from_volume(ntfs_file *f)
{
	list_del(&f->f_list);
	f->f_vol->nr_open_files--;
	f->f_count--;
}

/**
 * __allocate_ntfs_file - allocate a new ntfs_file and initialize it
 *
 * Allocate a new ntfs_file and initialize it. Return the new file or NULL on
 * error with errno set to ENOMEM.
 */
static __inline__ ntfs_file *__allocate_ntfs_file(void)
{
	ntfs_file *f = (ntfs_file*)malloc(sizeof(ntfs_file));
	if (!f)
		return NULL;
	memset(f, 0, sizeof(ntfs_file));
	INIT_LIST_HEAD(&f->f_list);
	return f;
}

/**
 * __free_ntfs_file - free a closed ntfs_file
 * @f:		ntfs_file to free
 *
 * Check that file is closed and that it doesn't reference any mft_entries then
 * remove it from the volume if necessary. Finally free() it.
 *
 * Return 0 on success and -1 if the file is busy.
 */
static __inline__ int __free_ntfs_file(ntfs_file *f)
{
	if (f->f_count > 1 || f->nr_m_refs)
		return -1;
	if (f->f_count == 1)
		__remove_ntfs_file_from_volume(f);
	free(f);
	return 0;
}

/**
 * ntfs_open_by_mref - open an ntfs_file corresponding to base mft record @mref
 * @vol:	ntfs_volume @mref is located on
 * @mref:	base mft record number (and sequence number optional) to open
 *
 * Open the ntfs file (using file to mean any base mft record) given the mft
 * reference of the mft record (i.e. the inode number) on the ntfs volume @vol.
 *
 * If the mft_entry corresponding to @mref is not cached yet, load it from
 * storage medium. Otherwise obtain it from the cache directly.
 *
 * If the file is not open yet, allocate a new file and connect it with @vol
 * and the (now) mapped mft_entry. Finally, increase the use count and return
 * a pointer to the file.
 *
 * On error, return NULL with errno set to the error code. Possible codes are:
 *
 * 	ENOENT - Mft record number @mref is not present on volume or is not a
 * 		 base mft record.
 * 	EEXIST - The MREF(@mref) does exist but it has a different sequence
 * 		 number from MSEQNO(@mref). Something is corrupt somewhere! )-:
 * 	EMFILE - Too many open files already. Close some files or increase the
 * 		 maximum number of open files, then try again.
 * 	ENOMEM - Not enough memory to allocate necessary structure(s).
 * 	Any others (made positive) returned by __ntfs_load_mft_entry(),
 * 	__allocate_ntfs_file() or __insert_base_mft_entry_in_file().
 *
 * Note: The returned ntfs_file pointer should be treated completely opaquely
 * by the caller. Never use it for anything except as a parameter to functions
 * taking a ntfs_file pointer as a parameter. You have been warned.
 */
ntfs_file *ntfs_open_by_mref(ntfs_volume *vol, const MFT_REF mref)
{
	ntfs_file *f;
	mft_entry *m;
	int err;

#define RET_ERR(_errno, _label)		{ errno = _errno; \
					  goto _label; }

	/*
	 * Map the entry, i.e. load into cache if not there and increment
	 * use count. - We have to decrement again if aborting, if the file
	 * was already open and when closing the file.
	 */
	m = ntfs_map_file_entry(vol, mref);
	if (!m)
		return NULL;
	/* Is a file associated with this mft entry already? */
	if ((f = m->m_file)) {
		/* Only base mft records can be opened. */
		if (f->f_m_entries[0] != m) {
			fprintf(stderr, "ERROR Linux-NTFS: "
					"ntfs_open_by_mref(): openning "
					"extension mft record as file "
					"not allowed.\n");
			RET_ERR(ENOENT, ret_mcnt);
		}
		/* Open again. */
		f->f_count++;
		/* Release the superfluous mft_entry mapping. */
		m->m_count--;
		return f;
	}
	/*
	 * We now have @mref's mft_entry mapped (m) and we know the file isn't
	 * open yet.
	 */
	if (!MftEntryIsBaseRecord(m))
		RET_ERR(ENOENT, ret_mcnt);
	if (vol->nr_open_files >= vol->max_open_files)
		RET_ERR(EMFILE, ret_mcnt);
	if (!(f = __allocate_ntfs_file()))
		/* errno is set by __allocate_ntfs_file(). */
		goto ret_mcnt;
	__insert_ntfs_file_in_volume(vol, f);
	if ((err = __insert_base_mft_entry_in_file(f, m)) < 0) {
		/*
		 * No need to check error code. It _will_ succeed. This takes
		 * care of removing f from @vol, too.
		 */
		__free_ntfs_file(f);
		RET_ERR(-err, ret_mcnt);
	}
	/*
	 * We now have the file (f) in the @vol's open_files list and we have
	 * the base mft entry (m) corresponding to @mref connected with f.
	 * Just increment the useage count and we are done!
	 */
	f->f_count++;
	return f;
ret_mcnt:
	m->m_count--;
	return NULL;

#undef RET_ERR
}

/**
 * ntfs_close - close an open ntfs_file
 * @f:		ntfs_file to close
 *
 * Close a previously opened ntfs file. Return 0 on success or -1 if file is
 * busy.
 */
int ntfs_close(ntfs_file *f)
{
	ntfs_volume *v;

	if (--f->f_count > 1)
		return 0;
	/* Remove all mft_refs/_entries from @f and, if no errors, remove @f
	 * from its volume and kill the file. */
	if (!__remove_all_mft_entries_from_file(f))
		return __free_ntfs_file(f);
	return -1;
}

