#include <stdlib.h>
#include <ctype.h>
#include "internal.h"
#include "samba.m"
#include "keyword.h"
#include <module_apis/status_api.h>
#include <string.h>

extern SAMBA_HELP_FILE help_samba;
static SAMBA_HELP_FILE help_share ("share");
static SAMBA_HELP_FILE help_homes ("homes");
static SAMBA_HELP_FILE help_printers ("printers");
static SAMBA_HELP_FILE help_global ("global");
static SAMBA_HELP_FILE help_netlogon ("netlogon");

// Dialog IDs for the virtual registry
static char ID_MAIN[]="main";
static char ID_SHARE[]="share";
static char ID_LOGON[]="logon";

struct SMB_FIELD_LOOKUP{
	const char **keywords;
	const char **titles;
};
static int samedit_lookup (
	const char *val,
	const char *lk[])
{
	int ret = 0;
	// Remove extra spaces in the value so the lookup works all the time
	char tmp[strlen(val)+1];
	char *pt = tmp;
	bool last_was_blank = false;
	while (*val != '\0'){
		if (isspace(*val)){
			if (!last_was_blank){
				*pt++ = ' ';
				last_was_blank=true;
			}
		}else{
			*pt++ = *val;
			last_was_blank=false;
		}
		val++;
	}
	*pt = '\0';
	for (int i=0; lk[i] != NULL; i++){
		if (strcmp(tmp,lk[i])==0){
			ret = i;
			break;
		}	   	
	}
	return ret;
}

PRIVATE void SMB_SHARE::setdia (
	DIALOG &dia, 
	SMB_FIELD tbf[],
	int nbf,
	SMB_FIELD_VAL tbv[])
{
	for (int i=0; i<nbf; i++){
		SMB_FIELD *pt = tbf + i;
		SMB_FIELD_TYPE type = pt->type;
		const char *title = pt->title;
		if (type == SMB_BOOK){
			dia.newf_title (title,1,"",title);
		}else{
			const char *val = getval (pt->key1,pt->key2,pt->key3);
			if (type == SMB_STR){
				tbv[i].str.setfrom (val);
				dia.newf_str (title,tbv[i].str);
			}else if (type == SMB_NUM){
				tbv[i].num = atoi(val);
				dia.newf_num (title,tbv[i].num);
			}else if (type == SMB_CHK){
				tbv[i].chk = stricmp(val,K_YES)==0;
				dia.newf_chk ("",tbv[i].chk,title);
			}else if (type == SMB_CHKON){
				// Like SMB_CHK, except that the default is yes
				tbv[i].chk = stricmp(val,K_YES)==0 || val[0] == '\0';
				dia.newf_chk ("",tbv[i].chk,title);
			}else if (type == SMB_ENUMH){
				SMB_FIELD_LOOKUP *lk = (SMB_FIELD_LOOKUP*)pt->extra;
				tbv[i].enm = samedit_lookup (val,lk->keywords);
				dia.newf_chkm (title,tbv[i].enm,lk->titles);
			}else if (type == SMB_COMBO){
				const char **opts = (const char **)pt->extra;
				tbv[i].str.setfrom (val);
				FIELD_COMBO *comb = dia.newf_combo (title,tbv[i].str);
				for (int i=0; opts[i] != NULL; i++){
					comb->addopt (opts[i]);
				}
			}
		}
	}
}

PRIVATE void SMB_SHARE::savedia(
	SMB_FIELD tbf[],
	int nbf,
	SMB_FIELD_VAL tbv[])
{
	for (int i=0; i<nbf; i++){
		SMB_FIELD *pt = tbf + i;
		SMB_FIELD_TYPE type = pt->type;
		const char *key1 = pt->key1;
		const char *key2 = pt->key2;
		const char *key3 = pt->key3;
		if (type == SMB_STR || type == SMB_COMBO){
			setval (key1,key2,key3,tbv[i].str);
		}else if (type == SMB_NUM){
			setval (key1,key2,key3,tbv[i].num);
		}else if (type == SMB_CHK
			|| type == SMB_CHKON){
			setval (key1,key2,key3,tbv[i].chk ? K_YES : K_NO);
		}else if (type == SMB_ENUMH){
			SMB_FIELD_LOOKUP *lk = (SMB_FIELD_LOOKUP*)pt->extra;
			const char *val = lk->keywords[tbv[i].enm];
			setval (key1,key2,key3,val);
		}
	}
}		

PUBLIC int SMB_SHARE::editgen(
	DIALOG &dia,
	SMB_FIELD tbf[],
	int nbf,
	bool may_del,		// May delete this share
	SAMBA_HELP_FILE &help_file)
{
	int ret = -1;
	SMB_FIELD_VAL tbv[nbf];
	setdia (dia,tbf,nbf,tbv);
	int nof = 0;
	while (1){
		int menubut = MENUBUT_ACCEPT|MENUBUT_CANCEL;
		if (may_del) menubut |= MENUBUT_DEL;
		MENU_STATUS code = dia.edit (MSG_U(T_SHARE,"Share setup")
			,MSG_U(I_SHARE,"You are allowed to enter configurations\n"
				"for one disk share\n")
			,help_file
			,nof,menubut);
		if (code == MENU_ESCAPE || code == MENU_CANCEL){
			break;
		}else if (code == MENU_DEL){
			if (xconf_delok()){
				ret = 1;
				break;
			}
		}else{
			savedia (tbf,nbf,tbv);
			ret = 0;
			break;
		}
	}
	if (ret != 0) dia.restore();
	return ret;
}

/*
	Edit a standard share (not homes, global, netlogon or printers)
*/
PUBLIC int SMB_SHARE::edit()
{
	DIALOG dia;
	dia.set_registry_id (ID_SHARE);
	dia.newf_str (MSG_U(F_SHARENAME,"Share name"),name);

	static SMB_FIELD tbf[]={
		{K_COMMENT,NULL,NULL,SMB_STR,NULL,MSG_U(F_COMMENT,"Comment/description")},

		{NULL,NULL,NULL,SMB_BOOK,NULL,MSG_U(T_BASE,"Base info")},
		{K_AVAILABLE,NULL,NULL,SMB_CHKON,NULL,MSG_U(F_ENABLE,"This share is enabled")},
		{K_BROWSEABLE,NULL,NULL,SMB_CHK,NULL,MSG_U(F_BROWSEABLE,"Browsable")},
		{K_COPY,NULL,NULL,SMB_STR,NULL,MSG_U(F_INHERIT,"Inherit settings from share")},
		{K_PATH,NULL,NULL,SMB_STR,NULL,MSG_U(F_PATH,"Directory to export")},

		
		{NULL,NULL,NULL,SMB_BOOK,NULL,MSG_U(T_ACCESS,"Access")},
		{K_PUBLIC,NULL,NULL,SMB_CHK,NULL,MSG_U(F_PUBLIC,"Public access")},
		{K_GUEST,K_ONLY,NULL,SMB_CHK,NULL,MSG_U(F_GUESTONLY,"Guest access only")},
		{K_WRITABLE,NULL,NULL,SMB_CHK,NULL,MSG_U(F_WRITABLE,"Writable")},
		{K_ALLOW,K_HOSTS,NULL,SMB_STR,NULL,MSG_U(F_ALLOWHOSTS,"Allow hosts")},
		{K_DENY,K_HOSTS,NULL,SMB_STR,NULL,MSG_U(F_DENYHOSTS,"Deny hosts")},

		{NULL,NULL,NULL,SMB_BOOK,NULL,MSG_U(T_USERS,"Users")},
		{K_USER,NULL,NULL,SMB_STR,NULL,MSG_U(F_USERLIST,"User list")},
		{K_ONLY,K_USER,NULL,SMB_CHK,NULL,MSG_U(F_ONLYUSER,"Only users may connect")},
		{K_ADMIN,K_USERS,NULL,SMB_STR,NULL,MSG_U(F_ADMINUSERS,"Admin users")},
		{K_WRITE,K_LIST,NULL,SMB_STR,NULL,MSG_U(F_WRITELIST,"Write list")},
		{K_VALID,K_USERS,NULL,SMB_STR,NULL,MSG_U(F_VALIDUSERS,"Valid users")},
		{K_INVALID,K_USERS,NULL,SMB_STR,NULL,MSG_U(F_IVLDUSERS,"Invalid users")},
		{K_READ,K_LIST,NULL,SMB_STR,NULL,MSG_U(F_READLIST,"Read only user list")},

		{NULL,NULL,NULL,SMB_BOOK,NULL,MSG_U(T_SCRIPTS,"Scripts")},
		{K_PREEXEC,NULL,NULL,SMB_STR,NULL,MSG_U(F_PREEXEC,"Setup command")},
		{K_PREEXEC,K_CLOSE,NULL,SMB_CHK,NULL,MSG_U(F_EXECCLOSE,"Close on failure")},
		{K_ROOT,K_PREEXEC,NULL,SMB_STR,NULL,MSG_U(F_ROOTPREEXEC,"Setup command (root)")},
		{K_ROOT,K_PREEXEC,K_CLOSE,SMB_CHK,NULL,MSG_U(F_ROOTEXECCLOSE,"Close on failure")},
		{K_POSTEXEC,NULL,NULL,SMB_STR,NULL,MSG_U(F_POSTEXEC,"Cleanup command")},
		{K_ROOT,K_POSTEXEC,NULL,SMB_STR,NULL,MSG_U(F_ROOTPOSTEXEC,"Cleanup command (root)")},
		{K_MAGIC,K_SCRIPT,NULL,SMB_STR,NULL,MSG_U(F_MAGICSCRIPT,"Magic script")},

		{NULL,NULL,NULL,SMB_BOOK,NULL,MSG_U(T_FEATURES,"Features")},
		{K_FORCE,K_USER,NULL,SMB_STR,NULL,MSG_U(F_FORCEUSER,"Force user")},
		{K_FORCE,K_GROUP,NULL,SMB_STR,NULL,MSG_U(F_FORCEGROUP,"Force group")},
		{K_CREATE,K_MASK,NULL,SMB_STR,NULL,MSG_U(F_CREATEMASK,"File creation mode")},
		{K_DIRECTORY,K_MASK,NULL,SMB_STR,NULL,MSG_U(F_DIRMASK,"Directory creation mode")},
		{K_DONT,K_DESCEND,NULL,SMB_STR,NULL,MSG_U(F_DONTDESCEND,"Don't descent")},
		{K_GUEST,K_ACCOUNT,NULL,SMB_STR,NULL,MSG_U(F_GUESTACCOUNTSH,"Guest account (this share)")},
		{K_MAGIC,K_OUTPUT,NULL,SMB_STR,NULL,MSG_U(F_MAGICOUTPUT,"Magic output")},
		{K_MAX,K_CONNECTIONS,NULL,SMB_STR,NULL,MSG_U(F_MAXCONNECT,"Max. connections")},
	};
	return editgen (dia,tbf,(int)(sizeof(tbf)/sizeof(tbf[0])),true,help_share);
}

PUBLIC int SMB_SHARE::editglobal(SMB_CONF *samba)
{
	DIALOG dia;
	dia.set_registry_id (ID_MAIN);

	dia.newf_title (MSG_U(T_BASIC,"Base config"),1,"",MSG_R(T_BASIC));
	static const char *tbsmbpass[]={
		MSG_U(I_NOTMANAGED,"Not managed"),
		MSG_U(I_MNGPASSWD,"Acct. & passwords"),
		MSG_U(I_MNGRECORDS,"Acct. only"),
		NULL,
	};
	dia.newf_chkm (MSG_U(F_MNGSMBPASS,"SMB account management")
		,samba->syncpass,tbsmbpass);

	static const char *tbsecurity_keywords[]={
		K_USER,
		K_SERVER,
		K_DOMAIN,
		K_SHARE,
		NULL,
	};
	static const char *tbsecurity_titles[]={
		MSG_U(I_SECURITYUSER,"User"),
		MSG_U(I_SECURITYSERVER,"Server"),
		MSG_U(I_SECURITYDOMAIN,"Domain"),
		MSG_U(I_SECURITYSHARE,"Share"),
		NULL
	};
	static SMB_FIELD_LOOKUP tbsecurity={
		tbsecurity_keywords,
		tbsecurity_titles
	};
	static const char *tbmaptoguest_keywords[]={
		K_NEVER,
		K_BADUSER,
		K_BADPASSWORD
	};
	static const char *tbmaptoguest_titles[]={
		MSG_U(I_NEVER,"Never"),
		MSG_U(I_BADUSER,"Bad user"),
		MSG_U(I_BADPASSWORD,"Bad password"),
		NULL
	};
	static SMB_FIELD_LOOKUP tbmaptoguest={
		tbmaptoguest_keywords,
		tbmaptoguest_titles
	};

	static const char *adduserscripts[]={
		"/usr/sbin/adduser -s /bin/false -g popusers -c \"smb account %u\" %u",
		NULL
	};
	static const char *deluserscripts[]={
		"/usr/sbin/deluser %u",
		NULL
	};

	static char *nameorder[]={
		"lmhosts, host, wins, bcast",
		"lmhosts, wins, bcast",
		"lmhosts, bcast",
		"lmhosts",
		NULL
	};	

	static char *winpopupcommand[]={
		"/bin/mail -s 'Message from %f at %m' root < %s; rm %s",
		NULL
	};
	
	static SMB_FIELD tbf[]={
		{K_UNIX,K_PASSWORD,K_SYNC,SMB_CHK,NULL,MSG_U(F_SYNCPASSFROMSMB,"Synchronise Linux from SMB passwords")},
		{K_SERVER,K_STRING,NULL,SMB_STR,NULL,MSG_U(F_SERVERDESC,"Server description")},
		{K_WORKGROUP,NULL,NULL,SMB_STR,NULL,MSG_U(F_WORKGROUP,"Work group")},
		{K_NETBIOS,K_NAME,NULL,SMB_STR,NULL,MSG_U(F_NETBIOSNAME,"Netbios name (opt)")},
		{K_NETBIOS,K_ALIASES,NULL,SMB_STR,NULL,MSG_U(F_NETBIOSALIASES,"Netbios aliases (opt)")},

		{NULL,NULL,NULL,SMB_BOOK,NULL,MSG_U(T_PASSWORDS,"Passwords")},
		{K_ENCRYPT,K_PASSWORDS,NULL,SMB_CHK,NULL,MSG_U(F_ENGRYPTPASS,"Encrypted password required")},
		{K_SECURITY,NULL,NULL,SMB_ENUMH,(void*)&tbsecurity
			,MSG_U(F_SECURITY,"Authentication mode")},
		{K_MAP,K_TO,K_GUEST,SMB_ENUMH,(void*)&tbmaptoguest
			,MSG_U(F_MAPGUEST,"Map to guest")},
		{K_PASSWORD,K_SERVER,NULL,SMB_STR,NULL,MSG_U(F_PASSSERVER,"Password server")},
		{K_PASSWORD,K_LEVEL,NULL,SMB_NUM,NULL,MSG_U(F_PASSLEVEL,"Password level")},
		{K_PASSWD,K_PROGRAM,NULL,SMB_STR,NULL,MSG_U(F_PASSPROG,"Passwd program")},
		{K_NULL,K_PASSWORDS,NULL,SMB_CHK,NULL,MSG_U(F_NULLPASSWORDS,"Allow null passwords account")},

		{NULL,NULL,NULL,SMB_BOOK,NULL,MSG_R(T_ACCESS)},
		{K_ALLOW,K_HOSTS,NULL,SMB_STR,NULL,MSG_R(F_ALLOWHOSTS)},
		{K_DENY,K_HOSTS,NULL,SMB_STR,NULL,MSG_R(F_DENYHOSTS)},

		{NULL,NULL,NULL,SMB_BOOK,NULL,MSG_U(T_NETWORKING,"Networking")},
		{K_OS,K_LEVEL,NULL,SMB_NUM,NULL,MSG_U(F_OSLEVEL,"OS level")},
		{K_PREFERRED,K_MASTER,NULL,SMB_CHK,NULL,MSG_U(F_PREFMASTER,"Preferred master")},
		{K_DOMAIN,K_MASTER,NULL,SMB_CHK,NULL,MSG_U(F_DOMMASTER,"Domain master")},
		{K_REMOTE,K_ANNOUNCE,NULL,SMB_STR,NULL,MSG_U(F_REMANNOUNCE,"Remote announce")},
		{K_REMOTE,K_BROWSE,K_SYNC,SMB_STR,NULL,MSG_U(F_BROWSESYNC,"Sync. remote browsers")},
		{K_WINS,K_SUPPORT,NULL,SMB_CHK,NULL,MSG_U(F_WINSSUPPORT,"Enable samba as a WINS server")},
		{K_WINS,K_SERVER,NULL,SMB_STR,NULL,MSG_U(F_WINSSERVER,"WINS server")},
		{K_INTERFACES,NULL,NULL,SMB_STR,NULL,MSG_U(F_INTERFACES,"Interfaces")},
		{K_NAME,K_RESOLVE,K_ORDER,SMB_COMBO,(void*)nameorder
			,MSG_U(F_RESOLVEORDER,"Name resolve order")},

		{NULL,NULL,NULL,SMB_BOOK,NULL,MSG_U(T_AUTOCREATE,"Auto-accounts")},
		{K_ADD,K_USER,K_SCRIPT,SMB_COMBO,(void*)adduserscripts
			,MSG_U(F_ADDUSERSCRIPT,"Add user script")},
		{K_DELETE,K_USER,K_SCRIPT,SMB_COMBO,(void*)deluserscripts
			,MSG_U(F_DELUSERSCRIPT,"Delete user script")},


		{NULL,NULL,NULL,SMB_BOOK,NULL,MSG_R(T_FEATURES)},
		{K_GUEST,K_ACCOUNT,NULL,SMB_STR,NULL,MSG_U(F_GUESTACCOUNT,"Guest account")},
		{K_DEAD,K_TIME,NULL,SMB_NUM,NULL,MSG_U(F_DEADTIME,"Dead time")},
		{K_DEBUG,K_LEVEL,NULL,SMB_NUM,NULL,MSG_U(F_DBGLEVEL,"Debug level")},
		{K_DEFAULT,K_SERVICE,NULL,SMB_STR,NULL,MSG_U(F_DEFSERVICE,"Default service")},
		{K_LOAD,K_PRINTERS,NULL,SMB_CHKON,NULL,MSG_U(F_LOADPRINTERS,"Show all available printers")},
		{K_MESSAGE,K_COMMAND,NULL,SMB_COMBO,(void*)winpopupcommand,
	 	 MSG_U(F_MSGCOMMAND,"WinPopup command")},
		{K_STATUS,NULL,NULL,SMB_CHKON,NULL,MSG_U(F_STATUSFILE,"Update the status database")},

	};
	int ret = editgen (dia,tbf,(int)(sizeof(tbf)/sizeof(tbf[0])),false,help_global);
	if (ret == 0) samba->write();
	return ret;
}

PUBLIC int SMB_SHARE::editnetlogon(SMB_SHARE *glob, SMB_CONF *samba)
{
	/* #Specification: netlogon / logon fields
		smb.conf requires the various logon information to be stored
		in the global section. It seems more appropriate to present this
		along the netlogon share definition. This is why we play
		some magic. We move this information from the global section
		to the netlogon section, edit it and move it back to the
		global section later.
	*/
	static const char *tbmove[][2]={
		{K_LOGON,K_SCRIPT},
		{K_LOGON,K_PATH},
		{K_LOGON,K_DRIVE},
		{K_LOGON,K_HOME},
		{K_DOMAIN,K_LOGONS},
	};
	const int NBGLOBAL=sizeof(tbmove)/sizeof(tbmove[0]);
	if (glob != NULL){
		for (int i=0; i<NBGLOBAL; i++){
			const char *val = glob->getval (tbmove[i][0],tbmove[i][1],NULL);
			setval (tbmove[i][0],tbmove[i][1],NULL,val);
		}
	}
	DIALOG dia;
	dia.set_registry_id (ID_LOGON);
	static SMB_FIELD tbf[]={
		{K_COMMENT,NULL,NULL,SMB_STR
				,NULL,MSG_U(F_NETLOGONCOMMENT,"netlogon share title")},
		{K_AVAILABLE,NULL,NULL,SMB_CHKON,NULL,MSG_R(F_ENABLE)},
		{K_PATH,NULL,NULL,SMB_STR,NULL,MSG_R(F_PATH)},

		{NULL,NULL,NULL,SMB_BOOK,NULL,MSG_U(T_LOGON,"Logon")},
		{K_DOMAIN,K_LOGONS,NULL,SMB_CHK,NULL,MSG_U(F_DOMLOGONS,"Workgroup Logon server")},
		{K_LOGON,K_SCRIPT,NULL,SMB_STR,NULL,MSG_U(F_LOGONSCRIPT,"Logon script")},
		{K_LOGON,K_PATH,NULL,SMB_STR,NULL,MSG_U(F_LOGONPATH,"Logon path")},
		{K_LOGON,K_DRIVE,NULL,SMB_STR,NULL,MSG_U(F_LOGONDRIVE,"Logon drive")},
		{K_LOGON,K_HOME,NULL,SMB_STR,NULL,MSG_U(F_LOGONHOME,"Logon home")},

		{NULL,NULL,NULL,SMB_BOOK,NULL,MSG_R(T_SCRIPTS)},
		{K_PREEXEC,NULL,NULL,SMB_STR,NULL,MSG_R(F_PREEXEC)},
		{K_PREEXEC,K_CLOSE,NULL,SMB_CHK,NULL,MSG_R(F_EXECCLOSE)},
		{K_ROOT,K_PREEXEC,NULL,SMB_STR,NULL,MSG_R(F_ROOTPREEXEC)},
		{K_ROOT,K_PREEXEC,K_CLOSE,SMB_CHK,NULL,MSG_R(F_ROOTEXECCLOSE)},
		{K_POSTEXEC,NULL,NULL,SMB_STR,NULL,MSG_R(F_POSTEXEC)},
		{K_ROOT,K_POSTEXEC,NULL,SMB_STR,NULL,MSG_R(F_ROOTPOSTEXEC)},
	};
	int ret = editgen (dia,tbf,(int)(sizeof(tbf)/sizeof(tbf[0])),false,help_netlogon);
	if (ret == 0){
		for (int i=0; i<NBGLOBAL; i++){
			const char *val = getval (tbmove[i][0],tbmove[i][1],NULL);
			if (glob != NULL) glob->setval (tbmove[i][0],tbmove[i][1],NULL,val);
			setval (tbmove[i][0],tbmove[i][1],NULL,"");
		}
		samba->write();
	}
	return ret;
}

PUBLIC int SMB_SHARE::edithomes(SMB_CONF *samba)
{
	DIALOG dia;
	static SMB_FIELD tbf[]={
		{K_COMMENT,NULL,NULL,SMB_STR,NULL,MSG_R(F_COMMENT)},
		{K_AVAILABLE,NULL,NULL,SMB_CHK,NULL,MSG_R(F_ENABLE)},
		{K_BROWSEABLE,NULL,NULL,SMB_CHK,NULL,MSG_R(F_BROWSEABLE)},

		{NULL,NULL,NULL,SMB_BOOK,NULL,MSG_R(T_ACCESS)},
		{K_PUBLIC,NULL,NULL,SMB_CHK,NULL,MSG_R(F_PUBLIC)},
		{K_WRITABLE,NULL,NULL,SMB_CHK,NULL,MSG_R(F_WRITABLE)},
		{K_ALLOW,K_HOSTS,NULL,SMB_STR,NULL,MSG_R(F_ALLOWHOSTS)},
		{K_DENY,K_HOSTS,NULL,SMB_STR,NULL,MSG_R(F_DENYHOSTS)},


		{NULL,NULL,NULL,SMB_BOOK,NULL,MSG_R(T_USERS)},
		{K_USER,NULL,NULL,SMB_STR,NULL,MSG_R(F_USERLIST)},
		{K_ONLY,K_USER,NULL,SMB_CHK,NULL,MSG_R(F_ONLYUSER)},
		{K_WRITE,K_LIST,NULL,SMB_STR,NULL,MSG_R(F_WRITELIST)},
		{K_VALID,K_USERS,NULL,SMB_STR,NULL,MSG_R(F_VALIDUSERS)},
		{K_INVALID,K_USERS,NULL,SMB_STR,NULL,MSG_R(F_IVLDUSERS)},
		{K_MAX,K_CONNECTIONS,NULL,SMB_STR,NULL,MSG_R(F_MAXCONNECT)},
		{K_READ,K_LIST,NULL,SMB_STR,NULL,MSG_R(F_READLIST)},

		{NULL,NULL,NULL,SMB_BOOK,NULL,MSG_R(T_SCRIPTS)},
		{K_PREEXEC,NULL,NULL,SMB_STR,NULL,MSG_R(F_PREEXEC)},
		{K_PREEXEC,K_CLOSE,NULL,SMB_CHK,NULL,MSG_R(F_EXECCLOSE)},
		{K_ROOT,K_PREEXEC,NULL,SMB_STR,NULL,MSG_R(F_ROOTPREEXEC)},
		{K_ROOT,K_PREEXEC,K_CLOSE,SMB_CHK,NULL,MSG_R(F_ROOTEXECCLOSE)},
		{K_POSTEXEC,NULL,NULL,SMB_STR,NULL,MSG_R(F_POSTEXEC)},
		{K_ROOT,K_POSTEXEC,NULL,SMB_STR,NULL,MSG_R(F_ROOTPOSTEXEC)},

		{NULL,NULL,NULL,SMB_BOOK,NULL,MSG_R(T_FEATURES)},
		{K_CREATE,K_MASK,NULL,SMB_STR,NULL,MSG_R(F_CREATEMASK)},
		{K_DIRECTORY,K_MASK,NULL,SMB_STR,NULL,MSG_R(F_DIRMASK)},

	};
	int ret = editgen (dia,tbf,(int)(sizeof(tbf)/sizeof(tbf[0])),false,help_homes);
	if (ret == 0) samba->write();
	return ret;
}

PUBLIC int SMB_SHARE::editprinters(SMB_CONF *samba)
{
	DIALOG dia;
	static SMB_FIELD tbf[]={
		{K_COMMENT,NULL,NULL,SMB_STR,NULL,MSG_R(F_COMMENT)},
		{K_AVAILABLE,NULL,NULL,SMB_CHK,NULL,MSG_R(F_ENABLE)},
		{K_PRINTABLE,NULL,NULL,SMB_CHKON,NULL,MSG_U(F_PRINTABLE,"Share is printable")},
		{K_BROWSEABLE,NULL,NULL,SMB_CHKON,NULL,MSG_R(F_BROWSEABLE)},
		{K_PUBLIC,NULL,NULL,SMB_CHK,NULL,MSG_R(F_PUBLIC)},
	};
	int ret = editgen (dia,tbf,(int)(sizeof(tbf)/sizeof(tbf[0])),false,help_printers);
	if (ret == 0) samba->write();
	return 0;
}

PUBLIC int SMB_SHARES::write()
{
	return samba->write();
}

static int cmp_by_name (const ARRAY_OBJ *o1, const ARRAY_OBJ *o2)
{
	SMB_SHARE *s1 = (SMB_SHARE*)o1;
	SMB_SHARE *s2 = (SMB_SHARE*)o2;
	return s1->name.cmp(s2->name);
}

PUBLIC int SMB_CONF::editshares()
{
	int ret = -1;
	int nof = 0;
	DIALOG_LISTE dia;
	dia.newf_head ("",MSG_U(H_SHARES,"Share name\tDescription"));
	dia.addwhat (MSG_U(I_NEWSHARE,"Select [Add] to add a new disk share"));
	while (1){
		int n = shares.getnb();
		SMB_SHARES sorted (this);
		sorted.neverdelete();	// We present the share sorted, but
								// keep the original ordering in smb.conf
		for (int i=0; i<n; i++){
			SMB_SHARE *d = shares.getitem(i);
			const char *name = d->name.get();
			if (strcmp(name,K_GLOBAL)!=0
				&& strcmp(name,K_PRINTERS)!=0
				&& strcmp(name,K_NETLOGON)!=0
				&& strcmp(name,K_HOMES)!=0){
				sorted.add (d);
			}
		}
		sorted.sort (cmp_by_name);
		for (int i=0; i<sorted.getnb(); i++){
			SMB_SHARE *d = sorted.getitem(i);
			const char *name = d->name.get();
			dia.set_menuitem(i,name,d->getval(K_COMMENT));
		}
		dia.remove_last (sorted.getnb()+1);
		MENU_STATUS code = dia.editmenu (MSG_U(T_SHARES,"Disk shares")
			,MSG_U(I_SHARES
				,"You can define multiple independant entry\n"
				 "points in your file server")
			,help_share
			,nof,MENUBUT_ADD);
		if (code == MENU_ESCAPE || code == MENU_QUIT){
			break;
		}else if (code == MENU_ADD){
			SMB_SHARE *d = new SMB_SHARE;
			ret = shares.editone (d);
		}else if (nof < sorted.getnb() && nof >= 0){
			ret = shares.editone (sorted.getitem(nof));
		}
	}
	return ret;
}

PUBLIC void SMB_CONF::editdefaults()
{
	SMB_SHARE *d = shares.getitem(K_GLOBAL);
	if (d != NULL) d->editglobal(this);
}
PUBLIC void SMB_CONF::edithomes()
{
	SMB_SHARE *d = shares.getitem(K_HOMES);
	if (d != NULL) d->edithomes(this);
}
PUBLIC void SMB_CONF::editprinters()
{
	SMB_SHARE *d = shares.getitem(K_PRINTERS);
	if (d != NULL) d->editprinters(this);
}

PUBLIC void SMB_CONF::editnetlogon()
{
	bool was_created = false;
	SMB_SHARE *d = shares.getitem(K_NETLOGON);
	if (d == NULL){
		d = new SMB_SHARE ("","netlogon");
		shares.add (d);
		was_created = true;
	}
	if (d != NULL){
		SMB_SHARE *glob = shares.getitem(K_GLOBAL);
		if (d->editnetlogon(glob,this) != 0 && was_created) shares.remove_del (d);
	}
}

PUBLIC void SMB_CONF::edit()
{
	static const char *m_def = MSG_U(M_DEFAULT,"Defaults");
	static const char *m_homes = MSG_U(M_HOMES,"Default setup for users's home");
	static const char *m_printers = MSG_U(M_PRINTERS,"Default setup for printers");
	static const char *m_netlogon = MSG_U(M_NETLOGON,"Netlogon setup");
	static const char *m_shares = MSG_U(M_SHARES,"Disk shares");
	static const char *m_status = MSG_R(M_SAMBASTATUS);
	static const char *menuopt[]={
		"",		m_def,
		"",		m_homes,
		"",		m_printers,
		"",		m_netlogon,
		"",		m_shares,
		NULL,
	};
	
	DIALOG_MENU dia;
	dia.new_menuitems (menuopt);
	if (status_api_available("samba")){
		static const char *menuopt2[]={
			"",		m_status,
			NULL,
		};
		dia.new_menuitems (menuopt2);
	};
	int choice=0;
	while (1){
		MENU_STATUS code = dia.editmenu (MSG_U(T_SAMBA,"Samba administration")
			,MSG_U(I_SAMBA,"This menu allows to configure the Samba SMB file server")
			,help_samba
			,choice,0);
		if (code == MENU_QUIT || code == MENU_ESCAPE){
			break;
		}else if (netconf_mainaccess()){
			const char *key = dia.getmenustr(choice);
			if (key == m_def){
				editdefaults();
			}else if (key == m_homes){
				edithomes();
			}else if (key == m_printers){
				editprinters();
			}else if (key == m_shares){
				editshares();
			}else if (key == m_netlogon){
				editnetlogon();
			}else if (key == m_status){
				samba_status();
			}
		}
	}	
}

void samba_edit ()
{
	CONTEXT_LOCK lk ("samba");
	if (lk.isok()){
		SMB_CONF smb;
		smb.edit();
		if (dialog_mode != DIALOG_TREE) smb.sanitycheck();
	}
}

static void samedit_global()
{
	SMB_CONF smb;
	smb.editdefaults();
}
static void samedit_netlogon()
{
	SMB_CONF smb;
	smb.editnetlogon();
}

static void samedit_editshare(const char *share, bool setting)
{
	SMB_CONF smb;
	SMB_SHARE *d = smb.shares.getitem(share);
	if (d != NULL){
		smb.shares.editone(d);
	}else if (setting){
		SMB_SHARE *d = new SMB_SHARE;
		smb.shares.editone(d);
	}
}

static void samedit_listshare(SSTRINGS &tb)
{
	SMB_CONF smb;
	for (int i=0; i<smb.shares.getnb(); i++){
		SMB_SHARE *d = smb.shares.getitem(i);
		const char *name = d->name.get();
		if (strcmp(name,K_GLOBAL)!=0
			&& strcmp(name,K_PRINTERS)!=0
			&& strcmp(name,K_NETLOGON)!=0
			&& strcmp(name,K_HOMES)!=0){
			tb.add (new SSTRING(name));
		}
	}
}

#include <modregister.h>

static PUBLISH_VARIABLES_MSG samba_var_list1[]={
	{"mngsmbpasswd",P_MSG_R(F_MNGSMBPASS)},
	{"syncpasswds",P_MSG_R(F_SYNCPASSFROMSMB)},
	{"servdesc",P_MSG_R(F_SERVERDESC)},
	{"workgroup",P_MSG_R(F_WORKGROUP)},
	{"netbiosname",P_MSG_R(F_NETBIOSNAME)},
	{"netbiosaliases",P_MSG_R(F_NETBIOSALIASES)},
	{"encryptpasswd",P_MSG_R(F_ENGRYPTPASS)},
	{"security",P_MSG_R(F_SECURITY)},
	{"maptoguest",P_MSG_R(F_MAPGUEST)},
	{"passwdserver",P_MSG_R(F_PASSSERVER)},
	{"passwdlevel",P_MSG_R(F_PASSLEVEL)},
	{"passwdprogram",P_MSG_R(F_PASSPROG)},
	{"nullpasswds",P_MSG_R(F_NULLPASSWORDS)},
	{"allowhosts",P_MSG_R(F_ALLOWHOSTS)},
	{"denyhosts",P_MSG_R(F_DENYHOSTS)},
	{"oslevel",P_MSG_R(F_OSLEVEL)},
	{"preferedmaster",P_MSG_R(F_PREFMASTER)},
	{"domainmaster",P_MSG_R(F_DOMMASTER)},
	{"remoteannounce",P_MSG_R(F_REMANNOUNCE)},
	{"remotebrowsesync",P_MSG_R(F_BROWSESYNC)},
	{"winssupport",P_MSG_R(F_WINSSUPPORT)},
	{"winsserver",P_MSG_R(F_WINSSERVER)},
	{"interfaces",P_MSG_R(F_INTERFACES)},
	{"resolveorder",P_MSG_R(F_RESOLVEORDER)},
	{"adduserscript",P_MSG_R(F_ADDUSERSCRIPT)},
	{"deluserscript",P_MSG_R(F_DELUSERSCRIPT)},
	{"guestaccount",P_MSG_R(F_GUESTACCOUNT)},
	{"deadtime",P_MSG_R(F_DEADTIME)},
	{"debuglevel",P_MSG_R(F_DBGLEVEL)},
	{"defaultservice",P_MSG_R(F_DEFSERVICE)},
	{"loadprinters",P_MSG_R(F_LOADPRINTERS)},
	{"msgcommand",P_MSG_R(F_MSGCOMMAND)},
	{NULL,NULL},
};

static REGISTER_VARIABLES samba_registry1("samba",samba_var_list1, ID_MAIN,samedit_global);


static PUBLISH_VARIABLES_MSG samba_var_list2[]={
	{"netlogon_comment",P_MSG_R(F_NETLOGONCOMMENT)},
	{"netlogon_available",P_MSG_R(F_ENABLE)},
	{"netlogon_path",P_MSG_R(F_PATH)},
	{"netlogon_domain",P_MSG_R(F_DOMLOGONS)},
	{"netlogon_logonscript",P_MSG_R(F_LOGONSCRIPT)},
	{"netlogon_logonpath",P_MSG_R(F_LOGONPATH)},
	{"netlogon_logondrive",P_MSG_R(F_LOGONDRIVE)},
	{"netlogon_logonhome",P_MSG_R(F_LOGONHOME)},
	{NULL,NULL},
};

static REGISTER_VARIABLES samba_registry2("samba",samba_var_list2, ID_LOGON,samedit_netlogon);

static PUBLISH_VARIABLES_MSG samba_var_list3[]={
	{"share_path",P_MSG_R(F_PATH)},
	{"share_available",P_MSG_R(F_ENABLE)},
	{"share_browseable",P_MSG_R(F_BROWSEABLE)},
	{"share_forceuser",P_MSG_R(F_FORCEUSER)},
	{"share_comment",P_MSG_R(F_COMMENT)},
	{"share_copy",P_MSG_R(F_INHERIT)},
	{"share_public",P_MSG_R(F_PUBLIC)},
	{"share_guestonly",P_MSG_R(F_GUESTONLY)},
	{"share_writable",P_MSG_R(F_WRITABLE)},
	{"share_allowhosts",P_MSG_R(F_ALLOWHOSTS)},
	{"share_denyhosts",P_MSG_R(F_DENYHOSTS)},
	{"share_users",P_MSG_R(F_USERLIST)},
	{"share_onlyusers",P_MSG_R(F_ONLYUSER)},
	{"share_adminusers",P_MSG_R(F_ADMINUSERS)},
	{"share_writelist",P_MSG_R(F_WRITELIST)},
	{"share_validusers",P_MSG_R(F_VALIDUSERS)},
	{"share_invalidusers",P_MSG_R(F_IVLDUSERS)},
	{"share_readlist",P_MSG_R(F_READLIST)},
	{"share_preexec",P_MSG_R(F_PREEXEC)},
	{"share_preexecclose",P_MSG_R(F_EXECCLOSE)},
	{"share_rootpreexec",P_MSG_R(F_ROOTPREEXEC)},
	{"share_rootpreexecclose",P_MSG_R(F_ROOTEXECCLOSE)},
	{"share_postexec",P_MSG_R(F_POSTEXEC)},
	{"share_rootpostexec",P_MSG_R(F_ROOTPOSTEXEC)},
	{"share_magicscript",P_MSG_R(F_MAGICSCRIPT)},
	{"share_forcegroup",P_MSG_R(F_FORCEGROUP)},
	{"share_dontdescend",P_MSG_R(F_DONTDESCEND)},
	{"share_guestaccount",P_MSG_R(F_GUESTACCOUNTSH)},
	{"share_magicoutput",P_MSG_R(F_MAGICOUTPUT)},
	{"share_maxconnections",P_MSG_R(F_MAXCONNECT)},
	{"share_createmask",P_MSG_R(F_CREATEMASK)},
	{"share_dirmask",P_MSG_R(F_DIRMASK)},
	{ NULL, NULL }
};

static REGISTER_VARIABLES samba_registry3("samba","shares",samba_var_list3,	ID_SHARE
	,samedit_editshare,samedit_listshare);


