/* lsparisc -- Display sysfs information about PA native bus devices
 *
 *  Copyright (C) 2005-2006 Kyle McMartin <kyle@parisc-linux.org>
 *  Copyright (C) 2005-2006 Thibaut Varene <varenet@parisc-linux.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2
 *  as published by the Free Software Foundation.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <getopt.h>
#include <sysfs/libsysfs.h>

#include "hphw.h"

static const char *version_str = "0.2";
static const char *program_str = "lsparisc";

static int verbose = 0;
static short out_mask = 0;

static const char *parisc_device_name = "parisc";

#define IS_NOT_PARISC 	-1

#define MASK_LONG	0x0001

static struct option long_options[] = {
        { "help", 0, 0, 'h'},
        { "long", 0, 0, 'l'},
        { "verbose", 0, 0, 'v'},
        { "version", 0, 0, 'V'},
        { 0, 0, 0, 0}
};

static int
get_sdev_value(struct sysfs_device * sdevp, const char * search,
	       char * value, int max_value_len)
{
	int len;
	struct sysfs_attribute * sattrp;

	sattrp = sysfs_get_device_attr(sdevp, search);
	if (NULL == sattrp)
		return 0;

	if (max_value_len <= 0)
		return 1;

	strncpy(value, sattrp->value, max_value_len);
	value[max_value_len - 1] = '\0';

	len = strlen(value);
	if ((len > 0) && (value[len - 1] == '\n'))
		value[len - 1] = '\0';

	return 1;
}

static void
llonger_entry(struct sysfs_device *sdev) {
	char value[SYSFS_NAME_LEN];

	if (get_sdev_value(sdev, "hversion", value, SYSFS_NAME_LEN))
		printf("  hversion=%s", value);
	if (get_sdev_value(sdev, "sversion", value, SYSFS_NAME_LEN))
		printf(" sversion=%s", value);
	if (get_sdev_value(sdev, "hpa", value, SYSFS_NAME_LEN))
		printf(" hpa=%s", value);

	printf("\n");
}

int
read_parisc_device(struct sysfs_device *sdev) {
        struct sysfs_attribute *sattr;

        unsigned int hw_type = 0;
        unsigned int hversion = 0;
        unsigned int sversion = 0;

        sattr = sysfs_get_device_attr(sdev, "hw_type");
        if (NULL == sattr) {
                return IS_NOT_PARISC;
        } else if (1 != sscanf(sattr->value, "0x%x", &hw_type)) {
                return 0;
        } 

        sattr = sysfs_get_device_attr(sdev, "hversion");
        if (NULL == sattr) {
                return IS_NOT_PARISC;
        } else if (1 != sscanf(sattr->value, "0x%x", &hversion)) {
                return 0;
        }

        sattr = sysfs_get_device_attr(sdev, "sversion");
        if (NULL == sattr) {
                return IS_NOT_PARISC;
        } else if (1 != sscanf(sattr->value, "0x%x", &sversion)) {
                return 0;
        }

        if (parisc_print_mask(hw_type, hversion, sversion))
                return 0;

	printf("[%s] %s: %s\n", sdev->name, parisc_type(hw_type),
		parisc_hardware_description(hw_type, hversion, sversion));

	if (out_mask & MASK_LONG)
		llonger_entry(sdev);

	if (verbose)
		printf("  dir: %s\n", sdev->path);

        return 0;
}

void 
one_sdev_entry(struct sysfs_device *sdev) {
        int ret = 0;
        struct dlist *d = NULL;

        ret = read_parisc_device(sdev);

        if (sdev->children) {
                d = sdev->children;
                dlist_for_each_data(d, sdev, struct sysfs_device) {
                        one_sdev_entry(sdev);
                }
        }

        return;
}

void 
search_sysfs_tree(const char *dev_name) {
        struct sysfs_device *sdev;
	struct sysfs_bus *sbus;
        struct dlist *d = NULL;

	sbus = sysfs_open_bus(dev_name);
	if (NULL == sbus) {
		fprintf(stderr, "Unable to open parisc bus!\n");
		return;
	}

	d = sysfs_get_bus_devices(sbus);
        dlist_for_each_data(d, sdev, struct sysfs_device) {
                one_sdev_entry(sdev);
        }

        sysfs_close_bus(sbus);
}

void
usage(char **argv) {
        fprintf(stderr, "usage: %s [--verbose|-v] [--version|-V]"
			" [--help|-h] [--long|-l]\n",
                program_str);
	fprintf(stderr, "\t--help     this usage information\n");
	fprintf(stderr, "\t--long     additional information output\n");
	fprintf(stderr, "\t--verbose  output path names were data "
			"is found\n");
	fprintf(stderr, "\t--version  output version string and exit\n");
}

int
main(int argc, char *argv[])
{
        char c;
        int optind;
        char sysfsroot[SYSFS_PATH_MAX];

        while((c = getopt_long(argc, argv, "hlvV", long_options, &optind))) {
                if (c == -1)
                        break;

                switch (c) {
                case 'v':
                        verbose++;
                        break;
                case 'V':
                        fprintf(stderr, "%s version %s\n", program_str,
                                version_str);
                        return 0;
		case 'h':
			usage(argv);
			return 0;
		case 'l':
			out_mask |= MASK_LONG;
			break;
                default:
                        usage(argv);
                        return 1;
                }
        }

        if (sysfs_get_mnt_path(sysfsroot, SYSFS_PATH_MAX)) {
                fprintf(stderr, "Unable to locate sysfsroot.\n");
                return 1;
        }

        search_sysfs_tree(parisc_device_name);

        return 0;
}
