/** vim:sw=4:sts=4
 * Handle void* wrappers.
 * This is part of lua-gtk.
 */

#include "luagtk.h"
#include <lauxlib.h>
#include <string.h>	    // strcmp

#define LUAGTK_WRAPPER "void* wrapper"

#define VALUE_WRAPPER_MAGIC1 0x89737948
#define VALUE_WRAPPER_MAGIC2 0xa0d7dfaa

struct value_wrapper {
    unsigned int magic1;
    unsigned int magic2;
    int ref;
    int refcount;		/* how many Lua objects use this wrapper? */
};

/* content of a userdata that wraps the wrapper... duh. */
struct _value_wrapper2 {
    struct value_wrapper *wrapper;
};

/* Keep track of how many wrappers exist.  Use this to catch leaks. */
int vwrapper_count = 0, vwrapper_alloc_count = 0, vwrapper_objects = 0;


/**
 * Determine whether the void* points to a value wrapper; it relies on
 * a "magic" signature, which might give false positives in pathological
 * cases.
 */
int luagtk_is_vwrapper(void *p)
{
    struct value_wrapper *wrp = (struct value_wrapper*) p;
    return wrp->magic1 == VALUE_WRAPPER_MAGIC1
	&& wrp->magic2 == VALUE_WRAPPER_MAGIC2;
}
	
/**
 * Push the Lua object wrapped by the given value_wrapper onto the stack.
 */
int luagtk_vwrapper_get(lua_State *L, struct value_wrapper *wrp)
{
    lua_rawgeti(L, LUA_REGISTRYINDEX, wrp->ref);
    return 1;
}

/**
 * Try to convert a Userdata to a pointer.
 *
 * @param index    Position of the value on the Lua stack
 * @param dest     Where to store the converted value
 * @param only_ptr Only accept a pointer type; otherwise, ENUM (integer) is OK
 *
 * @return   0=ok, pointer  1=ok, integer
 */
int luagtk_userdata_to_ffi(lua_State *L, int index, union gtk_arg_types *dest,
    int only_ptr)
{
    void *p = (void*) lua_topointer(L, index);

    // NULL pointer or no metatable - pass pointer as is
    if (p == NULL || !lua_getmetatable(L, index)) {
	dest->p = p;
	return 0;
    }
    // stack: metatable

    // is this an enum/flag?
    // XXX this meta might not be initialized yet
    lua_getfield(L, LUA_REGISTRYINDEX, ENUM_META);
    if (lua_rawequal(L, -1, -2)) {
	if (only_ptr)
	    luaL_error(L, "ENUM given for a pointer parameter\n");
	dest->l = ((struct luagtk_enum_t*)p)->value;
	lua_pop(L, 2);
	return 1;
    }
    lua_pop(L, 1);

    // Is this a value wrapper wrapper?  If so, pass the wrapper.
    lua_getfield(L, LUA_REGISTRYINDEX, LUAGTK_WRAPPER);
    if (lua_rawequal(L, -1, -2)) {
	struct _value_wrapper2 *wrp = (struct _value_wrapper2*) p;
	dest->p = wrp->wrapper;
	lua_pop(L, 2);
	return 1;
    }
    lua_pop(L, 1);

    // is this a widget? if not, pass pointer as is
    lua_getfield(L, -1, "_classname");
    // stack: metatable, _classname/nil
    if (lua_isnil(L, -1)) {
	lua_pop(L, 2);
	dest->p = p;
	return 0;
    }

    // this is a widget - pass the pointer
    dest->p = ((struct widget*)p)->p;
    lua_pop(L, 2);
    return 0;
}


/**
 * Garbage collection of a Lua wrapper.  Free the referenced C wrapper if
 * its refcount now drops to zero.
 */
static int wrapper_gc(lua_State *L)
{
    struct _value_wrapper2 *a = (struct _value_wrapper2*) lua_topointer(L, 1);
    if (a->wrapper && --a->wrapper->refcount == 0) {
	luaL_unref(L, LUA_REGISTRYINDEX, a->wrapper->ref);
	a->wrapper->ref = 0;
	a->wrapper->magic1 = 0;
	a->wrapper->magic2 = 0;
	g_free(a->wrapper);
	vwrapper_count --;
    }
    a->wrapper = NULL;
    return 0;
}


/**
 * The :destroy method decreases the refcount by one, if it is > 1.  It must
 * not be set to 0, because the userdata object will be garbage collected
 * eventually, and then the refcount is decreased again.
 */
static int wrapper_destroy(lua_State *L)
{
    struct _value_wrapper2 *a = (struct _value_wrapper2*) lua_topointer(L, 1);
    if (a->wrapper && a->wrapper->refcount > 1)
	a->wrapper->refcount --;
    return 0;
}


/**
 * Access to a wrapper's fields - value or destroy().  __index in the metatable
 * points to this function instead of itself, because in case of value, a
 * function must be called.
 */
int wrapper_index(lua_State *L)
{
    const struct _value_wrapper2 *a = lua_topointer(L, 1);
    const char *key = lua_tostring(L, 2);
    if (!strcmp(key, "value")) {
	lua_rawgeti(L, LUA_REGISTRYINDEX, a->wrapper->ref);
	return 1;
    }
    if (!strcmp(key, "destroy")) {
	lua_pushcfunction(L, wrapper_destroy);
	return 1;
    }

    return 0;
}

/**
 * Debug function: show the wrapper's address and content.
 */
static int wrapper_tostring(lua_State *L)
{
    const struct _value_wrapper2 *a = lua_topointer(L, 1);
    lua_pushliteral(L, "[void* wrapper: ");
    lua_getglobal(L, "tostring");
    lua_rawgeti(L, LUA_REGISTRYINDEX, a->wrapper->ref);
    lua_call(L, 1, 1);
    lua_pushliteral(L, "]");
    lua_concat(L, 3);
    return 1;
}


static const luaL_reg wrapper_methods[] = {
    { "__index", wrapper_index },
    { "__gc", wrapper_gc },
    { "__tostring", wrapper_tostring },
    { NULL, NULL }
};


/**
 * A value should be passed to a Gtk function as void*.  This is most likely
 * a "data" argument that will be given to a callback, or a value in a
 * collection class like a tree etc.  Create a special userdata that transports
 * a reference.
 *
 * Note: This wrapper is initialized with a refcount of 0.  This function
 * is either called by gtk.void_ptr(), which immediately creates a Lua
 * wrapper for it, or by lua2ffi_void_ptr.  In the latter case, refcount
 * remains at 0 until Gtk chooses to pass the value to a Lua callback; then
 * ffi2lua_void_ptr calls luagtk_push_value_wrapper, which increases the
 * refcount.
 */
struct value_wrapper *luagtk_make_value_wrapper(lua_State *L, int index)
{
    lua_pushvalue(L, index);
    struct value_wrapper *wrp = (struct value_wrapper*) g_malloc(sizeof(*wrp));
    wrp->magic1 = VALUE_WRAPPER_MAGIC1;
    wrp->magic2 = VALUE_WRAPPER_MAGIC2;
    wrp->ref = luaL_ref(L, LUA_REGISTRYINDEX);
    wrp->refcount = 0;
    vwrapper_count ++;
    vwrapper_alloc_count ++;
    return wrp;
}

/**
 * Make a Lua wrapper for the C wrapper for the Lua value.
 */
int luagtk_push_value_wrapper(lua_State *L, struct value_wrapper *wrp)
{
    struct _value_wrapper2 *p = lua_newuserdata(L, sizeof(*p));
    p->wrapper = wrp;
    wrp->refcount ++;
    vwrapper_objects ++;

    // add a metatable with some methods
    if (luaL_newmetatable(L, LUAGTK_WRAPPER))
	luaL_register(L, NULL, wrapper_methods);

    lua_setmetatable(L, -2);
    return 1;
} 

/**
 * The C function expects a void* pointer.  Any datatype should be permissible;
 * nil and lightuserdata are easy; userdata may be ENUM or contain a widget.
 * For other data types, a reference is created, which is then wrapped in a
 * small memory block with a "magic" signature.  This signature will then be
 * recognized in ffi2lua_void_ptr.
 *
 * The problem is that it's unknown how long this value is required.  The C
 * function might store it somewhere (e.g. g_tree_insert key and value) or not
 * (e.g. g_tree_foreach user_data), so it can't be freed automatically.
 */
int lua2ffi_void_ptr(struct argconv_t *ar)
{
    lua_State *L = ar->L;

    switch (ar->lua_type) {
	case LUA_TNIL:
	    ar->arg->p = NULL;
	    break;
	
	case LUA_TLIGHTUSERDATA:
	    ar->arg->p = (void*) lua_topointer(L, ar->index);
	    break;
	
	case LUA_TUSERDATA:
	    luagtk_userdata_to_ffi(L, ar->index, ar->arg, 1);
	    break;
	
	default:;
	    struct value_wrapper *w = luagtk_make_value_wrapper(ar->L,
		ar->index);
	    /* this reference isn't owned...  leak unless :destroy is called */
	    w->refcount ++;
	    ar->arg->p = w;
    }

    return 1;
}


