/** vim:sw=4:sts=4
 *
 * Lua binding for the hash functions, so that a Lua script can compute
 * hash values.
 */

#include "module.h"
#include "lg-hash.h"
#include <string.h>

static struct hash_state state;
struct lg_module_api *api;

int generate_hash_cmph(lua_State *L, const char *datafile_name,
    const char *prefix, const char *ofname);
int generate_hash_simple(lua_State *L, const char *datafile_name,
    const char *prefix, const char *ofname);

/**
 * Compute a hash value for the given key.
 *
 * @luaparam key  The string to compute the hash value for
 */
static int l_compute_hash(lua_State *L)
{
    size_t key_len;
    const unsigned char *key = (const unsigned char*)
	luaL_checklstring(L, 1, &key_len);

    unsigned int hash_value = compute_hash(&state, key, key_len, NULL);
    lua_pushinteger(L, hash_value);
    return 1;
}

/**
 * Load the given file using the cmph_load function of the cmph library,
 * then write the hash function data; then read the (key,value) file and
 * write that out, too, in the correct order.
 *
 * @luaparam cmphfile  Name of the file generated by cmph (suffix .mph)
 * @luaparam datafile  Name of the key/value pair file
 * @luaparam prefix  How to prefix the variables in the output
 * @luaparam ofile  Name of the output file to write to
 */
static int l_generate_hash(lua_State *L)
{
    const char *datafile_name = luaL_checkstring(L, 1);
    const char *prefix = luaL_checkstring(L, 2);
    const char *ofname = luaL_checkstring(L, 3);

#if (defined(LG_CMPH_ALGO))
    return generate_hash_cmph(L, datafile_name, prefix, ofname);
#else
    return generate_hash_simple(L, datafile_name, prefix, ofname);
#endif

}


static const luaL_Reg methods[] = {
    { "compute_hash", l_compute_hash },
    { "generate_hash", l_generate_hash },
    { NULL, NULL }
};

// The module gnome is loaded, so check API compatibility.
static int check_api_compat(lua_State *L)
{
    lua_getfield(L, -1, "api");
    if (lua_isnil(L, -1))
	return luaL_error(L, "gnome.api not found");
    api = (struct lg_module_api*) lua_topointer(L, -1);
    if (!api)
	return luaL_error(L, "gnome.api is NULL");

    if (api->major != LUAGNOME_MODULE_MAJOR
	|| api->minor < LUAGNOME_MODULE_MINOR)
	return luaL_error(L, "incompatible API versions of gnome %d.%d and "
	    "%s %d.%d.", api->major, api->minor, "gnomedev",
	    LUAGNOME_MODULE_MAJOR, LUAGNOME_MODULE_MINOR);

    return 0;
}


int luaopen_gnomedev(lua_State *L)
{
    const char *lib_name = lua_tostring(L, 1);
    luaL_register(L, lib_name, methods);

    // XXX get hashfunc and seed from api
    state.hashfunc = HASHFUNC_JENKINS;
    state.seed = 0;

    // if the module gnome is not loaded, we're building the core module.
    lua_pushliteral(L, "gnome");
    lua_rawget(L, LUA_GLOBALSINDEX);
    if (lua_isnil(L, -1)) {
	lua_pop(L, 1);
	return 1;
    }

    check_api_compat(L);

    lua_pop(L, 1);	// stack: gnomedev

    return 1;
}

