//
//  libavg - Media Playback Engine. 
//  Copyright (C) 2003-2006 Ulrich von Zadow
//
//  This library is free software; you can redistribute it and/or
//  modify it under the terms of the GNU Lesser General Public
//  License as published by the Free Software Foundation; either
//  version 2 of the License, or (at your option) any later version.
//
//  This library is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Lesser General Public License for more details.
//
//  You should have received a copy of the GNU Lesser General Public
//  License along with this library; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//  Current versions can be found at www.libavg.de
//

#include "AsyncVideoDecoder.h"
#include "EOFVideoMsg.h"
#include "InfoVideoMsg.h"
#include "ErrorVideoMsg.h"
#include "SeekDoneVideoMsg.h"

#include "../base/ObjectCounter.h"

#include <boost/thread/thread.hpp>
#include <boost/bind.hpp>

#include <math.h>
#include <iostream>

using namespace boost;
using namespace std;

namespace avg {

AsyncVideoDecoder::AsyncVideoDecoder(VideoDecoderPtr pSyncDecoder)
    : m_pSyncDecoder(pSyncDecoder),
      m_pDecoderThread(0),
      m_Size(0,0),
      m_bUseStreamFPS(true),
      m_FPS(0),
      m_bEOF(false),
      m_bSeekPending(false),
      m_LastFrameTime(-1000)
{
    ObjectCounter::get()->incRef(&typeid(*this));
}

AsyncVideoDecoder::~AsyncVideoDecoder()
{
    if (m_pDecoderThread) {
        close();
    }
    ObjectCounter::get()->decRef(&typeid(*this));
}

void AsyncVideoDecoder::open(const std::string& sFilename, YCbCrMode ycbcrMode,
        bool bThreadedDemuxer)
{
    m_bEOF = false;
    m_bSeekPending = false;
    m_sFilename = sFilename;
    m_pCmdQ = VideoDecoderThread::CmdQueuePtr(new VideoDecoderThread::CmdQueue);
    m_pMsgQ = VideoMsgQueuePtr(new VideoMsgQueue(8));
    m_pDecoderThread = new boost::thread(
            VideoDecoderThread(*m_pMsgQ, *m_pCmdQ, m_pSyncDecoder, sFilename, 
                    ycbcrMode, bThreadedDemuxer));
    VideoMsgPtr pMsg = m_pMsgQ->pop(true);
    getInfoMsg(pMsg);
    m_LastFrameTime = -1000;
}

void AsyncVideoDecoder::close()
{
    if (m_pDecoderThread) {
        m_pCmdQ->push(Command<VideoDecoderThread>(boost::bind(
                &VideoDecoderThread::stop, _1)));
        getNextBmps(false); // If the Queue is full, this breaks the lock in the thread.
        m_pDecoderThread->join();
        delete m_pDecoderThread;
        m_pDecoderThread = 0;
    }
}

void AsyncVideoDecoder::seek(int DestFrame)
{
    waitForSeekDone();
    m_bEOF = false;
    m_pCmdQ->push(Command<VideoDecoderThread>(boost::bind(
                &VideoDecoderThread::seek, _1, DestFrame)));
    m_bSeekPending = true;
    try {
        bool bDone = false;
        do {
            VideoMsgPtr pMsg = m_pMsgQ->pop(false);
            SeekDoneVideoMsgPtr pSeekDoneMsg = dynamic_pointer_cast<SeekDoneVideoMsg>(pMsg);
            if (pSeekDoneMsg) {
                bDone = true;
            }
        } while (!bDone);
        m_bSeekPending = false;
    } catch (Exception& ex) {
        // The queue is empty.
    }
}

IntPoint AsyncVideoDecoder::getSize()
{
    return m_Size;
}

int AsyncVideoDecoder::getNumFrames()
{
    assert(m_pDecoderThread);
    return m_NumFrames;
}

double AsyncVideoDecoder::getFPS()
{
    assert(m_pDecoderThread);
    return m_FPS;
}

void AsyncVideoDecoder::setFPS(double FPS)
{
    m_pCmdQ->push(Command<VideoDecoderThread>(boost::bind(
                &VideoDecoderThread::setFPS, _1, FPS)));
    m_bUseStreamFPS = (FPS == 0);
    if (FPS != 0) {
        m_FPS = FPS;
    }
}

PixelFormat AsyncVideoDecoder::getPixelFormat()
{
    assert(m_pDecoderThread);
    return m_PF;
}

FrameAvailableCode AsyncVideoDecoder::renderToBmp(BitmapPtr pBmp, long long TimeWanted)
{
    FrameAvailableCode FrameAvailable;
    FrameVideoMsgPtr pFrameMsg = getBmpsForTime(TimeWanted, FrameAvailable);
    if (FrameAvailable == FA_NEW_FRAME) {
        *pBmp = *(pFrameMsg->getBitmap(0));
    }
    return FrameAvailable;
}

FrameAvailableCode AsyncVideoDecoder::renderToYCbCr420p(BitmapPtr pBmpY, BitmapPtr pBmpCb, 
       BitmapPtr pBmpCr, long long TimeWanted)
{
    FrameAvailableCode FrameAvailable;
    FrameVideoMsgPtr pFrameMsg = getBmpsForTime(TimeWanted, FrameAvailable);
    if (FrameAvailable == FA_NEW_FRAME) {
        pBmpY->copyPixels(*(pFrameMsg->getBitmap(0)));
        pBmpCb->copyPixels(*(pFrameMsg->getBitmap(1)));
        pBmpCr->copyPixels(*(pFrameMsg->getBitmap(2)));
    }
    return FrameAvailable;
}

long long AsyncVideoDecoder::getCurFrameTime()
{
    return m_LastFrameTime;
}

bool AsyncVideoDecoder::isEOF()
{
    return m_bEOF;
}

void AsyncVideoDecoder::getInfoMsg(VideoMsgPtr pMsg)
{
    InfoVideoMsgPtr pInfoMsg = dynamic_pointer_cast<InfoVideoMsg>(pMsg);
    ErrorVideoMsgPtr pErrorMsg(dynamic_pointer_cast<ErrorVideoMsg>(pMsg));
    if (pErrorMsg) {
        close();
        throw(pErrorMsg->getException());
    } else {
        assert(pInfoMsg); // If this isn't true, the function shouldn't have 
                          // been called.
        m_Size = pInfoMsg->getSize();
        m_NumFrames = pInfoMsg->getNumFrames();
        if (m_bUseStreamFPS) {
            m_FPS = pInfoMsg->getFPS();
        }
        m_PF = pInfoMsg->getPF();
    }
}
        
FrameVideoMsgPtr AsyncVideoDecoder::getBmpsForTime(long long TimeWanted, 
        FrameAvailableCode& FrameAvailable)
{
    // XXX: This code is sort-of duplicated in FFMpegDecoder::readFrameForTime()
    long long FrameTime = -1000;
    FrameVideoMsgPtr pFrameMsg;
//    cerr << "getBmpsForTime " << TimeWanted << ", LastFrameTime= " << m_LastFrameTime 
//            << ", diff= " << TimeWanted-m_LastFrameTime <<  endl;
    if (TimeWanted == -1) {
        pFrameMsg = getNextBmps(true);
        FrameAvailable = FA_NEW_FRAME;
    } else {
        double TimePerFrame = 1000.0/m_FPS;
        if (fabs(TimeWanted-m_LastFrameTime) < 0.5*TimePerFrame) {
//            cerr << "   LastFrameTime = " << m_LastFrameTime << ", display again." <<  endl;
            // The last frame is still current. Display it again.
            FrameAvailable = FA_USE_LAST_FRAME;
            return FrameVideoMsgPtr();
        } else {
            if (m_bEOF) {
//                cerr << "  EOF." << endl;
                FrameAvailable = FA_USE_LAST_FRAME;
                return FrameVideoMsgPtr();
            }
            while (FrameTime-TimeWanted < -0.5*TimePerFrame && !m_bEOF) {
                pFrameMsg = getNextBmps(false);
                if (pFrameMsg) {
                    FrameTime = pFrameMsg->getFrameTime();
//                    cerr << "   readFrame returned time " << FrameTime << "." <<  endl;
                } else {
//                    cerr << "   no frame available." <<  endl;
                    FrameAvailable = FA_STILL_DECODING;
                    return FrameVideoMsgPtr();
                }
            }
            FrameAvailable = FA_NEW_FRAME;
//            cerr << "  frame ok." << endl;
        }
    }
    if (pFrameMsg) {
        m_LastFrameTime = pFrameMsg->getFrameTime();
    }
    return pFrameMsg;
}

FrameVideoMsgPtr AsyncVideoDecoder::getNextBmps(bool bWait)
{
    try {
        waitForSeekDone();
        VideoMsgPtr pMsg = m_pMsgQ->pop(bWait);
        FrameVideoMsgPtr pFrameMsg = dynamic_pointer_cast<FrameVideoMsg>(pMsg);
        while (!pFrameMsg) {
            EOFVideoMsgPtr pEOFMsg(dynamic_pointer_cast<EOFVideoMsg>(pMsg));
            ErrorVideoMsgPtr pErrorMsg(dynamic_pointer_cast<ErrorVideoMsg>(pMsg));
            InfoVideoMsgPtr pInfoMsg(dynamic_pointer_cast<InfoVideoMsg>(pMsg));
            if (pEOFMsg) {
                m_bEOF = true;
                return FrameVideoMsgPtr();
            } else if(pErrorMsg) {
                m_bEOF = true;
                close();
                return FrameVideoMsgPtr();
            } else if (pInfoMsg) {
                getInfoMsg(pInfoMsg);
            } else {
                // Unhandled message type.
                assert(false);
            }
            VideoMsgPtr pMsg = m_pMsgQ->pop(bWait);
            FrameVideoMsgPtr pFrameMsg = dynamic_pointer_cast<FrameVideoMsg>(pMsg);
        }
        return pFrameMsg;
    } catch (Exception& e) {
        return FrameVideoMsgPtr();
    }
}

void AsyncVideoDecoder::waitForSeekDone()
{
    if (m_bSeekPending) {
        m_bSeekPending = false;
        VideoMsgPtr pMsg;
        bool bDone = false;
        do {
            VideoMsgPtr pMsg = m_pMsgQ->pop(true);
            SeekDoneVideoMsgPtr pSeekDoneMsg = dynamic_pointer_cast<SeekDoneVideoMsg>(pMsg);
            if (pSeekDoneMsg) {
                bDone = true;
            }
        } while (!bDone);
    }
}

}
