/*
 * Mailbox folder access
 *
 * Copyright (C) 2004  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#pragma implementation

#include "MailboxMailFolder.h"
#include "Exception.h"
#include "from.h"

#include <sys/types.h>  // stat, opendir, readdir, utimes
#include <sys/stat.h>   // stat
#include <dirent.h>		// opendir, readdir
#include <sys/time.h>	// utimes

#include <errno.h>

using namespace std;

MailboxMailFolder::MailboxMailFolder(const string& path) throw ()
	: _path(path)
{
	_name = _path;

	/// Normalize the folder name

	// Remove trailing '/'
	while (_name[_name.size() - 1] == '/')
		_name.resize(_name.size() - 1);

	// Remove leading path
	size_t lastslash = _name.find_last_of('/');
	if (lastslash != string::npos)
		_name = _name.substr(lastslash + 1);

	// Remove leading dot
	if (_name[0] == '.')
		_name = _name.substr(1);
}

MailboxMailFolder::MailboxMailFolder(const string& name, const string& path) throw ()
	: _name(name), _path(path), _stat_total(-1), _stat_unread(-1), _stat_new(-1), _stat_flagged(-1), _mbox_mtime(0) {}


static const int BUFFY_NEW		(1);
static const int BUFFY_READ		(1 << 1);
static const int BUFFY_FLAGGED	(1 << 2);

static int parse_mime_header (FILE *in, char* buf, int bufsize)
{
	bool status = false;
	int res = 0;

	while (fgets(buf, bufsize, in)) 
		if (buf[0] == '\n')
			break;
		else if (strncmp(buf, "Status:", 7) == 0) 
		{
//			fprintf(stderr, "Has status: %s\n", buf);
			status = true;
			if (strchr(buf+7, 'R'))
				res |= BUFFY_READ;
			else if (!strchr (buf+7, 'O')) 
				res |= BUFFY_NEW;
		}
		else if (strncmp(buf, "X-Status:", 9) == 0) 
		{
//			fprintf(stderr, "Has x-status: %s\n", buf);
			if (strchr (buf+9, 'F')) 
				res |= BUFFY_FLAGGED;
		}

	if (!status)
		res |= BUFFY_NEW;

	return res;
}

bool MailboxMailFolder::changed()
{
	struct stat st;
	if (stat(_path.c_str(), &st) != 0)
		throw SystemException(errno, "getting informations on " + _path);

	return st.st_mtime > _mbox_mtime;
}

void MailboxMailFolder::updateStatistics()
{
	int res_total = 0;
	int res_read = 0;
	int res_new = 0;
	int res_flagged = 0;
	FILE* in = 0;

	/// Count messages in the 'new' directory
	
	// Perform consistency checks on the directory
	struct stat st;
	if (stat(_path.c_str(), &st) != 0)
		throw SystemException(errno, "getting informations on " + _path);
	if (S_ISDIR(st.st_mode) != 0)
		throw ConsistencyCheckException(_path + " is a directory");

	_mbox_mtime = st.st_mtime;

//	if (!force && (s.st_size == mbox->file_size) && (s.st_mtime == mbox->file_mtime))
//		return 0;

	if (st.st_size == 0)
	{
		//mbox->file_size = 0;
		//mbox->file_mtime = 0;
		goto end2;
	}

	// TODO: lock the file
	// Count the messages
	in = fopen(_path.c_str(), "rt");
	if (in == NULL)
		throw SystemException(errno, "opening " + _path);

	static const int bufsize = 1024;
	char buf[bufsize];

	/* Check if a folder by checking for From as first thing in file */
	fgets(buf, bufsize, in);
	if (!is_from(buf)) 
	{
		fclose(in);
		goto end1;
	}

	//mbox->file_mtime = s.st_mtime;
	//mbox->file_size = s.st_size;

	res_total = 1;
	if (int t = parse_mime_header(in, buf, bufsize)) 
	{
		if (t & BUFFY_NEW) res_new++;
		if (t & BUFFY_READ) res_read++;
		if (t & BUFFY_FLAGGED) res_flagged++;
	}

	while (fgets(buf, bufsize, in))
		if (is_from(buf))
		{
			res_total++;
			if (int t = parse_mime_header (in, buf, bufsize))
			{
				if (t & BUFFY_NEW) res_new++;
				if (t & BUFFY_READ) res_read++;
				if (t & BUFFY_FLAGGED) res_flagged++;
			}
		}

end1:
	fclose (in);


	// Restore the access time of the mailbox for other checking programs
	struct timeval t[2];
	t[0].tv_sec = st.st_atime;
	t[0].tv_usec = 0;
	t[1].tv_sec = st.st_mtime;
	t[1].tv_usec = 0;
	utimes(_path.c_str(), t);

	// Return the values
end2:
	_stat_total = res_total;
	_stat_unread = (res_total - res_read);// + res_new;
	_stat_new = res_new;
	_stat_flagged = res_flagged;
}


static bool isMailbox(const std::string& pathname)
{
	// Perform consistency checks on the file
	struct stat st;
	if (stat(pathname.c_str(), &st) != 0)
		throw SystemException(errno, "getting informations on " + pathname);
	if (S_ISDIR(st.st_mode) != 0)
		return false;

	// Can be an empty file
	if (st.st_size == 0)
		return true;

	// Check if it starts with a From line
	FILE* in = fopen(pathname.c_str(), "rt");
	if (in == NULL)
		throw SystemException(errno, "opening " + pathname);

	static const int bufsize = 1024;
	char buf[bufsize];

	/* Check if a folder by checking for From as first thing in file */
	fgets(buf, bufsize, in);
	fclose(in);

	return is_from(buf);
}

void MailboxMailFolder::enumerateFolders(const std::string& parent, MailFolderConsumer& cons)
{
	// Perform consistency checks on the parent directory
	struct stat st;
	if (stat(parent.c_str(), &st) != 0)
		return;
//		throw SystemException(errno, "getting informations on " + parent);

	if (isMailbox(parent))
	{
		MailFolder f(new MailboxMailFolder(parent));
		cons.consume(f);
	}

	if (S_ISDIR(st.st_mode) == 0)
		return;

	// Enumerate the Mailboxs in it
	DIR *dir = opendir(parent.c_str());
	if (dir == NULL)
		throw SystemException(errno, "opening " + parent);

	struct dirent *d;
	while ((d = readdir (dir)) != NULL)
	{
		if (strcmp(d->d_name, ".") == 0)
			continue;
		if (strcmp(d->d_name, "..") == 0)
			continue;
		string candidate = parent + "/" + d->d_name;
		if (access(candidate.c_str(), R_OK) != 0)
			continue;
		if (isMailbox(candidate))
		{
			MailFolder f(new MailboxMailFolder(candidate));
			cons.consume(f);
		}
	}
	closedir(dir);
}

// vim:set ts=4 sw=4:
