/*
  CLAW - a C++ Library Absolutely Wonderful

  CLAW is a free library without any particular aim but being useful to 
  anyone.

  Copyright (C) 2005-2008 Julien Jorge

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: julien_jorge@yahoo.fr
*/
/**
 * \file box_2d.tpp
 * \brief Implementation of claw::math::box_2d class.
 * \author Julien Jorge
 */
#include <claw/rectangle.hpp>

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
template<class T>
claw::math::box_2d<T>::box_2d()
{

} // box_2d::box_2d() [constructor]

/*----------------------------------------------------------------------------*/
/**
 * \brief Copy constructor.
 * \param that Box to copy from.
 */
template<class T>
claw::math::box_2d<T>::box_2d( const self_type& that )
  : first_point(that.first_point), second_point(that.second_point)
{

} // box_2d::box_2d() [copy constructor]

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor from a rectangle.
 * \param that Rectangle to copy from.
 */
template<class T>
claw::math::box_2d<T>::box_2d( const rectangle<value_type>& that )
  : first_point(that.position),
    second_point(that.right(), that.bottom())
{

} // box_2d::box_2d() [constructor from rectangle]

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor from two points.
 * \param p1 The first point.
 * \param p2 The second point.
 */
template<class T>
claw::math::box_2d<T>::box_2d( const point_type& p1, const point_type& p2 )
  : first_point(p1), second_point(p2)
{

} // box_2d::box_2d() [constructor from coordinates]

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor with initialization.
 * \param x1 X-coordinate of the first point.
 * \param y1 Y-coordinate of the first point.
 * \param x2 X-coordinate of the second point.
 * \param y2 Y-coordinate of the second point.
 */
template<class T>
claw::math::box_2d<T>::box_2d( const value_type& x1, const value_type& y1,
                               const value_type& x2, const value_type& y2 )
  : first_point(x1, y1), second_point(x2, y2)
{

} // box_2d::box_2d() [constructor with values]

/*----------------------------------------------------------------------------*/
/**
 * \brief Equality operator
 * \param that Box to compare to.
 */
template<class T>
bool claw::math::box_2d<T>::operator==(const self_type& that) const
{
  return (first_point == that.first_point)
    && (second_point == that.second_point);
} // box_2d::operator==()

/*----------------------------------------------------------------------------*/
/**
 * \brief Difference operator.
 * \param that Box to compare to.
 */
template<class T>
bool claw::math::box_2d<T>::operator!=(const self_type& that) const
{
  return !( *this == that );
} // box_2d::operator!=()

/*----------------------------------------------------------------------------*/
/**
 * \brief Translation.
 * \param vect The vector to add to points.
 */
template<class T>
claw::math::box_2d<T>
claw::math::box_2d<T>::operator+(const point_type& vect) const
{
  return self_type( first_point + vect, second_point + vect );
} // box_2d::operator+()

/*----------------------------------------------------------------------------*/
/**
 * \brief Translation.
 * \param vect The vector to substract to points.
 */
template<class T>
claw::math::box_2d<T>
claw::math::box_2d<T>::operator-(const point_type& vect) const
{
  return self_type( first_point - vect, second_point - vect );
} // box_2d::operator-()

/*----------------------------------------------------------------------------*/
/**
 * \brief Translation.
 * \param vect The vector to add to points.
 */
template<class T>
claw::math::box_2d<T>&
claw::math::box_2d<T>::operator+=(const point_type& vect)
{
  first_point += vect;
  second_point += vect;
} // box_2d::operator+=()

/*----------------------------------------------------------------------------*/
/**
 * \brief Translation.
 * \param vect The vector to substract to points.
 */
template<class T>
claw::math::box_2d<T>&
claw::math::box_2d<T>::operator-=(const point_type& vect)
{
  first_point -= vect;
  second_point -= vect;
} // box_2d::operator-=()

/*----------------------------------------------------------------------------*/
/**
 * \brief Return box' width.
 */
template<class T>
typename claw::math::box_2d<T>::value_type
claw::math::box_2d<T>::width() const
{
  if (first_point.x > second_point.x)
    return first_point.x - second_point.x + 1;
  else
    return second_point.x - first_point.x + 1;
} // box_2d::width()

/*----------------------------------------------------------------------------*/
/**
 * \brief Return box' height.
 */
template<class T>
typename claw::math::box_2d<T>::value_type
claw::math::box_2d<T>::height() const
{
  if (first_point.y > second_point.y)
    return first_point.y - second_point.y + 1;
  else
    return second_point.y - first_point.y + 1;
} // box_2d::height()
