// -*- C++ -*- (c) 2007 Enrico Zini <enrico@enricozini.org>
//                      Petr Rockai <me@mornfall.net>

/*
 * Copyright (C) 2007  Enrico Zini <enrico@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <ept/core/xapian.h>
#include <ept/core/apt.h>
#include <ept/core/apt/record.h>

#include <ept/textsearch/maint/path.h>

#include <wibble/regexp.h>
#include <cctype>
#include <cmath>

#include <xapian/queryparser.h>
#include <xapian/base.h>

#include <algorithm>

#include <iostream>

using namespace std;
using namespace ept;
// using namespace ept::debtags;

namespace ept {
namespace core {
namespace xapian {

size_t max_index = 0;

Source::Source()
    : m_timestamp(0), m_stem("en")
{
    // m_timestamp = Path::indexTimestamp();
    // if (m_timestamp)
        // FIXME
    m_db.add_database(Xapian::Database(textsearch::Path::index()));
}

std::string Source::toLower(const std::string& str)
{
    std::string res;
    res.reserve(str.size());
    for (std::string::const_iterator i = str.begin(); i != str.end(); ++i)
        res += tolower(*i);
    return res;
}

void Source::normalize_and_add(Xapian::Document& doc, const std::string& term,
                               int& pos) const
{
    string t = Source::toLower(term);
    string s = m_stem(t);
    doc.add_term(t);
    if (s != t)
        doc.add_term(s);
}

Xapian::Query Source::makeORQuery(const std::string& keywords) const
{
    wibble::Tokenizer tok(keywords, "[A-Za-z0-9_-]+", REG_EXTENDED);
    return makeORQuery(tok.begin(), tok.end());
}

Xapian::Query Source::makePartialORQuery(const std::string& keywords) const
{
    wibble::Tokenizer tok(keywords, "[A-Za-z0-9_-]+", REG_EXTENDED);
    vector<string> tokens;
    std::copy( tok.begin(), tok.end(), std::back_inserter( tokens ) );
    // Add all the terms starting with 'last'
    if (!tokens.empty())
    {
        std::string last = tokens.back();
        if (last.size() == 1)
            // Ignore one-letter partial terms: they make the query uselessly
            // large and slow, and it's worth just to wait for more characters
            // to come
            tokens.pop_back();
        else
            std::copy(
                m_db.allterms_begin(last),
                m_db.allterms_end(last),
                std::back_inserter(tokens) );
    }
    return makeORQuery(tokens.begin(), tokens.end());
}

Xapian::docid Source::docidByName(const std::string& pkgname) const
{
    Xapian::PostingIterator i = m_db.postlist_begin("pkg:"+pkgname);
    if (i == m_db.postlist_end("pkg:"+pkgname))
        return 0;
    else
        return *i;
}

 struct TagFilter : public Xapian::ExpandDecider
 {
     virtual bool operator()(const std::string &term) const {
         return term[0] == 'T';
     }
 };

static TagFilter tagFilter;

vector<string> Source::expand(Xapian::Enquire& enq) const
{
    Xapian::RSet rset;
    // Get the top 5 results as 'good ones' to compute the search expansion
    Xapian::MSet mset = enq.get_mset(0, 5);
    for (Xapian::MSet::iterator i = mset.begin(); i != mset.end(); ++i)
        rset.add_document(i);
    // Get the expanded set, only expanding the query with tag names
    Xapian::ESet eset = enq.get_eset(5, rset, &tagFilter);
    vector<string> res;
    for (Xapian::ESetIterator i = eset.begin(); i != eset.end(); ++i)
        res.push_back(*i);
    return res;
}

Xapian::Query Source::makeRelatedQuery(const std::string& pkgname) const
{
    Xapian::Enquire enquire(db());

    // Retrieve the document for the given package
    enquire.set_query(Xapian::Query("pkg:"+pkgname));
    Xapian::MSet matches = enquire.get_mset(0, 1);
    Xapian::MSetIterator mi = matches.begin();
    if (mi == matches.end()) return Xapian::Query();
    Xapian::Document doc = mi.get_document();

    // Return the query to get the list of similar documents
    return Xapian::Query(Xapian::Query::OP_OR, doc.termlist_begin(), doc.termlist_end());
}

double Source::getDoubleValue(const std::string& pkgname,
                              Xapian::valueno val_id) const
{
    Xapian::docid id = docidByName(pkgname);
    if (id == 0)
        return 0.0;
    Xapian::Document doc = db().get_document(id);
    string val = doc.get_value(val_id);
    if (val.empty())
        return 0.0;
    else
        return Xapian::sortable_unserialise(val);
}

int Source::getIntValue(const std::string& pkgname,
                        Xapian::valueno val_id) const
{
    Xapian::docid id = docidByName(pkgname);
    if (id == 0)
        return 0;
    Xapian::Document doc = db().get_document(id);
    string val = doc.get_value(val_id);
    if (val.empty())
        return 0;
    else
        return (int)nearbyint(Xapian::sortable_unserialise(val));
}

}
}
}

// vim:set ts=4 sw=4:
