/* Copyright (C) 1999 Hans Petter K. Jansson
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * You can contact the library's author by sending e-mail to <hpj@styx.net>.
 */

#ifndef _SOCK_H
#define _SOCK_H 1

#include <stdio.h>

#include <sys/types.h>
#include <netinet/in.h>

#ifndef LIBFLUX_BUILD
#  include <flux/fifobuf.h>
#else
#  include "fifobuf.h"
#endif

#define SOCK_TIMEOUT_DEFAULT 20        /* Maximum number of seconds to wait for connections
                                          or transfers. */

/* --- Socket call flags --- */

#define SOCK_RETRY (1 << 0)       /* If initially unsuccessful, retry bind for
                                     a maximum of 30 seconds */
#define SOCK_SECURE (1 << 1)      /* Prevent spoofing attacks */
#define SOCK_IMMEDIATE (1 << 2)   /* Don't flush or linger */
#define SOCK_DEBUG (1 << 3)       /* Print debug messages */
#define SOCK_LOG (1 << 4)         /* Log most socket operations */

/* --- Socket status flags --- */

#define SOCK_CONNECTED   (1 << 5)
#define SOCK_PIPE_PARENT (1 << 6)
#define SOCK_PIPE_CHILD  (1 << 7)
#define SOCK_PIPE        (SOCK_PIPE_PARENT | SOCK_PIPE_CHILD)
#define SOCK_CALLER      (1 << 8)
#define SOCK_ACCEPT      (1 << 9)
#define SOCK_INTERNAL    (1 << 10)
#define SOCK_CONNECTING  (1 << 11)

/* TRUE if process is at both reading and writing end. */
#define SOCK_PIPE_LOOP(flags)  ((flags & (SOCK_PIPE_PARENT | SOCK_PIPE_CHILD)) \
                                ==  (SOCK_PIPE_PARENT | SOCK_PIPE_CHILD))

/* Determine reading filedes for pipe. */
#define SOCK_PIPE_FDREAD(s) (SOCK_PIPE_LOOP((s)->flags) ? (s)->pipe_fd0[0] : \
                             ((s)->flags & SOCK_PIPE_PARENT) ? (s)->pipe_fd1[0] : \
                             (s)->pipe_fd0[0])

/* Determine writing filedes for pipe. */
#define SOCK_PIPE_FDWRITE(s) (SOCK_PIPE_LOOP((s)->flags) ? (s)->pipe_fd0[1] : \
                              ((s)->flags & SOCK_PIPE_PARENT) ? (s)->pipe_fd0[1] : \
                              (s)->pipe_fd1[1])

/* Determine filedes for any socket type. */
#define SOCK_FDREAD(s)  ((s)->flags & SOCK_PIPE ? SOCK_PIPE_FDREAD(s) : (s)->handle)
#define SOCK_FDWRITE(s) ((s)->flags & SOCK_PIPE ? SOCK_PIPE_FDWRITE(s) : (s)->handle)

/* --- Socket status values --- */

#define SOCK_STAT_OK 0
#define SOCK_STAT_PARADOX -1       /* Error condition caused by caller, like calling
                                      sock_connect() on an already connected socket. */
#define SOCK_STAT_INTERRUPT -2     /* Interrupted by signal. */
#define SOCK_STAT_DISCONNECTED -3  /* Remote host closed connection. */
#define SOCK_STAT_CONNREFUSED -4   /* Connection refused by remote. */
#define SOCK_STAT_TIMEOUT -5       /* Transaction timed out. */
#define SOCK_STAT_NONAME -6        /* Name lookup failed. */
#define SOCK_STAT_OVERRUN -7       /* Buffer exhausted. */
#define SOCK_STAT_NOMEM -8         /* Failed allocation attempt. */
#define SOCK_STAT_MEMFAULT -9      /* Addressing error. Doesn't occur. */
#define SOCK_STAT_ERR -100         /* Unspecified error. */

/* --- Socket structures and typedefs --- */

struct crypto_1
{
  unsigned char xor_mask;
};

struct crypto_2  /* NOTE: Currently unused. Kept for union validation. */
{
  unsigned long pad;
};

struct crypto
{
  unsigned int level;
  union
  {
    struct crypto_1 c1;
    struct crypto_2 c2;
  } spec;
};


/* --- The new socket structure --- */

typedef struct socket_ext
{
  /* Handles/pointers */

  int handle;
  int pipe_fd0[2], pipe_fd1[2];
  struct socket_ext *parent;

  /* Addresses */

  struct sockaddr_in addr_remote;
  struct hostent *info_remote;
  char ip_remote[18];          /* Statically allocated cache */
  int port_bound;              /* Last-bound-to port */

  /* Buffers */

  FIFOBUF *fib_in, *fib_out;

  /* Status/misc */

  int timeout_sec;
  unsigned long flags;
  long status;
}
SOCK;

/* --- Function prototypes --- */

/* Internal */

int _sock_interrupted;
int _sock_initialized;

void _sock_interrupt(int signal);
int _sock_chkerrno(SOCK *s);
int _sock_feed(SOCK *s);
int _sock_spill(SOCK *s);

/* Misc interface */

SOCK *sock_new();
SOCK *sock_pipe_new();
SOCK *sock_new_with_port(int port);
SOCK *sock_accepting_new(int port);
SOCK *sock_new_from_handles(int in, int out);
SOCK *sock_new_from_files(FILE *in, FILE *out);
void sock_del(SOCK *s);

char *sock_status2str(int status);
char *sock_trans_primary_name(char *host_str);

#define sock_is_connecting(s) (((SOCK *) s)->flags & SOCK_CONNECTING)
#define sock_is_connected(s) (((SOCK *) s)->flags & SOCK_CONNECTED)
#define sock_is_accepting(s) (((SOCK *) s)->flags & SOCK_ACCEPT)
#define sock_is_active(s) (sock_is_connected(s) | sock_is_accepting(s))

#define sock_get_fd_write(s) SOCK_FDWRITE((SOCK *) s)
#define sock_get_fd_read(s) SOCK_FDREAD((SOCK *) s)

#define sock_buf_out_size(s) (((SOCK *) s)->fib_out->enqueued)
#define sock_buf_in_size(s) (((SOCK *) s)->fib_in->enqueued)

#define sock_get_max_write(s) (((SOCK *) s)->fib_out->buffers_max * \
                               ((SOCK *) s)->fib_out->bufsize - \
                               sock_buf_out_size(s))
#define sock_get_max_read(s) sock_buf_in_size(s)

/* Session */

SOCK *sock_open(int port_local, unsigned long flags);
void sock_close(SOCK *s, unsigned long flags);

SOCK *sock_pipe_from_handles(int in, int out);
SOCK *sock_pipe(unsigned long flags);
int sock_pipe_parent(SOCK *s);
int sock_pipe_child(SOCK *s);

int sock_connect(SOCK *s, char *desthost_str, int destport);
int sock_reconnect(SOCK *s);
int sock_disconnect(SOCK *s, unsigned long flags);
SOCK *sock_wait(unsigned long timeout_usec, unsigned int options, SOCK *s, ...);
SOCK *sock_wait_arr(unsigned long timeout_usec, unsigned int options, SOCK **s);

SOCK *sock_accept(SOCK *s);

int sock_poll(SOCK *s);

void sock_set_timeout(SOCK *s, int timeout_sec);
void sock_set_options(SOCK *s, unsigned long flags);

const char *sock_get_remote_name(SOCK *s);
const char *sock_get_remote_ip(SOCK *s);
const char *sock_get_remote_name_or_ip(SOCK *s);

/* Crypto */

int sock_set_encryption_in(SOCK *s, unsigned short level);
int sock_set_encryption_out(SOCK *s, unsigned short level);
void sock_encrypt(SOCK *s, unsigned char *data, unsigned long size);
void sock_decrypt(SOCK *s, unsigned char *data, unsigned long size);

/* Write */

int sock_flush(SOCK *s);
int sock_write(SOCK *s, void *data, unsigned int size);
int sock_putc(SOCK *s, char c);
int sock_puts(SOCK *s, char *str);
int sock_printf(SOCK *s, char *format, ...);

/* Read */

int sock_read(SOCK *s, void *dest, unsigned int size);
void *sock_dread(SOCK *s, unsigned int size);
int sock_gets(SOCK *s, char *dest, unsigned int size_max);
int sock_getc(SOCK *s);
char *sock_dgets(SOCK *s);
int sock_scanf(SOCK *s, char *format, ...);

#endif  /* _SOCK_H */
