/* Copyright (C) 1999 Hans Petter K. Jansson
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * You can contact the library's author by sending e-mail to <hpj@styx.net>.
 */

/* memory.c  -  memory allocation
 *  Copyright (C) 1998 Free Software Foundation, Inc.
 *
 * This file is part of Flux.
 *
 * Flux is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Flux is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

/* This file was lifted from the GNUPG source and slightly modified by
 * Hans Petter Jansson (hpj@styx.net) to work better with the rest of the
 * library and API.
 */

#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>

#include "types.h"
#include "log.h"
#include "mem.h"

#define DBG_MEMSTAT 0

#define MAGIC_NOR_BYTE 0x55
#define MAGIC_SEC_BYTE 0xcc
#define MAGIC_END_BYTE 0xaa

#if SIZEOF_UNSIGNED_LONG == 8
#  define EXTRA_ALIGN 4
#else
#  define EXTRA_ALIGN 0
#endif

const void mem_bug( const char *fmt, ... );

#define store_len(p,n,m) do { _mem_debug_add_entry(p,n,m, info, __FUNCTION__);  } while(0)

int _mem_opt_initialized = 0;

struct _mem_opt
{
  int debug;
  int abort;
  int log;
} _mem_opt;



long mem_used_guard;


#define INFO_BUCKETS 53
#define info_hash(p)  ( *(u32*)((p)) % INFO_BUCKETS )

struct mem_info_entry *mem_info_strings[INFO_BUCKETS]; /* hash table */

struct mem_table_entry *mem_table;  /* the table with the memory info */
unsigned mem_table_size;  /* number of allocated entries */
unsigned mem_table_len;  /* number of used entries */
struct mem_table_entry *mem_table_unused;/* to keep track of unused entries */

void mem_table_dump_on_exit(void);
void mem_table_dump(const char *prefix);
/* void mem_check_all( const char *info ); */

void _mem_free(void *a, const char *info, const unsigned int line);
void _mem_check(const void *a, const char *info);




const void mem_bug(const char *fmt, ...)
{
  va_list arg_ptr;

  va_start( arg_ptr, fmt ) ;
  log_vput_opt(LOG_DEBUG, 0, fmt, arg_ptr);
  va_end(arg_ptr);
  if (DBG_MEMSTAT) _mem_table_dump();
  if (_mem_opt.abort) abort();
}


void mem_stats_print(const char *prefix)
{
  unsigned n;
  struct mem_table_entry *e;
  ulong sum = 0, chunks =0;

  if (_mem_opt.debug)
  {
    for( e = mem_table, n = 0; n < mem_table_len; n++, e++ ) 
    {
      if(e->inuse) 
      {
        sum += e->user_n;
        chunks++;
      }
    }

    log_put_opt(LOG_DEBUG, 0, "(Mem) %s%stats: %8lu bytes in %ld chunks used.",
                prefix ? prefix : "", prefix ? " s" : "S", sum, chunks);
  }
}

void mem_table_dump(const char *prefix)
{
  if (_mem_opt.debug)
  {
    fprintf(stderr,"Memory-Table-Dump: %s\n", prefix);
    _mem_table_dump();
  }
  
  mem_stats_print(prefix);
}


void out_of_core(size_t n, int secure)
{
  log_put_opt(LOG_DEBUG, 0, "(Mem) Out of %smemory while allocating %u bytes.",
              secure ? "secure " : "", (unsigned int) n);
  
  if (_mem_opt.abort) abort();

  /* log_fatal("out of %s memory while allocating %u bytes\n",
            secure? "secure":"", (unsigned) n ); */
}

/****************
 * Allocate memory of size n.
 * This function gives up if we do not have enough memory
 */
void *_mem_alloc(size_t n, const char *info_text, const unsigned int line)
{
  char *p, *info;
  int t;
  
  if (!_mem_opt_initialized) mem_init(0, 20480, MEM_ABORT);

  if (_mem_opt.debug)
  {
    if (!(p = malloc(n + EXTRA_ALIGN + 5))) out_of_core(n,0);
  
    t = strlen(info_text);
    info = malloc(t + 10);
    sprintf(info, "%s->%d", info_text, line);
  
    store_len(p, n, 0);
  
    free(info);
  
    mem_used_guard += n;
    p[4 + EXTRA_ALIGN+n] = MAGIC_END_BYTE;
    return p + EXTRA_ALIGN + 4;
  }
  else
  {
    if (!(p = malloc(n))) out_of_core(n,0);
    return p;
  }
}

/****************
 * Allocate memory of size n from the secure memory pool.
 * This function gives up if we do not have enough memory
 */
void *_mem_alloc_secure(size_t n, const char *info_text, const unsigned int line)
{
  char *p, *info;
  int t;

  if (_mem_opt.debug)
  {
    if (!(p = mem_sec_alloc(n + EXTRA_ALIGN + 5))) out_of_core(n, 1);

    t = strlen(info_text);
    info = malloc(t + 10);
    sprintf(info, "%s->%d", info_text, line);
  
    store_len(p, n, 1);
  
    free(info);
    
    p[4 + EXTRA_ALIGN + n] = MAGIC_END_BYTE;
    return p+EXTRA_ALIGN+4;
  }
  else
  {
    if (!(p = mem_sec_alloc(n))) out_of_core(n, 1);
    return p;
  }
}

void *_mem_alloc_clear(size_t n, const char *info, const unsigned int line)
{
  void *p;
  p = _mem_alloc(n, info, line);
  memset(p, 0, n );
  return p;
}

void *_mem_alloc_secure_clear(size_t n, const char *info, const unsigned int line)
{
  void *p;
  p = _mem_alloc_secure(n, info, line);
  memset(p, 0, n );
  return p;
}


/****************
 * realloc and clear the old space
 */
void *_mem_realloc(void *a, size_t n, const char *info, const unsigned int line)
{
  unsigned char *p = a;
  void *b;
  size_t len;

  if (_mem_opt.debug)
  {
    len = mem_size(a);
    
    if( len >= n ) /* we don't shrink for now */
      return a;

    if (p[-1] == MAGIC_SEC_BYTE) b = _mem_alloc_secure_clear(n, info, line);
    else b = _mem_alloc_clear(n, info, line);
    _mem_check(NULL, info);
    memcpy(b, a, len );
    _mem_free(p, info, line);
  }
  else
  {
    if (m_is_secure(a))
    {
      if (!(b = mem_sec_realloc(a, n))) out_of_core(n,1);
    }
    else
    {
      if (!(b = realloc( a, n ))) out_of_core(n,0);
    }
  }

  return b;
}



/****************
 * Free a pointer
 */
void _mem_free(void *a, const char *info_text, const unsigned int line)
{
  byte *p = a;
  char *info;
  int t;
  
  if (!p)
  {
    if (_mem_opt.debug || _mem_opt.log)
      log_put_opt(LOG_DEBUG, 0, "[Mem] (%s->%d) tried to free null pointer.", info_text, line);
    if (_mem_opt.abort) abort();
    return;
  }

  if (_mem_opt.debug)
  {
    t = strlen(info_text);
    info = malloc(t + 10);
    sprintf(info, "%s->%d", info_text, line);
    
    mem_entry_free(p - EXTRA_ALIGN - 4, info);
    
    if (m_is_secure(p)) mem_sec_free(p - EXTRA_ALIGN - 4);
    else free(p);
    
    free(info);
  }
  else
  {
    if (m_is_secure(a)) mem_sec_free(p);
    else free(p);
  }
}


void _mem_check(const void *a, const char *info)
{
  const byte *p = a;

  if (!_mem_opt.debug)
  {
    log_put_opt(LOG_DEBUG, 0, "[Mem] Oops - _mem_check() called outside debug mode.");
    return;
  }
  
  if (p) mem_check(p-EXTRA_ALIGN-4, info);
  else mem_check_all(info);
}


size_t mem_size(const void *a)
{
  const byte *p = a;
  size_t n;
  struct mem_table_entry *e;

  if (!_mem_opt.debug)
  {
    log_put_opt(LOG_DEBUG, 0, "[Mem] Oops - mem_size() called outside debug mode.");
    return(0);
  }

  e = mem_check(p - EXTRA_ALIGN - 4, "mem_size");
  if (e) n = e->user_n;
  else n = 0;

  return n;
}


char *_mem_strdup(const char *a, const char *info, const unsigned int line)
{
  size_t n = strlen(a);
  char *p = _mem_alloc(n + 1, info, line);
  strcpy(p, a);
  return p;
}


void *_mem_alloc_opt(unsigned long size, unsigned long options,
                     const char *info_text, const unsigned int line)
{
  if (!_mem_opt_initialized) mem_init(0, 20480, MEM_ABORT);

  if (options & MEM_SECURE)
  {
    if (options & MEM_CLEAR)
      return(_mem_alloc_secure_clear(size, info_text, line));
    else
      return(_mem_alloc_secure(size, info_text, line));
  }
  else if (options & MEM_CLEAR)
    return(_mem_alloc_clear(size, info_text, line));

  return(_mem_alloc(size, info_text, line));
}


/* Options allowed: MEM_DEBUG | MEM_LOG | MEM_ABORT */

int mem_init(size_t shared_size, size_t secure_size, unsigned long options)
{  
  if (_mem_opt_initialized) return(0);

  _mem_opt_initialized = 1;
  memset(&_mem_opt, 0, sizeof(_mem_opt));
  
  if (options & MEM_DEBUG) _mem_opt.debug = 1;
  if (options & MEM_LOG) _mem_opt.log = 1;
  if (options & MEM_ABORT) _mem_opt.abort = 1;

  if (secure_size) mem_sec_init(secure_size);
  
  return(1);
}
