/* Copyright (C) 1999 Hans Petter K. Jansson
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * You can contact the library's author by sending e-mail to <hpj@styx.net>.
 */

#include "config.h"
#include "flux.h"

#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/time.h>
#ifdef TIME_WITH_SYS_TIME
# include <time.h>
#endif


/* ---------------- *
 * Proxy allocation *
 * ---------------- */


PROXY *proxy_new()
{
  PROXY *p;
  
  p = malloc(sizeof(*p));
  memset(p, 0, sizeof(*p));

  p->socks = tt_new_with_data("sock", 4);
  p->comms = tt_new_with_data("comm", 4);
  p->timers = tt_new_with_data("time", 4);
  p->signals = tt_new_with_data("sig", 3);
  p->changed = 1;
  return(p);
}


void proxy_del(PROXY *p)
{
  tt_del(p->socks);
  tt_del(p->comms);
  tt_del(p->timers);
  tt_del(p->signals);
  free(p);
}


/* ------------- *
 * Proxy running *
 * ------------- */


void proxy_signal_catch(int sig)
{
  /* WORKING ON THIS
  
  TT_FOR_EACH(

  */
}


void proxy_loop(PROXY *p)
{
  TT *tt0, *tt1;
  int fd, fd_num;
  struct timeval tv0, *tv1, tv2, tv4;
  fd_set fd_read, fd_write;
  SOCK *s, *s_child;
  COMM *c, *c_child;
  u16 trans;

  p->exiting = 0;
  
  while (!p->exiting)
  {
    /* --- Check comm queues --- */

    TT_FOR_EACH(p->comms, tt0)
    {
      c = (COMM *) tt_data_get_ptr(tt0);
       while (comm_queue_in_len(c))
      {
        tt1 = comm_dequeue(c, &trans);
        proxy_call_comm(p, c, tt1, trans, 1);
        tt_del(tt1);
      }
    }

    /* --- Update filedes list if needed --- */

    if (p->changed)
    {
      FD_ZERO(&(p->fd_read));
      FD_ZERO(&(p->fd_write));
      p->fd_max = -1;

      /* --- Find all sockets --- */

      TT_FOR_EACH(p->socks, tt0)
      {
        if (sock_is_active(tt_data_get_ptr(tt0)))
        {
          fd = sock_get_fd_read(tt_data_get_ptr(tt0));
          FD_SET(fd, &(p->fd_read));
          if (fd > p->fd_max) p->fd_max = fd;

          if (sock_buf_out_size(tt_data_get_ptr(tt0)) ||
              sock_is_connecting(tt_data_get_ptr(tt0)))
          {
            fd = sock_get_fd_write(tt_data_get_ptr(tt0));
            FD_SET(fd, &(p->fd_write));
            if (fd > p->fd_max) p->fd_max = fd;
          }
        }
      }

      /* --- Find all comms --- */

      TT_FOR_EACH(p->comms, tt0)
      {
        if (comm_is_active(tt_data_get_ptr(tt0)))
        {
          fd = comm_get_fd_read(tt_data_get_ptr(tt0));
          if (fd > p->fd_max) p->fd_max = fd;
          FD_SET(fd, &(p->fd_read));

          if (comm_buf_out_size(tt_data_get_ptr(tt0)) ||
              comm_is_connecting(tt_data_get_ptr(tt0)))
          {
            fd = comm_get_fd_write(tt_data_get_ptr(tt0));
            FD_SET(fd, &(p->fd_write));
            if (fd > p->fd_max) p->fd_max = fd;
          }
        }
      }

      p->fd_max += 1;
      p->changed = 0;
    }

    /* --- Find closest timeout --- */

    gettimeofday(&tv0, 0);
    tv4.tv_sec = 0x7ffffffe;

    TT_FOR_EACH(p->timers, tt0)
    {
      tt1 = tt_get_first_child(tt_find_first_child(tt0, "last", 4));
      tv1 = (struct timeval *) tt_data_get(tt1);
      memcpy(&tv2, tv1, sizeof(tv2));

      tt1 = tt_get_first_child(tt_find_first_child(tt0, "interval", 8));
      tv1 = (struct timeval *) tt_data_get(tt1);
      timeradd(&tv2, tv1, &tv2);

      /* Now: tv2 is absolute time, at which to trigger */

      timersub(&tv2, &tv0, &tv2);

      /* Now: tv2 is time from now, at which to trigger */

      if (timercmp(&tv2, &tv4, <)) memcpy(&tv4, &tv2, sizeof(tv4));

      /* Now: tv4 is closest timeout found */
    }

    /* --- Wait for something to happen --- */

    memcpy(&fd_read, &(p->fd_read), sizeof(fd_read));
    memcpy(&fd_write, &(p->fd_write), sizeof(fd_write));
    fd_num = select(p->fd_max, &fd_read, &fd_write, 0, &tv4);

    /* --- Check comms and sockets --- */

    if (fd_num > 0)
    {
      /* --- Check comms --- */

      TT_FOR_EACH(p->comms, tt0)
      {
        c = (COMM *) tt_data_get_ptr(tt0);

        /* Write */

        if (FD_ISSET(comm_get_fd_write(c), &fd_write))
        {
          if (comm_is_connected(c)) comm_push(c);
          else  /* if (comm_is_connecting(c)) */
          {
            /* Connection attempt resolved */
            c_child = comm_accept(c);

            /* FIXME: if (c_child && !connect_handler) comm_del(c_child) */
            if (c_child && (tt1 = tt_find_first_child(tt0, "connect", 7)))
              proxy_call_comm_connect(c, c_child, tt1);
          }
        }

        /* Read */

        if (FD_ISSET(comm_get_fd_read(c), &fd_read))
        {
          if (comm_is_connected(c))
          {
            if (_sock_feed(comm_get_sock(c)) < 1)
            {
              /* Disconnected (EOF) */
              if ((tt1 = tt_find_first_child(tt0, "close", 5)))
                proxy_call_comm_close(c, tt1);
              else
              {
                proxy_del_comm(p, c);
                comm_del(c);
              }
            }
            else if (tt_find_first_child(tt0, "incomplete", 10))
            {
              while ((tt1 = comm_pull_one(c)))
              {
                proxy_call_comm(p, c, tt1, comm_get_trans_in(c), 0);
                if (p->changed && !tt_has_child(p->comms, tt0)) break;
              }
            }
            else comm_pull_many(c);
          }
          else /* if (comm_is_accepting(s)) */
          {
            /* Incoming connection (probably) */
            c_child = comm_accept(c);

            /* FIXME: if (c_child && !connect_handler) comm_del(c_child) */
            if (c_child && (tt1 = tt_find_first_child(tt0, "connect", 7)))
              proxy_call_comm_connect(c, c_child, tt1);
          }
        }
        
        if (p->changed) break;
      }

      /* --- Check sockets --- */

      TT_FOR_EACH(p->socks, tt0)
      {
        s = (SOCK *) tt_data_get_ptr(tt0);

        /* Write */

        if (FD_ISSET(sock_get_fd_write(s), &fd_write))
        {
          if (sock_is_connected(s))
          {
            /* Outgoing data (hungry socket) */
            _sock_spill(s);
            if ((tt1 = tt_find_first_child(tt0, "write", 5)))
              proxy_call_sock(s, tt1);
          }
          else /* if (sock_is_connecting(s)) */
          {
            /* Connection attempt resolved */
            s_child = sock_accept(s);

            if (s_child && (tt1 = tt_find_first_child(tt0, "connect", 7)))
              proxy_call_sock_connect(s, s_child, tt1);
          }
        }

        if (p->changed) break;

        /* Read */

        if (FD_ISSET(sock_get_fd_read(s), &fd_read))
        {
          if (sock_is_connected(s))
          {
            if (_sock_feed(s) < 1)
            {
              /* Disconnected (EOF) */
              if ((tt1 = tt_find_first_child(tt0, "close", 5)))
                proxy_call_sock(s, tt1);
              else
                if (!sock_disconnect(s, 0)) proxy_del_sock(p, s);
            }
            else if ((tt1 = tt_find_first_child(tt0, "read", 4)))
            {
              /* Incoming data */
              proxy_call_sock(s, tt1);
            }
          }
          else /* if (sock_is_accepting(s)) */
          {
            /* Incoming connection (probably) */
            s_child = sock_accept(s);

            if (s_child && (tt1 = tt_find_first_child(tt0, "connect", 7)))
              proxy_call_sock_connect(s, s_child, tt1);
          }
        }
        
        if (p->changed) break;
      }
    }  /* (fd_num > 0) */

    /* --- Check timeouts --- */
    
    gettimeofday(&tv0, 0);
    tv4.tv_sec = 0x7ffffffe;

    TT_FOR_EACH(p->timers, tt0)
    {
      tt1 = tt_get_first_child(tt_find_first_child(tt0, "last", 4));
      tv1 = (struct timeval *) tt_data_get(tt1);
      memcpy(&tv2, tv1, sizeof(tv2));

      tt1 = tt_get_first_child(tt_find_first_child(tt0, "interval", 8));
      tv1 = (struct timeval *) tt_data_get(tt1);
      timeradd(&tv2, tv1, &tv2);

      /* Now: tv2 is absolute time, at which to trigger */
      
      if (timercmp(&tv2, &tv0, <)) proxy_call_timer(p, tt0);
    }
  }
}


void proxy_break(PROXY *p)
{
  p->exiting = 1;
}


/* ------ *
 * Adding *
 * ------ */


void proxy_add_sock_read(PROXY *p, SOCK *s, void *priv,
                         int (*read)(SOCK *, void *))
{
  TT *tt0, *tt1;
  
  tt0 = tt_find_first_child(p->socks, &s, sizeof(s));
  if (!tt0) tt0 = tt_new_with_parent_and_data(p->socks, &s, sizeof(s));

  tt1 = tt_find_first_child(tt0, "read", 4);
  if (tt1) tt_del(tt1);
  tt1 = tt_new_with_parent_and_data(tt0, "read", 4);
  tt1 = tt_new_with_parent_and_data(tt1, &read, sizeof(read));
  tt_new_with_parent_and_data(tt1, &priv, sizeof(priv));
}


void proxy_add_sock_write(PROXY *p, SOCK *s, void *priv,
                          int (*write)(SOCK *, void *))
{
  TT *tt0, *tt1;
  
  tt0 = tt_find_first_child(p->socks, &s, sizeof(s));
  if (!tt0) tt0 = tt_new_with_parent_and_data(p->socks, &s, sizeof(s));

  tt1 = tt_find_first_child(tt0, "write", 5);
  if (tt1) tt_del(tt1);
  tt1 = tt_new_with_parent_and_data(tt0, "write", 5);
  tt1 = tt_new_with_parent_and_data(tt1, &write, sizeof(write));
  tt_new_with_parent_and_data(tt1, &priv, sizeof(priv));
}


void proxy_add_sock_connect(PROXY *p, SOCK *s, void *priv,
                            int (*connect)(SOCK *, void *))
{
  TT *tt0, *tt1;
  
  tt0 = tt_find_first_child(p->socks, &s, sizeof(s));
  if (!tt0) tt0 = tt_new_with_parent_and_data(p->socks, &s, sizeof(s));

  tt1 = tt_find_first_child(tt0, "connect", 7);
  if (tt1) tt_del(tt1);
  tt1 = tt_new_with_parent_and_data(tt0, "connect", 7);
  tt1 = tt_new_with_parent_and_data(tt1, &connect, sizeof(connect));
  tt_new_with_parent_and_data(tt1, &priv, sizeof(priv));
}


void proxy_add_sock_close(PROXY *p, SOCK *s, void *priv,
                          int (*close)(SOCK *, void *))
{
  TT *tt0, *tt1;
  
  tt0 = tt_find_first_child(p->socks, &s, sizeof(s));
  if (!tt0) tt0 = tt_new_with_parent_and_data(p->socks, &s, sizeof(s));

  tt1 = tt_find_first_child(tt0, "close", 5);
  if (tt1) tt_del(tt1);
  tt1 = tt_new_with_parent_and_data(tt0, "close", 5);
  tt1 = tt_new_with_parent_and_data(tt1, &close, sizeof(close));
  tt_new_with_parent_and_data(tt1, &priv, sizeof(priv));
}


void proxy_add_sock(PROXY *p, SOCK *s, void *priv,
                    int (*read)(SOCK *, void *),
                    int (*write)(SOCK *, void *),
                    int (*connect)(SOCK *, SOCK *, void *),
                    int (*close)(SOCK *, void *))
{
  TT *tt0, *tt1;

  p->changed = 1;

  tt0 = tt_find_first_child(p->socks, &s, sizeof(s));
  if (!tt0) tt0 = tt_new_with_parent_and_data(p->socks, &s, sizeof(s));

  if (read)
  {
    tt1 = tt_find_first_child(tt0, "read", 4);
    if (tt1) tt_del(tt1);
    tt1 = tt_new_with_parent_and_data(tt0, "read", 4);
    tt1 = tt_new_with_parent_and_data(tt1, &read, sizeof(read));
    tt_new_with_parent_and_data(tt1, &priv, sizeof(priv));
  }

  if (write)
  {
    tt1 = tt_find_first_child(tt0, "write", 5);
    if (tt1) tt_del(tt1);
    tt1 = tt_new_with_parent_and_data(tt0, "write", 5);
    tt1 = tt_new_with_parent_and_data(tt1, &write, sizeof(write));
    tt_new_with_parent_and_data(tt1, &priv, sizeof(priv));
  }

  if (connect)
  {
    tt1 = tt_find_first_child(tt0, "connect", 7);
    if (tt1) tt_del(tt1);
    tt1 = tt_new_with_parent_and_data(tt0, "connect", 7);
    tt1 = tt_new_with_parent_and_data(tt1, &connect, sizeof(connect));
    tt_new_with_parent_and_data(tt1, &priv, sizeof(priv));
  }

  if (close)
  {
    tt1 = tt_find_first_child(tt0, "close", 5);
    if (tt1) tt_del(tt1);
    tt1 = tt_new_with_parent_and_data(tt0, "close", 5);
    tt1 = tt_new_with_parent_and_data(tt1, &close, sizeof(close));
    tt_new_with_parent_and_data(tt1, &priv, sizeof(priv));
  }
}


/* <comm> -> <completeness> -> root   -> <root>
 *                          -> trans  -> <trans>
 *                          -> recv   -> <func>  -> <priv>
 *        -> <completeness> -> ...
 *        -> connect        -> <func> -> <priv>
 *        -> close          -> <func> -> <priv>
 *        -> ...                                 */

void proxy_add_comm(PROXY *p, COMM *c, char *root, ushort trans, int incomplete,
                    void *priv, int (*recv)(COMM *, TT *, ushort, int, void *))
{
  TT *tt0, *tt1, *tt2;
  
  p->changed = 1;

  tt0 = tt_find_first_child(p->comms, &c, sizeof(c));
  if (!tt0) tt0 = tt_new_with_parent_and_data(p->comms, &c, sizeof(c));

  tt1 = tt_new();
  if (incomplete) tt_data_set_str(tt1, "incomplete");
  tt_add_as_first_child(tt0, tt1);
  
  if (root)
  {
    tt2 = tt_new_with_parent_and_data(tt1, "root", 4);
    tt_new_with_parent_and_data(tt2, root, strlen(root));
  }

  if (trans)
  {
    tt2 = tt_new_with_parent_and_data(tt1, "trans", 5);
    tt_new_with_parent_and_data(tt2, &trans, sizeof(trans));
  }

  tt2 = tt_new_with_parent_and_data(tt1, "recv", 4);
  tt2 = tt_new_with_parent_and_data(tt2, &recv, sizeof(recv));
  tt_new_with_parent_and_data(tt2, &priv, sizeof(priv));
}


void proxy_add_comm_connect(PROXY *p, COMM *c, void *priv,
                            int (*connect)(COMM *, COMM *, void *))
{
  TT *tt0, *tt1;
  
  p->changed = 1;

  tt0 = tt_find_first_child(p->comms, &c, sizeof(c));
  if (!tt0) tt0 = tt_new_with_parent_and_data(p->comms, &c, sizeof(c));

  tt1 = tt_find_first_child(tt0, "connect", 7);
  if (tt1) tt_del(tt1);
  tt1 = tt_new_with_parent_and_data(tt0, "connect", 7);

  tt1 = tt_new_with_parent_and_data(tt1, &connect, sizeof(connect));
  tt_new_with_parent_and_data(tt1, &priv, sizeof(priv));
}


void proxy_add_comm_close(PROXY *p, COMM *c, void *priv,
                          int (*close)(COMM *, void *))
{
  TT *tt0, *tt1;
  
  p->changed = 1;

  tt0 = tt_find_first_child(p->comms, &c, sizeof(c));
  if (!tt0) tt0 = tt_new_with_parent_and_data(p->comms, &c, sizeof(c));

  tt1 = tt_find_first_child(tt0, "close", 5);
  if (tt1) tt_del(tt1);
  tt1 = tt_new_with_parent_and_data(tt0, "close", 5);
  
  tt1 = tt_new_with_parent_and_data(tt1, &close, sizeof(close));
  tt_new_with_parent_and_data(tt1, &priv, sizeof(priv));
}


void proxy_add_timer(PROXY *p, char *name, struct timeval *interval,
                     int recurrent, void *priv, int (*timeout)(char *, void *))
{
  TT *tt0, *tt1;
  struct timeval tv;

  /* Don't set changed flag, as this only affects sockets */

  gettimeofday(&tv, 0);

  tt0 = tt_find_first_child(p->timers, name, strlen(name));
  if (tt0) tt_del(tt0);

  tt0 = tt_new_with_parent_and_data(p->timers, name, strlen(name));

  tt1 = tt_new_with_parent_and_data(tt0, "priv", 4);
  tt_new_with_parent_and_data(tt1, &priv, sizeof(priv));

  tt1 = tt_new_with_parent_and_data(tt0, "interval", 8);
  tt_new_with_parent_and_data(tt1, interval, sizeof(*interval));

  if (recurrent) tt_new_with_parent_and_data(tt0, "recurrent", 9);

  tt1 = tt_new_with_parent_and_data(tt0, "last", 4);
  tt_new_with_parent_and_data(tt1, &tv, sizeof(tv));
  
  tt1 = tt_new_with_parent_and_data(tt0, "timeout", 7);
  tt_new_with_parent_and_data(tt1, &timeout, sizeof(timeout));
}


/* -------- *
 * Removing *
 * -------- */


void proxy_del_sock(PROXY *p, SOCK *s)
{
  TT *tt0;
  
  tt0 = tt_find_first_child(p->socks, &s, sizeof(s));
  if (tt0)
  {
    p->changed = 1;
    tt_del(tt0);
  }
}


void proxy_del_comm(PROXY *p, COMM *c)
{
  TT *tt0;

  if ((tt0 = tt_find_first_child(p->comms, &c, sizeof(c))))
  {
    /* Should only need doing once */
    p->changed = 1;
    tt_del(tt0);
  }
}


void proxy_del_comm_trans(PROXY *p, COMM *c, u16 trans)
{
  TT *tt0, *tt1, *tt2;
  
  tt0 = tt_find_first_child(p->comms, &c, sizeof(c));
  if (!tt0) return;

  tt2 = 0;

  TT_FOR_EACH(tt0, tt1)
  {
    if (tt2) tt_del(tt2);  /* Delete after we followed ref. to next */

    tt2 = tt_find_first_child(tt1, "trans", 5);
    if (!tt2) continue;
    
    if (tt_find_first_child(tt2, &trans, sizeof(trans))) tt2 = tt1;
    else tt2 = 0;
  }
}


void proxy_del_comm_block(PROXY *p, COMM *c, char *root)
{
  TT *tt0, *tt1, *tt2;
  int len;
  
  tt0 = tt_find_first_child(p->comms, &c, sizeof(c));
  if (!tt0) return;

  len = strlen(root);
  tt2 = 0;

  TT_FOR_EACH(tt0, tt1)
  {
    if (tt2) tt_del(tt2);  /* Delete after we followed ref. to next */

    tt2 = tt_find_first_child(tt1, "root", 4);
    if (!tt2) continue;
    
    if (tt_find_first_child(tt2, root, len)) tt2 = tt1;
    else tt2 = 0;
  }
}


void proxy_del_timer(PROXY *p, char *name)
{
  TT *tt0;
  
  /* Don't set changed flag, as this only affects sockets */

  tt0 = tt_find_first_child(p->timers, name, strlen(name));
  if (tt0) tt_del(tt0);
}


/* ------------------ *
 * Special interfaces *
 * ------------------ */


void proxy_reset_timer(PROXY *p, char *name)
{
  TT *tt0;
  struct timeval tv;
  
  tt0 = tt_find_first_child(p->timers, name, strlen(name));
  if (!tt0) return;
  
  gettimeofday(&tv, 0);

  tt0 = tt_find_first_child(tt0, "last", 4);
  tt_del(tt_get_first_child(tt0));
  tt_new_with_parent_and_data(tt0, &tv, sizeof(tv));
}


/* ------- *
 * Calling *
 * ------- */


void proxy_call_sock(SOCK *s, TT *tt)
{
  int (*func_sock)(SOCK *, void *);
  void *priv;

  tt = tt_get_first_child(tt);
  func_sock = tt_data_get_ptr(tt);
  tt = tt_get_first_child(tt);
  if (tt) priv = tt_data_get_ptr(tt);
  else priv = 0;

  func_sock(s, priv);
}


void proxy_call_sock_connect(SOCK *s, SOCK *s_child, TT *tt)
{
  int (*func_sock)(SOCK *, SOCK *, void *);
  void *priv;

  tt = tt_get_first_child(tt);
  func_sock = tt_data_get_ptr(tt);
  tt = tt_get_first_child(tt);
  if (tt) priv = tt_data_get_ptr(tt);
  else priv = 0;

  func_sock(s, s_child, priv);
}


void proxy_call_comm(PROXY *p, COMM *c, TT *node, ushort trans, int complete)
{
  TT *root;
  TT *this_root, *this_trans, *this_priv;
  TT *tt0, *tt1, *tt2, *tt3;
  int (*pcc_recv)(COMM *, TT *, ushort, int, void *);

  root = tt_get_root(node);

  tt0 = tt_find_first_child(p->comms, &c, sizeof(c));
  if (!tt0) return;

  TT_FOR_EACH(tt0, tt1) tt_set_ready(tt1, 0);

  for (;;)
  {
    tt3 = 0;  /* Makes compiler happy and keeps us safe */
    for (tt1 = tt_get_first_child(tt0); tt1; tt1 = tt3)
    {
      tt3 = tt_get_next(tt1);  /* Prepare for next iteration */
      if (tt_is_ready(tt1)) continue;
      tt_set_ready(tt1, 1);

      if ((!complete && tt_memcmp(tt1, "incomplete", 10)) ||
          !tt_memcmp(tt1, "close", 5) ||
          !tt_memcmp(tt1, "connect", 7)) continue;

      this_root = tt_find_first_child(tt1, "root", 4);
      this_trans = tt_find_first_child(tt1, "trans", 5);
      tt2 = tt_get_first_child(tt_find_first_child(tt1, "recv", 4));
      pcc_recv = tt_data_get_ptr(tt2);
      if (tt_get_first_child(tt2)) this_priv = tt_data_get_ptr(tt_get_first_child(tt2));
      else this_priv = 0;
    
      if (this_root)
      {
        if (!tt_cmp(tt_get_first_child(this_root), root))
        {
          if (this_trans)
          {
            if (!tt_memcmp(tt_get_first_child(this_trans), &trans, sizeof(trans)))
              /* Matching root, matching trans */
              if (pcc_recv(c, node, trans, complete, this_priv) == TRUE) break;
          }
          else
            /* Matching root, no trans */
            if (pcc_recv(c, node, trans, complete, this_priv) == TRUE) break;
        }
      }
      else
      {
        if (this_trans)
        {
          if (!tt_memcmp(tt_get_first_child(this_trans), &trans, sizeof(trans)))
            /* No root, matching trans */
            if (pcc_recv(c, node, trans, complete, this_priv) == TRUE) break;
        }
        else
          /* No root, no trans */
          if (pcc_recv(c, node, trans, complete, this_priv) == TRUE) break;
      }
    
      if (p->changed)
      {
        if (!tt_has_child(p->comms, tt0)) return;  /* We were completely removed */
        if (!tt_has_child(tt0, tt3)) break;  /* Next handler was removed */
      }
    }
    
    if (!tt3) break;  /* Finished */

    /* Some change forces us to rescan handlers, loop will run again */
  }
  
}


void proxy_call_comm_connect(COMM *c, COMM *c_child, TT *tt)
{
  int (*func_comm)(COMM *, COMM *, void *);
  void *priv;

  tt = tt_get_first_child(tt);
  func_comm = tt_data_get_ptr(tt);
  tt = tt_get_first_child(tt);
  if (tt) priv = tt_data_get_ptr(tt);
  else priv = 0;

  func_comm(c, c_child, priv);
}


void proxy_call_comm_close(COMM *c, TT *tt)
{
  int (*func_comm)(COMM *, void *);
  void *priv;

  tt = tt_get_first_child(tt);
  func_comm = tt_data_get_ptr(tt);
  tt = tt_get_first_child(tt);
  if (tt) priv = tt_data_get_ptr(tt);
  else priv = 0;

  func_comm(c, priv);
}


void proxy_call_timer(PROXY *p, TT *timer)
{
  TT *tt0;
  int (*timeout)(char *, void *);
  void *priv;
  char *timer_name_str;
  struct timeval tv;

  tt0 = tt_get_first_child(tt_find_first_child(timer, "timeout", 7));
  if (!tt0) return;  /* Would be an internal error */
  tt_data_get_bytes(tt0, &timeout, 0, sizeof(timeout));

  tt0 = tt_get_first_child(tt_find_first_child(timer, "priv", 4));
  tt_data_get_bytes(tt0, &priv, 0, sizeof(priv));

  gettimeofday(&tv, 0);
  tt0 = tt_find_first_child(timer, "last", 4);
  tt_del(tt_get_first_child(tt0));
  tt_new_with_parent_and_data(tt0, &tv, sizeof(tv));

  timer_name_str = tt_data_get_str(timer);
  if (!tt_find_first_child(timer, "recurrent", 9)) tt_del(timer);
  
  timeout(timer_name_str, priv);
  free(timer_name_str);
}
